# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import re

import common
from com.huawei.ism.exception import IsmException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ITEM_ID = "getElectonicTag"

NODE_IP = "节点："
SERVER_TYPE = "server_type="
TARGET_VERSION = "目标版本："
DEFAULT_SERVER = "default"


# 各产品对应的电子标签全集
SERVER_MAP = {
    '8.0.1': ("TaiShan 2280 V2", "200-2280", "TaiShan 200 (Model 2280)", "T22R-02"),
    '8.0.3': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor 100D C180"),
    '8.1.0': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280" "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550"),
    '8.1.1': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550"),
    '8.1.2': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550" "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520",
              "DPC30101", "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120"),
    '8.1.3': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550", "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520", "DPC30101",
              "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120", "OceanStor Pacific 9640",
              "OceanStor Pacific 9546", "OceanStor Pacific 9340", "OceanStor Pacific 9346", "OceanStor Pacific 9350"),
    '8.1.RC6': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550", "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520", "DPC30101",
              "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120", "OceanStor Pacific 9640",
              "OceanStor Pacific 9546", "OceanStor Pacific 9340", "OceanStor Pacific 9346", "OceanStor Pacific 9350"),
    '8.1.5': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550", "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520",
              "DPC30101", "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120",
              "OceanStor Pacific 9640",
              "OceanStor Pacific 9546", "OceanStor Pacific 9340", "OceanStor Pacific 9346", "OceanStor Pacific 9350"),
    '8.1.RC7': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100",
              "200-2280", "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550", "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520",
              "DPC30101", "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120",
              "OceanStor Pacific 9640",
              "OceanStor Pacific 9546", "OceanStor Pacific 9340", "OceanStor Pacific 9346", "OceanStor Pacific 9350"),
    '8.2.0': ("TaiShan 2280 V2", "OceanStor 100D P100", "OceanStor 100D-K P100", "OceanStor 100D-A P100", "200-2280",
              "TaiShan 200 (Model 2280)", "OceanStor Pacific 9520", "OceanStor Pacific 9920",
              "OceanStor Pacific 17510", "OceanStor Pacific 18510", "OceanStor 100D C180", "OceanStor Pacific 9950",
              "OceanStor Pacific 9550", "DPE30101", "QinDing100-2U12", "QinDing100s-2U12", "RS9520",
              "DPC30101", "HDPE30101", "DPE90000", "QinDing200-5U80", "DPE80000", "QinDing200-5U120",
              "OceanStor Pacific 9640",
              "OceanStor Pacific 9546", "OceanStor Pacific 9340", "OceanStor Pacific 9346", "OceanStor Pacific 9350"),
}


def check_product_name(product_version, product_name):
    """
    判断环境电子标签在版本白名单中所对应的机型
    """
    if product_name in SERVER_MAP.get(product_version, DEFAULT_SERVER):
        #通用服务器2280机型
        return 2280
    else:
        #通用服务器5280机型
        return 5280


def check_service(old_product_version, new_product_version):
    """
    检查升级前后电子标签对应的机型是否一致
    """
    node_dictionary = {}
    err_node_list = []
    node_ip = ""

    # 获取预检查脚本结果
    info = (py_java_env.get("sourceInfo")).splitlines()
    for info_item in info:
        LOGGER.logInfo("[getElectonicTag] info_item:%s", info_item)
        if NODE_IP in info_item:
            node_ip = info_item.split(NODE_IP)[-1].strip()
            continue
        
        if SERVER_TYPE in info_item:
            node_dictionary[node_ip] = info_item.split(SERVER_TYPE)[-1].strip()


    for node in node_dictionary.keys():
        LOGGER.logInfo("[getElectonicTag] node: %s", node)
        # 版本不存在，检查通过
        if not old_product_version or not new_product_version:
            continue
        
        old_server = check_product_name(old_product_version, node_dictionary[node])
        new_server = check_product_name(new_product_version, node_dictionary[node])
        if old_server != new_server:
            LOGGER.logInfo("NodeIP:%s E-tag upgrade risks:%s", (str(node), str(node_dictionary[node])))
            err_node_list.append(node)
    str_node_ip = str(err_node_list).replace('u\'', '\'')
    LOGGER.logInfo("[getElectonicTag] str_node_ip: %s", str_node_ip)
    if not err_node_list:
        return common.INSPECT_PASS, common.get_err_msg(LANG, "pool.storage.tag.check.pass",
                                                    (new_product_version)), ""  # 升级前后槽位号不会变更，无风险
    else:
        return common.INSPECT_UNNORMAL, common.get_err_msg(LANG, "pool.storage.tag.check.unnormal",
                                                        (new_product_version, str_node_ip)), common.get_err_msg(
            LANG, "pool.storage.tag.check", (
                str_node_ip, old_product_version, new_product_version))  # 升级前后槽位号会变更，需提前做适配


def check_major_release(full_version):
    # 只匹配大版本
    for major_version in SERVER_MAP.keys():
        if re.search(major_version, full_version):
            return major_version
    return ""


def execute(rest):
    """
    检查升级前后版本电子标签不一致风险
    如果不在风险版本范围内，则检查通过，否则继续执行
    """
    dev_node = py_java_env.get("devInfo")
    try:
        LOGGER.logInfo("start getElectonicTag")
        # 查询当前产品版本
        old_product_version = check_major_release(dev_node.getProductVersion())
        # 查询升级目标版本
        new_version = py_java_env.get("version").split(TARGET_VERSION)[-1].split("）")[0].strip()
        new_product_version = check_major_release(new_version)
        # 查询电子标签升级是否存在风险
        return check_service(old_product_version, new_product_version)
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        return (
            common.INSPECT_UNNORMAL,
            "",
            common.get_err_msg(LANG, "query.result.abnormal"),
        )
