# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.

"""
@version: SmartKit 22.0.0
@time: 2022/04/24
@file: check_pool_num.py
@function: 检查项脚本check_item_hardware_storage_node_num_*.py的关于池的相关方法封装
@modify:
"""

import math

# 卡换算率
RATE_CARD = 1000
# 盘换算率
RATE_DISK = 1024
# 单节点最小磁盘数
MIN_DISK_NUM = 4
# 池中最小节点数
MIN_NODE_NUM = 3
# 单节点最小缓存比/池中最大节点数
MIN_CATCHE_FOR_STORAGE = 40
# 扩容后节点与扩容前节点数最大比值
MAX_ALL_NODE_THAN_ORIGIN_NODE = 6
# 建议池内最大节点数
SUGGESTION_MAX_NODE_NUM = 40


def get_disk_type_enum():
    from com.huawei.ism.tool.obase.entity import DiskTypeEnum
    return DiskTypeEnum


def get_dev_node_class():
    from com.huawei.ism.tool.obase.entity import DevNode
    return DevNode


def get_common():
    import common
    return common


class CheckPoolNumUtil:
    def __init__(self, pool, msg_head, lang, logger, dev_node=None):
        self.msg_head = msg_head
        self.lang = lang
        self.logger = logger
        self.pool = pool
        self.pool_id = pool.getId()
        self.is_create_pool = pool.isExpansionCreate()
        self.main_type = pool.getMainStorageDiskType().name()
        self.cache_type = self.get_pool_cache_type()
        self.origin_nodes = list(pool.getJoinedClusterNode())
        self.expansion_nodes = list(self.pool.getExpansionClusterNode()) + list(
            self.pool.getExpansionNodeList())
        self.all_nodes = self.origin_nodes + self.expansion_nodes
        self._dev_node = dev_node

    def get_pool_cache_type(self):
        return self.pool.getCacheDiskType().name() if is_involve_cache(
            self.main_type) else None

    def check_pool_node_num_info(self):
        """检查项：检查扩容存储节点数，检查内容的实现封装

        检查内容基于单个池
        基于错误消息分步封装了检查标准
        :return: (errors, warnings) 错误类消息和建议类消息
        """
        errors = []
        warnings = []
        # 1、扩容的单个硬盘池中，扩容存储节点数不超过已有节点数的5倍
        if self._is_expansion_over_origin():
            errors.append(self._get_err_msg(
                'hardware.storage.node.num.five', self.msg_head))

        # 2、新建池时，存储节点不少于3个
        # 新增cube设备扩容评估，cube场景最小节点数为2
        min_node_num = MIN_NODE_NUM
        if self._dev_node and self._dev_node.isFusionCube():
            min_node_num = 2
        self.logger.logInfo("node :{} min_node_num :{}".format(self._dev_node, min_node_num))
        if self.is_create_pool and len(self.all_nodes) < min_node_num:
            errors.append(self._get_err_msg(
                'hardware.storage.node.num.new.min', (self.msg_head, min_node_num,
                                                      len(self.all_nodes))))
        # 3、节点扩了缓存，则扩主存数不能为0
        zero_exp_main_ips = self._get_zero_exp_main_ips()
        for ip in zero_exp_main_ips:
            errors.append(self._get_err_msg(
                "hardware.pool.cache.storage.only", ip))

        # 4、单个节点主存盘不少于4块
        little_num_desc = self._get_little_main_num_desc()
        if little_num_desc:
            errors.append(self._get_err_msg(
                "hardware.storage.node.disk.num.total",
                (self.msg_head, ",".join(little_num_desc))))

        # 5、新建池时，单个主存盘不少于40GB的缓存盘
        if self.is_create_pool:
            size_desc = self._get_assign_little_cache_size_desc()
            if size_desc:
                errors.append(self._get_err_msg(
                    "hardware.storage.node.disk.average",
                    (self.msg_head, ",".join(size_desc))))

        # 6、扩容池时，新扩节点缓存盘总容量大于或等于当前池内缓存盘总容量最小的节点的缓存盘总容量
        if not self.is_create_pool:
            size_desc = self._get_little_cache_disk_desc_than_origin()
            if size_desc:
                errors.append(self._get_err_msg(
                    "hardware.storage.node.cache.size",
                    (self.msg_head, ",".join(size_desc))))

        # 7、单个硬盘池的总存储节点数不建议超过40个
        if len(self.all_nodes) > SUGGESTION_MAX_NODE_NUM:
            warnings.append(self._get_err_msg(
                "hardware.storage.node.num.max", (self.msg_head,
                                                  len(self.all_nodes))))
        return errors, warnings

    def _is_expansion_over_origin(self, multi=MAX_ALL_NODE_THAN_ORIGIN_NODE):
        return not self.is_create_pool and \
               len(self.all_nodes) > len(self.origin_nodes) * multi

    def _get_zero_exp_main_ips(self):
        ip_list = []
        for node in self.all_nodes:
            if CacheDiskUtil(node).get_expansion_num() > 0 \
                    and MainDiskUtil(node).get_expansion_num() == 0:
                ip_list.append(get_node_ip(node))
        return ip_list

    def _get_little_main_num_desc(self, threshold=MIN_DISK_NUM):
        little_num_desc = list()
        for ip, disk_num in self._get_ip_disk_num_dict(self.all_nodes).items():
            if disk_num < threshold:
                little_num_desc.append("[{}]:{}".format(ip, disk_num))
        return little_num_desc

    @staticmethod
    def _get_ip_disk_num_dict(all_nodes):
        ip_2_num = dict()
        for node in all_nodes:
            ip_2_num[get_node_ip(node)] = \
                MainDiskUtil(node).get_total_num()
        return ip_2_num

    def _get_assign_little_cache_size_desc(self,
                                           threshold=MIN_CATCHE_FOR_STORAGE):
        """获取主存盘分配缓存容量小于阈值的节点，并返回描述信息

        :return: 格式化描述信息
        """
        size_desc = list()
        for ip, average in self._get_ip_average_cache_dict(
                self.all_nodes).items():
            if average < threshold:
                size_desc.append("[{}]:{}GB".format(ip, average))
        return size_desc

    def _get_ip_average_cache_dict(self, all_nodes):
        ip_2_average_cache = dict()
        for node in all_nodes:
            main_util = MainDiskUtil(node)
            cache_util = CacheDiskUtil(node)
            main_num = main_util.get_total_num()
            cache_size = cache_util.get_total_size()
            cache_rate = cache_util.get_rate(self.cache_type)
            if is_involve_cache(self.main_type) \
                    and cache_rate != 0 and main_num != 0:
                ip_2_average_cache[get_node_ip(node)] = \
                    round(cache_size / cache_rate / main_num, 2)
        return ip_2_average_cache

    def _get_little_cache_disk_desc_than_origin(self):
        """获取比原有节点缓存容量小的新扩节点，并返回格式化描述信息

        :return: 格式化描述信息
        """
        desc_list = list()
        min_cache_size = self._get_min_cache_size()
        for node in self.expansion_nodes:
            cache_util = CacheDiskUtil(node)
            if not is_involve_cache(self.main_type):
                continue
            tag_size, tag_unit = cache_util.get_expansion_size_and_unit()
            min_cache_tag_size = get_common().get_tag_size(
                float(min_cache_size), tag_unit, get_common().get_scale(
                    tag_size))
            self.logger.logInfo("node {} cache expansion config: size: {}, "
                                "unit {}".format(get_node_ip(node),
                                                 tag_size, tag_unit))
            self.logger.logInfo("min_cache_size_tag_size : {}".format(
                min_cache_tag_size))
            if float(tag_size) < float(min_cache_tag_size):
                desc_list.append("[{}]:{}{}"
                                 .format(get_node_ip(node), tag_size, tag_unit))
        return desc_list

    def _get_min_cache_size(self):
        return min([CacheDiskUtil(node).get_origin_size()
                    for node in self.origin_nodes])

    def _get_err_msg(self, key, arg):
        return get_common().get_err_msg(self.lang, key, arg)


def is_exp_node_out_cluster(node):
    """判断是否是集群外的新扩节点

    集群外的新扩节点类型为DevNode, 集群内的节点类型为ClusterNode
    :return: True/False
    """
    return isinstance(node, get_dev_node_class())


def get_node_ip(node):
    """获取节点的管理IP

    :return: 节点管理IP
    """
    return node.getIp() if is_exp_node_out_cluster(node) \
        else node.getManagementIp()


def is_involve_cache(main_type):
    """判断当前主存盘类型是否能配置缓存盘

    SSD盘或SSD卡做主存时，节点不能配置缓存盘
    :param main_type: 主存盘类型
    :return: True/False 是/否能配置缓存盘
    """
    return main_type != get_disk_type_enum().SSD_CARD_AND_NVME_SSD.name() \
        and main_type != get_disk_type_enum().SSD.name()


class MainDiskUtil:
    """解析单个节点上的主存盘配置的工具类

    基于单个节点进行分析
    """
    def __init__(self, node):
        self.node = node

    def get_origin_num(self):
        if is_exp_node_out_cluster(self.node):
            return 0
        return len(self.node.getMainStorageDisk()) \
            if self.node.getMainStorageDisk() else 0

    def get_expansion_num(self):
        return sum(
            dict(get_common().get_expansion_main_storage_disk(self.node)).values())

    def get_total_num(self):
        return self.get_origin_num() + self.get_expansion_num()


class CacheDiskUtil:
    """解析单个节点上的缓存盘配置的工具类

    基于单个节点进行分析
    """
    def __init__(self, node):
        self.node = node
        self.expansion_dict = dict(get_common().get_expansion_cache_disk(self.node))

    def get_expansion_num(self):
        return sum(self.expansion_dict.values())

    def get_origin_size(self):
        if is_exp_node_out_cluster(self.node):
            return 0
        return sum([int(disk.getCapacity())
                    for disk in self.node.getCacheDisk()])

    def get_expansion_size(self):
        return sum([int(disk.getCapacity()) * num
                    for disk, num in self.expansion_dict.items()])

    def get_total_size(self):
        return self.get_origin_size() + self.get_expansion_size()

    def get_expansion_size_and_unit(self):
        """
        取扩容盘的总容量和单位，单位为最大的单位
        """
        max_unit, scale = self._get_expansion_max_unit_and_scale()
        total_size = 0
        for disk, num in self.expansion_dict.items():
            ori_unit = disk.getCapacityUnit()
            size = float(disk.getOriCapacity())
            if ori_unit != max_unit:
                # 需要将容量转换为对应的单位的值，
                size = float(disk.getCapacityAssignUnit(max_unit, scale))
            total_size += (size * num)
        return total_size, max_unit

    def _get_expansion_max_unit_and_scale(self):
        """
        针对扩容配置中存在多种容量，获取最大的容量单位和保留的小数位数
        """
        max_unit = None
        scale = 0
        for disk in self.expansion_dict.keys():
            if max_unit is None or disk.getCapacityUnit().getMultiples() > \
                    max_unit.getMultiples():
                max_unit = disk.getCapacityUnit()
                scale = get_common().get_scale(str(disk.getCapacity()))
        return max_unit, scale

    @staticmethod
    def get_rate(cache_type):
        """获取指定类型的缓存盘由Byte转为GB的换算率

        :param cache_type: 缓存盘类型(SSD盘或SSD卡)
        :return: Byte->GB的换算率
        """
        if cache_type == get_disk_type_enum().SSD.name():
            return math.pow(RATE_DISK, 3)
        return math.pow(RATE_CARD, 3)
