# -*- coding: UTF-8 -*-
import codecs
import os
import re
import threading
# noinspection PyUnresolvedReferences
from decimal import Decimal

import java.util.MissingResourceException as MissingResourceException
from com.huawei.ism.tool.inspect.utils import InspectContext
from com.huawei.ism.tool.obase.connection import SshConnectionManager
from com.huawei.ism.tool.protocol.factory import SftpConnector

import cli_util
# noinspection PyUnresolvedReferences
import com.huawei.ism.tool.framework.platform.runtime.SystemProperties as sysProperties
# noinspection PyUnresolvedReferences
import com.huawei.ism.tool.inspect.bmcconn.redfish.BmcRedfishConnManagement as RedFishPool
# noinspection PyUnresolvedReferences
import com.huawei.ism.tool.inspect.bmcconn.ssh.BmcShellConnectionManagement as SSHPool
import err_msg_dict
import logger
from resource_client import ChassisResourceClient
from resource_client import SystemsResourceClient

MUTEX = threading.Lock()
FILE_SUFFIX = "."
INSPECT_PASS = "0"
INSPECT_UNNORMAL = "1"
INSPECT_NOCHECK = "2"
INSPECT_NOSUPPORT = "3"
INSPECT_WARNING = "4"
DEFAULT_MAX_THREAD_NUM_ONE_NODE = 10
CLI_RET_END_FLAG = ":/>"
HOST_CMD_SHORT_TIMEOUT = 2 * 60
MAIN_STORAGE_DISK_TYPE = "main_storage_disk"
CACHE_DISK_TYPE = "cache_disk"


def getLang(py_java_env):
    """
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    """
    return py_java_env.get("lang")


def getBaseName(file_path):
    """
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    """
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name


def getLogger(loggerInstance, pyFilePath):
    """
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    """
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)


def get_err_msg(lang, msg, args="", resource=err_msg_dict.ERROR_MSG_DICT):
    """
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    """
    err_msg = "\n--"

    local_dict = resource.get(msg)
    if not local_dict:
        return err_msg

    local_msg = local_dict.get(lang)
    if not local_msg:
        return err_msg
    if "%s" in local_msg or "%i" in local_msg:
        return local_msg % args
    else:
        return local_msg


def get_device_node(env):
    """
    获取dev node
    :param env:
    :return:
    """
    return env.get("devInfo")


def get_progress_observer(env):
    """
    获取进度观察者
    :param env:
    :return:
    """
    return env.get("progressObserver")


def get_cluster_nodes(dev_node):
    """
    获取集群node
    :param dev_node:
    :return:
    """
    return dev_node.getClusterNodes()


def get_node_ip(dev_node):
    """
    获取Ip
    :param dev_node:
    :return:
    """
    return dev_node.getManagementIp()


def get_ibmc_ssh_conn(dev_node):
    """
    获取ibmc链接
    :param dev_node: 阵列dev node,非Ibmc dev node
    :return:
    """
    return SSHPool.getSSHAndSFTPConnection(dev_node)


def release_bmc_ssh_conn(dev_node):
    """
    释放ssh链接
    :param dev_node:
    :return:
    """
    SSHPool.releaseConnection(dev_node)


def get_ssh_conn(dev_node):
    """
    获取ssh链接
    :param dev_node: 单节点
    :return:
    """
    return SshConnectionManager.getSshConnection(dev_node)


def get_sftp_connection(dev_node):
    """
    获取sftp链接
    :param dev_node: 单节点
    :return: 单节点连接
    """
    return SftpConnector(dev_node)


def release_ssh_conn(con):
    """
    释放ssh链接
    :param con:
    :return:
    """
    SshConnectionManager.releaseConnection(con)


def release_sftp_conn(sftp_connection):
    """
    释放sftp链接
    :param sftp_connection: sftp连接
    """
    if sftp_connection:
        sftp_connection.getConnection().closeSshConnection()


def get_redfish_conn(cluster_node):
    """
    获取redfish 链接
    :param cluster_node:
    :return:
    """
    return RedFishPool.getRedFishConnection(cluster_node)


def is_not_storage_node_or_do_not_support_redfish(node, ret_list, ip):
    """
    不是存储节点（不含存储服务）或不支持redfish接口
    :param node:
    :param ret_list:
    :param ip:
    :return:
    """
    if not node.includedStorageService() or node.isHGNode():
        err_msg = cli_util.get_format_header_ret(
            ip, "this is not support node type")
        if isinstance(ret_list, list):
            ret_list.append(err_msg)
        if isinstance(ret_list, dict):
            ret_list[ip] = [err_msg]
        return True
    return False


def check_all_ibmc_node(func):
    """
    遍历所有节点执行检查
    分批巡检，每个10个节点为一批。
    :param func: 业务检查逻辑
    :return: 全部节点的检查结果
    """

    def wraper(param_dict, ibmc_node, err_info_list, ret_list):
        env = param_dict.get("env")
        item_id = param_dict.get("ITEM_ID")
        progress_map = param_dict.get("progress_map")
        device_node = get_device_node(env)
        observer = get_progress_observer(env)
        cur_progress = progress_map.get(item_id)
        logger = param_dict.get("logger")
        all_func_res = []
        all_nodes = get_cluster_nodes(device_node)
        eve_progress = len(all_nodes) if len(all_nodes) > 0 else 1
        node_progress = (99.0 - cur_progress) / eve_progress
        p_observer = ProgressObserver(progress_map, item_id, observer)
        max_thread_num = get_max_work_thread_num_from_sys_properties(logger)
        for i_index in range(0, len(all_nodes), max_thread_num):
            thread_list = []
            next_step = (
                i_index + max_thread_num
                if i_index + max_thread_num < len(all_nodes)
                else len(all_nodes)
            )
            for j_index in range(i_index, next_step):
                node = all_nodes[j_index]
                if node.getBmcNode():
                    logger.logInfo("dev_node.getBmcNode().getLoginUser():{}".
                                   format(node.getBmcNode().getLoginUser()))
                if not node.isSelected():
                    continue
                ip_addr = get_node_ip(node)
                if is_not_storage_node_or_do_not_support_redfish(node, ret_list, ip_addr):
                    continue
                if node.isSelected() and not has_ibmc_auth(node):
                    err_info_list.append(build_no_auth_msg(env, ip_addr))
                    continue
                logger.logInfo("start new thread check: node {}".format(ip_addr))
                check_thread = threading.Thread(
                    target=one_task,
                    args=(
                        param_dict, node, func, all_func_res, p_observer,
                        node_progress, err_info_list, ret_list, logger,
                    ),
                )
                thread_list.append(check_thread)
            # 启动所有线程
            start_cur_threads(thread_list, logger)
        return all_func_res

    def build_no_auth_msg(env, ip_address):
        return get_err_msg(getLang(env), get_error_code(env), ip_address)

    def get_error_code(env):
        return "hardware.node.has.no.auth" if not env.get("inspect_context").isDetachCheck() \
            else "hardware.node.has.no.auth.detach.check"

    return wraper


def start_cur_threads(thread_list, py_log):
    """
    工具方法，启动线程
    :param thread_list: 线程
    :param py_log: 日志
    :return:
    """
    py_log.logInfo("thread_list: thread_list {}".format(thread_list))
    for t in thread_list:
        t.start()
    # 主线程中等待所有子线程退出
    for t in thread_list:
        t.join()


def one_task(
        param_dict,
        node,
        func,
        all_func_res,
        p_observer,
        node_progress,
        err_info_list,
        ret_list,
        logger,
):
    """
    工具方法，执行命令，刷新进度
    """
    logger.logInfo("start func ip:{}".format(get_node_ip(node)))
    func_res = func(param_dict, node, err_info_list, ret_list)
    logger.logInfo("ip:{}, ret:{}".format(get_node_ip(node), func_res))
    all_func_res.append(func_res)
    p_observer.refresh_progress(node_progress)


class ProgressObserver:
    """
    自定义多线程进度刷新
    """

    def __init__(self, progress_map, item_id, observer):
        self.progress_map = progress_map
        self.item_id = item_id
        self.observer = observer
        self.current_progress = self.progress_map.get(self.item_id)

    def refresh_progress(self, node_progress):
        with MUTEX:
            self.current_progress += node_progress
            self.progress_map[self.item_id] = int(self.current_progress)
            self.observer.updateProgress(self.progress_map)


class ItemEntity:
    """
    检查项通用属性的封装
    """

    def __init__(self, env, id, logger, script_name):
        self.env = env
        self.id = id
        self.logger = getLogger(logger, script_name)
        self.progress_map = {id: 1}
        self.observer = env.get("progressObserver")
        self.ret_list = []
        self.err_info_list = []

    def build_param_dict(self):
        """
        兼容已有结构的参数封装，将属性封装至字典中
        :return: 参数字典
        """
        param_dict = dict()
        param_dict["env"] = self.env
        param_dict["logger"] = self.logger
        param_dict["ITEM_ID"] = self.id
        param_dict["progress_map"] = self.progress_map
        return param_dict

    def update_progress(self, progress):
        """
        更新检查进度
        :param progress:进度值
        :return: void
        """
        self.progress_map[self.id] = progress
        self.observer.updateProgress(self.progress_map)

    def record_ret(self):
        """
        记录检查结果至回显文件中，最终持久化到报告
        :return: 回显文本内容，超过一定阈值转储至文件
        """
        return save_cli_ret_to_file(self.ret_list, self.id, self.env, self.logger)

    def get_failed_result(self, err_msg):
        """
        获取巡检结果失败的结果
        :param status: 巡检状态
        :param err_msg: 错误消息
        :return: 巡检状态，原始信息，错误消息
        """
        return INSPECT_UNNORMAL, self.record_ret(), err_msg

    def get_check_result(self):
        """
        获取巡检结果，如果错误消息不为空则巡检不通过，否则巡检通过
        :return: 巡检状态，原始信息，错误消息
        """
        record = self.record_ret()
        if self.err_info_list:
            self.logger.logInfo(str(self.err_info_list))
            return INSPECT_UNNORMAL, record, "".join(self.err_info_list)
        return INSPECT_PASS, record, ""


def get_max_work_thread_num_from_sys_properties(logger):
    """
    从配置文件中读取线程配置数。默认10个
    :param logger:
    :return:
    """
    sys_properties_path = os.path.abspath(FILE_SUFFIX)
    work_thread_num_key = "app.inspect.max.work.thread.num"
    app_conf = sysProperties(sys_properties_path)
    try:
        wtn_value = app_conf.getValue(work_thread_num_key)
        wtn_value = str(wtn_value).lower().strip()
        if wtn_value.isdigit():
            return int(wtn_value)
    except MissingResourceException as e:
        logger.logError("no max work thread num key found.{}".format(str(e)))
    return DEFAULT_MAX_THREAD_NUM_ONE_NODE


def get_system_over_view_info(cluster_dev_node):
    """
    获取系统总体信息
    :param cluster_dev_node:
    :return:
    """
    redfish_conn = get_redfish_conn(cluster_dev_node)
    cmd_url = "/SystemOverview"
    return cli_util.execute_redfish_get(redfish_conn, cmd_url)


def get_hardware_health_status(cluster_dev_node, log, ret_list, hardware_summary):
    """
    通过接口/redfish/v1/Systems/获取指定硬件健康状态
    :param cluster_dev_node: 节点
    :param log: 日志
    :param ret_list: 回文
    :param hardware_summary: 需要查询的硬件摘要
    :return: 是否获取到
    """
    redfish = get_redfish_conn(cluster_dev_node)
    client = SystemsResourceClient(redfish, log)
    path = client.get_assign_resource_path()
    flag, ret_json = cli_util.direct_execute_redfish_get(redfish, path)
    cluster_node_ip = get_node_ip(cluster_dev_node)
    show_ret = "{}\n{}".format(path, ret_json)
    ret_list.append(cli_util.get_format_header_ret(cluster_node_ip, show_ret))
    if not flag:
        return False, ""
    summary = ret_json.get(hardware_summary, {})
    status = summary.get("Status", {})
    health_status = status.get("HealthRollup", {})
    return True, str(health_status).upper()


def get_chassis_hardware_resource(cluster_dev_node, log, resource_path_suffix):
    """
    通过接口/redfish/v1/Chassis/获取指定硬件资源
    :param cluster_dev_node: 节点
    :param log: 日志
    :param resource_path_suffix: 指定硬件资源路径后缀
    :return:
    """
    redfish = get_redfish_conn(cluster_dev_node)
    client = ChassisResourceClient(redfish, log)
    path = client.get_fan_resource_path(resource_path_suffix)
    flag, ret_json = cli_util.direct_execute_redfish_get(redfish, path)
    return flag, ret_json, path


def get_alam_major_certical(cluster_dev_node):
    """
    获取告警信息，多线程re模块会概率报错，所以在方法中引用re
    :param cluster_dev_node:
    :return:
    """
    ibmc_ssh = get_ibmc_ssh_conn(cluster_dev_node)
    flag, ret = cli_util.execute_cmd(ibmc_ssh, "ipmcget -d healthevents")
    reg_x = (
        "\d+\s*\|\s*\d{4}-\d{2}-\d{2}\s\d{2}:*\d{2}:*\d"
        "{2}:*\s*\|\s*(Minor|Major|Critical)\s*\|(.*)"
    )
    release_bmc_ssh_conn(cluster_dev_node)
    res_list = re.compile(reg_x).findall(ret)
    return flag, ret, res_list


def is_dict_obj(t_obj):
    """
    是否是字典类型
    :param t_obj:
    :return:
    """
    return type(t_obj) == dict


def save_cli_ret_to_file(ret, item_id, env, log):
    """
    分布式回文都写入文件
    :param ret: 回文
    :param item_id: 检查项ID
    :param env: env
    :param log: logger
    :return: 保存路径
    """
    show_line_num = 10
    if len(ret) < show_line_num:
        log.logInfo("line num smaller than {}".format(show_line_num))
        return u"\n".join(ret)

    dev_node = get_device_node(env)
    ip = str(dev_node.getIp())
    sn = dev_node.getDeviceSerialNumber()
    base_path = os.path.abspath(".")
    tmp_path = os.path.join(base_path, "temp", "Dstorage", "dsrecord")
    save_dir = "{ip}_{sn}".format(ip=ip.replace(":", "_"), sn=sn)
    dump_path = os.path.join(tmp_path, save_dir)
    if not os.path.exists(dump_path):
        os.makedirs(dump_path)

    file_name = "{itemId}{tail}".format(itemId=item_id, tail=".txt")
    file_path = os.path.join(dump_path, file_name)
    add_ret = u""
    with codecs.open(file_path, "w", encoding="utf-8") as f:
        if isinstance(ret, list):
            add_ret = u"\n".join(ret[:show_line_num])
            for line in ret:
                f.write("\n" + line)
        else:
            add_ret = str(ret[:show_line_num])
            f.write(str(ret))
    show_dir = os.path.join("data", "dsrecord", save_dir, file_name)
    return u"{}{}".format(
        add_ret,
        get_err_msg(
            getLang(env),
            "ret.show.more.information",
            (show_line_num, show_dir),
        ),
    )


def has_ibmc_auth(dev_node):
    return dev_node.getBmcNode().getLoginUser() is not None


def get_vertical_cliret(ret):
    """
    回文转换字典
    :param ret: 命令回文
    :return: 字典
    """
    ret_list = ret.encode("utf8").splitlines()
    line_dict = {}
    for line in ret_list:
        if CLI_RET_END_FLAG in line:
            break
        fields = line.split(":")
        if len(fields) < 2:
            continue
        key = fields[0].strip().decode("utf8")
        value = fields[1].strip().decode("utf8")
        line_dict.setdefault(key, value)
    return line_dict


def get_east_node_and_disk_num_dic(disk_pool, disk_tpe):
    """
    获取硬盘池内，东海节点IP和某类型盘数量的字典
    :param disk_pool: 节点信息
    :param disk_tpe 硬盘类型，枚举，主存或缓存
    :return:
        single_node_and_disk_num_dic： 单控节点IP和盘数量的字典
        dual_node_and_disk_num_dic： 双控节点IP和盘数量的字典
    """
    single_node_and_disk_num_dic = {}
    dual_node_and_disk_num_dic = {}

    # 分别处理 设备实际池内节点、加入了集群的扩容新增节点、未加入集群的扩容新增节点
    # 获取单控和双控 节点IP和盘数量的字典
    nodes_ls = [disk_pool.getJoinedClusterNode(), disk_pool.getExpansionClusterNode()]  # 抽取临时变量 防止代码长度过长
    for nodes in nodes_ls:
        parse_east_disk_number(nodes, disk_tpe, single_node_and_disk_num_dic, dual_node_and_disk_num_dic)
    parse_east_disk_number(disk_pool.getExpansionNodeList(), disk_tpe, single_node_and_disk_num_dic,
                           dual_node_and_disk_num_dic, is_cluster_nodes=False)

    return single_node_and_disk_num_dic, dual_node_and_disk_num_dic


def parse_east_disk_number(
        nodes, disk_tpe, single_node_and_disk_num_dic, dual_node_and_disk_num_dic, is_cluster_nodes=True):
    """
    获取集群内或集群外东海节点的存储盘数量
    :param nodes: 节点信息
    :param disk_tpe 硬盘类型，枚举，主存或缓存
    :param single_node_and_disk_num_dic: 单控节点IP和盘数量的字典
    :param dual_node_and_disk_num_dic: 双控节点IP和盘数量的字典
    :param is_cluster_nodes: 是否为集群内节点
    :return:
    """
    for node in nodes:
        disk_num, item_storage = get_disk_num_and_item_storage(disk_tpe, is_cluster_nodes, node)
        if is_cluster_nodes:
            node_ip = node.getManagementIp()
        else:
            node_ip = node.getIp()
        if node.isEastSeaSingleNode():
            parse_disk_num(node_ip, item_storage, disk_num, single_node_and_disk_num_dic)
        elif node.isEastSeaDoubleNode():
            parse_disk_num(node_ip, item_storage, disk_num, dual_node_and_disk_num_dic)
    return


def get_disk_num_and_item_storage(disk_tpe, is_cluster_node, node):
    disk_num = 0
    config = get_expansion_config(node)
    if disk_tpe == MAIN_STORAGE_DISK_TYPE:
        item_storage = config.getExpansionMainStorageDisk().entrySet().iterator()
        if is_cluster_node:
            disk_num = node.getMainStorageDisk().size()
    else:
        item_storage = config.getExpansionCacheDisk().entrySet().iterator()
        if is_cluster_node:
            disk_num = node.getCacheDisk().size()
    return disk_num, item_storage


def get_pacific_node_and_disk_num_dic(disk_pool, disk_tpe):
    """
    获取硬盘池内，太平洋节点ip和某类型盘数量的字典
    :param disk_pool: 节点信息
    :param disk_tpe 硬盘类型，枚举，主存或缓存
    :return: node_and_disk_num_dic: 节点ip和主存盘数量的字典
    """
    node_and_disk_num_dic = {}
    parse_pacific_disk_number_in_cluster_node(disk_pool.getJoinedClusterNode(),
                                              disk_tpe,
                                              node_and_disk_num_dic)
    parse_pacific_disk_number_in_cluster_node(
        disk_pool.getExpansionClusterNode(), disk_tpe, node_and_disk_num_dic)
    parse_pacific_disk_number_in_node(disk_pool.getExpansionNodeList(), disk_tpe,
                                      node_and_disk_num_dic)
    return node_and_disk_num_dic


def parse_pacific_disk_number_in_cluster_node(cluster_node_list, disk_tpe,
                                              node_and_disk_num_dic):
    """
    获取集群内太平洋节点的主存储盘数量
    :param cluster_node_list: 节点信息
    :param disk_tpe 硬盘类型，枚举，主存或缓存
    :param node_and_disk_num_dic: 节点ip和盘数量的字典
    :return:
    """
    for node in cluster_node_list:
        if not node.isPacificNode():
            continue
        config = get_expansion_config(node)
        if disk_tpe == MAIN_STORAGE_DISK_TYPE:
            disk_num = node.getMainStorageDisk().size()
            item_storage = config.getExpansionMainStorageDisk().entrySet().iterator()
        else:
            disk_num = node.getCacheDisk().size()
            item_storage = config.getExpansionCacheDisk().entrySet().iterator()
        parse_disk_num(node.getManagementIp(), item_storage, disk_num, node_and_disk_num_dic)
    return


def parse_pacific_disk_number_in_node(node_list, disk_tpe,
                                      node_and_disk_num_dic):
    """
    获取集群外太平洋节点的主存储盘数量
    :param node_list: 节点信息
    :param disk_tpe 硬盘类型，枚举，主存或缓存
    :param node_and_disk_num_dic: 节点ip和盘数量的字典
    :return:
    """
    for node in node_list:
        if not node.isPacificNode():
            continue
        config = get_expansion_config(node)
        if disk_tpe == MAIN_STORAGE_DISK_TYPE:
            item_storage = config.getExpansionMainStorageDisk().entrySet().iterator()
        else:
            item_storage = config.getExpansionCacheDisk().entrySet().iterator()
        parse_disk_num(node.getIp(), item_storage, 0, node_and_disk_num_dic)
    return


def parse_disk_num(node_ip, item_storage, disk_num,
                   node_and_disk_num_dic):
    disk_num += get_disk_num_by_map_value(item_storage)
    node_and_disk_num_dic[node_ip] = disk_num
    return


def get_disk_num_by_map_value(item_storage):
    disk_num = 0
    while item_storage.hasNext():
        entry = item_storage.next()
        disk_num += entry.getValue()
    return disk_num


def get_expansion_config(node):
    return InspectContext.getInstance().getFsExpansionConfig(node)


def get_expansion_main_storage_disk(node):
    return get_expansion_config(node).getExpansionMainStorageDisk()


def get_expansion_cache_disk(node):
    return get_expansion_config(node).getExpansionCacheDisk()


def get_scale(ori_capacity):
    """
    获取容量的小数位数
    eg: 8 -> 0; 1.1 -> 1; 1.23 -> 2; 1.300 -> 3
    :param ori_capacity: 容量
    :return: 小数位数
    """
    if "." not in str(ori_capacity):
        return 0
    return len(str(ori_capacity).split(".")[1])


def get_tag_size(ori_bytes, target_unit, scale):
    """
    将实际容量转换为标签容量,换算率为1000
    :param ori_bytes: 实际容量，单位Byte
    :param target_unit: 目标单位
    :param scale: 保留小数位数
    """
    target_size = ori_bytes
    for _ in range(1, target_unit.getMultiples()):
        target_size = target_size / 1000
    rate_str = "0."
    for _ in range(0, scale):
        rate_str += "0"
    return Decimal(target_size).quantize(Decimal(rate_str),
                                         rounding="ROUND_DOWN")


def compare_patch_version(version1, version2):
    return compare_str_version(format_patch_version(version1),
                               format_patch_version(version2))


def compare_str_version(version1, version2):
    if version1 == version2:
        return 0
    return 1 if version1 > version2 else -1


def format_patch_version(version):
    find_versions = re.findall(r"SPH(\d+)", version)
    sph_ver = find_versions[0] if find_versions else "0"
    return "SPH{}".format(sph_ver.zfill(5))


def is_product_version_80(dev_node):
    product_version = str(dev_node.getProductVersion())
    version = product_version.replace('.', "")
    return version[0:2] == "80"


def is_product_version_ge_than_821(dev_node):
    """  判断设备存储软件版本是否大于等于821  """
    # OceanStor A310 1.0.0继承自 OceanStor Pacific 8.2.1
    # 8.2.0L15Y01是8.2.6改名，是8.2.1之后的版本
    if "a310" in dev_node.getProductModel().lower() or "l15y01" in dev_node.getProductVersion().lower():
        return True
    product_version = str(dev_node.getProductVersion())
    version = product_version.replace('.', "")
    # 81X的版本中存在8.1.RCX的情况，判断第三位非R可以排除掉81X的版本
    # 82X的版本中存在8.2.XRCX的情况，此处正常判断即可
    return version[2] != "R" and version[0:3].isdigit() and int(version[0:3]) >= 821


def get_disk_pool_by_dev_node(dev_node):
    """  获取设备中所有的硬盘池（8.0.X版本为存储池）  """
    if is_product_version_80(dev_node):
        return dev_node.getStoragePools()
    disk_pools = []
    for storage_pool in dev_node.getStoragePools():
        disk_pools.extend(storage_pool.getDiskPools())
    return disk_pools
