# coding: UTF-8

import re
import threading

import com.huawei.ism.tool.protocol.utils.RestUtil as RestUtil
from com.huawei.ism.exception import IsmException
from com.huawei.ism.tool.inspect.logic.config import FusionStorageConfig
from com.huawei.ism.tool.obase.entity import DiskTypeEnum
from com.huawei.ism.tool.obase.entity import FusionStorDiskInfo
from com.huawei.ism.tool.obase.entity import FusionStorStoragePool
from com.huawei.ism.tool.protocol.rest import RestConnectionManager

import common
from ds_rest_util import CommonRestService

LOGGER = common.getLogger(PY_LOGGER, __file__)
# 主存盘主类型
MEDIA_TYPE = {
    "sas_disk": DiskTypeEnum.SAS,
    "sata_disk": DiskTypeEnum.SATA,
    "ssd_card": DiskTypeEnum.SSD_CARD_AND_NVME_SSD,
    "ssd_disk": DiskTypeEnum.SSD
}
# 主存盘额外类型
MEDIA_TYPE_EXTEND = {
    "ssd_card_zero_op": DiskTypeEnum.SSD_CARD_AND_NVME_SSD_ZERO_OP,
    "ssd_card_qlc": DiskTypeEnum.SSD_CARD_AND_NVME_SSD_QLC,
    "sas_disk_encrypted": DiskTypeEnum.SAS_ENCRYPTED,
    "sata_disk_encrypted": DiskTypeEnum.SATA_ENCRYPTED,
    "ssd_card_encrypted": DiskTypeEnum.SSD_CARD_AND_NVME_SSD_ENCRYPTED,
    "ssd_card_zero_op_encrypted": DiskTypeEnum.SSD_CARD_AND_NVME_SSD_ZERO_OP_ENCRYPTED,
    "ssd_card_qlc_encrypted": DiskTypeEnum.SSD_CARD_AND_NVME_SSD_QLC_ENCRYPTED
}
CACHE_MEDIA_TYPE = {
    "ssd_card": DiskTypeEnum.SSD_CARD_AND_NVME_SSD,
    "ssd_disk": DiskTypeEnum.SSD
}
# 目前只用于给集群的硬盘类型赋值，仅在FusionStorage Block扩容评估场景会用到，不再改动
ENCRYPTED_MEDIA_TYPE = {
    "sas_disk": DiskTypeEnum.SAS_ENCRYPTED,
    "sata_disk": DiskTypeEnum.SATA_ENCRYPTED,
    "ssd_card": DiskTypeEnum.SSD_CARD_AND_NVME_SSD
}
MAX_REPlACE_TIME = 4
ONLY_BLOCK_VERSION = str(FusionStorageConfig.getNfvVersion(
    "directlyExpandingStoragePool"))
NO_SUPPORT_DISK_TYPE = ["m2_disk"]
NO_SUPPORT_DISK_ROLE = ["system_disk", "zk_disk"]

SLOT_TO_DISK_ZONE_ID = {(0, 6, 15, 22): "0",
                        (7, 14, 23, 29): "1",
                        (30, 36, 45, 52): "2",
                        (37, 44, 53, 59): "3",
                        (60, 66, 75, 82): "4",
                        (67, 74, 83, 89): "5",
                        (90, 96, 105, 112): "6",
                        (97, 104, 113, 119): "7"
                        }

EXPANDER_PATTERN = re.compile(r'.*/expander-(.*?)/', flags=re.IGNORECASE)
HOST_PATTERN = re.compile(r'.*/host(.*?)/port', flags=re.IGNORECASE)
LOCATION_PATTERN = re.compile(r'.*/(.*?)\]', flags=re.IGNORECASE)

zero_op_disk_info = dict()
qlc_disk_info = dict()


def execute_storage():
    """
    获取存储池信息
    :param env:
    :return:
    """
    dev_nodes = py_java_env.get("devInfoList")
    for dev_node in dev_nodes:
        get_node_pool(dev_node)
        get_node_not_in_pool(dev_node)


def execute_node():
    """
    获取待扩节点信息
    :param env:
    :return:
    """
    dev_nodes = py_java_env.get("devInfoList")
    max_thread_num = common.get_max_work_thread_num_from_sys_properties(LOGGER)
    for i_index in range(0, len(dev_nodes), max_thread_num):
        thread_list = []
        next_step = (
            i_index + max_thread_num
            if i_index + max_thread_num < len(dev_nodes)
            else len(dev_nodes)
        )
        for j_index in range(i_index, next_step):
            node = dev_nodes[j_index]
            check_thread = threading.Thread(
                target=get_node_disks,
                args=(node,)
            )
            thread_list.append(check_thread)
        # 启动所有线程
        common.start_cur_threads(thread_list, LOGGER)


def get_node_disks(dev_node):
    """
    获取环境下所有的盘
    :param dev_node:环境信息
    :return:
    """
    # 1、获取节点下所有的盘
    ssh = common.get_ssh_conn(dev_node)
    cmd = "dmidecode|grep 'System Information' -A9|egrep  'Manufacturer|Product'|cat"
    ret = ssh.execCmdWithTimout(cmd, common.HOST_CMD_SHORT_TIMEOUT)
    if "Product Name" in ret:
        product_info = common.get_vertical_cliret(ret)
        dev_node.setProductModel(product_info["Product Name"])
    query_base_board(ssh, dev_node)
    if dev_node.isPacificNode():
        disk_name_to_disk_zone_id = get_disk_list_by_expander(ssh, EXPANDER_PATTERN)
    elif dev_node.isEastSeaSingleNode() or dev_node.isEastSeaDoubleNode():
        disk_name_to_disk_zone_id = get_disk_list_by_expander(ssh, HOST_PATTERN)
    else:
        disk_name_to_disk_zone_id = get_disk_list_by_scsi(ssh)
    # 2、分析每块盘 foreach
    for disk_name, disk_zone_id in disk_name_to_disk_zone_id.items():
        get_new_disk_info(disk_name, disk_zone_id, ssh, dev_node)
    common.release_ssh_conn(ssh)


def query_base_board(ssh, dev_node):
    cmd = "dmidecode|grep 'Product Name:'|cat"
    ret = ssh.execCmdWithTimout(cmd, common.HOST_CMD_SHORT_TIMEOUT)
    ret_list = ret.encode("utf8").splitlines()
    for line in ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        if "Product Name" == fields[0].strip().decode("utf8"):
            dev_node.setBaseBoardType(fields[1].strip().decode("utf8"))


def get_disk_list_by_expander(ssh, pattern):
    disk_name_to_disk_zone_id = {}
    cmd = "lsscsi -v"
    ret = ssh.execCmdWithTimout(cmd, common.HOST_CMD_SHORT_TIMEOUT)
    if any(["Can't find" in ret, "Unknown command" in ret,
            "command not found" in ret, "command-not-found" in ret,
            "No such device or address" in ret]):
        return disk_name_to_disk_zone_id
    info_ret_list = ret.encode("utf8").splitlines()
    # 解析出位置和盘符的对应关系
    disk_name_to_location_id = get_disk_name_to_location_id(info_ret_list)
    # 解析出位置和expander的对应关系
    location_id_to_expander_id = get_location_id_to_expander_id(info_ret_list, pattern)
    for disk_name, location_id in disk_name_to_location_id.items():
        disk_name_to_disk_zone_id[disk_name] = location_id_to_expander_id.get(
            location_id, "")
    return disk_name_to_disk_zone_id


def get_disk_name_to_location_id(info_ret_list):
    disk_name_location_id = {}
    for info_ret in info_ret_list:
        if re.search(".*di?sk.*/dev.*", info_ret):
            disk_id = "/dev" + info_ret.split("/dev")[1]
            location_id = re.compile(r'\[(.*?)\].*',
                                     flags=re.IGNORECASE).findall(info_ret)
            if location_id:
                disk_name_location_id[disk_id] = location_id[0]
    return disk_name_location_id


def get_location_id_to_expander_id(info_ret_list, pattern):
    location_id_to_expander_id = {}
    for info_ret in info_ret_list:
        location_id = LOCATION_PATTERN.findall(info_ret)
        expander_id = pattern.findall(info_ret)
        if location_id and expander_id:
            location_id_to_expander_id[location_id[0]] = str(expander_id[0])
    return location_id_to_expander_id


def get_disk_zone_id_by_slot_id(slot_id):
    for slot_list, disk_zone_id in SLOT_TO_DISK_ZONE_ID.items():
        if check_slot_id(slot_list[0], slot_list[1], slot_list[2],
                         slot_list[3], slot_id):
            return disk_zone_id
    return ""


def check_slot_id(min_1, max_1, min_2, max_2, slot_id):
    return (slot_id >= min_1 and slot_id <= max_1) or (
        slot_id >= min_2 and slot_id <= max_2)


def get_disk_list_by_scsi(ssh):
    disk_name_to_disk_zone_id = {}
    cmd = "lsscsi"
    ret = ssh.execCmdWithTimout(cmd, common.HOST_CMD_SHORT_TIMEOUT)
    if any(["Can't find" in ret, "Unknown command" in ret,
            "command not found" in ret, "command-not-found" in ret,
            "No such device or address" in ret]):
        return disk_name_to_disk_zone_id
    info_ret_list = ret.encode("utf8").splitlines()
    for info_ret in info_ret_list:
        if re.search(".*disk.*/dev.*", info_ret):
            disk_id = "/dev" + info_ret.split("/dev")[1]
            disk_name_to_disk_zone_id[disk_id] = ""
    return disk_name_to_disk_zone_id


def get_new_disk_info(disk, disk_zone_id, ssh, dev_node):
    """
    获取新节点的盘信息
    :param disk: 盘名称
    :param disk_zone_id: 盘对应的disk zone的id
    :param ssh: 执行接口
    :param dev_node: 节点信息
    :return:
    """
    disk_java = FusionStorDiskInfo()
    capacity = ""
    disk_type = DiskTypeEnum.SAS
    cmd = "smartctl -i " + disk
    ret = ssh.execCmdWithTimout(cmd, common.HOST_CMD_SHORT_TIMEOUT)
    if any(["Can't find" in ret, "Unknown command" in ret,
            "command not found" in ret, "command-not-found" in ret,
            "No such device or address" in ret]):
        return
    disk_info = common.get_vertical_cliret(ret)
    sn = ""
    if disk_info.get("SATA Version is") and disk_info.get(
            "SATA Version is").startswith("SATA"):
        capacity = disk_info.get("User Capacity").split()[0]

        if disk_info.get("Rotation Rate") and disk_info.get(
                "Rotation Rate").startswith("Solid"):
            disk_type = DiskTypeEnum.SSD
        else:
            disk_type = DiskTypeEnum.SATA
            capacity = disk_info["User Capacity"].split()[0]
        sn = disk_info.get("Serial Number")
    if disk_info.get("Transport protocol") and disk_info.get(
            "Transport protocol").startswith("SAS"):
        disk_type = DiskTypeEnum.SAS
        capacity = disk_info.get("User Capacity").split()[0]
        if disk_info.get("Rotation Rate") and disk_info.get(
                "Rotation Rate").startswith("Solid"):
            disk_type = DiskTypeEnum.SSD
        sn = disk_info.get("Serial number")
    if disk_info.get("Total NVM Capacity"):
        disk_type = DiskTypeEnum.SSD_CARD_AND_NVME_SSD
        capacity = disk_info.get("Total NVM Capacity").split()[0]
        sn = disk_info.get("Serial Number")
    # 自研盘不显示
    if not sn:
        LOGGER.logInfo(
            "{}:{}".format(dev_node.getIp(), disk_info.get("Product")))
        return
    disk_java.setSn(sn)
    disk_java.setType(disk_type)
    disk_java.setCapacity(capacity.replace(",", "", MAX_REPlACE_TIME))
    disk_java.setDiskZoneId(disk_zone_id)
    dev_node.getAllDiskInfo().add(disk_java)


def get_node_pool(dev_node):
    """
    获取环境所有的池
    :param dev_node: 环境信息
    :return:
    """
    try:
        rest = RestConnectionManager.getRestConnection(dev_node)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        cmd_str = "{}/dsware/service/resource/queryStoragePool".format(
            base_uri
        )
        pools_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        if not pools_json.get("storagePools"):
            return
        get_disk_pool_op_disk_info(rest, base_uri)
        storage_pools = pools_json.get("storagePools", [])

        build_pool(base_uri, dev_node, rest, storage_pools)
        RestConnectionManager.releaseConn(dev_node)
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        RestConnectionManager.releaseConn(dev_node)


def get_disk_pool_op_disk_info(rest, base_uri):
    cmd_str = "{}/dsware/service/resource/queryPoolBasicInfo".format(base_uri)
    pools_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not pools_json.get("pools"):
        return
    for disk_pool in pools_json.get("pools"):
        disk_pool_id = disk_pool.get("poolId")
        storage_pool_id = disk_pool.get("storagePoolId")
        is_zero_op_disk = disk_pool.get("zeroOPDiskPoolFlag")
        zero_op_disk_info["{}_{}".format(storage_pool_id, disk_pool_id)] = is_zero_op_disk
        sub_types = disk_pool.get("subTypes", [])
        qlc_disk_info["{}_{}".format(storage_pool_id, disk_pool_id)] = sub_types


def build_pool(base_uri, dev_node, rest, storage_pools):
    for record in storage_pools:
        # 获取池id
        pool_id = record.get("poolId")
        if pool_id is None:
            continue
        java_pool = FusionStorStoragePool()
        parse_pool_info(java_pool, record)
        java_pool.setServiceType(str(record.get("serviceType", "")))
        # 8.0的块设备，结构为：一个存储池-n个硬盘
        # 8.1及之后的设备，结构为：一个存储池-n个硬盘池-m个硬盘
        product_version = str(dev_node.getProductVersion())
        if product_version.startswith(ONLY_BLOCK_VERSION):
            build_pool_by_disk(base_uri, dev_node, rest, pool_id, java_pool)
        else:
            build_pool_by_disk_pool(base_uri, dev_node, rest, pool_id,
                                    java_pool)


def build_pool_by_disk_pool(base_uri, dev_node, rest, pool_id,
                            java_storage_pool):
    cmd_str = (
        "{}/api/v2/data_service/diskpool?storagePoolId={}".format(base_uri,
                                                                  pool_id)
    )
    disk_pool_json = \
        CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not disk_pool_json.get("diskPools"):
        return
    disk_pool_info_list = disk_pool_json.get("diskPools", [])
    for disk_pool_record in disk_pool_info_list:
        build_disk_pool(disk_pool_record, java_storage_pool, base_uri, rest,
                        dev_node)
    dev_node.getStoragePools().add(java_storage_pool)


def build_disk_pool(disk_pool_record, java_storage_pool, base_uri, rest,
                    dev_node):
    disk_pool_id = disk_pool_record.get("poolId")
    if disk_pool_id is None:
        return
    storage_pool_id = disk_pool_record.get("storagePoolId")
    pase_op_disk_info(disk_pool_id, disk_pool_record, storage_pool_id)
    java_disk_pool = FusionStorStoragePool()
    java_storage_pool.addDiskPool(java_disk_pool)
    java_disk_pool.setParentPool(java_storage_pool)
    parse_pool_info(java_disk_pool, disk_pool_record)
    cmd_str = (
        "{}/dsware/service/cluster/diskpool/"
        "queryNodeDiskInfo?diskPoolId={}".format(base_uri, disk_pool_id)
    )
    node_json = \
        CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    if not node_json.get("nodeInfo"):
        return
    nodes_info = node_json.get("nodeInfo", [])
    cluster_nodes = []
    # 对每个池中的ip进行遍历
    get_node_disk_info(nodes_info, cluster_nodes)
    change_node_info(dev_node, cluster_nodes, None, java_disk_pool)


def pase_op_disk_info(disk_pool_id, disk_pool_record, storage_pool_id):
    zero_op_disk = zero_op_disk_info.get("{}_{}".format(storage_pool_id, disk_pool_id), 0)
    sub_types = qlc_disk_info.get("{}_{}".format(storage_pool_id, disk_pool_id), [])
    disk_pool_record["zeroOPDiskPoolFlag"] = zero_op_disk
    disk_pool_record["subTypes"] = sub_types


def build_pool_by_disk(base_uri, dev_node, rest, pool_id, java_pool):
    cmd_str = (
        "{}/dsware/service/cluster/storagepool/"
        "queryNodeDiskInfo?poolId={}".format(base_uri, pool_id)
    )
    node_json = \
        CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
    # 池中没有子节点
    if not node_json.get("nodeInfo"):
        return
    nodes_info = node_json.get("nodeInfo", [])
    cluster_nodes = []
    # 对每个池中的ip进行遍历
    get_node_disk_info(nodes_info, cluster_nodes)
    change_node_info(dev_node, cluster_nodes, java_pool, None)
    dev_node.getStoragePools().add(java_pool)


def get_node_disk_info(nodes_info, cluster_nodes):
    """
    解析盘信息
    :param nodes_info: 返回的节点信息
    :param cluster_nodes: 节点信息
    :return:
    """
    for node_info in nodes_info:
        disk_list = node_info.get("mediaInfo", [])
        if (disk_list is None) or \
                (node_info.get("errorCode", 0) != 0):
            # 待处理
            continue
        # 子节点信息
        cluster_node = dict()
        all_disk_info = []
        main_storage_disk = []
        cache_disk = []
        cluster_node["nodeMgrIp"] = node_info.get("nodeMgrIp")
        for disk_info in disk_list:
            tran_disk_info(disk_info, all_disk_info, main_storage_disk,
                           cache_disk)
        cluster_node["allDiskInfo"] = all_disk_info
        cluster_node["mainStorageDisk"] = main_storage_disk
        cluster_node["cacheDisk"] = cache_disk
        cluster_nodes.append(cluster_node)


def parse_pool_info(java_pool, record):
    """
    池信息转换
    :param java_pool: 池信息
    :param record: record信息
    :return:
    """
    java_pool.setId(str(record.get("poolId")))
    java_pool.setName(record.get("poolName"))
    java_pool.setStatus(str(record.get("poolStatus")))
    java_pool.setTotalCapacity(str(record.get("totalCapacity")))
    java_pool.setUsedCapacity(str(record.get("usedCapacity")))
    main_type = record.get("mediaType")
    java_pool.setMainStorageDiskType(MEDIA_TYPE.get(main_type))
    main_extend_type = get_main_extend_type(record)
    if main_extend_type != main_type:
        java_pool.setMainStorageDiskExtend(MEDIA_TYPE_EXTEND.get(main_extend_type))
    if "none" != record.get("cacheMediaType", "none"):
        java_pool.setCacheDiskType(
            CACHE_MEDIA_TYPE.get(record.get("cacheMediaType")))


def get_main_extend_type(record):
    """
    判断是否是特殊盘（加密盘、零op盘、QLC盘）场景，如果是，生成对应的枚举Key值
    :param record: record信息
    :return:主存盘额外类型key值
    """
    main_extend_type = record.get("mediaType")
    if main_extend_type == "ssd_disk":  # ssd_disk没有特殊盘场景
        return main_extend_type
    if record.get("zeroOPDiskPoolFlag", 0) == 1:  # zeroOPDiskPoolFlag值为1，说明主存盘是零OP盘（动态OP盘）
        main_extend_type = "ssd_card_zero_op"
    elif 2 in record.get("subTypes", []):  # subTypes包含2 表示主存盘是容量型SSD
        main_extend_type = "ssd_card_qlc"
    if 1 == record.get("supportEncryptForMainStorageMedia"):  # supportEncryptForMainStorageMedia等于1代表加密场景
        main_extend_type = main_extend_type + "_encrypted"
    return main_extend_type


def tran_disk_info(disk_info, all_disk_info, main_storage_disk, cache_disk):
    """
    转换盘信息
    :param disk_info: 盘信息
    :param all_disk_info: 所有的盘
    :param main_storage_disk: 主存盘
    :param cache_disk: 缓存盘
    :return:
    """
    if disk_info.get("diskExist") == 1:
        return
    disk = FusionStorDiskInfo()
    encrypted = disk_info.get("supportEncrypt")
    if 1 == encrypted:
        disk.setType(ENCRYPTED_MEDIA_TYPE.get(disk_info.get("diskType")))
    else:
        disk.setType(MEDIA_TYPE.get(disk_info.get("diskType")))
    disk.setSn(disk_info.get("diskSn"))
    disk.setCapacity(str(disk_info.get("mediaCapacityForByte")))
    role = disk_info.get("diskRole")
    disk.setDiskZoneId(str(disk_info.get("vNodeId", "")))
    if "no_use" == role:
        disk.setFree(True)
        all_disk_info.append(disk)
    else:
        disk.setFree(False)
    if "main_storage" == role:
        main_storage_disk.append(disk)
        all_disk_info.append(disk)
    if "osd_cache" == role:
        cache_disk.append(disk)
        all_disk_info.append(disk)


def change_node_info(dev_node, cluster_nodes, java_storage_pool,
                     java_disk_pool):
    """
    转换节点信息
    :param dev_node: 原始节点信息
    :param cluster_nodes: 查询的节点信息
    :param java_storage_pool: 存储池信息
    :param java_disk_pool: 硬盘池信息
    :return:
    """
    for node in dev_node.getClusterNodes():
        for cluster_node in cluster_nodes:
            associate_pool_and_node(cluster_node, java_disk_pool,
                                    java_storage_pool, node)


def associate_pool_and_node(cluster_node, java_disk_pool, java_storage_pool,
                            node):
    if node.getManagementIp() == cluster_node.get("nodeMgrIp"):
        set_disk_to_node(node.getAllDiskInfo(),
                         cluster_node["allDiskInfo"])
        set_disk_to_node(node.getMainStorageDisk(),
                         cluster_node["mainStorageDisk"])
        set_disk_to_node(node.getCacheDisk(),
                         cluster_node["cacheDisk"])
        if java_storage_pool:
            java_storage_pool.getJoinedClusterNode().add(node)
            node.setJoinedStoragePool(java_storage_pool)
        if java_disk_pool:
            java_disk_pool.getJoinedClusterNode().add(node)
            node.setJoinedStoragePool(java_disk_pool)


def set_disk_to_node(disk_list, disks):
    """
    盘列表转换
    :param disk_list: 盘列表
    :param disks: 盘信息
    :return:
    """
    for disk in disks:
        disk_list.add(disk)


def get_node_not_in_pool(dev_node):
    """
    获取未加入池的节点
    :param dev_node: 集群信息
    :return:
    """
    try:
        rest = RestConnectionManager.getRestConnection(dev_node)
        base_uri = RestUtil.getDstorageUrlHead(dev_node)
        cmd_str = "{}/dsware/service/resource/queryAllDisk".format(
            base_uri
        )
        disks_json = CommonRestService.exec_get_gor_big_by_ds(rest, cmd_str)
        if not disks_json.get("disks"):
            return

        disks_all = disks_json.get("disks", {})
        for node_ip, disks in disks_all.items():
            get_equals_node_ip(dev_node.getClusterNodes(), node_ip, disks)
        RestConnectionManager.releaseConn(dev_node)
    except (IsmException, Exception) as exception:
        LOGGER.logException(exception)
        RestConnectionManager.releaseConn(dev_node)


def get_equals_node_ip(cluster_nodes, node_ip, disks):
    """
    获取ip相同的待扩节点
    :param cluster_nodes: 非池中节点信息
    :param node_ip: ip地址
    :param disks: 盘信息
    :return:
    """
    for node in cluster_nodes:
        if node.getManagementIp() == node_ip and not node.getMainStorageDisk():
            tran_disk_not_in_pool(disks, node)


def tran_disk_not_in_pool(disks, node):
    """
    转换未加入池的盘信息
    :param disks: 盘信息
    :param node: 节点信息
    :return:
    """
    for disk_info in disks:
        disk = FusionStorDiskInfo()
        disk_type = disk_info.get("devType")
        disk_role = disk_info.get("diskRole")
        if is_not_support_disk(disk_role, disk_type):
            continue
        encrypted = disk_info.get("supportEncrypt", 0)
        if 1 == encrypted:
            disk.setType(ENCRYPTED_MEDIA_TYPE.get(disk_type))
        else:
            disk.setType(MEDIA_TYPE.get(disk_type))
        if is_involve_disk_zone(node) and disk_info.get("devSlot") is not None:
            disk.setDiskZoneId(
                get_disk_zone_id_by_slot_id(int(disk_info.get("devSlot"))))
        disk.setSn(disk_info.get("devEsn"))
        disk.setCapacity(str(disk_info.get("devTotalCapacityForByte")))
        node.getAllDiskInfo().add(disk)


def is_involve_disk_zone(node):
    return node.isPacificNode() or node.isEastSeaSingleNode() or node.isEastSeaDoubleNode()


def is_not_support_disk(disk_role, disk_type):
    return disk_role in NO_SUPPORT_DISK_ROLE or disk_type in NO_SUPPORT_DISK_TYPE
