#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2020/06/08
@file: resource_client.py
@function:
"""

import cli_util


class ResourcePath(object):
    MANAGER = "/redfish/v1/Managers/"
    ETHERNET_INTERFACE = MANAGER + "{}/EthernetInterfaces"
    ASSIGN_ETHERNET_INTERFACE = ETHERNET_INTERFACE + "{}"
    SYSTEMS = "/redfish/v1/Systems/"
    STORAGES = SYSTEMS + "1/Storages/"
    BIOS = SYSTEMS + "{}/Bios/"
    BIOS_SET = BIOS + "Settings"
    SERVER_REBOOT = SYSTEMS + "{}/Actions/ComputerSystem.Reset"
    USER_COLLECTS = "/redfish/v1/AccountService/Accounts/"
    SYSTEM_OVERVIEW = "/redfish/v1/SystemOverview/"
    CHASSIS = "/redfish/v1/Chassis/"
    CHASSIS_BOARDS = CHASSIS + "{}/Boards/"
    CHASSIS_MAIN_BOARD = CHASSIS + "{}/Boards/chassisMainboard"
    CHASSIS_BY_ID = "/redfish/v1/Chassis/{}"


class ManagerResourceClient:
    """
    管理资源
    """

    def __init__(self, redfish_conn, logger):
        self.redfish = redfish_conn
        self._logger = logger

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 资源路径
        """
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.MANAGER)
        path = response.get("Members")[0].get("@odata.id")
        self._logger.logInfo("assign managers resource path: {}".format(path))
        return path

    def get_assign_resource_info(self):
        """
        获取指定资源的信息
        :return: 资源信息
        """
        path = self.get_assign_resource_path()
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, path)
        return response

    def get_ethernet_interface_gather_resource_info(self):
        """
        获取网口集合资源信息
        :return: 资源信息
        """
        resource_path = self.get_assign_resource_path()
        resource_id = resource_path.split("/")[-1]
        flag, response = cli_util.direct_execute_redfish_get(self.redfish,
                                                             ResourcePath.ETHERNET_INTERFACE.format(resource_id))
        return response

    def get_assign_ethernet_interfaces_resource_path(self):
        """
        获取指定网口的资源访问路径
        :return: 路径
        """
        info = self.get_ethernet_interface_gather_resource_info()
        path = info.get("Members")[0].get("@odata.id")
        self._logger.logInfo("managers ethernet interface path: {}".format(path))
        return path

    def get_assign_ethernet_interfaces_info(self):
        """
        获取指定网口的资源信息
        :return: 信息
        """
        resource_path = self.get_assign_ethernet_interfaces_resource_path()
        flag, response = cli_util.direct_execute_redfish_get(self.get_assign_resource_path(), resource_path)
        return response

    def get_sp_server_info(self):
        resource_path = self.get_assign_resource_path()
        sp_server_path = "{}/SPService".format(resource_path)
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, sp_server_path)
        return response


class SystemsResourceClient:
    """
    系统资源
    """

    def __init__(self, redfish_conn, logger):
        self.redfish = redfish_conn
        self._logger = logger

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 路径
        """
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.SYSTEMS)
        path = response.get("Members")[0].get("@odata.id")
        self._logger.logInfo("assign systems resource path: {}".format(path))
        return path

    def get_assign_resource_info(self):
        """
        获取指定资源的信息
        :return: 资源信息
        """
        path = self.get_assign_resource_path()
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, path)
        return response

    def get_storage_resource_paths(self):
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.STORAGES)
        resources = response.get("Members")
        return [resource.get("@odata.id") for resource in resources], response

    def get_storage_resource_infos(self):
        original_infos = list()
        resource_paths, resource = self.get_storage_resource_paths()
        original_infos.append(str(resource))
        resource_info_dict = dict()
        for path in resource_paths:
            storage_id = path.split("/")[-1]
            flag, resource = cli_util.direct_execute_redfish_get(self.redfish, path)
            resource_info_dict[storage_id] = resource
            original_infos.append(str(resource))
        return resource_info_dict, "\n\n".join(original_infos)

    def get_storage_logical_drive_resource_infos(self):
        original_infos = list()
        resource_paths, resource = self.get_storage_resource_paths()
        original_infos.append(str(resource))
        storage_logical_drive_resources = dict()
        for path in resource_paths:
            flag, resource = cli_util.direct_execute_redfish_get(self.redfish, path + "/Volumes")
            original_infos.append(str(resource))
            for member in resource.get("Members"):
                logical_drive_path = member.get("@odata.id")
                flag, resource = cli_util.direct_execute_redfish_get(self.redfish, logical_drive_path)
                storage_logical_drive_resources[logical_drive_path] = resource
                original_infos.append(str(resource))
        return storage_logical_drive_resources, "\n\n".join(original_infos)

    def get_bios_info(self):
        """
        获取系统BIOS信息
        :return: BIOS信息
        """
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        bios_resource_path = ResourcePath.BIOS.format(resource_id)
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, bios_resource_path)
        return response.get("Attributes"), response


class SystemOverviewResourceClient:
    """
    只有Arm服务器只支持
    """

    def __init__(self, redfish_conn, logger):
        self.redfish = redfish_conn
        self._logger = logger

    def get_system_overview_info(self):
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.SYSTEM_OVERVIEW)
        return response


class ChassisResourceClient:

    def __init__(self, redfish_conn, logger):
        self.redfish = redfish_conn
        self._logger = logger

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 路径
        """
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.CHASSIS)
        path = response.get("Members")[0].get("@odata.id")
        self._logger.logInfo("assign chassis resource path: {}".format(path))
        return path

    def get_chassis_members(self):
        """
        获取所有机箱资源资源
        :return: 路径
        """
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.CHASSIS)
        return response.get("Members")

    def get_resource_by_blade(self, blade_id):
        flag, response = cli_util.direct_execute_redfish_get(self.redfish, ResourcePath.CHASSIS_BY_ID.format(blade_id))
        return response

    def get_boards_resource_info(self):
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        flag, response = cli_util.direct_execute_redfish_get(self.redfish,
                                                             ResourcePath.CHASSIS_BOARDS.format(resource_id))
        return response

    def get_main_board_resource_info(self):
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        flag, response = cli_util.direct_execute_redfish_get(self.redfish,
                                                             ResourcePath.CHASSIS_MAIN_BOARD.format(resource_id))
        return response

    def get_assign_board_infos(self, board_name):
        resource = self.get_boards_resource_info()
        if "Oem" in resource:
            return resource.get("Oem").values()[0].get("Boards").get(board_name)

    def get_fan_resource_path(self, resource_path_suffix):
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        source_url = ResourcePath.CHASSIS + "{}/" + resource_path_suffix
        return source_url.format(resource_id)
