# -*- coding: UTF-8 -*-

import re 
import common

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 硬盘容量管理检查
# 输入参数: cli
# 输出参数: 无
# 返 回 值: flag, cliRet, errMsg
# **************************************************************************** # 
def execute(cli):
    try:
        """
        1 设备版本号低于V100R001C00SPCb00，巡检不通过。
        2 如果MaxSizePerDisk的值为600G则检查通过，否则不通过。
        """
        flag = True
        lang = py_java_env.get("lang")
        logger = PY_LOGGER
        errMsg = ""
        allErrMsg = ""
        allCliRet = ""
        #步骤一：执行showdisk -logic命令，查询硬盘容量
        flag, cliRet, errMsg = checkDiskTotalUsableCapacity(cli,lang,logger)
        allErrMsg += errMsg
        allCliRet += cliRet
        if flag == True:
            return (flag, allCliRet, allErrMsg)
        
        #步骤二：执行showupgradepkginfo -t 1命令查看设备版本号，判断是否为问题版本
        flag, cliRet, errMsg = checkRiskVersion(cli,lang)
        allErrMsg += errMsg
        allCliRet += cliRet
        if flag != True:
            return (flag, allCliRet, allErrMsg)
        
        #步骤三：developer模式执行showextconf -section SPECIFICATION -item MaxSizePerDisk查看配置的单盘最大容量配置
        flag, cliRet, errMsg = checkMaxSizePerDisk(cli,lang)
        allErrMsg += errMsg
        allCliRet += cliRet
        
        return (flag, allCliRet, allErrMsg)
    except:
        logger.error("check_item_Management_Disk_Capacity, Exception:" + traceback.format_exc())
        if lang == "zh":
            errMsg = u"\n获取信息失败。"
        else:
            errMsg = "\nThere is an error in getting information."
        return (False, allCliRet, errMsg)
        
def checkDiskTotalUsableCapacity(cli,lang,logger):
    '''
    @summary: 检查硬盘已使用容量
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, cliRet, errMsg)
        flag:
            True: 使用总容量低于35660GB
            False： 使用总容量高于35660GB
        cliRet:cli命令回显
        errMsg: 错误消息
    '''
    flag = True
    errMsg = ""
    cliRet = cli.execCmd("showdisk -logic")
    list = cliRet.splitlines()
    if len(list) < 7:
        #判断cli信息是否有效
        flag = False
        if lang == "zh":
            errMsg = u"\nCli信息无效。"
        else:
            errMsg = "\nInvalid Cli information."
        return (flag, cliRet, errMsg)
    
    totalUsableCapacity = []
    for line in list[6:-2]:
        linelist = line.split()
        if len(linelist) < 2:
            continue
        
        usableCapacity = linelist[-1]
        if "Member" in line and usableCapacity.isdigit():
            totalUsableCapacity.append(int(usableCapacity))
            
    logger.info("totalUsableCapacity = %s"  %sum(totalUsableCapacity))
    if int(sum(totalUsableCapacity)) > int(35660):
        flag = False
        if lang == "zh":
            errMsg = u"\nLogic Type为Member的硬盘Usable Capacity(GB)容量和为%sGB，高于35660GB。" %sum(totalUsableCapacity)
        else:
            errMsg = "\nThe sum of Usable Capacity(GB) for all disks whose values of Logic Type are Member is %s GB and larger than 35660 GB." %sum(totalUsableCapacity)
    
    return (flag, cliRet, errMsg)

def checkRiskVersion(cli,lang):
    '''
    @summary: 检查当前版本是否为风险版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, cliRet, errMsg)
        flag:
            True: 非风险版本
            False： 风险版本
        cliRet:cli命令回显
        errMsg: 错误消息
    '''
    flag = True
    errMsg = ""
    cliRet = ""
    solutionVersion = "V100R001C00SPCb00"
    
    cliRet = cli.execCmd("showupgradepkginfo -t 1")
    lines = cliRet.splitlines()
    if len(lines) < 7:
        flag = False
        if lang == "zh":
            errMsg = u"\nCli信息无效。"
        else:
            errMsg = "\nInvalid CLI information."
        return (flag, cliRet, errMsg)
    
    # 从第7行开始，取出有效信息
    productVersion = lines[6].split()[-2]
    validVersion = common.switchValidVersion(productVersion)
    if validVersion < solutionVersion :
        flag = False
        if lang == "zh":
            errMsg += u"\n当前产品版本为%s,低于版本%s。" %(validVersion,solutionVersion)
        else:
            errMsg += "\nThe current system version is %s and earlier than %s." %(validVersion,solutionVersion)
        
    return (flag, cliRet, errMsg)

def checkMaxSizePerDisk(cli,lang):
    '''
    @summary: 单盘最大容量配置
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, cliRet, errMsg)
        flag:
            True: 单盘最大容量配置等于600
            False： 单盘最大容量配置不等于600
        cliRet:cli命令回显
        errMsg: 错误消息
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    
    if not common.changeCli2Developer(cli, py_java_env):
        if lang == "zh":
            errMsg += (u"\n进入developer模式失败，失败的原因可能为：" 
                       + u"\n（1）添加设备时未输入调试密码。"
                       + u"\n（2）添加设备时输入的调试密码无效。")
        else:
            errMsg += ("\nFailed to log in to the developer mode, the causes may be:" 
                       + "\n(1) You did not enter a debugging password when adding the device."
                       + "\n(2) The debug password entered is incorrect.")
        return (False, cliRet, errMsg)
    
    cliRet = cli.execCmd("showextconf -section SPECIFICATION -item MaxSizePerDisk")
    
    lineDict = common.getHorizontalStandardCliRet(cliRet)
    #获取单盘最大容量配置
    maxSizePerDiskItemValue = lineDict.get("Item Value")
    if not maxSizePerDiskItemValue:
        flag = False
        if lang == "zh":
            errMsg = u"\n获取单盘最大容量配置信息失败。"
        else:
            errMsg = "\nFailed to obtain the maximum capacity configuration information about a single disk."
        return (flag, cliRet, errMsg) 
        
    if maxSizePerDiskItemValue != "600":
        flag = False
        if lang == "zh":
            errMsg += u"\nMaxSizePerDisk的值为：%s。" %maxSizePerDiskItemValue
        else:
            errMsg += "\nThe value of MaxSizePerDisk is %s." %maxSizePerDiskItemValue
    
    #developer模式切换到CLI模式
    common.changeAnyModel2Cli(cli)
        
    return (flag, cliRet, errMsg)
    