#!/bin/bash

#################################################
#描述：一键式日志收集
#信息包含：FSA，vbs，mdc，osd，omm，sys，conf
#################################################

##############################
#error:error返回值
##############################
PARAMETER_WRONG=3
TASK_BLOCK=4
INVALID_PATH=5
PATH_NOT_EXIST=6

##############################################################
#公共变量
##############################################################
g_infocollect_path="/opt/dsware/tools/ops_tool/infoCollect"
g_collect_log_dir="/opt/dsware/tools/ops_tool/Collected_log_dir"
output_dir=${g_collect_log_dir}
lockFile="/opt/dsware/tools/ops_tool/infoCollect/lockFile.txt"
g_tool_kit_path="/opt/dsware/tools/ops_tool/tool_kit"
g_log_file="/opt/dsware/tools/ops_tool/infoCollect/log-collected.log"

################################################
#确保fsm测信息上传目录不存在或者被人为删除
#如果目录不存在，创建并赋值相应的权限
################################################
if [ ! -d ${g_collect_log_dir} ]
then
    mkdir -p ${g_collect_log_dir}  >/dev/null 2>&1
fi

chown omm:omm -h ${g_collect_log_dir}  >/dev/null 2>&1
chmod 750 ${g_collect_log_dir}  >/dev/null 2>&1

#清除环境中的日志。
function clean_global_log_file()
{
    if [ ! -f ${g_log_file} ]; then
        touch ${g_log_file}
        chmod 640 ${g_log_file}
        return 0
    fi
    local result=$(du -BK ${g_log_file}|awk '{print $1}'|sed 's/[K,k]//g')
    if [ ${result} -gt 1024 ]
    then
        echo -n >${g_log_file}
    fi
    chmod 640 ${g_log_file}
}

function help_manual()
{
cat <<EOF
Usage : sh CollectInfo.sh -{nodeip|poolid} ip_option -nodetype node_option -period period_option -type type_option [-nodeUserName usename_option] [output_dir]
-{nodeip|poolid}: the manager ip address or the pool from which you will collect info
  nodeip_option : supported ip format, you can specify one ip list or single ip or ip region or one type ip with special role
                : below is all format supported
                  ip1,ip2,ip3,ip4  : ip list
                  ip1-ip2,ip3-ip4  : ip region
                  all_node         : all node
                  all_vbs          : all vbs node
                  all_mdc          : all mdc node
                  all_zk           : all zk node
  poolid_option : poolid range from 0 to 127
-nodetype       : the node type
  node_option   : supported node type, you can specify one type one time
                  FSA              : storage node
                  FSM              : manager node 
-period         : time period for collect info
  period_option : supported period format must be yyyymmddHHMM-yyyymmddHHMM.
                  When the type is Collect_sys_cfg,Collect_mem_all,Collect_mem_vbd,Collect_mem_osd,Collect_mem_mdc,
                  Collect_meta_all,Collect_meta_vbs,Collect_meta_mdc,Collect_meta_osd
                  we may not use period_option.
-type           : supported info types
  type_option   : you can specify one type or multi types with ","
                  Collect_mod_all         : collect running log of all module which include vbs,mdc,osd and fsa.
                                            this option can't be used together with Collect_mod_vbs,Collect_mod_mdc,
                                            Collect_mod_osd and Collect_mod_fsa
                  Collect_mod_vbs         : collect running log(log-VBS-*) of vbs
                  Collect_mod_mdc         : collect running log(log-MDC*) of mdc
                  Collect_mod_osd         : collect running log(log-OSD-*) of osd
                  Collect_mod_fsa         : collect running log(dsware_agent.log) of fsa
                  Collect_sys_all         : collect system env info and config file
                                            this option can't be used together with Collect_sys_env and Collect_sys_cfg
                  Collect_sys_env         : collect system ennv info which include iostat,top,sar,vmstat
                  Collect_sys_cfg         : collect config info which include db table and config file
                  Collect_mem_all         : collect important info of process which saved in memory. these info include below:
                                            mdc : topo,ptview,ioview,poolmaping
                                            vbs : io statistics,attach point,important map
                                            osd : cache hit ratio,rsm task queue,io statistics
                                            fsm :the info in fsm memory
                                            fsa :the info  in fsa memory
                                            this option can't be used together with Collect_mem_vbs,Collect_mem_osd
                                            and Collect_mem_mdc
                  Collect_mem_vbs         : collect important info of vbs process
                  Collect_mem_osd[_nid]         : collect important info of osd process
                  Collect_mem_mdc         : collect important info of mdc process
                  Collect_core_all        : collect the core file. this option can't be used together with Collect_core_dsw
                                            and Collect_core_omm
                  Collect_core_dsw        : collect the core file of mdc,vbs and osd
                  Collect_core_omm        : collect the core file of omm
                  Collect_meta_all        : collect all metadata of mdc,vbs and osd, but metada maybe very big 
                                            so please collect necessary meta only. and this option can't be used together with
                                            Collect_meta_osd_nid, Collect_meta_mdc and Collect_meta_vbs_treeid
                  Collect_meta_osd[_NID]  : collect osd metadata, you can specify one osd with NID
                  Collect_meta_mdc        : collect mdc metadata
                  Collect_meta_vbs[_TREE] : collect vbs metadata, you can specify one meta tree with TREE
                  Collect_match_REG       : specify log file name with reg expression and collect special file
                  Collect_os_all          : collect all log and diagnose info of os platform, including infrastructure modules 
                                            and replication modules
                  Collect_lld_vol_[VOL-NAME]    : collect lazyloading vol info, the param VOL-NAME is necessary 
                  Collect_lld_img_[IMAGE-ID]    : collect lazyloading image info, the param IMAGE-ID in decimal is necessary 
-nodeUserName   :if want to Collect Standby_fsm_node ,we should enter the node_user_name.
    usename_option:The usename is user name of Standby_fsm_node.
output_dir      :the directory for storing files
example:
    sh CollectInfo.sh -nodeip 192.168.0.2 -nodetype FSA -period 201603090000-201603100000 -type Collect_sys_cfg {output_dir}
    sh CollectInfo.sh -poolid 0 -nodetype FSA -period 201603090000-201603100000 -type Collect_sys_cfg {output_dir}
    sh CollectInfo.sh -nodeip 192.168.0.125-128 -nodetype FSA -period 201603090000-201603100000 -type Collect_mod_all {output_dir}
    sh CollectInfo.sh -nodeip all_vbs -nodetype FSA -period 201603090000-201603100000 -type Collect_mod_vbs {output_dir}
    sh CollectInfo.sh -nodeip all_vbs -nodetype FSA -period 201603090000-201603100000 -type Collect_match_user_vbs_meta {output_dir}
    sh CollectInfo.sh -nodeip all_mdc -nodetype FSA -period 201603090000-201603100000 -type Collect_match_user_mdc_view {output_dir}
    sh CollectInfo.sh -nodeip all_node -nodetype FSA -period 201603090000-201603100000 -type Collect_mod_all {output_dir}
    sh CollectInfo.sh -nodeip all_node -nodetype FSA -period 201603090000-201603100000 -type Collect_mod_all,Collect_sys_all,Collect_mem_all {output_dir}
    sh CollectInfo.sh -nodeip 192.168.0.2 -nodetype FSA -type Collect_lld_vol_LLD_VOL_NAME {output_dir}
    sh CollectInfo.sh -nodeip 192.168.0.2 -nodetype FSA -type Collect_lld_img_2001992 {output_dir}
EOF
}

#trace输出函数，只向log文件输出日志而不向终端输出
function output_trace()
{
    local result=$1
    shift 1
    local message=$@
    local pos_func=${FUNCNAME[1]}
    local pos_lineno=${BASH_LINENO}

    _WriteLogBase "no" ${result} ${pos_func} ${pos_lineno} "$message" ${g_log_file}
}

#LOG输出函数调用的内部函数
function _WriteLogBase()
{
    local out_flag=$1
    local result=$2
    local func=$3
    local lineno=$4
    local message=$5
    local log_file=$6
    local time=`date +%Y-%m-%d-%H:%M:%S:%N`

    local fmt_time=$(printf "[%s]" "$time")
    local fmt_result=$(printf "[%-8s]" "$result")
    local fmt_pos=$(printf "[Func:%-15s  LineNo:%-4s]" "$func" "$lineno")
    local fmt="%s %s %s %s\n"
    local all_str=$(printf "$fmt" "$fmt_time" "$fmt_result" "$fmt_pos" "${message}")

    if [ "x$out_flag" = "xyes" ]
    then
        echo -e "$all_str" |tee -a ${log_file}
    else
        echo -e "$all_str"  >>${log_file}
    fi
}

#检查输入IP的合法性
function checkip()
{
    if [ "$1" == "all_node" ]
    then
        all_node_ip_list=($(sh ${g_tool_kit_path}/get_all_node_ip.sh))
        nodeip=$(echo ${all_node_ip_list[@]} |sed 's/ /,/g')    
        return 0
    elif [ "$1" == "all_vbs" ]
    then
        all_vbs_ip_list=($(sh ${g_tool_kit_path}/get_all_vbs_ip.sh))
        nodeip=$(echo ${all_vbs_ip_list[@]} |sed 's/ /,/g')
        return 0
    elif [ "$1" == "all_mdc" ]
    then
        all_mdc_ip_list=($(sh ${g_tool_kit_path}/get_all_mdc_ip.sh))
        nodeip=$(echo ${all_mdc_ip_list[@]} |sed 's/ /,/g')
        return 0
    elif [ "$1" == "all_zk" ]
    then
        all_zk_ip_list=($(sh ${g_tool_kit_path}/get_all_zk_ip.sh))
        nodeip=$(echo ${all_zk_ip_list[@]} |sed 's/ /,/g')
        return 0
    fi
    
    #将IP段展开
    iparray=$1
    iparray=${iparray//,/ }
    nodeip=()
    ip_cnt=0
    for ip in $iparray
    do
        nodeip[$ip_cnt]=$ip
        ((ip_cnt++))
    done

    #构造以逗号分割的IP列表
    nodeip=$(echo ${nodeip[@]} |sed 's/ /,/g')
    
    return 0
}

function getpoolip()
{
    local poolid=$1
    # ?â?ï²»?£?é³?IDµ????§·¶?§£¬±??â³??©?¹ºó??·¨?é?¤?¨¹ý¡£?ç¹û³?ID³¬³ö???§·¶?§£¬?òDI?ü?î»á·µ»?´í?ó?á?¾
    local is_digital=$(echo $poolid |sed 's/[0-9]//g')
    if [ "$is_digital" != "" ]
    then
        echo "poolid is illegal."
        return 1
    fi

   if [ ${poolid} -lt 0 -o ${poolid} -gt 127 ]
   then
       echo "pool id para error."
       echo "usage: sh CollectInfo.sh -h"
       exit 1
   fi

    local timetag=`date +%Y%m%d_%H%M%S_%N`
    local tmp_log=${g_infocollect_path}/"tmp_collect_info_${timetag}.log"
    sh ${g_tool_kit_path}/get_pool_all_node_ip.sh $poolid 1 > ${tmp_log}
    local error_num=`cat ${tmp_log} | grep -i error | wc -l`
    if [ $error_num -ne 0 ]
    then
        cat ${tmp_log}
        rm -f ${tmp_log}
        return 1       
    fi

    local all_pool_ip_list=($(cat ${tmp_log} | grep -v -i "ip"))
    nodeip=$(echo ${all_pool_ip_list[@]} |sed 's/ /,/g') 

    if [ "$nodeip" == "" ]
    then
        echo "The pool ${poolid} has no node"
        rm -f ${tmp_log}
        return 1
    fi
   
    rm -f ${tmp_log}
    return 0
} 

#检查输入时间的合法性
function checkperiod()
{
    local time=$1
    beginTime=$(echo $time|awk -F "-" '{print $1}')
    endTime=$(echo $time |awk -F "-" '{print $2}')

    local isdigital_1=$(echo $beginTime |sed 's/[0-9]//g')
    local isdigital_2=$(echo $endTime |sed 's/[0-9]//g')
    if [ "$isdigital_1" != "" -o "$isdigital_2" != "" ]
    then
        echo "period time is illegal."
        return 1
    fi

    if [ $endTime -lt $beginTime ]
    then
        echo "period time is illegal."
        return 1
    fi
    
    return 0
}

#检查收集类型的合法性
function check_type()
{
    local items=$1
    local items=${items//,/ }

    for item in $items
    do
        local string=$item
        local result=$(echo $string|grep "Collect_")
        if [ "$result" = "" ]
        then
            echo "$string is illegal."
            return 1
        fi
        
        local tem_string=$(echo $string|sed 's/Collect_//')

        result=$(echo $tem_string|awk -F "_" '{print $1}')
        if [ "$result" != "lld" -a "$result" != "os" -a "$result" != "mod" -a "$result" != "sys" -a "$result" != "mem" -a "$result" != "core" -a "$result" != "meta" -a "$result" != "subhealth" ]
        then
            if [ "$result" = "match" ]
            then
                return 0
            fi
            echo "$string is illegal."
            return 1
        fi
        
        if [ "$result" = "lld" ];then
            tem_string=$(echo $tem_string|sed 's/'$result'_//')
            result=$(echo $tem_string|awk -F "_" '{print $1}')
            if [ "$result" != "vol" -a "$result" != "img" ];then
                echo "$string is illegal."
                return 1
            fi
            return 0
        fi

        tem_string=$(echo $tem_string|sed 's/'$result'_//g')
        result=$(echo $tem_string|awk -F "_" '{print $1}')
        if [ "$result" != "all" -a "$result" != "vbs" -a "$result" != "mdc" -a "$result" != "osd" -a "$result" != "fsa" -a "$result" != "omm" -a "$result" != "fsm" -a "$result" != "dsw" -a "$result" != "env" -a "$result" != "cfg" ]
        then
            echo "$string is illegal."
            return 1
        fi
        
        tem_string=$(echo $tem_string|sed 's/'$result'//g')
        if [ "$tem_string" != "" ]
        then
            tem_string=$(echo $tem_string|sed 's/_//g')
            local isdigital=$(echo $tem_string|sed 's/[0-9]//g')
            if [ "$tem_string" != "" -a "$isdigital" != "" ]
            then
                echo "$string is illegal."
                return 1
            fi
        fi
    done
    
    return 0
}

#检查type之间的冲突性，如Collect_mod_all与Collect_mod_vbs之间有冲突
function check_type_conflict()
{
    if echo $1 |grep Collect_mod_all > /dev/null
    then
        if echo $1 |grep -E "Collect_mod_vbs|Collect_mod_mdc|Collect_mod_osd|Collect_mod_fsa" > /dev/null
        then
            echo "option Collect_mod_all conflict with other option"
            return 1
        fi
    fi
   
    if echo $1 |grep Collect_sys_all > /dev/null
    then
        if echo $1 |grep -E "Collect_sys_env|Collect_sys_cfg" > /dev/null
        then
            echo "option Collect_sys_all conflict with other option"
            return 1
        fi
    fi
   
    if echo $1 |grep Collect_mem_all > /dev/null
    then
        if echo $1 |grep -E "Collect_mem_mdc|Collect_mem_osd|Collect_mem_vbs" > /dev/null
        then
            echo "option Collect_mem_all conflict with other option"
            return 1
        fi
    fi

    if echo $1 |grep Collect_core_all > /dev/null
    then
        if echo $1 |grep -E "Collect_core_dsw|Collect_core_omm" > /dev/null
        then
            echo "option Collect_core_all conflict with other option"
            return 1
        fi
    fi
    
    if echo $1 |grep Collect_meta_all > /dev/null
    then
        if echo $1 |grep -E "Collect_meta_mdc|Collect_meta_vbs|Collect_meta_osd" > /dev/null
        then
            echo "option Collect_meta_all conflict with other option"
            return 1
        fi
    fi
    
    return 0
}

#检查输入参数的合法性
function checkparameter()
{   
    #check ip address
    if [ "$1" != "-nodeip" -a "$1" != "-poolid" ]
    then
        echo "the parameter is illegal."
        return 1
    fi

    if [ "$1" == "-nodeip" ]
    then
        checkip $2
        [ $? -ne 0 ] && return 1
    else
        getpoolip $2
        [ $? -ne 0 ] && return 1
    fi
    
    #check nodetype
    if [ $3 != "-nodetype" ]
    then
        echo "B_nodetype is illegal."
        return 1
    fi
    
    if [ $4 != "FSA" -a $4 != "FSM" ]
    then
        echo "E_nodetype is illegal."
        return 1
    fi
    if [ $para_num -eq 6 -o $para_num -eq 7 ]
    then
    	#check type
    	if [ $5 != "-type" ]
    	then
            echo "type is illegal."
            return 1
    	fi
        if [ $need_period -eq 1 ]
        then
            echo "this type need -peroid paramater"
            return 1
        fi
        period="201603090000-201603100000"
        types="$6"  
    	check_type $6
    	[ $? -ne 0 ] && return 1

    	check_type_conflict $6
    	[ $? -ne 0 ] && return 1
    fi
	
    if [ $para_num -eq 8 -o $para_num -eq 9 ]
    then
    	if [ $5 != "-period" -a $5 != "-type" ]
        then
            echo "parameter 7 must be -period or -type."
            return 1
        fi
    	#check period time
    	if [ $5 = "-period" ]
    	then
            period=$6
    	    checkperiod $6
    	    [ $? -ne 0 ] && return 1
            
            if [ $7 != "-type" ]
            then
                echo "type is illegal."
                return 1
            fi
            types=${8}
            check_type ${8}
            [ $? -ne 0 ] && return 1
    
            check_type_conflict ${8}
            [ $? -ne 0 ] && return 1
    	fi
    

	#check type
    	if [ $5 = "-type" ]
    	then
            if [ $need_period -eq 1 ]  
            then
                echo "we need peroid paramater"
            return 1
            fi
            period="201603090000-201603100000"
            types=$6
            check_type $6
            [ $? -ne 0 ] && return 1
                
            check_type_conflict $6
            [ $? -ne 0 ] && return 1
       
            if [ $7 != "-nodeUserName" ]
            then
                echo "type nodeUserName is illegal."
            fi
            nodeUserName=${8}
    	fi
    fi

	
    if [ $para_num -eq 10 -o $para_num -eq 11 ]
    then
        if [ $5 != "-period" ]	
        then
            echo "period  is illegal."
            return 1
        fi
        period=$6
        checkperiod $6
        [ $? -ne 0 ] && return 1
            
        if [ $7 != "-type" ]
        then
            echo "type is illegal"
        fi
        types=${8}
        check_type ${8}
        [ $? -ne 0 ] && return 1

        check_type_conflict ${8}
        [ $? -ne 0 ] && return 1

        if [ "${9}" != "-nodeUserName" ]
        then
            echo "username is illegal."
            return 1
        fi
        nodeUserName=${10}
    fi
    return 0
}

#确认是否为手动直接调用信息收集功能
function check_isCalled_by_Manual()
{
    local parent_name=$(ps -o args $PPID)
    local result=$(echo ${parent_name}|awk '{print $2}')
    if [ "${result}" = "bash" -o "${result}" = "-bash" ]
    then
        return 1
    else
        return 0
    fi
}

#校验目录
function check_output_dir_valid()
{
    local output_dir=$1
    if [ "${output_dir}" = "" ]
    then
    	return 1
    fi
    local parent_name=$(ps -o args $PPID)
    local result=$(echo ${parent_name}|awk '{print $2}')
    if [ "${result}" = "bash" -o "${result}" = "-bash" ]
    then
        return 0
    fi
    
    if [ ${output_dir} = ${g_collect_log_dir} -o ${output_dir} = "${g_collect_log_dir}/" ]
    then
        echo  "the path[${output_dir}] is invalid"
        return 1
    fi
    return 0
}

#清空默认路径下的文件信息
del_files_collected_before()
{
    filelist=$(ls ${g_collect_log_dir}|grep "tar.gz$")
    for file in ${filelist}
    do
        rm -f ${g_collect_log_dir}/${file}  >/dev/null 2>&1
    done
}

function fun_main()
{
    if [ $# -eq 1 ]
    then
        if [ "$1" == "--help" -o $1 == "-h" ]
        then
            help_manual
            return 1
        fi
    fi


    if [ $# -lt 6 -o $# -gt 11 ]
    then
        echo "wrong input parameters."
        echo "sh CollectInfo.sh -h"
        return 1
    fi

    para_num=$#
    nodeip=$2
    nodetype=$4
    period=""
    types=""
    nodeUserName=""
    type_string=$(echo $@|grep -E "mod|sys_all|sys_env|core|match"|wc -l)
    if [ ${type_string} -ne 0 ]
    then
        need_period=1
    else
        need_period=0
    fi
    
    #校验参数
    checkparameter $@
    [ $? -ne 0 ] &&  return 1
    
    #赋值转储路径
    if [ ${para_num} -eq 7 -o ${para_num} -eq 9 -o ${para_num} -eq 11 ]
    then
        output_dir=$(echo $@|awk '{print $NF}')
        [ ! -d ${output_dir} ] && echo "output_dir is is not a directory." && return ${PATH_NOt_EXIST}
    fi
    
    #参数检查，目录指定，只有父进程-bash才能使用路径：/opt/dsware/tools/ops_tool/Collected_log_dir
    check_output_dir_valid $output_dir
    status=$?
    if [ ${status} -ne 0 ]
    then
        output_trace "ERROR" "the input para is invalid."
        return ${INVALID_PATH}
    fi
    
    #判断日志文件是否已满，超过1M清除。
    clean_global_log_file
    
    #手动调用需要判断是否需要删除默认路径下的文件
    default_flag=0
    result=$(ls ${g_collect_log_dir}|grep "tar.gz$")
    check_isCalled_by_Manual
    manual_operation=$?
    [ "${output_dir}" = "${g_collect_log_dir}" -o "${output_dir}" = "${g_collect_log_dir}/" ] && default_flag=1
    if [ ${manual_operation} -ne 0 -a "${result}" != "" -a ${default_flag} -eq 1 ]
    then
        echo -n "Do you want to delete the previously collected files?(y/n):"
        read user_choice
        if [ "$user_choice" != "yes" -a "$user_choice" != "y" -a "$user_choice" != "no" -a "$user_choice" != "n" ]
        then
            echo "input choice is wrong type."
            return "${PARAMETER_WRONG}"
        fi
        
        if [ "$user_choice" = "yes" -o "$user_choice" = "y" ]
        then 
            del_files_collected_before
        fi
    fi
    
    result=0
    task_num=0
    
    file_date=$(date +%Y-%m-%d_%H-%M-%S)
    nodeip=$(echo ${nodeip}|sed 's/,/ /g')
    echo "start to collect infomation."
    output_trace "INFO" "start to collect infomation."

    adminpwd=`cat /opt/dsware/manager/webapps/dsware/WEB-INF/manager.properties | grep "usrPwd=" | sed 's/usrPwd=//'`
    user=`cat /opt/dsware/manager/webapps/dsware/WEB-INF/manager.properties | grep "user=" | grep -v "hauser=" |  sed 's/user=//'`

    #$nodeUserName不为空，则发给备FSM节点；为空，发给FSA节点或者主FSM节点
    if [ -z "$nodeUserName" ] 
    then
        for ip in ${nodeip};do
            while [ 1 ];do
                result=$(ps -efww|grep "dswareTool.sh"|grep -E "infoCollectorToFsm|infoCollector"|grep -v "grep"|wc -l)
                #10个进程后台并发，一个进程ps出两条结果
                if [ ${result} -lt 20 ]
                then
                    task_num=$((${task_num}+1))
                    #上传之前要把原来的文件删掉
                    rm -f ${g_collect_log_dir}/${nodetype}_${ip}.tar.gz >/dev/null 2>&1
                    su - dsware -c "export DswareTool_username=${user};export DswareTool_password=${adminpwd};sh /opt/dsware/client/bin/dswareTool.sh --op infoCollectorToFsm -nodeIp $ip -nodeType $nodetype -period $period -items $types 100>&- &"  >>${g_infocollect_path}/Collectinfo_result_${file_date}.txt
                    break
                fi
                sleep 5
            done
        done 
    else
        while [ 1 ];do
            #并发控制，可能有人手动收集信息调用dswareTool.sh
            result=$(ps -efww|grep "dswareTool.sh"|grep -E "infoCollectorToFsm|infoCollector"|grep -v "grep"|wc -l)
            if [ ${result} -lt 20 ]
            then
                task_num=$((${task_num}+1))
                rm -f ${g_collect_log_dir}/${nodetype}_${ip}.tar.gz >/dev/null 2>&1
                su - dsware -c "export DswareTool_username=${user};export DswareTool_password=${adminpwd};sh /opt/dsware/client/bin/dswareTool.sh --op infoCollectorToFsm -nodeIp $nodeip -nodeType $nodetype -period $period -items $types  -nodeUserName $nodeUserName 100>&-"  
                if [ $? -ne 0 ]
                then
                    output_trace "ERROR" "failed to collected info from standby fsm node."
                fi
                break
            fi
            sleep 5     
        done    
    fi
    
    if [ -z "$nodeUserName" ]  
    then
        #每三秒查询一次后台任务是否执行完
        while [ 1 ]
        do
            tmp_num=$(cat ${g_infocollect_path}/Collectinfo_result_${file_date}.txt|grep "DswareTool operation end"|wc -l)
            #proc_num为了防止返回信息丢失导致死循环
            proc_num=$(ps -efww|grep "dswareTool.sh"|grep -E "infoCollectorToFsm|infoCollector"|grep -v "grep"|wc -l)  
            if [ ${tmp_num} -eq ${task_num} -o ${proc_num} -eq 0 ]
            then
                break;
            fi
            sleep 3
        done

        #在屏幕上显示结果
        cat ${g_infocollect_path}/Collectinfo_result_${file_date}.txt|grep -v "DswareTool operation start"|grep -v "DswareTool operation end"
        cat ${g_infocollect_path}/Collectinfo_result_${file_date}.txt|grep -v "DswareTool operation start"|grep -v "DswareTool operation end"|grep -v "Operation finish" >>${g_log_file}
        #删除临时文件
        rm -f ${g_infocollect_path}/Collectinfo_result_*.txt >/dev/null 2>&1
    fi
    
    #将上传文件移植指定目录
    if [ ${output_dir} != ${g_collect_log_dir} ]
    then
        for ip in ${nodeip}
        do
            mv -f ${g_collect_log_dir}/${nodetype}_${ip}.tar.gz ${output_dir}  >/dev/null 2>&1
        done
    fi
    
    #如果是默认路径，就将本次收集的信息打包成一个tar包
    if [ ${output_dir} = ${g_collect_log_dir} ]
    then
        cd ${g_collect_log_dir}  >/dev/null 2>&1
        for ip in ${nodeip}
        do
            if [ -f "${nodetype}_${ip}.tar.gz" ]
            then
                tar -rvf ${g_collect_log_dir}/CollectInfo_${file_date}.tar ${nodetype}_${ip}.tar.gz  >/dev/null 2>&1
                chmod 440 ${g_collect_log_dir}/CollectInfo_${file_date}.tar
                rm -f ${g_collect_log_dir}/${nodetype}_${ip}.tar.gz  >/dev/null 2>&1
            fi
        done
        cd - >/dev/null 2>&1
    fi
      
    echo "End to collect information"
    output_trace "INFO" "End to collect information"
    #红色前景蓝色背景
    echo -e "\033[31;44;1m [Please delete the Collected files after using them.] \033[0m"

    return $?
}

#CollectInfo.sh程序只能一个一个进行
function flock_check()
{
    {
        flock -no 100
        if [ $? -eq 1 ];then
            echo "Already has one Collect task run."
            exit ${TASK_BLOCK}
        fi
        
        fun_main $@
        local result=$?
        rm -f ${lockFile} >/dev/null 2>&1
        return $result
        
    } 100<>${lockFile}
}

flock_check $@
exit $?


