#!/bin/bash

#############################################################
#
# 名  称：	检查服务器PMC背板和日立盘兼容性
# 错误码：
#         0000 无兼容性风险
#         0001 存在兼容性风险
#
#############################################################
UPG_LIB_PATH="/opt/omm/oma/atoms/Inspect/lib"
PATH="/sbin:/usr/sbin:/usr/local/sbin:/root/bin:/usr/local/bin:/usr/bin:/bin:/opt/omm/oma/workspace/tools"
source "${UPG_LIB_PATH}/log.sh" || { echo "source ${UPG_LIB_PATH}/log.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/out_put.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/out_put.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/version.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/version.sh failed."; exit 130; }

checkItemId="4423"
resultCode=0
errorKey=""
params=""
originalInfo=""
#存储节点:storage; 计算节点:compute; 管理节点:management
agentMonitorFile="/opt/dsware/agent/conf/agentMonitor"

#脚本入参，节点类型
NODE_TYPE=$1

# --------------------------------------------------------------------------- #
#函数名：judge_whether_storage_node
#用  途: 判断服务器是否是存储节点。
#说  明：通过查询背板network_config.ini文件查询是否为存储节点
# --------------------------------------------------------------------------- #
function judge_whether_storage_node()
{
    # 监控文件不存在
    if [ ! -f ${agentMonitorFile} ]
    then
        return 1
    fi
    
    local CNT_MONITOR=$(cat ${agentMonitorFile}  | grep osd |grep -v alarm | grep -E "True" | wc -l)
    if [ "${CNT_MONITOR}" == "0" ]
    then
        # 当前节点没有OSD
        return 1
    fi
    
    return 0
}

# --------------------------------------------------------------------------- #
#函数名：judge_whether_storage_proprietary_dev
#用  途: 判断是否是存储专有硬件
#说  明：判断是否是存储专有硬件，这里主要指三大洋的设备
# --------------------------------------------------------------------------- #
function judge_whether_storage_proprietary_dev()
{

	# 检查是否为太平洋/大西洋/北冰洋设备 #
	check_Pacific_envir=$(dmidecode -t2 | grep "STL6SPCM")
	check_Atlantic_envir=$(dmidecode -t2 | grep "STL6SPCN")
	check_Arctic_envir=$(dmidecode -t2 | grep "STL6SPCO")

	if [ ! -z "$check_Pacific_envir" ] || [ ! -z "$check_Atlantic_envir" ] || [ ! -z "$check_Arctic_envir" ]; then
		return 0
	fi
	
	return 1
}

# --------------------------------------------------------------------------- #
#函数名：judge_risk_of_disk_fw
#用  途: 判断硬盘fw版本
#说  明：判断硬盘fw版本是否存在问题
# --------------------------------------------------------------------------- #
function judge_risk_of_disk_fw()
{
	local disk_6T_type_cnt=0
	local disk_10T_type_cnt=0
	# 检查版本是否为 930 950 980#
	for disk_fw in "930" "950" "980"
	do
		# 6T盘在环境上有风险版本的数量 #
		disk_6T_type_cnt=`lsscsi -g | grep "HUS726T6TALE600" | awk -F " " '{print $5}' | grep $disk_fw | wc -l 2>/dev/null`
		# 10T盘在环境上有风险版本的数量 #
		disk_10T_type_cnt=`lsscsi -g | grep "WUS721010ALE6L4" | awk -F " " '{print $5}' | grep $disk_fw | wc -l 2>/dev/null`
		if [ "${disk_6T_type_cnt}" != "0" -o "${disk_10T_type_cnt}" != "0" ]; then
			return 1
		fi
	done
	return 0	
 
}

# --------------------------------------------------------------------------- #
#函数名：is_ipmitool_load
#用  途: 检查ipmitool工具
#说  明：检查ipmitool工具是否有加载驱动
# --------------------------------------------------------------------------- #
function is_ipmitool_load()
{
	# 检查驱动是否存在 #
	ipmitool lan print 2>ipmitool_temp_check.txt 1>&2
	check_ipmitool_error=`cat ipmitool_temp_check.txt | grep "not open" | grep "/dev/ipmi"`
	if [ ! -z "$check_ipmitool_error" ]; then
		rm -f ipmitool_temp_check.txt
		return 0
	fi

	rm -f ipmitool_temp_check.txt
	return 1
}

# --------------------------------------------------------------------------- #
#函数名：judge_backplane_type_by_boardid
#用  途: 判断服务器背板型号。
#说  明：通过查询背板boardId来判断是否是存在兼容性问题的设备背板
# --------------------------------------------------------------------------- #
function judge_backplane_type_by_boardid()
{
	local boardid=0;

	for((i=0; i<256; i++))
	do
		FruValue=`printf "0x%x" $i`
		boardid=`ipmitool raw 0x30 0x90 0x29 $FruValue 0x2 2>/dev/null`
		if [ $? -ne 0 ]; then
			continue
		fi
		case $boardid in
		#RH2288 V3/2288H V5/taishan 2280/taishan 2280v2 12盘背板
		" 73")
			return 1
			;;
		#5288X 36盘背板
		" 83")
			return 2
			;;
		*)
			;;
		esac
	done
	return 0
}

# --------------------------------------------------------------------------- #
#函数名: main
#用  途: 巡检项主函数，判断环境是否存在PMC+日立盘配套问题
#说  明: 判断是否有PMC背板配套日立盘，并且存在兼容性问题
# --------------------------------------------------------------------------- #
function main()
{
	local expander_version=""
	local server_type=""
	
	if [ "${NODE_TYPE}" == "Agent" ]; then
		agent_items_check ${checkItemId}
		FS_json_output=FSA_json_output
		if [ $? -ne 0 ]; then
		    log INFO "${checkItemId} do not select, pass"
		    return 0
		fi
	elif [ "${NODE_TYPE}" == "Manager" ]; then
		manager_items_check ${checkItemId}
		FS_json_output=FSM_json_output
		if [ $? -ne 0 ]; then
		    log INFO "${checkItemId} do not select, pass"
		    return 0
		fi
	fi
	
	# 检查是否为服务器设备 #
	server_type=`dmidecode -t2 | grep "Manufacturer" | awk -F ":" '{print $2}' | sed 's/^[ ]*//g'`
	if [ "${server_type}" != "Huawei" ]; then
		log INFO "It's not huawei server, no need check."
		originalInfo="It's not huawei server, no need check."
		${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
		return 0
	fi
	
	# 检查是否是三大洋设备 #
	judge_whether_storage_proprietary_dev
	if [ $? -eq 0 ]; then
		log INFO "It's storage proprietary device, no need check."
		originalInfo="It's storage proprietary device, no need check."
		${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
		return 0
	fi
	
	# 检查是否是存储节点 #
	judge_whether_storage_node
	if [ $? -ne 0 ]; then
		log INFO "Not storage node, no need check."
		originalInfo="Not storage node, no need check."
		${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
		return 0
	fi

	# 检查ipmitool的工具 #
	is_ipmitool_load
	if [ $? -ne 1 ]; then
		log INFO "Ipmitool is not load, skip check."
		originalInfo="Ipmitool is not load, skip check."
		${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
		return 0
	fi
	# 检查是否是pmc 12盘/36盘背板
	judge_backplane_type_by_boardid
	ret=$?
	if [ $ret -eq 1 ]; then
		# 判断expander版本 #
		expander_version=`lsscsi -g | grep "enclosu" | grep "Expander" | awk -F " " '{print $6}' 2>/dev/null`
		if [ "${expander_version}" != "331" ]; then
			judge_risk_of_disk_fw
			if [ $? -ne 0 ]; then
				resultCode=1
				log ERROR "There's compatibility risks in 12 disk backplane."
				originalInfo="There's compatibility risks in 12 disk backplane."
				errorKey="${checkItemId}0001"
				${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
				return 1
			fi	
		fi
	elif [ $ret -eq 2 ]; then
		judge_risk_of_disk_fw
		if [ $? -ne 0 ]; then
			resultCode=1
			log ERROR "There's compatibility risks in 36 disk backplane."
			originalInfo="There's compatibility risks in 36 disk backplane."
			errorKey="${checkItemId}0001"
			${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
			return 1
		fi
	fi

	${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
	return 0
}

log MUST "enter [$0],para=[$@]"
main $@
retValue=$?
log MUST "leave [$0],retValue=${retValue}"
exit ${retValue}
