#!/bin/bash

########################################################################################
# 描述：检查检查环境证书有效期
#  错误码：
#         0000 正常
#         0001 证书有效时间大于3年，并在2034年12月30日前到期，建议优化
#         0002 证书剩余有效时间小于3年，检查不通过 
########################################################################################
UPG_LIB_PATH="/opt/omm/oma/atoms/Inspect/lib"
PATH="/sbin:/usr/sbin:/usr/local/sbin:/root/bin:/usr/local/bin:/usr/bin:/bin:/opt/omm/oma/workspace/tools"
LOG_FILE="/var/log/dsware/fusionstorage_inspect.log"

source "${UPG_LIB_PATH}/func.sh" >> ${LOG_FILE} 2>&1 || { echo "source ${UPG_LIB_PATH}/version.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/out_put.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/out_put.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/version.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/version.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/log.sh" || { log ERROR "source ${UPG_LIB_PATH}/log.sh failed."; exit 130; }

declare -r checkItemId="4430"
resultCode=0
errorKey=""
params=""
originalInfo=""
OK_LIST=""
HALF_OK_LIST=""
UNOK_LIST=""
#脚本入参，节点类型
declare -r NODE_TYPE=$1

declare -r KEY_TOOL="${OMS_RUN_PATH}/jre/bin/keytool"

declare -r TOMCAT_SSL_ROOT_CERT="${OMS_RUN_PATH}/workspace/webapps/dsware/WEB-INF/tomcat_client.key"
declare -r TOMCAT_SSL_ROOT_PASSWD_CONF="${OMS_RUN_PATH}/workspace/webapps/dsware/WEB-INF/manager.properties"

declare -r TOMCAT_XML="${OMS_RUN_PATH}/workspace/tomcat/conf/server.xml"
declare -r SSL_CONFIG="${OMS_RUN_PATH}/workspace/conf/swms/config/application.properties"
declare -r FMWS_CONF_DIR="${OMS_RUN_PATH}/workspace/webapps/fm/WEB-INF/config/"
declare -r FMWS_CERT_CONF="${FMWS_CONF_DIR}/certification.properties"

declare -r OMA_PATH="/opt/omm/oma"
declare -r OMA_SERCFGFILE="${OMA_PATH}/workspace/conf/servercfg.cfg"

declare -r FSA_SSL_ROOT_CERT_FILE="/opt/omm/oms/workspace/webapps/dsware/WEB-INF/fsm_server.cer"
declare -r FSA_SSL_SELF_CERT_FILE="/opt/dsware/agent/conf/ssl/fsa_server.cer"

declare -r HA_SSL_SELF_CERT_FILE="${OMS_RUN_PATH}/workspace/ha/local/cert/server.crt"
declare -r HA_SSL_ROOT_CERT_FILE="${OMS_RUN_PATH}/workspace/ha/local/cert/root-ca.crt"

declare -r VBS_API_CERT_FILE="/opt/dsware/vbs/conf/ssl/api_gatewayca.crt"
declare -r VBS_IAM_CERT_FILE="/opt/dsware/vbs/conf/ssl/iam_ca.crt"

declare -r CURRENT_TIME=$(date +'%s')
# 当前时间 + 3年
declare -r AFTER_DATY_3_YEAY_TIME=$((${CURRENT_TIME}+94608000))
# 2034-12-30 00:00:01（int最大值为2,147,483,647，shell会自动转换）
declare -r AFTER_DATY_2034_TIME=2051020800



function addResultList()
{
    local certificate_name=$1
    local listType=$2
    case "$listType" in
        ok)
            OK_LIST="$OK_LIST""$certificate_name, "
            ;;
        half_ok)
            HALF_OK_LIST="$HALF_OK_LIST""$certificate_name, "
            ;;
        unok)
            UNOK_LIST="$UNOK_LIST""$certificate_name, "
            ;;
    	*)
            log "ERROR" "never be there. function addResultList $certificate_name $listType"
            ;;
    esac  
}

##############################################################
## checkKeyCertByKMC.
##############################################################
function checkKeyCertByKMC()
{
    local certificate=$1
    local passwd_file=$2
    local passwd_key=$3
    local certificate_name=$4

    local encryptd_passwd=$(cat ${passwd_file} | grep "^$passwd_key" | awk -F "$passwd_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${passwd_file} | grep \\\"^$passwd_key\\\" | awk -F \\\"$passwd_key=\\\" '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Result: encryptd_passwd=\$encryptd_passwd\n"
    
    if [ -z "${encryptd_passwd}" ];then
        originalInfo="$originalInfo""$certificate_name encryptd password is null.\n"
        log "ERROR" "$certificate_name encryptd password is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    passwd=$(decrptByKMC ${encryptd_passwd})

    if [ -z "${passwd}" ];then
        originalInfo="$originalInfo""$certificate_name password is null.\n"
        log "ERROR" "$certificate_name passwd is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    keyToolCheckValidTime "$certificate" "$passwd" "$certificate_name"
    local ret=$?
    return $ret
}


##############################################################
## checkKeyCert.
##############################################################
function checkKeyCert()
{
    local certificate_file=$1
    local certificate_key=$2
    local passwd_file=$3
    local passwd_key=$4
    local certificate_name=$5

    local encryptd_passwd=$(cat ${passwd_file} | grep "$passwd_key" | awk -F '"' '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${passwd_file} | grep \\\"$passwd_key\\\" | awk -F '\\\"' '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Result: encryptd_passwd=\$encryptd_passwd\n"
    if [ -z "${encryptd_passwd}" ];then
        originalInfo="$originalInfo""$certificate_name encryptd password is null.\n"
        log "ERROR" "$certificate_name encryptd password is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    passwd=$(decrpt ${encryptd_passwd})
    if [ -z "${passwd}" ];then
        originalInfo="$originalInfo""$certificate_name password is null.\n"
        log "ERROR" "$certificate_name passwd is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    if [ -z "$certificate_key" ];then
      local certificate=${certificate_file}
    else
      local certificate=$(cat "${certificate_file}" | grep "$certificate_key" | awk -F '"' '{print $2}' | sed 's/ //g')
      originalInfo="$originalInfo""Command: cat ${certificate_file} | grep \\\"$certificate_key\\\" | awk -F '\\\"' '{print \$2}' | sed 's/ //g' \n"
      originalInfo="$originalInfo""Result: certificate=$certificate\n"
    fi
    if [ -z "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name certificate is null.\n"
        log "ERROR" "$certificate_name certificate is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    if [ ! -e "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name not find, file path is ${certificate}.\n"
        log "ERROR" "$certificate_name not find, file path is ${certificate}."
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    keyToolCheckValidTime "$certificate" "$passwd" "$certificate_name"
    local ret=$?
    return $ret
}

##############################################################
## checkJksCert.
##############################################################
function checkJksCert()
{
    local certificate_file=$1
    local certificate_key=$2
    local passwd_file=$3
    local passwd_key=$4
    local certificate_name=$5

    local encryptd_passwd=$(cat ${passwd_file} | grep "^$passwd_key" | awk -F "$passwd_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${passwd_file} | grep \\\"^$passwd_key\\\" | awk -F "$passwd_key=" '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Result: encryptd_passwd=\$encryptd_passwd\n"
    if [ -z "${encryptd_passwd}" ];then
        originalInfo="$originalInfo""$certificate_name encryptd password is null.\n"
        log "ERROR" "$certificate_name encryptd password is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    passwd=$(decrpt ${encryptd_passwd})
    if [ -z "${passwd}" ];then
        originalInfo="$originalInfo""$certificate_name password is null.\n"
        log "ERROR" "$certificate_name passwd is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    local certificate=$(cat ${certificate_file} | grep "^$certificate_key" | awk -F "$certificate_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${certificate_file} | grep \\\"^$certificate_key\\\" | awk -F "$certificate_key=" '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Reulst: certificate=$certificate\n"
    if [ -z "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name certificate is null.\n"
        log "ERROR" "$certificate_name certificate is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    if [ ! -e "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name not find, file path is ${certificate}.\n"
        log "ERROR" "$certificate_name not find, file path is ${certificate}."
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    keyToolCheckValidTime "$certificate" "$passwd" "$certificate_name"
    local ret=$?
    return $ret
}

##############################################################
## checkFMWSCert.
##############################################################
function checkFMWSCert()
{
    local certificate_file=$1
    local certificate_key=$2
    local passwd_file=$3
    local passwd_key=$4
    local certificate_name=$5

    local encryptd_passwd=$(cat ${passwd_file} | grep "^$passwd_key" | awk -F "$passwd_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${passwd_file} | grep \\\"^$passwd_key\\\" | awk -F "$passwd_key=" '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Result: encryptd_passwd=\$encryptd_passwd\n"
    if [ -z "${encryptd_passwd}" ];then
        originalInfo="$originalInfo""$certificate_name encryptd password is null.\n"
        log "ERROR" "$certificate_name encryptd password is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    passwd=$(decrpt ${encryptd_passwd})
    if [ -z "${passwd}" ];then
        originalInfo="$originalInfo""$certificate_name password is null.\n"
        log "ERROR" "$certificate_name passwd is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    local certificate=$(cat ${certificate_file} | grep "^$certificate_key" | awk -F "$certificate_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${certificate_file} | grep \\\"^$certificate_key\\\" | awk -F "$certificate_key=" '{print \$2}' | sed 's/ //g'\n"
    originalInfo="$originalInfo""Result: certificate=$certificate\n"
    if [ -z "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name certificate is null.\n"
        log "ERROR" "$certificate_name certificate is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    if [ ! -e "${FMWS_CONF_DIR}/${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name not find, file path is ${FMWS_CONF_DIR}/${certificate}.\n"
        log "ERROR" "$certificate_name not find, file path is ${FMWS_CONF_DIR}/${certificate}."
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    keyToolCheckValidTime "${FMWS_CONF_DIR}/${certificate}" "$passwd" "$certificate_name"
    local ret=$?
    return $ret
}

function keyToolCheckValidTime()
{
    local certificate=$1
    local passwd=$2
    local certificate_name=$3

    originalInfo="$originalInfo""Command: ${KEY_TOOL} -list -v -keystore ${certificate} \n"
    local certInfo=`${KEY_TOOL} -list -v -keystore ${certificate} <<EOF     
${passwd}
EOF`

    originalInfo="$originalInfo""Command: echo "\${certInfo}"  | grep "Valid" | awk -F 'until: ' '{print \$2}' | tail -n 1 | sed 's/PET/UTC-5/g' \n"
    local invalid_time_str=$(echo "${certInfo}"  | grep "Valid" | awk -F 'until: ' '{print $2}' | tail -n 1 | sed 's/PET/UTC-5/g' )
    if [ -z "${invalid_time_str}" ];then
        originalInfo="$originalInfo""invalid_time_str is null \n"
        log "ERROR"  "invalid_time_str is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    checkValidTime "${invalid_time_str}" "${certificate_name}"
    local ret=$?
    return $ret
}

function checkSSlCert()
{
    local certificate_file=$1
    local certificate_key=$2
    local certificate_name=$3

    local certificate=$(cat ${certificate_file} | grep "^$certificate_key" | awk -F "^$certificate_key=" '{print $2}' | sed 's/ //g')
    originalInfo="$originalInfo""Command: cat ${certificate_file} | grep \\\"$certificate_key\\\" | awk -F \\\"^$certificate_key=\\\" '{print \$2}' | sed 's/ //g' \n"
    originalInfo="$originalInfo""Review: certificate=$certificate\n"
    if [ -z "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name $certificate certificate is null.\n"
        addResultList "$certificate_name" "unok"
        return 1
    fi

    if [ ! -e "${certificate}" ];then
        originalInfo="$originalInfo""$certificate_name not find, file path is ${certificate}.\n"
        log "ERROR" "$certificate_name not find, file path is ${certificate}."
        addResultList "$certificate_name" "unok"
        return 1
    fi
    
    opensslCheckValidTime "${certificate}" "${certificate_name}"
    local ret=$?
    return $ret
}

function opensslCheckValidTime()
{
    local certificate=$1
    local certificate_name=$2

    local invalid_time_str=$(openssl x509 -enddate -noout -in $certificate | grep "notAfter" | awk -F 'notAfter=' '{print $2}' | tail -n 1 | sed 's/PET/UTC-5/g' )
    originalInfo="$originalInfo""Command: openssl x509 -enddate -noout -in $certificate | grep \\\"notAfter\\\" | awk -F 'notAfter=' '{print \$2}' | tail -n 1 | sed 's/PET/UTC-5/g'\n"
    originalInfo="$originalInfo""Review: invalid_time_str=$invalid_time_str\n"
    if [ -z "${invalid_time_str}" ];then
        originalInfo="$originalInfo""$certificate_name invalid_time_str is null.\n"
        log "ERROR"  "$certificate_name invalid_time_str is null." "$SELF_NAME:$LINENO($FUNCNAME)"
        addResultList "$certificate_name" "unok"
        return 1
    fi
    checkValidTime "${invalid_time_str}" "${certificate_name}"
    local ret=$?
    return $ret
}

function checkValidTime()
{
    local invalid_time_str=$1
    local certificate_name=$2
    local temp_time=$(timeTranslate "${invalid_time_str}")
    local invalid_time=$(date -d "$temp_time" +'%s')

    if [ "${invalid_time}" -ge "${AFTER_DATY_3_YEAY_TIME}" -a "${invalid_time}" -ge ${AFTER_DATY_2034_TIME} ];then
        # 证书有效期在3年以上，并且过期时间在2034年12月之后，检查通过
        originalInfo="$originalInfo""Review: $certificate_name valid time: $invalid_time_str\n"
        originalInfo="$originalInfo""Result: $certificate_name ok, left time greater than 3 years and greater than 2034-12-30.\n"
        log INFO "${checkItemId}: Review: $certificate_name valid time: $invalid_time_str\n"
        log INFO "${checkItemId}: Result: $certificate_name ok, left time greater than 3 years and greater than 2034-12-30.\n "
        addResultList "$certificate_name" "ok"
        return 0
    fi

    if [ "${invalid_time}" -ge "${AFTER_DATY_3_YEAY_TIME}" -a "${invalid_time}" -lt ${AFTER_DATY_2034_TIME} ];then
        # 证书有效期在3年以上，但是过期时间在2034年12月之前，建议更换
        originalInfo="$originalInfo""Review: $certificate_name valid time: $invalid_time_str\n"
        originalInfo="$originalInfo""Result: $certificate_name advice replace certificate, left time greater than 3 years, but less than 2034-12-30.\n"
        log INFO "${checkItemId}: Review: $certificate_name valid time: $invalid_time_str"
        log INFO "${checkItemId}: Result: $certificate_name advice replace certificate, left time greater than 3 years, but less than 2034-12-30.\n"
        addResultList "$certificate_name" "half_ok"
        return 0
    fi
    
    #检查不通过，证书过期时间在3年以下
    originalInfo="$originalInfo""Review: $certificate_name valid time: $invalid_time_str\n"
    originalInfo="$originalInfo""Result: $certificate_name unok,  left time less than 3 years.\n"
    log INFO "${checkItemId}: Review: $certificate_name valid time: $invalid_time_str"
    log INFO "${checkItemId}: Result: $certificate_name unok,  left time less than 3 years."
    addResultList "$certificate_name" "unok"
    return 1
}

function checkVBSCert()
{
    local certificate_file=$1
    local certificate_name=$2

    if [ ! -e "${certificate_file}" ];
    then
        originalInfo="$originalInfo""Review: $certificate_name not find, file path is $certificate_file\n"
        originalInfo="$originalInfo""Result: $certificate_name ok\n"
        log INFO "Review: $certificate_name not find, file path is $certificate_file"
        log INFO "Result: $certificate_name ok"
        addResultList "$certificate_name" "ok"
        return 0
    fi

    opensslCheckValidTime "$certificate_file" "$certificate_name"
}

function collectCheckResult()
{
    originalInfo="$originalInfo""CHECK_CERT_RESULT:\n"
    originalInfo="$originalInfo""OK_LIST: $OK_LIST\n"
    originalInfo="$originalInfo""HALF_OK_LIST: $HALF_OK_LIST\n"
    originalInfo="$originalInfo""UNOK_LIST: $UNOK_LIST\n"
    if [ -n "$UNOK_LIST" ];then
        # 不通过
        resultCode=1
        return 0
    elif [ -n "$HALF_OK_LIST" ];then
        # 建议替换.
        errorKey="${checkItemId}0001"
        resultCode=2
        return 0
    else
        # 通过
        resultCode=0
        return 0
    fi
}

function check_agent_main()
{
    opensslCheckValidTime $FSA_SSL_SELF_CERT_FILE "FSA_SSL_SELF"

    checkVBSCert $VBS_API_CERT_FILE "VBS_API_GATEWAY_ROOT"

    checkVBSCert $VBS_API_CERT_FILE "VBS_IAM_ROOT"
    
    # 总结检查结果，详细信息在原始信息中展示
    collectCheckResult

    return 0
}

function check_manager_main()
{
    checkKeyCert $TOMCAT_SSL_ROOT_CERT "" $TOMCAT_XML "keystorePass" "TOMCAT_SSL_ROOT"

    checkKeyCert $TOMCAT_XML "keystoreFile" $TOMCAT_XML "keystorePass" "TOMCAT_SSL_SELF"

    checkJksCert $SSL_CONFIG "KeyStore" $SSL_CONFIG "KeySequence" "OMS_SSL_SELF"

    checkJksCert $SSL_CONFIG "TrustStore" $SSL_CONFIG "TrustSequence" "OMS_SSL_ROOT"

    checkFMWSCert $FMWS_CERT_CONF "certName" $FMWS_CERT_CONF "certPwd" "OMM_OpenstackOm_Certificate"

    checkSSlCert $OMA_SERCFGFILE "TRUSTEDCERTIFICATE" "OMA_SSL_ROOT"

    checkSSlCert $OMA_SERCFGFILE "CERTIFICATE" "OMA_SSL_SELF"

    opensslCheckValidTime $HA_SSL_SELF_CERT_FILE "HA_SSL_SELF"

    opensslCheckValidTime $HA_SSL_ROOT_CERT_FILE "HA_SSL_ROOT"

    opensslCheckValidTime $FSA_SSL_ROOT_CERT_FILE "FSA_SSL_ROOT"

    # 总结检查结果，详细信息在原始信息中展示
    collectCheckResult
    
    return 0
}

function main()
{
    if [ "${NODE_TYPE}" == "Agent" ]; then
        FS_json_output=FSA_json_output
        agent_items_check ${checkItemId}
        if [ $? -ne 0 ]; then
            log INFO "${checkItemId} do not select, pass"
            return 0
        fi
        check_agent_main
    elif [ "${NODE_TYPE}" == "Manager" ]; then
        FS_json_output=FSM_json_output

        manager_items_check ${checkItemId}
        if [ $? -ne 0 ]; then
            log INFO "${checkItemId} do not select, pass"
            return 0
        fi

        #主FSM执行
        get_harole="/opt/omm/oms/workspace/ha/module/hacom/script/get_harole.sh"
        local harole=$(sh ${get_harole} | grep active)
        if [ -z "$harole" ];then
            log INFO "${checkItemId} not master FSM, pass"
            originalInfo="${checkItemId} not master FSM, pass"
            ${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
            return 0
        fi

        check_manager_main
    fi

    ${FS_json_output} "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
    return ${resultCode}
}

log MUST "enter [$0],para=[$@]"
main $@
retValue=$?
log MUST "leave [$0],retValue=${retValue}"
exit ${retValue}