#!/bin/bash

#############################################################
#
# 名  称：IB驱动版本检查
# 错误码：
#         0000 正常
#         0001 IB驱动版本低于3.0
#         0002 IB bond模式不是主备模式
#         0003 IB bond包含相同的MAC地址
#         0004 IB bond查询不到工作速率
#############################################################

UPG_LIB_PATH="/opt/omm/oma/atoms/Inspect/lib"
PATH="/sbin:/usr/sbin:/usr/local/sbin:/root/bin:/usr/local/bin:/usr/bin:/bin:/opt/omm/oma/workspace/tools"

source "${UPG_LIB_PATH}/log.sh" || { echo "source ${UPG_LIB_PATH}/log.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/out_put.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/out_put.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/version.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/version.sh failed."; exit 130; }
source "${UPG_LIB_PATH}/dswareTool_lib.sh" >> ${LOG_FILE} 2>&1 || { log ERROR "source ${UPG_LIB_PATH}/dswareTool_lib.sh failed."; exit 130; }

checkItemId="4421"
resultCode=0
errorKey=""
params=""
originalInfo=""
function check_driver_version()
{
    tmp=`ofed_info -s|tr -cd "[0-9]"` >/dev/null
    if [ ${tmp:0:1} -lt 3 ]; then
        log INFO "IB driver version is not correct"
        return 1
    fi
    return 0
}

function is_storage_ip()
{
    storage_ip_list=$1
    for_check_ip=$2

    for ip in `echo ${storage_ip_list}|tr '@' ' '`
    do
        if [ $ip = $for_check_ip ];then 
            return 0
        fi
    done
    return 1
}

function query_storage_original_nic_name_by_ipv4()
{
    storage_ips=$1
    if [ -z $storage_ips ];then
        log ERROR "can't query storage ip"
        return 1
    fi

    #获取到节点上所有网络设备
    all_net_dev_name=$(ip addr show|grep "scope global" | awk '{print $NF}')
    for dev_name in $all_net_dev_name ; do
        dev_ip=$(ifconfig $dev_name 2>&1 | grep "inet " | awk '{print $2}' | sed 's/addr://')
        if [ -n "$dev_ip" ] ; then
            # 查找存储IP所在网口
            is_storage_ip $storage_ips "$dev_ip"
            flag=$?
            if [ $flag -eq 0 ]; then
                if [ -n "$all_storage_port" ]; then  
                    all_storage_port=${all_storage_port}@${dev_name}
                else
                    all_storage_port=${dev_name}
                fi
            fi
        fi
    done
    return 0 
}

function query_storage_original_nic_name_by_ipv6()
{
    storage_ips=$1
    if [ -z $storage_ips ];then
        log ERROR "can't query storage ip"
        return 1
    fi
	
    #获取到节点上所有网络设备
    all_net_dev_name=$(ip -6 addr show|grep -E "scope global|site" -B 1 | grep mtu|awk -F ': ' '{print $2}')
    for dev_name in $all_net_dev_name ; do
        dev_ip=$(ip -6 addr show dev ${dev_name} | grep -v -i -e '[^:]FE80' | grep inet6 | awk '{print $2}' | awk -F/ '{print $1}')
        if [ -n "$dev_ip" ] ; then
            # 查找存储IP所在网口
            is_storage_ip $storage_ips "$dev_ip"
            flag=$?
            if [ $flag -eq 0 ]; then
                if [ -n "$all_storage_port" ]; then
                    all_storage_port=${all_storage_port}@${dev_name}
                else
                    all_storage_port=${dev_name}
                fi
            fi
        fi
    done
    return 0
}

function query_storage_original_nic_name_by_ip()
{
    local storage_ipv6=`cat /opt/dsware/agent/conf/network.cfg |grep storage_ipv6 | awk -F '=' '{print $2}'`
    
    if [ -z "$storage_ipv6" ];then
        query_storage_original_nic_name_by_ipv4 $1
        return $?
    fi

    if [ "$storage_ipv6" == "0" ];then
        query_storage_original_nic_name_by_ipv4 $1
        return $?
    elif [ "$storage_ipv6" == "1" ];then
        query_storage_original_nic_name_by_ipv6 $1
        return $?
    else
        log ERROR "never be here"
        return 1
    fi
}

function get_phy_interface_by_bond()
{
    phy_eth=''
    tmp_phy_eth=''
    bond_list=($(echo $1|tr '@' ' '))
    
    bond_file="/proc/net/bonding/$bond_name"
    tmp_phy_eth=`cat $bond_file|grep "Slave Interface:" |awk -F: '{print $2}'`
    tmp_phy_eth=`echo $tmp_phy_eth|tr ' ' '@'`
	
    if [ -z "$phy_eth" ];
    then
        phy_eth=$tmp_phy_eth
    else
        phy_eth=${phy_eth}@$tmp_phy_eth
    fi

    echo $phy_eth
}

function query_port_speed()
{
    bond_name=$1
    phy_eth_list=`get_phy_interface_by_bond $bond_name`
    phy_eth_list=($(echo $phy_eth_list|tr '@' ' '))
    for port in ${phy_eth_list[@]}
    do
        local nic_name=$(ibdev2netdev | grep $port | awk '{print $1}')  
        local port_id=$(ibdev2netdev | grep $port | awk '{print $3}')    
        local base_lid=$(ibstat ${nic_name} ${port_id} | grep "Base lid" | awk '{print $3}')
        local port_guid=$(ibstat ${nic_name} ${port_id} | grep "Port GUID" | awk '{print $3}')
        local cur_spd_str=$(iblinkinfo --cas-only | grep ${port_guid})
        if [ -z "${cur_spd_str}" ];then
            return 1
        fi
    done
    return 0
}

function query_bond_from_logical_netcard()
{
    interface_list=$1 

    if [ -z "$interface_list" ];then
        log ERROR "no find storage network card"
        return 0
    fi
    
    bond_list=`ls /proc/net/bonding 2>/dev/null`
    if [ -z "$bond_list" ];then
        return 0
    fi
    
    bond_file=''
    port_list=($(echo ${interface_list}|tr '@' ' '))
    ret=''
    i=0
    for storage_port in ${port_list[@]}
    do    
        hwaddr=`ip addr show $storage_port|grep "link/infiniband"|awk '{print $2}'`
        for bond_name in $bond_list
        do
            bond_hw_addr_list=`cat /proc/net/bonding/$bond_name|grep "Permanent HW addr:"|awk '{print $4}'`
            for sing_hw in ${bond_hw_addr_list[@]}
            do
                #共用一个物理口，说明是storage的bond
                echo $hwaddr|grep -i $sing_hw > /dev/null 2>&1
                if [ $? -eq 0 ];then
                    if [ -z "$(cat /sys/class/net/${bond_name}/bonding/mode 2>/dev/null | grep -i 'active-backup')" ];then #判断bond的模式是否是主备模式
                        return 1
                    fi
                    query_port_speed $bond_name
                    if [ $? -ne 0 ];then
                        return 3
                    fi
                    # 去重
                    if [[ "$ret" =~ "$bond_name@" ]];then
                        continue
                    fi
                    ret=$ret$bond_name@
                    let i=i+1
                fi
            done
        done
    done
    if [ $i -gt 1 ];then  #判断是否有多个bond的MAC地址相同
        return 2
    fi
    return 0
}

function check_network_config()
{
    storage_ip=`cat /opt/dsware/agent/conf/node_storage_ip | awk -F '=' '{print$2}'`
    query_storage_original_nic_name_by_ip $storage_ip
    if [ $? -ne 0 ];then
        return 0
    fi
    query_bond_from_logical_netcard $all_storage_port
    ret=$?
    return $ret
}

function modify_result()
{
    resultCode=$1
    errorKey=$2
    msg=$3
    local result=""
    [ $resultCode -eq 0 ] && { result="ok"; } || { result="unok";}
    originalInfo="${originalInfo}""result:${msg}"
    log INFO "FSA_${checkItemId}:${msg}"
    log INFO "$result"
    FSA_json_output "${checkItemId}" "${resultCode}" "${errorKey}" "${params}" "${originalInfo}"
}

function main()
{
    agent_items_check ${checkItemId}
    if [ $? -ne 0 ]
    then
        log INFO "${checkItemId} do not select, pass"
        return 0
    fi

    product_version=`cat /opt/dsware/DSwareAgentNodeVersion | grep -i version | awk -F '=' '{print$2}'`
    if [[ ! "${product_version}" =~ "V100R006C30" ]]
    then
        modify_result 0 "" "NodeVersion is not 6.3, no need to check"
        return 0
    fi

    local IB_net_judg=`cat /opt/dsware/agent/conf/network.cfg | grep -i infiniband`
    if [[ -z "${IB_net_judg}" ]]
    then
        modify_result 0 "" "this FSA network type is not infiniband, no need to check"
        return 0
    else
        ret=`check_driver_version`
        if [ $? -ne 0 ];then
            modify_result 1 "${checkItemId}0001" "The IB driver version is lower than 3.0."
            return 1
        fi
        ret=`check_network_config`
        ret1=$?
        if [ $ret1 -eq 1 ];then
            modify_result 1 "${checkItemId}0002" "bond mode is not active-backup"
            return 1  
        elif [ $ret1 -eq 2 ];then
            modify_result 1 "${checkItemId}0003" "The same mac address exists in different bonds"
            return 1
        elif [ $ret1 -eq 3 ];then
            modify_result 1 "${checkItemId}0004" "Fail to query negotiation speed of ports"
            return 1
        else
            modify_result 0 "" "check ib driver:ok"
            return 0
        fi
    fi

    return 0
}

log MUST "enter [$0],para=[$@]"
main $@
retValue=$?
log MUST "leave [$0],retValue=${retValue}"
exit ${retValue}
