# -*- coding: UTF-8 -*-
import os
import sys
import cliUtil
import common
import preCheck
import re

scriptpath = os.path.dirname(os.path.abspath(__file__))
hostCommonPath = os.path.join(scriptpath, "..\\..\\HOST_Common")
sys.path.append(hostCommonPath)
import host_common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

queryWWNFailures = []
queryAluaFailures = []

robinNotPassDisks = []
evaluateNotPassDisks = []
cliRets = []

INDEX_INT6 = 6
LENGTH_INT38 = 38
lineBreak = '\n'
LINE_SIZE_INT2 = 2
PATH_STATUS_LOC = 3


def execute(ssh):
    '''
    @summary: AIX自带多路径双活LUN配置检查
    '''
    try:
        is_upadmin_designated, echos = \
            common.mark_host_upadmin_hyper_metro_luns(py_java_env,
                                                      ssh, LOGGER)
        if is_upadmin_designated:
            return cliUtil.RESULT_NOSUPPORT, echos, ''
        # 白名单检查
        py_java_env["logger"] = PY_LOGGER
        py_java_env["ssh"] = ssh
        checkFlag, cliRet, errMsg = preCheck.execute_doradov6(py_java_env)
        cliRet += echos
        if checkFlag is not True:
            return checkFlag, cliRet, errMsg
        arrayWwnDict = getArrayWwns(py_java_env)
        if not arrayWwnDict:
            return (True, '', '')
        cliRets.append(cliRet)
        executeStatus, disks = queryHuaweiDisk(py_java_env, ssh)
        if not executeStatus:
            return (cliUtil.RESULT_NOCHECK, lineBreak.join(cliRets),
                    common.getMsg(LANG,
                                  "hyper.metro.host.disks.query.failure"))
        if not disks:
            err_key = "hyper.metro.host.query.multipath.not.take.lun"
            return (False, lineBreak.join(cliRets),
                    common.getMsg(LANG, err_key))
        diskWwnDict = getValidDiskDict(disks, arrayWwnDict, ssh)
        LOGGER.logInfo("***[diskWwnDict is: %s]***" % diskWwnDict)
        if not diskWwnDict:
            err_key = "hyper.metro.host.query.multipath.not.take.lun"
            return (False, lineBreak.join(cliRets),
                    common.getMsg(LANG, err_key))

        # 将上下文中获取访问模式的回显加入cliRets
        access_mode_cli = host_common.get_host_access_cli_ret(py_java_env)
        for sn in access_mode_cli:
            cliRets.append(sn)
            cliRets.append(access_mode_cli[sn])


        # 获取主机访问模式和优先路径配置
        hyper_path_info = host_common.get_hyper_path_opt(py_java_env)
        hyper_path_list = host_common.get_hyper_path_list(hyper_path_info)
        flag, access_mode = host_common.get_cur_access_mode(py_java_env)
        if not flag:
            msg = common.getMsg(LANG,
                                "hyper.storage.host.access.mode.not.equal")
            return False, lineBreak.join(cliRets), msg

        return handleResultAndErrorMsg(
            evaluateDiskAlua(diskWwnDict, arrayWwnDict, ssh, hyper_path_list,
                             access_mode))
    except BaseException as exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, lineBreak.join(cliRets),
                common.getMsg(LANG, "query.result.abnormal"))


def updateHostAluaStatus(arrayWwnDict, wwn, status):
    if wwn not in arrayWwnDict:
        return
    LOGGER.logInfo(
        "***[update the host alua status for the lun wwn: %s is: %s]***" % (
            wwn, status))
    arrayWwnDict[wwn]['hostAluaStatus'] = status


def getArrayWwns(py_java_env):
    arrayWwnDict = py_java_env.get('allStrgHyprMtrLns')
    return arrayWwnDict


def handleResultAndErrorMsg(evaluateResult):
    '''
    @summary: handle result and construct error message
    '''
    errorMsg = ''
    if queryWWNFailures:
        evaluateResult = cliUtil.RESULT_NOCHECK
        errorMsg += common.getMsg(LANG, "hyper.metro.disk.wwn.query.failure",
                                  ','.join(queryWWNFailures))
    if queryAluaFailures:
        evaluateResult = cliUtil.RESULT_NOCHECK
        errorMsg += common.getMsg(LANG, "hyper.metro.alua.query.failure",
                                  ','.join(queryAluaFailures))
    if robinNotPassDisks:
        evaluateResult = False
        errorMsg += common.getMsg(LANG,
                                  "hyper.metro.alua.evaluate.notpass.robin",
                                  ','.join(robinNotPassDisks))
    if evaluateNotPassDisks:
        evaluateResult = False
        errorMsg += common.getMsg(LANG,
                                  "hyper.metro.alua.evaluate.notpass.step6",
                                  ','.join(evaluateNotPassDisks))
    return (evaluateResult, lineBreak.join(cliRets), errorMsg)


def evaluateDiskAlua(diskWwnDict, arrayWwnDict, ssh, hyper_path_list,
                     access_mode):
    global robinNotPassDisks, evaluateNotPassDisks, queryAluaFailures, cliRets

    evaluateResult = True
    for diskWwn in diskWwnDict:
        disk = diskWwnDict.get(diskWwn)

        # 步骤5检查 磁盘多路径配置信息中是否包含 algorithm round robin
        executeStatus, checkResult = queryAndCheckDiskAlua(disk, py_java_env,
                                                           ssh)
        evaluateResult = evaluateResult and checkResult
        if not executeStatus:
            queryAluaFailures.append(disk)
            updateHostAluaStatus(arrayWwnDict, diskWwn, -1)
            LOGGER.logInfo(
                "***[query the disk : %s alua info failure]***" % disk)
            continue

        if not checkResult:
            LOGGER.logInfo("***[the disk: %s is not pass]***" % disk)
            robinNotPassDisks.append(disk)
            updateHostAluaStatus(arrayWwnDict, diskWwn, -1)
            continue

        # 检查MPIO ALUA配置OPT和Non
        result, ret = check_access_mode(ssh, disk, access_mode,
                                        hyper_path_list)
        cliRets.append(ret)
        if result == cliUtil.RESULT_NOCHECK:
            evaluateResult = False
            queryAluaFailures.append(disk)
            updateHostAluaStatus(arrayWwnDict, diskWwn, -1)
            LOGGER.logInfo(
                "***[query the disk : %s alua info failure]***" % disk)
            continue

        if result is False:
            LOGGER.logInfo("***[the disk: %s is not pass]***" % disk)
            evaluateResult = False
            evaluateNotPassDisks.append(disk)
            updateHostAluaStatus(arrayWwnDict, diskWwn, -1)
            continue

        # 如果配置都正确则，更新为正确状态1
        updateHostAluaStatus(arrayWwnDict, diskWwn, 1)

    return evaluateResult


def getValidDiskDict(disks, arrayWwnDict, ssh):
    global queryWWNFailures
    diskWwnDict = {}
    for disk in disks:
        executeStatus, wwn = queryDiskWwn(disk, py_java_env, ssh)
        LOGGER.logInfo("[ %s,  the executeStatus: %s, %s]***" % (
            disk, executeStatus, wwn))
        if not executeStatus:
            queryWWNFailures.append(disk)
            continue
        valildWwn = filter(lambda arrayWwn: arrayWwn.lower() == wwn.lower(),
                           arrayWwnDict.keys())
        if valildWwn:
            updateHostAluaStatus(arrayWwnDict, valildWwn[0], 1)
            diskWwnDict[valildWwn[0]] = disk
    return diskWwnDict


def queryHuaweiDisk(context, ssh):
    '''
    @summary: query the available huawei disks
    @return: (execute status: True or False, disk list)
    '''
    try:
        disks = []
        cmd = 'lsdev -Cc disk'
        executeStatus, cliRet, errorMsg = cliUtil.executeHostCmd(context, ssh,
                                                                 cmd)
        cliRets.append(cliRet)
        if executeStatus is not True:
            LOGGER.logInfo("***[queryHuaweiDisk : %s]***" % errorMsg)
            return False, disks
        for line in cliRet.splitlines():
            avilableDisk = filter(lambda item: item.lower() == 'available',
                                  line.split())
            if not avilableDisk:
                continue
            diskName = line.strip().split()[0]
            if diskName:
                disks.append(diskName)
        LOGGER.logInfo("execute queryHuaweiDisk, disks is : %s" % disks)
        return True, disks
    except Exception as e:
        LOGGER.logError("queryHuaweiDisk error:%s " % str(e))
        return False, []


def queryDiskWwn(disk, context, ssh):
    '''
    @summary: query disk wwpn from host
    @return: (execute status: True or False, WWN)
    '''
    try:
        cmd = 'odmget -q "name=%s and attribute=unique_id" CuAt' % disk
        wwn = ''
        executeStatus, cliRet, errorMsg = cliUtil.executeHostCmd(context, ssh,
                                                                 cmd)
        cliRets.append(cliRet)
        if executeStatus is not True:
            LOGGER.logInfo(
                "***[queryDiskWwn failure, cliRet is: %s, "
                "errorMsg is: %s]***" % (cliRet, errorMsg))
            return False, wwn
        if len(cliRet.splitlines()) <= LINE_SIZE_INT2:
            LOGGER.logInfo(
                "***[the disk: %s is not taken over by nmp]***" % disk)
            return True, wwn
        for line in cliRet.splitlines():
            stripLine = line.strip()
            if not stripLine.lower().startswith('value'):
                continue
            splitDatas = stripLine.split('=')
            valuePart = splitDatas[-1].strip()
            if len(splitDatas) > 1 and len(valuePart) >= LENGTH_INT38:
                wwn = valuePart[INDEX_INT6:LENGTH_INT38]
                return True, wwn
        return False, wwn
    except Exception as e:
        LOGGER.logError("queryDiskWwn error:%s " % str(e))
        return False, False


def queryAndCheckDiskAlua(disk, context, ssh):
    '''
    @summary: query the alua configure information
    of disk and check the information
    @return: (execute status: True or False, check result: True or False)
    '''
    try:
        cmd = 'lsattr -EHl %s | cat' % disk
        executeStatus, cliRet, errorMsg = cliUtil.executeHostCmd(context, ssh,
                                                                 cmd)
        cliRets.append(cliRet)
        if executeStatus is not True:
            LOGGER.logInfo(
                "***[query the disk : %s  : %s]***" % (disk, errorMsg))
            return False, False
        for line in cliRet.splitlines():
            stripLine = line.strip()
            if not stripLine.lower().startswith('algorithm'):
                continue
            if 'round_robin' in stripLine.lower():
                return True, True
            else:
                return True, False
        return False, False
    except Exception as e:
        LOGGER.logError("queryAndCheckDiskAlua error:%s " % str(e))
        return False, False


def get_path_status(ssh, disk):
    """ 获取路径状态

    @param context:
    @param ssh:
    @param disk_num:
    @return: is_ok, path_status, cli_ret
    """
    cli_ret = ""
    path_status = []
    cmd = "lsmpio -l " + str(disk)
    try:
        flag, cli_ret, _ = cliUtil.executeHostCmd(py_java_env, ssh, cmd)
        if not flag:
            LOGGER.logInfo(
                "Failed to query path status of the disk[%s]" % str(disk))
            return False, path_status, cli_ret

        lines = cli_ret.splitlines()
        for line in lines:
            split_flag = re.compile(r"[a-zA-Z0-9_,#]+\s*")
            items = []
            start_pos = 0
            end_pos = 0
            while (start_pos <= len(line)):
                match = split_flag.search(line, start_pos)
                if match:
                    end_pos = match.end()
                    items.append(
                        line[start_pos:end_pos].strip().decode("utf8"))
                    start_pos = end_pos
                else:
                    break
            if len(items) <= PATH_STATUS_LOC:
                continue
            status = items[PATH_STATUS_LOC].lower()

            if "sel" not in status and \
                    "opt" not in status and "non" not in status:
                continue
            path_status.append(status)
        return True, path_status, cli_ret
    except Exception as e:
        LOGGER.logError(
            "checking disk's mpio status but failed:%s" % str(e))
        return False, path_status, cli_ret


def check_path_status(path_status):
    is_opt = False
    is_non = False
    for status in path_status:
        if "opt" in status:
            is_opt = True

        if "non" in status:
            is_non = True

        if is_opt and is_non:
            return True
    return False


def check_asymmetric_mode(path_status, hyper_path_list):
    if not check_path_status(path_status):
        LOGGER.logInfo("[Asymmetric]The path status is wrong.")
        return False

    if len(hyper_path_list) > 1:
        return True

    LOGGER.logInfo(
        "[Asymmetric]The hyper path is wrong:%s" % str(hyper_path_list))
    return False


def check_balanced_mode(path_status):
    path_status_set = set(path_status)
    if len(path_status_set) > 1:
        LOGGER.logInfo("The path status is not only:%s" % str(path_status_set))
        return False
    if "sel" in path_status_set:
        return True, ""
    else:
        LOGGER.logInfo("The path status is wrong:%s" % str(path_status_set))
        return False


def check_access_mode(ssh, disk, access_mode, hyper_path_list):
    flag, path_status, cli_ret = get_path_status(ssh, disk)
    if not flag:
        return cliUtil.RESULT_NOCHECK, cli_ret

    if access_mode == "balanced":
        flag = check_balanced_mode(path_status)
        return flag, cli_ret
    elif access_mode == "asymmetric":
        flag = check_asymmetric_mode(path_status, hyper_path_list)
        return flag, cli_ret
    else:
        LOGGER.logInfo("The access mode is not supported:%s" % access_mode)
        return True, cli_ret

    return True, cli_ret
