# coding:utf-8
"""
@version: Toolkit V200R006C00
@time: 2020/01/04
@file: host_common.py
"""
import re

from cbb.frame.cli import running_data_analyze_util as ana_util
from cbb.frame.context import sqlite_context
from cbb.frame.cli.cli_with_cache import query_one_cmd
from cbb.frame.cli import cliUtil
from cbb.frame.base import product

import HostResource

RUNNING_DATA = "running data {0}"
CONFIG_CLI_RET = "Get Path from config.txt, Path information:\n{0}"
RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"
ALUA_CONFIGURED_PROPERLY = 1
ALUA_CONFIGURED_FALSELY = -1
ALUA_CONFIGURED_NOT_CHECK = 0

def get_access_mode(context):
    """获取访问模式
    :param context:上下文
    :return:{阵列SN:access_mode, 阵列SN2:access_mode}
    """
    array_ini_info = get_array_ini_info(context)
    ini_list = get_ini_info(context)
    res_dict = {}
    for array_sn in array_ini_info:
        info_dict = array_ini_info.get(array_sn, {})
        ini_info_dict = info_dict.get("initiator_info", {})
        for ini in ini_info_dict:
            if ini in ini_list:
                access_mode = ini_info_dict.get(ini, {}).get("access_mode", "")
                res_dict[array_sn] = access_mode
                break
    return res_dict


def get_hyper_path_opt(context):
    """获取双活路径模式
    :param context:上下文
    :return:结果字典
    """
    array_ini_info = get_array_ini_info(context)
    ini_list = get_ini_info(context)
    res_dict = {}
    for array_sn in array_ini_info:
        info_dict = array_ini_info.get(array_sn, {})
        ini_info_dict = info_dict.get("initiator_info", {})
        for ini in ini_info_dict:
            if ini in ini_list:
                path_opt = ini_info_dict.get(ini, {}).get("hyper_path_opt", "")
                res_dict[array_sn] = path_opt
                break
    return res_dict


def get_host_access_cli_ret(context):
    """
    获取主机访问模式
    :param context:上下文
    :return: 结果
    """
    array_ini_info = get_array_ini_info(context)
    ini_list = get_ini_info(context)
    res_dict = {}
    for array_sn in array_ini_info:
        info_dict = array_ini_info.get(array_sn, {})
        cli_ret = info_dict.get("cli_ret", "")
        ini_info_dict = info_dict.get("initiator_info", {})
        for ini in ini_info_dict:
            if ini in ini_list:
                res_dict[array_sn] = cli_ret
                break

    return res_dict


def get_array_ini_info(context):
    """获取定义的内存数据
    :param context: 上下文
    :return: 内存数据
    """
    object_for_py = context.get("objectForPy")
    return object_for_py.get("array_initiator_info")


def get_ini_info(context):
    """
    获取启动器信息
    :param context:上下文
    :return: 启动器
    """
    dev_node = context.get("devInfo")
    # 主机侧FC启动器信息
    fc_wwn_list = dev_node.getHostFCLauncherWwns()
    # 主机侧ISCSI启动器信息
    iscsi_wwn_list = dev_node.getHostISCSILauncherWwns()
    if not fc_wwn_list:
        fc_wwn_list = []
    if not iscsi_wwn_list:
        iscsi_wwn_list = []
    fc_wwn_list = list(fc_wwn_list)
    iscsiList = list(iscsi_wwn_list)
    fc_wwn_list.extend(iscsiList)
    fc_wwn_list = [ini.lower() for ini in fc_wwn_list]
    return fc_wwn_list


def get_cur_access_mode(context):
    """ 获取访问模式，原则上访问模式是一致的
    :param context:
    :return:
    """
    logger = context["logger"]
    access_mode_info = get_access_mode(context)
    logger.info("access_mode_info:%s" % str(access_mode_info))
    access_mode_set = set(access_mode_info.values())
    if len(access_mode_set) > 1:
        return False, access_mode_info
    logger.info("access_mode_set:%s" % str(access_mode_set))
    access_mode = list(access_mode_set)[0].lower()
    return True, access_mode


def get_hyper_path_list(hyper_path_info):
    """ 获取所有双活优先路径配置
    :param hyper_path_info: 当前主机关联的所有阵列双活优先路径配置
    :return:
    """
    hyper_path_set = set(hyper_path_info.values())
    return list(hyper_path_set)


def get_lun_info_from_config(mem_obj, logger, sn):
    """
    获取LUN信息
    :param mem_obj: 内存对象
    :param logger: 日志
    :param sn: 阵列SN
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    # 融合存储和dorado v6和CLI字段不存在差异。如果存在差异需要处理后返回
    return ana_util.get_lun_info_from_running_data(sqlite_conn)


def get_host_info_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    # 融合存储和dorado v6和CLI字段不存在差异。如果存在差异需要处理后返回
    result_dict = {}
    # 由于Dorado v6和融合存储和对应CLI回文存在字段差异，需要统一处理后返回
    host_info_dict = ana_util.get_host_info_from_running_data(sqlite_conn)
    if product.is_dorado_series_version(p_version):
        for host_info in host_info_dict:
            host_info["IP Address"] = host_info.get("IP", "")
            host_info["ID"] = host_info.get("Id", "")
            host_info["Access Mode"] = host_info.get("ALUA MODE", "")
            host_info["name"] = host_info.get("Name", "")
            host_info["operSys"] = host_info.get("Operating System", "")
            host_info["ipAddr"] = host_info.get("IP", "")
            result_dict[host_info.get("Id", "")] = host_info
    else:
        for host_info in host_info_dict:
            host_info["ID"] = host_info.get("Id", "")
            host_info["IP Address"] = host_info.get("Ip", "")
            host_info["Operating System"] = host_info.get("Os Type", "")
            host_info["name"] = host_info.get("Name", "")
            host_info["operSys"] = host_info.get("Os Type", "")
            host_info["ipAddr"] = host_info.get("Ip", "")
            result_dict[host_info.get("Id", "")] = host_info

    return result_dict


def get_host_lun_info_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机LUN信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    host_lun_dict = {}
    # 由于Dorado v6和融合存储和对应CLI回文存在字段差异，需要统一处理后返回
    if product.is_dorado_series_version(p_version):
        host_lun_info_list = ana_util.get_host_lun_info_from_running_data(
            sqlite_conn
        )
        for host_lun_info in host_lun_info_list:
            lun_id_list = host_lun_info.get("LUN ID List", "").split(",")
            host_id = host_lun_info.get("Host ID", "")
            host_lun_dict[host_id] = lun_id_list
    else:
        # 融合存储
        mapping_view_info_dict = ana_util.get_mapping_view_info_from_running_data(
            sqlite_conn
        )
        lun_group_info_dict = ana_util.get_lun_group_info_from_running_data(
            sqlite_conn
        )
        host_group_host_info_dict = ana_util.get_host_group_info_from_running_data(
            sqlite_conn
        )

        for mapping_view in mapping_view_info_dict:
            host_group_id = mapping_view.get("Host Group Id")
            lun_group_id = mapping_view.get("Lun Group Id")
            # 如果任一个group为--，表示此view要么没有映射LUN，要么没有主机
            if host_group_id == "--" or lun_group_id == "--":
                continue

            lun_id_list = []
            for lun_group in lun_group_info_dict:
                if lun_group_id != lun_group.get("Id"):
                    continue
                lun_info_list = lun_group.get("LUN INFO", [])
                for lun_info in lun_info_list:
                    lun_id = lun_info.get("LUN ID")
                    if not lun_id or lun_id == "--":
                        continue
                    lun_id_list.append(lun_id)
            if not lun_id_list:
                continue

            for host_group_info in host_group_host_info_dict:
                if host_group_id != host_group_info.get("Id"):
                    continue
                host_info_list = host_group_info.get("Host List", [])
                for host_info in host_info_list:
                    host_id = host_info.get("ID")
                    if not host_id or host_id == "--":
                        continue

                    tmp_lun_list = host_lun_dict.get(host_id, [])
                    tmp_lun_list.extend(lun_id_list)
                    host_lun_dict[host_id] = tmp_lun_list
    return host_lun_dict


def get_running_data(mem_obj, p_version, sn):
    """
    获取running data抽取的公共方法
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :return:
    """
    if is_use_running_data(p_version):
        running_data_dicts = mem_obj.get(RUNNING_DATA.format(str(sn)))
        if not running_data_dicts:
            return False

        tmp_dict = running_data_dicts.get(sn, {})
        return tmp_dict.get("running_data_dict", False)
    return False


def get_product_version_from_mem(mem_obj, sn):
    """
    获取running data抽取的公共方法
    :param mem_obj: 内存对象
    :param sn: 阵列SN
    :return:
    """
    running_data_dicts = mem_obj.get(RUNNING_DATA.format(str(sn)))
    if not running_data_dicts:
        return ""

    tmp_dict = running_data_dicts.get(sn, {})
    p_version = tmp_dict.get("p_version", "")
    return p_version


def get_fc_ini_from_config(mem_obj, p_version, sn, logger):
    """
    获取FC启动器信息。
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    host_fc_ini_dict = {}
    # 由于Dorado v6才能查询到FC启动器信息，融合存储config中没有
    if product.is_dorado_series_version(p_version) or product.com:
        fc_info_dict = ana_util.get_fc_initiator_info_from_running_data(
            sqlite_conn
        )
        for fc_info in fc_info_dict:
            host_id = fc_info.get("Host Id")
            wwn = fc_info.get("WWN")

            if host_id == "--":
                continue

            tmp_ini_list = host_fc_ini_dict.get(host_id, [])
            tmp_ini_list.append(wwn)
            host_fc_ini_dict[host_id] = tmp_ini_list
    return host_fc_ini_dict


def get_mapping_from_config_v6(mem_obj, p_version, sn, logger):
    """
    v6 获取映射信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    mapping = {}
    # 由于Dorado v6才有mapping信息
    if product.is_dorado_series_version(p_version):
        mapping_dict_list = ana_util.get_mapping_info_from_running_data(
            sqlite_conn
        )
        for mapping_dict in mapping_dict_list:
            mapping_id = mapping_dict.get("Id", "")
            host_group_id = mapping_dict.get("Host GroupId", "")
            lun_group_id = mapping_dict.get("Lun GroupId", "")
            host_id = mapping_dict.get("Host Id", "")
            lun_id_list = (
                mapping_dict.get("Lun Id List", "")
                .replace("{", "")
                .replace("}", "")
                .split(",")
            )
            port_group_id = mapping_dict.get("Port GroupId", "")
            mapping[mapping_id] = {
                "Host Group ID": host_group_id,
                "Port Group ID": port_group_id,
                "Host ID": host_id,
                "Lun Group ID": lun_group_id,
                "Lun ID List": lun_id_list,
            }

    return mapping


def get_mapping_view_from_config_ocean_stor(mem_obj, p_version, sn, logger):
    """
    融合存储获取映射信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return: 字典列表
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    # 融合存储才有mapping信息
    if product.is_dorado_series_version(p_version):
        return ana_util.get_mapping_view_info_from_running_data(sqlite_conn)
    return {}


def get_all_view_info_from_config_ocean_stor(
    hyper_lun_id, mem_obj, p_version, sn, logger
):
    # need_xx需要通过命令重新查询的对象
    need_host_group_list = []
    need_port_group_list = []
    need_lun_group_list = []
    # view信息时候解析成功
    is_view_exist_flag = True
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    mapping_view_dict = get_mapping_view_from_config_ocean_stor(
        mem_obj, p_version, sn, logger
    )
    if not mapping_view_dict:
        is_view_exist_flag = False
        return (
            is_view_exist_flag,
            need_host_group_list,
            need_port_group_list,
            need_lun_group_list,
        )

    # 从config中获取主机组、端口组、LUN组信息
    host_group_dict = get_host_group_host_from_config(
        mem_obj, p_version, sn, logger
    )
    port_group_dict = get_port_group_port_from_config(
        mem_obj, p_version, sn, logger
    )
    lun_group_info_dict = ana_util.get_lun_group_info_from_running_data(
        sqlite_conn
    )

    for record in mapping_view_dict:
        lun_group_id = record.get("Lun Group Id")
        host_group_id = record.get("Host Group Id")
        port_group_id = record.get("Port Group Id")
        # 如果任一个group为--，表示此view要么没有映射LUN，要么没有主机
        if host_group_id == "--" or lun_group_id == "--":
            continue

        lun_id_list = []
        for lun_group in lun_group_info_dict:
            if lun_group_id != lun_group.get("Id"):
                continue
            lun_info_list = lun_group.get("LUN INFO", [])
            for lun_info in lun_info_list:
                lun_id = lun_info.get("LUN ID")
                if not lun_id or lun_id == "--":
                    continue

                # 如果不是双活LUN则不处理
                if lun_id not in hyper_lun_id:
                    continue

                lun_id_list.append(lun_id)

        if not lun_id_list:
            continue

        # mapping view中有，但config中未查到则使用命令查询
        if port_group_id and port_group_id != "--":
            if not port_group_dict.get(port_group_id, []):
                need_port_group_list.append(port_group_id)

        # mapping view中有，但config中未查到则使用命令查询
        if host_group_id and host_group_id != "--":
            if not host_group_dict.get(host_group_id, []):
                need_host_group_list.append(host_group_id)

    return (
        is_view_exist_flag,
        need_host_group_list,
        need_port_group_list,
        need_lun_group_list,
    )


def get_host_group_host_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机主下主机信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志对象
    :return: 字典
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    host_group_host_dict = {}
    # 融合存储和dorado v6获取信息不一样
    if product.is_dorado_series_version(p_version):
        host_group_host_list = ana_util.get_host_group_host_info_from_running_data(
            sqlite_conn
        )
        for host_group_host in host_group_host_list:
            host_group_id = host_group_host.get("Host Group Id", "")
            # 防止1对多时，跟多个host id。
            host_id_list = host_group_host.get("Host Id", "").split(",")
            tmp_host_list = host_group_host_dict.get(host_group_id, [])
            tmp_host_list.extend(host_id_list)
            host_group_host_dict[host_group_id] = tmp_host_list
    else:
        host_group_list = ana_util.get_host_group_info_from_running_data(
            sqlite_conn
        )
        for host_group in host_group_list:
            host_group_id = host_group.get("Id", "")
            host_info_dict_list = host_group.get("Host List", [])
            tmp_host_list = host_group_host_dict.get(host_group_id, [])
            for host_info in host_info_dict_list:
                host_id = host_info.get("ID")
                tmp_host_list.append(host_id)
            host_group_host_dict[host_group_id] = tmp_host_list

    return host_group_host_dict


def get_port_group_port_from_config(mem_obj, p_version, sn, logger):
    """
    获取端口组下的端口信息。
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :return: 字典
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    port_group_port_dict = {}
    # 融合存储和dorado v6获取信息不一样
    if product.is_dorado_series_version(p_version):
        port_group_port_list = ana_util.get_port_group_port_info_from_running_data(
            sqlite_conn
        )
        for port_group_host in port_group_port_list:
            port_group_id = port_group_host.get("Port Group Id", "")
            # 防止1对个port
            port_id_list = port_group_host.get("Port Id", "").split(",")
            tmp_port_list = port_group_port_dict.get(port_group_id, [])
            tmp_port_list.extend(port_id_list)
            port_group_port_dict[port_group_id] = tmp_port_list

    # 融合存储查询出的port id 是：hysical Port Id:0x1110600无法使用
    return port_group_port_dict


def get_all_mapping_info_from_config_v6(
    hyper_lun_id_list, mem_obj, p_version, sn, logger
):
    """
    从config中获取mapping的全部对象（包含主机组、端口组、LUN组等对象信息）。
    哪个对象不存在就使用命令查哪个对象。
    mapping又分两种场景
    1. LUN 直接映射给主机
    2. LUN 通过LUN group 映射给host group.
    :param hyper_lun_id_list:
    :param mem_obj:
    :param p_version:
    :param sn:
    :return:
    """
    mapping_flag = True
    # 需要使用命令查询的对象
    port_group_list = []
    host_group_list = []
    mapping_info_dict = get_mapping_from_config_v6(
        mem_obj, p_version, sn, logger
    )
    if not mapping_info_dict:
        mapping_flag = False

    lun_host_dict = {}
    for mapping_id in mapping_info_dict:
        tmp_host_list = []
        tmp_dict = mapping_info_dict.get(mapping_id, {})
        host_group_id = tmp_dict.get("Host Group ID")
        port_group_id = tmp_dict.get("Port Group ID")
        lun_id_list = tmp_dict.get("Lun ID List")
        host_id = tmp_dict.get("Host ID")

        cur_hyper_list = []
        for lun_id in lun_id_list:
            if lun_id not in hyper_lun_id_list:
                continue
            cur_hyper_list.append(lun_id)

        # 当前mapping没有双活LUN
        if not cur_hyper_list:
            continue

        # 如果主机不为--，
        if host_id and host_id != "--":
            tmp_host_list.append(host_id)

        if host_group_id and host_group_id != "--":
            host_group_dict = get_host_group_host_from_config(
                mem_obj, p_version, sn, logger
            )
            if not host_group_dict:
                host_group_list.append(host_group_id)
            host_list = host_group_dict.get(host_group_id, [])
            tmp_host_list.extend(host_list)

        for lun_id in cur_hyper_list:
            # 更新LUN映射的主机，通过主机找启动器
            tmp_lun_host_list = lun_host_dict.get(lun_id, [])
            tmp_lun_host_list.extend(tmp_host_list)
            lun_host_dict[lun_id] = list(set(tmp_lun_host_list))

        if port_group_id and port_group_id != "--":
            port_group_info_dict = get_port_group_port_from_config(
                mem_obj, p_version, sn, logger
            )
            if not port_group_info_dict:
                port_group_list.append(port_group_id)

    return mapping_flag, lun_host_dict, port_group_list, host_group_list


def is_use_running_data(p_version):
    """
    原因：因维护版本FC驱动限制部分版本收集config.txt。
    结论：本次优化版本范围： Dorado V6、V500R007C60(包含kunpeng和非kunpeng)
        V300R006C61及其之后
    :param p_version:
    :return:
    """
    if not p_version:
        return False

    if product.is_dorado_series_version(p_version):
        return True

    if p_version.startswith("V500R007") and p_version >= "V500R007C60":
        return True

    if p_version.startswith("V300R006") and p_version >= "V300R006C61":
        return True

    return False


def get_host_ini_info(sqlite_manager, sqlite_conn, sn, lang, cmd, ini_type):
    """
    直接获取主机启动器对应关系需要区分主机的ISCSI和FC启动器
    :param sqlite_manager: 数据库管理对象
    :param sqlite_conn: 数据库连接
    :param sn: 阵列SN
    :param lang: 语言
    :param cmd: 命令
    :param ini_type: 类型
    :return:
    """
    host_ini_dict = {}
    flag, cliRet, errMsg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return host_ini_dict

    if flag is not True:
        raise UnCheckException(errMsg, cliRet, flag)

    tmp_res_dict = getHorizontalCliRet(cliRet)
    for record in tmp_res_dict:
        wwn = record.get(ini_type, "")
        host_id = record.get("Host ID", "")
        runningStatus = record.get("Running Status")
        if runningStatus != "Online":
            continue

        tmp_ini_list = host_ini_dict.get(host_id, [])
        tmp_ini_list.append(wwn)
        host_ini_dict[host_id] = tmp_ini_list

    return host_ini_dict


def get_cmd_res(sqlite_manager, sqlite_conn, cmd, sn, lang="en"):
    flag, cli_ret, err_msg = query_one_cmd(
        sqlite_manager, sqlite_conn, cmd, sn
    )
    if flag is None:
        # 不存在于数据库
        err_msg = HostResource.getMsg(lang, "get.hyper.data.failed", (sn, cmd))
        return RESULT_NOCHECK, "", err_msg
    return flag, cli_ret, err_msg


class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """

    def __init__(self, errorMsg, cliRet, flag=None):
        self.errorMsg = errorMsg
        self.cliRet = cliRet
        self.flag = flag


CLI_RET_END_FLAG = ":/>"


def getHorizontalCliRet(cliRet):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    headline = ""
    i = 0
    cliRetList = cliRet.encode("utf8").splitlines()
    for line in cliRetList:
        reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            headline = match_headline.group()
            break
        i += 1
    if headline == "" or i == 0 or i >= len(cliRetList) - 1:
        return []

    title = cliRetList[i - 1]
    field_words = cliRetList[(i + 1) :]
    reg_split = re.compile(r"\s*-+\s*")
    tuple_idxs = []
    start_pos = 0
    end_pos = 0

    while start_pos <= len(headline):
        match = reg_split.search(headline, start_pos)
        if match:
            end_pos = match.end()
            tuple_idxs.append((start_pos, end_pos))
            start_pos = end_pos
        else:
            break

    keys = []
    for item in tuple_idxs:
        key = title[item[0] : item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key.decode("utf8"))

    requiredLineLen = tuple_idxs[-1][0]
    dictList = []
    for line in field_words:
        if CLI_RET_END_FLAG in line:
            break

        # 标题换行的场景
        if re.search(r"^-+(\s+-+)*\s*$", line):
            continue

        if len(line.strip()) == 0:
            continue

        if len(line) <= requiredLineLen:
            continue

        vals = []
        for item in tuple_idxs:
            vals.append(line[item[0] : item[1]].strip().decode("utf8"))
        dictList.append(dict(zip(keys, vals)))

    return dictList


def get_horizontal_nostandard_cli_ret(cli_ret):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,
    此方法用来解析CLI回显未对其情况
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为
    键的字典集合,处理不正常时，返回空集合
    """
    head_line = ""
    i = 0
    cli_ret_list = cli_ret.encode("utf8").splitlines()
    for line in cli_ret_list:
        reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            head_line = match_headline.group()
            break
        i += 1
    if head_line == "" or i == 0 or i >= len(cli_ret_list) - 1:
        return []

    title = cli_ret_list[i - 1]
    field_words = cli_ret_list[(i + 1) :]
    reg_split = re.compile(r"\s*-+\s*")
    tuple_idxs = []
    start_pos = 0
    end_pos = 0

    while start_pos <= len(head_line):
        match = reg_split.search(head_line, start_pos)
        if match:
            end_pos = match.end()
            tuple_idxs.append((start_pos, end_pos))
            start_pos = end_pos
        else:
            break

    keys = []
    for item in tuple_idxs:
        key = title[item[0] : item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key.decode("utf8"))

    lenkeys = len(keys)
    dict_list = []
    for line in field_words:
        if line.find(":/>") >= 0:
            break

        if re.search(r"^-+(\s+-+)*\s*$", line):
            continue

        if len(line.strip()) == 0:
            continue

        vals = []
        value_list = line.strip().split("  ")

        value_space_list = []
        for value in value_list:
            if value != "":
                vals.append(value.strip().decode("utf8"))
                value_space_list.append(value)
        lenvalue = len(value_space_list)
        if lenvalue == lenkeys:
            dict_list.append(dict(zip(keys, vals)))

    return dict_list


def getVerticalCliRet(cliRet):
    """
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1 : len(fields)]).strip().decode("utf8")

        if key in lineDict:
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList


def get_mapping_initiator_from_config(
    mem_obj,
    p_version,
    sn,
    hyper_lun_dict,
    ini_port_dict,
    ini_on_host_list,
    host_fc_ini_dict,
    host_iscsi_ini_dict,
    logger,
):
    """
    从config中获取mapping下所有对象信息并计算路径
    :param mem_obj: 内存对象
    :param p_version: 存储版本
    :param sn: 存储SN
    :param hyper_lun_dict:双活LUN
    :param ini_port_dict:启动器和端口对应关系
    :param ini_on_host_list:当前主机上的实际启动器
    :param host_fc_ini_dict:阵列上fc启动器和主机对应关系
    :param host_iscsi_ini_dict:阵列上iscsi启动器和主机对应关系
    :param logger:logger
    :return:
    """
    hyper_lun_ini_port_dict = {}
    not_standard_view_list = []
    only_iscsi_view_lun_wwn_list = []
    lun_host_dict = {}
    mapping_info_dict = get_mapping_from_config_v6(
        mem_obj, p_version, sn, logger
    )
    for mapping_id in mapping_info_dict:
        tmp_host_list = []
        tmp_dict = mapping_info_dict.get(mapping_id, {})
        host_group_id = tmp_dict.get("Host Group ID")
        port_group_id = tmp_dict.get("Port Group ID")
        lun_id_list = tmp_dict.get("Lun ID List")
        host_id = tmp_dict.get("Host ID")

        cur_hyper_list = []
        for lun_id in lun_id_list:
            if lun_id not in hyper_lun_dict:
                continue
            cur_hyper_list.append(lun_id)

        # 当前mapping没有双活LUN
        if not cur_hyper_list:
            continue

        # 如果主机不为--，
        if host_id and host_id != "--":
            tmp_host_list.append(host_id)

        # 如果存在主机组，则需要查询对应的全部主机
        if host_group_id and host_group_id != "--":
            host_group_dict = get_host_group_host_from_config(
                mem_obj, p_version, sn, logger
            )
            host_list = host_group_dict.get(host_group_id, [])
            tmp_host_list.extend(host_list)

        # 如果存在端口组，需要查出对应端口信息
        port_group_port_list = []
        if port_group_id and port_group_id != "--":
            port_group_info_dict = get_port_group_port_from_config(
                mem_obj, p_version, sn, logger
            )
            port_group_port_list = port_group_info_dict.get(port_group_id, [])

        # 获取主机组下主机的启动器
        host_ini_fc_wwn_list, host_ini_iscsi_wwn_list = get_host_ini(
            tmp_host_list,
            host_fc_ini_dict,
            host_iscsi_ini_dict,
            ini_on_host_list,
        )

        # 问题引入点，如果所以主机启动器都不在当前主机上，则不检查。
        if not host_ini_fc_wwn_list and not host_ini_iscsi_wwn_list:
            continue

        # 获取FC 路径
        fc_link_list = get_path_ini(
            host_ini_fc_wwn_list, ini_port_dict, port_group_port_list
        )

        # 获取ISCSI路径
        iscsi_link_list = get_path_ini(
            host_ini_iscsi_wwn_list, ini_port_dict, port_group_port_list
        )
        for lun_id in cur_hyper_list:
            ns_list, oi_list = get_lun_path_info(
                hyper_lun_dict,
                lun_id,
                hyper_lun_ini_port_dict,
                fc_link_list,
                iscsi_link_list,
                lun_host_dict,
                tmp_host_list,
            )
            not_standard_view_list.extend(ns_list)
            only_iscsi_view_lun_wwn_list.extend(oi_list)

    # 综合全部的view 计算lun 的路径：
    lun_path_info_dict = {}
    for view_id in hyper_lun_ini_port_dict:
        view_dict = hyper_lun_ini_port_dict[view_id]
        for lun_wwn in view_dict:
            tmp_lun_path_list = view_dict[lun_wwn].get("fc", [])
            lun_path_list = lun_path_info_dict.get(lun_wwn, [])
            lun_path_list.extend(tmp_lun_path_list)
            lun_path_list = list(set(lun_path_list))
            lun_path_info_dict[lun_wwn] = lun_path_list

    return (
        not_standard_view_list,
        only_iscsi_view_lun_wwn_list,
        lun_path_info_dict,
        lun_host_dict,
    )


def joinLines(originLines, postLines):
    """
    @summary: 将postLines追加originLines后
    """
    if not (originLines or postLines):
        return ""

    if not originLines:
        return postLines

    if not postLines:
        return originLines

    return "\n".join([originLines, postLines])


def query_host_id_list(record, sqlite_manager, sqlite_conn, sn, all_ret, lang):
    """
    获取主机id列表
    :param record: 一行lun映射的回文
    :param sqlite_manager:
    :param sqlite_conn: sn对应的数据库连接
    :param sn: 设备SN
    :param all_ret: 汇总的回文
    :return:
    """
    host_id = record.get("Host ID")
    hostGroupId = record.get("Host Group ID")
    host_id_list = []
    if host_id != "--" and host_id != "":
        host_id_list.append(host_id)
    elif hostGroupId and hostGroupId != "--":
        cmd_show_host_group = "show host_group host host_group_id=%s"
        cmd = cmd_show_host_group % hostGroupId
        flag, cliRet, errMsg = get_cmd_res(
            sqlite_manager, sqlite_conn, cmd, sn, lang
        )
        all_ret = joinLines(all_ret, cliRet)
        tmpHostGroupDict = getHorizontalCliRet(cliRet)
        for tmpRecord in tmpHostGroupDict:
            host_id_list.append(tmpRecord.get("ID", ""))
    return host_id_list, all_ret


def query_port_group_list(
    port_group_id, sqlite_manager, sqlite_conn, sn, all_cli_ret, lang
):
    """
    查询端口组包含的端口列表
    :param port_group_id: 端口组ID
    :param sqlite_manager: 数据库管理对象
    :param sqlite_conn: 数据库连接
    :param sn: 阵列SN
    :param all_cli_ret: 回文
    :param lang: 中文
    :return:
    """
    portGroupPortList = []
    if port_group_id != "--" and port_group_id != "":
        # 存在端口组的场景
        cmd_show_port_group = "show port_group port port_group_id=%s"
        cmd = cmd_show_port_group % port_group_id
        flag, cliRet, errMsg = get_cmd_res(
            sqlite_manager, sqlite_conn, cmd, sn, lang
        )
        all_cli_ret = joinLines(all_cli_ret, cliRet)
        ethCliRet = cliRet[: cliRet.find("FC port:")]
        fcCliRet = cliRet[cliRet.find("FC port:") : cliRet.find("FCoE port:")]
        fcoeCliRet = cliRet[
            cliRet.find("FCoE port:") : cliRet.find("IB port:")
        ]
        ibCliRet = cliRet[cliRet.find("IB port:") :]

        # eth port
        tmpPortGroupPortDict = getHorizontalCliRet(ethCliRet)
        for portDict in tmpPortGroupPortDict:
            portGroupPortList.append(portDict.get("ID"))

        # fc port
        tmpPortGroupPortDict = getHorizontalCliRet(fcCliRet)
        for portDict in tmpPortGroupPortDict:
            portGroupPortList.append(portDict.get("ID"))

        # fcoe port
        tmpPortGroupPortDict = getHorizontalCliRet(fcoeCliRet)
        for portDict in tmpPortGroupPortDict:
            portGroupPortList.append(portDict.get("ID"))

        # ib port
        tmpPortGroupPortDict = getHorizontalCliRet(ibCliRet)
        for portDict in tmpPortGroupPortDict:
            portGroupPortList.append(portDict.get("ID"))

    return portGroupPortList, all_cli_ret


def get_host_group_host(host_id_list, host_fc_ini_dict, initiatorOnHostList,
                        host_iscsi_ini_dict, currentCheckHostList):
    # 获取主机组下主机的启动器
    hostIniFcWWNList = []
    hostIniIscsiWWNList = []
    for host_id in host_id_list:
        # 获取主机组下主机的启动器
        tmp_ini_list = host_fc_ini_dict.get(host_id, [])
        tmp_fc_list = [
            ini for ini in tmp_ini_list if ini in initiatorOnHostList
        ]
        hostIniFcWWNList.extend(tmp_fc_list)
        tmp_ini_list = host_iscsi_ini_dict.get(host_id, [])
        tmp_iscsi_list = [
            ini for ini in tmp_ini_list if ini in initiatorOnHostList
        ]
        hostIniIscsiWWNList.extend(tmp_iscsi_list)
        # 当前主机是否需要检查报错。
        if tmp_fc_list or tmp_iscsi_list:
            currentCheckHostList.append(host_id)
    return hostIniFcWWNList, hostIniIscsiWWNList


def get_mapping_view_initiator_dorado_v6(param_dict):
    """
    获取mappview 下的 LUN、主机、路径的对应关系
    :param param_dict:
    :return:
    """
    cli_ret_all = ""
    hyperLunDict = param_dict.get("hyperLunDict")
    notStandardViewList = []
    onlyIscsiViewLunWWNList = []
    lunHostDict = {}
    lun_id_list = hyperLunDict.keys()
    hyperLunIniAndPortLinkDict = {}
    for lun_id in lun_id_list:
        cmd = "show mapping general lun_id=%s" % lun_id
        flag, cliRet, errMsg = get_cmd_res(
            param_dict.get("sqlite_manager"),
            param_dict.get("sqlite_conn"), cmd,
            param_dict.get("sn"), param_dict.get("lang")
        )
        cli_ret_all = joinLines(cli_ret_all, cliRet)
        tmpDict = get_horizontal_nostandard_cli_ret(cliRet)

        cli_ret_all = get_all_view_info_v6(
            tmpDict, param_dict, lun_id, cli_ret_all,
            hyperLunIniAndPortLinkDict, lunHostDict, notStandardViewList,
            onlyIscsiViewLunWWNList
        )

    # 综合全部的view 计算lun 的路径：
    lunPathInfoDict = cal_lun_path(hyperLunIniAndPortLinkDict)
    return (
        notStandardViewList, onlyIscsiViewLunWWNList, lunPathInfoDict,
        lunHostDict, cli_ret_all,
    )


def get_all_view_info_v6(
        tmpDict, param_dict, lun_id, cli_ret_all, hyperLunIniAndPortLinkDict,
        lunHostDict, notStandardViewList, onlyIscsiViewLunWWNList
):
    hyperLunDict = param_dict.get("hyperLunDict")
    sqlite_manager = param_dict.get("sqlite_manager")
    sqlite_conn = param_dict.get("sqlite_conn")
    sn = param_dict.get("sn")
    initiatorAndPortLinkDict = param_dict.get("initiatorAndPortLinkDict")
    initiatorOnHostList = param_dict.get("initiatorOnHostList")
    host_fc_ini_dict = param_dict.get("host_fc_ini_dict")
    host_iscsi_ini_dict = param_dict.get("host_iscsi_ini_dict")
    lang = param_dict.get("lang")
    logger = param_dict.get("LOGGER")
    currentCheckHostList = []
    for record in tmpDict:
        lun_group_id = record.get("LUN Group ID")
        host_group_id = record.get("Host Group ID")
        if lun_group_id == "--" or host_group_id == "--":
            continue

        host_id_list, cli_ret_all = query_host_id_list(
            record, sqlite_manager, sqlite_conn, sn, cli_ret_all, lang
        )
        # 获取主机组下主机的启动器
        hostIniFcWWNList, hostIniIscsiWWNList = get_host_group_host(
            host_id_list, host_fc_ini_dict, initiatorOnHostList,
            host_iscsi_ini_dict, currentCheckHostList
        )
        # LUN的全部主机启动器是否在当前主机上，如果不在则不检查
        if not hostIniFcWWNList and not hostIniIscsiWWNList:
            logger.logInfo("view [%s] no initiator in host." % lun_id)
            continue

        if not currentCheckHostList:
            logger.logInfo("view [%s] no initiator in host." % lun_id)
            continue

        # 获取端口组信息
        portGroupId = record.get("Port Group ID")
        portGroupPortList, cli_ret_all = query_port_group_list(
            portGroupId, sqlite_manager, sqlite_conn, sn, cli_ret_all,
            lang,
        )
        # 获取FC 路径
        fcLinkeList = get_path_ini(
            hostIniFcWWNList, initiatorAndPortLinkDict,
            portGroupPortList
        )
        # 获取ISCSI路径
        iscsiLinkeList = get_path_ini(
            hostIniIscsiWWNList, initiatorAndPortLinkDict,
            portGroupPortList
        )
        # 记录每个view 下LUN对应的fc启动器、iscsi启动器
        tmpLunWWN = hyperLunDict.get(lun_id, "")
        if tmpLunWWN:
            ns_list, oi_list = get_lun_path_info(
                hyperLunDict, lun_id, hyperLunIniAndPortLinkDict,
                fcLinkeList, iscsiLinkeList, lunHostDict,
                currentCheckHostList,
            )
            notStandardViewList.extend(ns_list)
            onlyIscsiViewLunWWNList.extend(oi_list)
    return cli_ret_all


def get_all_mapping_view(sqlite_manager, sqlite_conn, sn, lang):
    view_list = []
    cmd = "show mapping_view general"
    flag, cli_ret, err_msg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    tmp_res_dict = getHorizontalCliRet(cli_ret)
    for tmp_dict in tmp_res_dict:
        mapp_id = tmp_dict.get("Mapping View ID", "")
        if mapp_id:
            view_list.append(mapp_id)
    return view_list, cli_ret


def get_host_group_host_cmd(sql_manager, sqlite_conn, sn, lang, group_id):
    cmd = "show host_group host host_group_id=%s" % group_id
    flag, cli_ret, msg = get_cmd_res(
        sql_manager, sqlite_conn, cmd, sn, lang
    )
    tmp_group_dict_list = getHorizontalCliRet(cli_ret)
    return [record.get("ID") for record in tmp_group_dict_list], cli_ret


def get_view_detail(sqlite_manager, sqlite_conn, sn, lang, mp_id):
    cmd = "show mapping_view general mapping_view_id=%s" % mp_id
    flag, ret, msg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    return getVerticalCliRet(ret), ret


def get_lun_group_lun(sqlite_manager, sqlite_conn, sn, lang, lg_id):
    cmd = "show lun_group lun lun_group_id=%s" % lg_id
    flag, ret, msg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    return getHorizontalCliRet(ret), ret


def get_mapping_view_initiator_ocean(param_dict):
    """
    获取mappview 下的 LUN、主机、路径的对应关系
    :param param_dict:
    :return:
    """
    cli_ret_all = ""
    sqlite_manager = param_dict.get("sqlite_manager")
    sqlite_conn = param_dict.get("sqlite_conn")
    sn = param_dict.get("sn")
    lang = param_dict.get("lang")
    notStandardViewList = []
    onlyIscsiViewLunWWNList = []
    lunHostDict = {}
    mappingViewList, ret = get_all_mapping_view(
        sqlite_manager, sqlite_conn, sn, lang)
    cli_ret_all = joinLines(cli_ret_all, ret)
    hyperLunIniAndPortLinkDict = {}
    for mappId in mappingViewList:
        tmpDict, ret = get_view_detail(
            sqlite_manager, sqlite_conn, sn, lang, mappId)
        cli_ret_all = joinLines(cli_ret_all, ret)
        currentCheckHostList = []
        cli_ret_all = get_all_view_info_ocean(
            tmpDict, param_dict, currentCheckHostList, mappId,
            hyperLunIniAndPortLinkDict, lunHostDict, notStandardViewList,
            onlyIscsiViewLunWWNList, cli_ret_all
        )
    # 综合全部的view 计算lun 的路径：
    lunPathInfoDict = cal_lun_path(hyperLunIniAndPortLinkDict)

    return (
        notStandardViewList, onlyIscsiViewLunWWNList, lunPathInfoDict,
        lunHostDict, cli_ret_all
    )


def get_all_view_info_ocean(
        tmpDict, param_dict, currentCheckHostList, mappId,
        hyperLunIniAndPortLinkDict, lunHostDict, notStandardViewList,
        onlyIscsiViewLunWWNList, cli_ret_all
):
    hyperLunDict = param_dict.get("hyper_lun_dict")
    sqlite_manager = param_dict.get("sqlite_manager")
    sqlite_conn = param_dict.get("sqlite_conn")
    sn = param_dict.get("sn")
    initiatorAndPortLinkDict = param_dict.get("ini_port_dict")
    initiatorOnHostList = param_dict.get("ini_on_host_list")
    host_fc_ini_dict = param_dict.get("host_fc_ini_dict")
    lang = param_dict.get("lang")
    for record in tmpDict:
        lunGroupId = record.get("LUN Group ID")
        hostGroupId = record.get("Host Group ID")
        if lunGroupId == "--" or hostGroupId == "--":
            continue
        tmpDictLun, ret = get_lun_group_lun(
            sqlite_manager, sqlite_conn, sn, lang, lunGroupId)
        cli_ret_all = joinLines(cli_ret_all, ret)
        hyperLunInViewList = []
        for lunDict in tmpDictLun:
            lunID = lunDict.get("ID")
            if lunID not in hyperLunDict:
                continue
            # hyperLunDict中的wwn已经更新为伪装LUN WWN
            hyperLunInViewList.append(lunID)
        # 如果该MappingView不存在双活LUN，则继续检查。
        if not hyperLunInViewList:
            param_dict.get("logger").logInfo("view no hyper lun.")
            continue
        host_id_list, ret = get_host_group_host_cmd(
            sqlite_manager, sqlite_conn, sn, lang, hostGroupId
        )
        cli_ret_all = joinLines(cli_ret_all, ret)
        # 获取主机组下主机的启动器
        hostIniFcWWNList, hostIniIscsiWWNList = get_host_group_host(
            host_id_list, host_fc_ini_dict, initiatorOnHostList,
            param_dict.get("host_iscsi_ini_dict"), currentCheckHostList
        )

        # LUN的全部主机启动器是否在当前主机上，如果不在则不检查
        if (not hostIniFcWWNList and not hostIniIscsiWWNList) or \
                (not currentCheckHostList):
            param_dict.get("logger").logInfo(
                "mapping view {} has no initiator".format(mappId)
            )
            continue

        # 获取端口组信息
        pg_id = record.get("Port Group ID")
        portGroupPortList, cli_ret_all = query_port_group_list(
            pg_id, sqlite_manager, sqlite_conn, sn, cli_ret_all, lang
        )
        # 获取FC 路径
        fcLinkeList = get_path_ini(
            hostIniFcWWNList, initiatorAndPortLinkDict, portGroupPortList
        )
        # 获取ISCSI路径
        iscsiLinkeList = get_path_ini(
            hostIniIscsiWWNList, initiatorAndPortLinkDict,
            portGroupPortList
        )
        for lun_id in hyperLunInViewList:
            ns_list, oi_list = get_lun_path_info(
                hyperLunDict, lun_id, hyperLunIniAndPortLinkDict,
                fcLinkeList, iscsiLinkeList, lunHostDict,
                currentCheckHostList,
            )
            notStandardViewList.extend(ns_list)
            onlyIscsiViewLunWWNList.extend(oi_list)
    return cli_ret_all


def cal_lun_path(lun_ini_and_port_dict):
    """
    综合全部的view 计算lun 的路径：
    :param lun_ini_and_port_dict: 全部元数据
    :return: 
    """
    lun_path_info_dict = {}
    for view_id in lun_ini_and_port_dict:
        view_dict = lun_ini_and_port_dict.get(view_id, {})
        for lun_wwn in view_dict:
            tmp_lun_path_list = view_dict[lun_wwn].get("fc", [])
            lun_path_list = lun_path_info_dict.get(lun_wwn, [])
            lun_path_list.extend(tmp_lun_path_list)
            lun_path_list = list(set(lun_path_list))
            lun_path_info_dict[lun_wwn] = lun_path_list
    return lun_path_info_dict


def get_port_group_port_by_cmd(
    sqlite_manager, sqlite_conn, arraySn, lang, port_group_id
):
    """
    融合存储config中无端口组端口信息，只能通过命令获取
    :param cmd_rets:
    :param port_group_id:
    :param lang:
    :return:
    """
    # 存在端口组的场景
    port_group_port_list = []
    cmd = "show port_group port port_group_id={0}".format(port_group_id)
    flag, cli_ret, err_msg = get_cmd_res(
        sqlite_manager, sqlite_conn, arraySn, lang, cmd
    )
    eth_ret = cli_ret[: cli_ret.find("FC port:")]
    fc_ret = cli_ret[cli_ret.find("FC port:") : cli_ret.find("FCoE port:")]
    fcoe_ret = cli_ret[cli_ret.find("FCoE port:") : cli_ret.find("IB port:")]
    ib_ret = cli_ret[cli_ret.find("IB port:") :]

    # eth port
    tmp_dict = getHorizontalCliRet(eth_ret)
    for port_dict in tmp_dict:
        port_group_port_list.append(port_dict.get("ID"))

    # fc port
    tmp_dict = getHorizontalCliRet(fc_ret)
    for port_dict in tmp_dict:
        port_group_port_list.append(port_dict.get("ID"))

    # fcoe port
    tmp_dict = getHorizontalCliRet(fcoe_ret)
    for port_dict in tmp_dict:
        port_group_port_list.append(port_dict.get("ID"))

    # ib port
    tmp_dict = getHorizontalCliRet(ib_ret)
    for port_dict in tmp_dict:
        port_group_port_list.append(port_dict.get("ID"))

    return port_group_port_list


def get_mapping_view_initiator_from_config_ocean(param_dict):
    """
    融合存储从config中获取mapping view下所有对象信息并计算路径
    :param param_dict: 内存对象
    :return:
    """
    hyper_lun_ini_port_dict = {}
    not_standard_view_list = []
    only_iscsi_view_lun_wwn_list = []
    lun_host_dict = {}
    mem_obj = param_dict.get("mem_obj")
    p_version = param_dict.get("p_version")
    sn = param_dict.get("sn")
    hyper_lun_dict = param_dict.get("hyper_lun_dict")
    ini_port_dict = param_dict.get("ini_port_dict")
    ini_on_host_list = param_dict.get("ini_on_host_list")
    host_fc_ini_dict = param_dict.get("host_fc_ini_dict")
    host_iscsi_ini_dict = param_dict.get("host_iscsi_ini_dict")
    sqlite_manager = param_dict.get("sqlite_manager")
    lang = param_dict.get("lang")
    logger = param_dict.get("logger")

    mapping_view_dict = get_mapping_view_from_config_ocean_stor(
        mem_obj, p_version, sn, logger
    )
    # 从config中获取主机组、端口组、LUN组信息
    host_group_dict = get_host_group_host_from_config(
        mem_obj, p_version, sn, logger
    )
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    lun_group_info_dict = ana_util.get_lun_group_info_from_running_data(
        sqlite_conn
    )

    for record in mapping_view_dict:
        tmp_host_list = []
        lun_group_id = record.get("Lun Group Id")
        host_group_id = record.get("Host Group Id")
        port_group_id = record.get("Port Group Id")
        # 如果任一个group为--，表示此view要么没有映射LUN，要么没有主机
        if host_group_id == "--" or lun_group_id == "--":
            continue

        # lun group 下是否有双活LUN，如果没有的话则不继续查主机组
        cur_hyper_list = get_hyper_host_group(
            lun_group_info_dict, hyper_lun_dict, lun_group_id
        )
        # 当前mapping没有双活LUN
        if not cur_hyper_list:
            continue

        # 如果存在主机组，则需要查询对应的全部主机
        if host_group_id and host_group_id != "--":
            tmp_host_list = host_group_dict.get(host_group_id, [])

        if not tmp_host_list:
            continue

        # 如果存在端口组，需要查出对应端口信息
        port_group_port_list = []
        if port_group_id and port_group_id != "--":
            tmp_p_list = get_port_group_port_by_cmd(
                sqlite_manager, sqlite_conn, sn, lang, port_group_id
            )
            port_group_port_list.extend(tmp_p_list)

        # 获取主机组下主机的启动器
        host_ini_fc_wwn_list, host_ini_iscsi_wwn_list = get_host_ini(
            tmp_host_list, host_fc_ini_dict, host_iscsi_ini_dict,
            ini_on_host_list,
        )

        # LUN的全部主机启动器是否在当前主机上，如果不在则不检查
        if not host_ini_fc_wwn_list and not host_ini_iscsi_wwn_list:
            continue

        # 获取FC 路径
        fc_link_list = get_path_ini(
            host_ini_fc_wwn_list, ini_port_dict, port_group_port_list
        )
        # 获取ISCSI路径
        iscsi_link_list = get_path_ini(
            host_ini_iscsi_wwn_list, ini_port_dict, port_group_port_list
        )

        for lun_id in cur_hyper_list:
            ns_list, oi_list = get_lun_path_info(
                hyper_lun_dict, lun_id, hyper_lun_ini_port_dict, fc_link_list,
                iscsi_link_list, lun_host_dict, tmp_host_list,
            )
            not_standard_view_list.extend(ns_list)
            only_iscsi_view_lun_wwn_list.extend(oi_list)

    # 综合全部的view 计算lun 的路径：
    lun_path_info_dict = cal_lun_path(hyper_lun_ini_port_dict)
    return (
        not_standard_view_list, only_iscsi_view_lun_wwn_list,
        lun_path_info_dict, lun_host_dict,
    )


def get_path_ini(host_ini_wwn_list, ini_port_dict, port_group_port_list):
    """
    将对应启动器的链路计算出来存放。
    :param host_ini_wwn_list:
    :param ini_port_dict:
    :param port_group_port_list:
    :return:
    """
    link_list = []
    for ini in host_ini_wwn_list:
        port_list = ini_port_dict.get(ini, [])
        for port in port_list:
            if port_group_port_list:
                if port in port_group_port_list:
                    link_list.append((ini, port))
                else:
                    continue
            else:
                link_list.append((ini, port))
    return link_list


def get_host_ini(
    tmp_host_list, host_fc_ini_dict, host_iscsi_ini_dict, ini_on_host_list
):
    """
    获取主机启动器信息
    :param tmp_host_list:
    :param host_fc_ini_dict:
    :param host_iscsi_ini_dict:
    :param ini_on_host_list:
    :return:
    """
    host_ini_fc_wwn_list = []
    host_ini_iscsi_wwn_list = []
    for host_id in tmp_host_list:
        tmp_ini_list = host_fc_ini_dict.get(host_id, [])
        host_ini_fc_wwn_list.extend(
            [ini for ini in tmp_ini_list if ini in ini_on_host_list]
        )
        tmp_ini_list = host_iscsi_ini_dict.get(host_id, [])
        host_ini_iscsi_wwn_list.extend(
            [ini for ini in tmp_ini_list if ini in ini_on_host_list]
        )

    return host_ini_fc_wwn_list, host_ini_iscsi_wwn_list


def get_hyper_host_group(lun_group_info_dict, hyper_lun_dict, lun_group_id):
    """
    获取双活主机组信息
    :param lun_group_info_dict:
    :param hyper_lun_dict:
    :param lun_group_id:
    :return:
    """
    cur_hyper_list = []
    for lun_group in lun_group_info_dict:
        if lun_group_id != lun_group.get("Id"):
            continue

        lun_info_list = lun_group.get("LUN INFO", [])
        for lun_info in lun_info_list:
            lun_id = lun_info.get("LUN ID")
            if not lun_id or lun_id == "--":
                continue

            # 如果不是双活LUN则不处理
            if lun_id not in hyper_lun_dict:
                continue

            cur_hyper_list.append(lun_id)
    return cur_hyper_list


def get_lun_path_info(
    hyper_lun_dict,
    lun_id,
    hyper_lun_ini_port_dict,
    fc_link_list,
    iscsi_link_list,
    lun_host_dict,
    tmp_host_list,
):
    """
    计算双活LUN路径信息
    :param hyper_lun_dict:
    :param lun_id:
    :param hyper_lun_ini_port_dict:
    :param fc_link_list:
    :param iscsi_link_list:
    :param lun_host_dict:
    :param tmp_host_list:
    :return:
    """
    not_standard_view_list = []
    only_iscsi_list = []
    # 记录每个view 下LUN对应的fc启动器、iscsi启动器
    tmp_lun_wwn = hyper_lun_dict.get(lun_id, "")
    hy_lun_ini_dict = hyper_lun_ini_port_dict.get(lun_id, {})
    hy_lun_ini_list_dict = hy_lun_ini_dict.get(tmp_lun_wwn, {})

    # 记录每个LUN映射的全部FC启动器
    fc_list = hy_lun_ini_list_dict.get("fc", [])
    fc_list.extend(fc_link_list)
    # 记录每个LUN映射的全部ISCSI启动器
    iscsi_list = hy_lun_ini_list_dict.get("iscsi", [])
    iscsi_list.extend(iscsi_link_list)
    hy_lun_ini_list_dict["fc"] = fc_list
    hy_lun_ini_list_dict["iscsi"] = iscsi_list
    hy_lun_ini_dict[tmp_lun_wwn] = hy_lun_ini_list_dict
    # 保存数据
    hyper_lun_ini_port_dict[lun_id] = hy_lun_ini_dict

    # 如果存在阵列侧多个主机对应同一个真实主机且路径数大于主
    # 机侧的情况，则全部主机都报出来。
    tmp_lun_host_list = lun_host_dict.get(tmp_lun_wwn, [])
    tmp_lun_host_list.extend(tmp_host_list)
    lun_host_dict[tmp_lun_wwn] = tmp_lun_host_list

    # 如果同时存在iscsi和fc
    if fc_list and iscsi_list:
        not_standard_view_list.append(lun_id)

    # 如果只存在iscsi
    if not fc_list and iscsi_list:
        only_iscsi_list.extend(tmp_lun_wwn)

    return not_standard_view_list, only_iscsi_list


def need_check_array(env, array_sn):
    """
    判断是否是关联存储设备
    :param env:
    :param array_sn:
    :return:
    """
    dev_node = env.get("devInfo")
    object_for_py = env.get("objectForPy")
    host_mapped_array_sn = object_for_py.get("host_mapped_array_sn")
    host_sn = str(dev_node.getDeviceSerialNumber())
    return array_sn in host_mapped_array_sn.get(host_sn, {}).get(
        "array_sn", []
    )


def getAllLinkedPortId(cliRet, bondPortEthPortIdList):
    """
    获取所有主机端口
    :param cliRet:
    :param bondPortEthPortIdList:
    :return:
    """
    portDict = {}
    ethList = []
    fcList = []
    fcoeList = []
    ethStr = cliRet[cliRet.find("ETH port"):cliRet.find("FC port")]
    if ethStr and "Host Port:" in ethStr:
        tmpResDict = getHorizontalCliRet(
            ethStr[ethStr.find("Host Port:"):])
        for record in tmpResDict:
            portId = record.get("ID", '')
            # 需要去除绑定端口中的eth端口。
            if portId and portId not in bondPortEthPortIdList:
                ethList.append(record.get("ID", ''))

    fcStr = cliRet[cliRet.find("FC port:"):cliRet.find("FCoE port:")]
    if fcStr:
        tmpResDict = getHorizontalCliRet(fcStr)
        for record in tmpResDict:
            if record.get("ID", ''):
                fcList.append(record.get("ID", ''))

    fcoeStr = cliRet[cliRet.find("FCoE port"):cliRet.find("RoCE port")]
    if fcoeStr:
        tmpResDict = getHorizontalCliRet(fcoeStr)
        for record in tmpResDict:
            if record.get("ID", ''):
                fcoeList.append(record.get("ID", ''))

    portDict["FC"] = fcList
    portDict["FCoE"] = fcoeList
    portDict["ETH"] = ethList

    return portDict


def getPortInfo(sqlite_manager, sqlite_conn, sn, lang):
    """
    获取LUN接管端口信息
    :param sqlite_manager: 数据库管理对象
    :param sqlite_conn: 当前阵列数据库连接
    :param sn: 当前阵列SN
    :param lang: 中英文
    :return:
    """
    ALL_CLI_RET = ''
    portDict = {}
    ibPortList = []
    bondPortEthPortIdList = []
    portDict["IB"] = ibPortList

    cmd = "show bond_port"
    flag, cliRet, errMsg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    ALL_CLI_RET = joinLines(ALL_CLI_RET, cliRet)
    if flag is not True:
        raise UnCheckException(errMsg, ALL_CLI_RET, flag)

    tmpResDict = getHorizontalCliRet(cliRet)
    bondPortList = []
    for record in tmpResDict:
        if record.get("ID", ''):
            bondPortList.append(record.get("ID", ''))
        if record.get("Port ID List", ''):
            bondPortEthPortIdList.extend(
                record.get("Port ID List", '').split(","))
    portDict["BOND_PORT"] = bondPortList

    cmd = "show port general logic_type=Host_Port running_status=link_up"
    flag, cliRet, errMsg = get_cmd_res(
        sqlite_manager, sqlite_conn, cmd, sn, lang
    )
    ALL_CLI_RET = joinLines(ALL_CLI_RET, cliRet)
    if flag is not True:
        raise UnCheckException(errMsg, ALL_CLI_RET, flag)

    portDictTemp = getAllLinkedPortId(cliRet, bondPortEthPortIdList)
    portDict.update(portDictTemp)

    cmd = "show port initiator port_type=%s port_id=%s"
    initiatorPortLinkDict = {}
    for portType in portDict:
        portList = portDict.get(portType, [])
        if not portList:
            continue

        for portId in portList:
            cmdStr = cmd % (portType, portId)
            flag, cliRet, errMsg = get_cmd_res(
                sqlite_manager, sqlite_conn, cmdStr, sn, lang
            )
            ALL_CLI_RET = joinLines(ALL_CLI_RET, cliRet)
            if flag is not True:
                raise UnCheckException(errMsg, ALL_CLI_RET, flag)

            tmpResDict = getHorizontalCliRet(cliRet)
            for record in tmpResDict:
                hostId = record.get("Host ID", '')
                runningStatus = record.get("Running Status", '')
                wwn = record.get("WWN", '')
                if portType == 'ETH':
                    wwn = record.get("iSCSI IQN", '')
                if hostId and hostId != "--" and runningStatus == \
                        "Online" and wwn:
                    # 根据主机来保留启动器信息，不以端口为单位。
                    # 遍历完全部端口则，全部主机的启动器的链路数量也都计算出来了。
                    # 只关注主机ID，启动器wwn,启动器链路数量
                    initiatorPortList = initiatorPortLinkDict.get(wwn, [])
                    initiatorPortList.append(portId)
                    initiatorPortLinkDict[wwn] = initiatorPortList

    return initiatorPortLinkDict, ALL_CLI_RET


def get_nof_lun_wwn_on_host(lun_wwn):
    """
    根据协议转换规则转换阵列侧lun wwn为nof启动器对应的主机上显示的lun wwwn
    新LUN以7开头
    BYTE    0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15
                  |        |
    字符：0 | 1 2 3 4 5 6 | 7 8 9 10 11 12 ...18 19 20 21 | 22 23 ... 32
    NGUID : 7 vendor specific identifier | OUI | vendor specific
    WWN   : 6 OUI | vendor specific identifier | vendor specific
    :param lun_wwn: 阵列侧lun wwn
    :return: 主机上的Lun wwn
    """
    # oui 取1至7个字符
    oui = lun_wwn[1:7]

    # vendor specific identifier 标识 7 至 22个字符。
    vendor_specific_identifier = lun_wwn[7:22]

    # vendor specific identifier extension 标识 取后半段，保持一致。
    vendor_specific_identifier_extension = lun_wwn[22:]
    new_wwn = "7{}{}{}".format(
        vendor_specific_identifier, oui, vendor_specific_identifier_extension
    )
    return new_wwn


def update_lun_alua_from_echo(lun_wwn_dicts, echos):
    """
    抽取公告函数 更新自研多路径ALUA信息
    :param lun_wwn_dicts: 阵列侧待更新双活LUN WWN
    :param echos: 主机侧查询的回文
    :return:
    """
    for lun_wwn in lun_wwn_dicts:
        lun_wwn_upper = lun_wwn.upper()
        if (
            lun_wwn_upper in echos.upper()
            or get_nof_lun_wwn_on_host(lun_wwn_upper)
            in echos.upper()
        ):
            if (
                lun_wwn_dicts.get(lun_wwn).get("hostAluaStatus")
                == ALUA_CONFIGURED_NOT_CHECK
            ):
                lun_wwn_dicts.get(lun_wwn)[
                    "hostAluaStatus"
                ] = ALUA_CONFIGURED_PROPERLY
