# -*- coding: UTF-8 -*-
import re
import os
import constant
import pyEnv
import HostResource
import traceback
from defusedxml import ElementTree

scriptpath = os.path.dirname(os.path.abspath(__file__))

lang = None
ssh = None
versionWhiteList = []
logger = None


def execute(context):
    '''
    @summary: 主机白名单查询
    @param context: 上下文对象
    @return: 检查结果
    '''
    global lang, ssh, versionWhiteList, logger
    lang = pyEnv.getLang(context)
    ssh = pyEnv.getSSH(context)
    logger = pyEnv.getLogger(context)
    # 白名单配置文件
    path_str = "./config/Host_WhiteList.xml"
    white_list_xml_path = os.path.join(scriptpath, path_str)
    # Windows系统版本号映射关系 注意：新增Windows白名单，必须同步修改此文件
    windows_version_map_path = "./config/WindowsVersion.ini"
    tmp_windows_path = os.path.join(scriptpath, windows_version_map_path)
    # 获取设备信息，通过devName获取指定类型的白名单信息
    devName = pyEnv.getDeviceName(context)
    versionWhiteList = _parseWhiteListXml(devName, white_list_xml_path)
    logger.info("[whiteListCheck]devName:%s,whiteList:%s" % (
        devName, str(versionWhiteList)))
    if not versionWhiteList:
        # 获取白名单信息失败，报未检查
        return (constant.RESULT_NOCHECK, constant.CLIRET_NO_INFORMATION,
                HostResource.getMsg(lang, "query.result.abnormal"))

    # 不同的系统，不同的处理流程
    if devName == "Linux":
        return _processHostLinux(context)
    elif devName == "VMware":
        return _processHostVmware(context)
    elif devName == "Windows":
        return _processHostWindows(context, tmp_windows_path)
    elif devName == "Solaris":
        return _processHostSolaris(context)
    elif devName == "HP-UX":
        return _processHostHPUX(context)
    elif devName == "AIX":
        return _processHostAIX(context)
    else:
        # 未匹配到任何操作系统，报未检查
        return (constant.RESULT_NOCHECK, constant.CLIRET_NO_INFORMATION,
                HostResource.getMsg(lang, "Host.NotSupport.error", devName))


def execute_doradov6(context):
    '''
    @summary: 主机白名单查询
    @param context: 上下文对象
    @return: 检查结果
    '''
    global lang, ssh, versionWhiteList, logger
    lang = pyEnv.getLang(context)
    ssh = pyEnv.getSSH(context)
    logger = pyEnv.getLogger(context)
    # 白名单配置文件
    path_str = "./config/Host_WhiteList_DoradoV6.xml"
    white_list_xml_path = os.path.join(scriptpath, path_str)
    # Windows系统版本号映射关系注意：新增Windows白名单，必须同步修改此文件
    windows_version_map_path = "./config/WindowsVersion.ini"
    tmp_windows_path = os.path.join(scriptpath, windows_version_map_path)
    # 获取设备信息，通过devName获取指定类型的白名单信息
    devName = pyEnv.getDeviceName(context)
    versionWhiteList = _parseWhiteListXml(devName, white_list_xml_path)
    logger.info("[whiteListCheck] devName:%s, whiteList: %s" % (devName, str(versionWhiteList)))
    if not versionWhiteList:
        # 获取白名单信息失败，报未检查
        return constant.RESULT_NOCHECK, constant.CLIRET_NO_INFORMATION, HostResource.getMsg(lang, "query.result.abnormal")

    # 不同的系统，不同的处理流程
    if devName == "Linux":
        return _processHostLinux(context)
    elif devName == "VMware":
        return _processHostVmware(context)
    elif devName == "Windows":
        return _processHostWindows(context, tmp_windows_path)
    elif devName == "Solaris":
        return _processHostSolaris(context)
    elif devName == "HP-UX":
        return _processHostHPUX(context)
    elif devName == "AIX":
        return _processHostAIX(context)
    else:
        # 未匹配到任何操作系统，报未检查
        return (constant.RESULT_NOCHECK, constant.CLIRET_NO_INFORMATION,
                HostResource.getMsg(lang, "Host.NotSupport.error", devName))


def _getSupportWindowsInnerVersions(file_path):
    '''
    @summary: 获取Windows系统支持列表（内部版本号）
    @return: innerVersionWhiteList：内部版本号的白名单列表
    '''

    innerVersionWhiteList = []
    try:
        mapFile = open(file_path)
        for line in mapFile.readlines():
            fields = line.split("=")
            if len(fields) != 2:
                continue
            verType = fields[0].strip()
            innerVer = fields[1].strip()
            if verType in versionWhiteList:
                innerVersionWhiteList.append(innerVer)
    except:
        logger.error("[_getSupportWindowsInnerVersions] exception:%s" % (traceback.format_exc()))

    return innerVersionWhiteList


def _parseWhiteListXml(devType, file_name):
    '''
    @summary: 解析白名单配置文件，获取白名单列表
    @param devType: 主机类型
    @return: verWhiteList：白名单列表
    '''
    verWhiteList = []
    # 获取所有的element
    root = ElementTree.parse(file_name)
    hostInfoList = root.getiterator("host")

    # 获取指定设备类型的白名单列表
    for hostInfo in hostInfoList:
        if hostInfo.attrib["type"] == devType:
            versionInfoList = hostInfo.getiterator("version")
            for versionInfo in versionInfoList:
                verWhiteList.append(versionInfo.text)
            break

    return verWhiteList


def _parseHostReleaseInfo(context, hostType):
    '''
    @summary: 确认主机是否在白名单中（公共的处理，部分Linux操作系统适用）
    @param context: 上下文
           hostType: 主机类型
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''

    errMsg = ""
    cliRet = ssh.execCmdWithTimout("cat /etc/*-release", constant.HOST_CMD_SHORT_TIMEOUT)
    if hostType not in cliRet:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    hostVersion = ""
    for line in cliRet.splitlines():
        if line.strip().startswith(hostType):
            versionList = re.findall(r"(\d+\.\d+)", line)
            if versionList:
                hostVersion = hostType + " " + versionList[0].strip()
            break

    # 获取信息有误，或不在白名单中，检查不通过
    logger.info("[whiteListCheck] hostVersion:%s" % hostVersion)
    if not hostVersion:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostLinux(context):
    '''
    @summary: Linux主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    # 获取设备信息和白名单
    devType = pyEnv.getDeviceType(context)
    logger.info("[whiteListCheck] devType:%s" % devType)

    # 不同的系统，不同的处理流程
    if "CentOS" in devType:
        return _processHostLinuxCentOS(context)
    elif "SUSE" in devType:
        return _processHostLinuxSUSE(context)
    elif "Red Hat" in devType:
        return _processHostLinuxRedHat(context)
    elif "Ubuntu" in devType:
        return _processHostLinuxUbuntu(context)
    else:
        #工具报不支持
        return constant.RESULT_FAILED, constant.CLIRET_NO_INFORMATION, \
               HostResource.getMsg(lang, "Host.NotSupport.error", devType)


def _processHostLinuxCentOS(context):
    '''
    @summary: Linux主机白名单检查（CentOS）
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    # 查询CentOS系统的白名单
    return _parseHostReleaseInfo(context, "CentOS")


def _processHostLinuxRedHat(context):
    '''
    @summary: Linux主机白名单检查（RedHat）
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    # 查询RedHat系统的白名单
    return _parseHostReleaseInfo(context, "Red Hat Enterprise Linux")


def _processHostLinuxSUSE(context):
    '''
    @summary: Linux主机白名单检查（SUSE）
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''

    errMsg = ""
    cliRet = ssh.execCmdWithTimout("cat /etc/*-release", constant.HOST_CMD_SHORT_TIMEOUT)
    if "SUSE Linux" not in cliRet:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    hostVersion = "SUSE Linux Enterprise Server"
    for line in cliRet.splitlines():
        if line.strip().startswith("VERSION"):
            hostVersion += " " + line.split("=")[-1].strip()
        elif line.strip().startswith("PATCHLEVEL"):
            hostVersion += " SP" + line.split("=")[-1].strip()
            break

    # 获取信息有误，或不在白名单中，检查不通过
    logger.info("[whiteListCheck] hostVersion:%s" % hostVersion)
    if not hostVersion:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostLinuxUbuntu(context):
    '''
    @summary: Linux主机白名单检查（Ubuntu）
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ssh.execCmdWithTimout("lsb_release -a", constant.HOST_CMD_SHORT_TIMEOUT)
    if "Ubuntu" not in cliRet:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    hostVersion = ""
    for line in cliRet.splitlines():
        if line.strip().startswith("Release:") and "LTS" in cliRet:
            hostVersion = "Ubuntu " + line.split(":")[-1].strip() + " LTS"
            break

    # 获取信息有误，或不在白名单中，检查不通过
    logger.info("[whiteListCheck] hostVersion:%s" % hostVersion)
    if not hostVersion:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostVmware(context):
    '''
    @summary: VMware主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ssh.execCmdWithTimout("vmware -l", constant.HOST_CMD_SHORT_TIMEOUT)
    if "VMware" not in cliRet:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    hostVersion = ""
    for line in cliRet.splitlines():
        if line.strip().startswith("VMware"):
            # 全量信息为：VMware ESXi 5.5.0 Update 3，需要获取去掉小版本（VMware ESXi 5.5 Update 3）
            line = line.replace("GA", "")
            versionList = re.findall(r"(\d+\.\d+\.*\d*)", line)
            if versionList:
                allVer = versionList[0]
                baseVer = re.findall(r"(\d+\.\d+)", allVer)[0]
                hostVersion = line.replace(allVer, baseVer).strip()
            break

    # 获取信息有误，或不在白名单中，检查不通过
    logger.info("[whiteListCheck] hostVersion:%s" % hostVersion)
    if not hostVersion:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostWindows(context, file_path):
    '''
    @summary: Windows主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ""
    # windows需要转化成内部版本号进行判断
    innerVersionWhiteList = _getSupportWindowsInnerVersions(file_path)
    if not innerVersionWhiteList:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    cliRet = ssh.execCmdRemoteTimout("wmic os get version", constant.HOST_CMD_SHORT_TIMEOUT)
    versionList = re.findall("(\d+\.\d+\.\d+)", cliRet)
    if not versionList:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    innerVersion = versionList[0]

    logger.info("[whiteListCheck] hostVersion:%s" % innerVersion)
    if innerVersion not in innerVersionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", innerVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostSolaris(context):
    '''
    @summary: Solaris主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ssh.execCmdWithTimout("cat /etc/release", constant.HOST_CMD_SHORT_TIMEOUT)
    lineList = cliRet.splitlines()
    if len(lineList) < 2:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    # 主机版本
    verFieldList = []
    versionLine = lineList[1]
    fields = versionLine.split()
    if len(fields) >= 3:
        # 获取基础版本
        verFieldList.append((" ").join(fields[0:3]))

    # 获取U版本
    uVersions = re.findall("_u(\d+)wos", versionLine)
    if uVersions:
        verFieldList.append("U" + uVersions[0])
    # 获取平台信息
    if "X86" in versionLine.upper():
        verFieldList.append("for X86")
    elif "SPARC" in versionLine.upper():
        verFieldList.append("for SPARC")

    # 拼装完整版本号
    hostVersion = (" ").join(verFieldList)
    logger.info("[whiteListCheck] hostVersion:%s" % hostVersion)
    if not hostVersion:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostHPUX(context):
    '''
    @summary: HP_UX主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ""
    # 获取设备版本号并转化成有效值
    devType = pyEnv.getDeviceType(context)
    devType = _switchDevType4HPUX(devType)
    if devType not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", devType)

    return constant.RESULT_PASS, cliRet, errMsg


def _processHostAIX(context):
    '''
    @summary: AIX主机白名单检查
    @param context: 上下文
    @return: flag: 白名单检查结果
            cliRet: 检查回文
            errMsg: 错误信息
    '''
    errMsg = ""
    cliRet = ssh.execCmdWithTimout("oslevel -s", constant.HOST_CMD_SHORT_TIMEOUT)
    # 版本信息格式：7100-01-10-1415
    versions = re.findall(r"(\d{2,})-(\d+)-", cliRet)
    if not versions:
        # 获取信息失败，报未检查
        return constant.RESULT_NOCHECK, cliRet, HostResource.getMsg(lang, "query.result.abnormal")

    baseVersion = versions[0][0]
    TLVersion = versions[0][1]

    # 构造的版本格式为：IBM AIX 7.1 TL3
    hostVersion = "IBM AIX " + baseVersion[0] + "." + baseVersion[1] + " TL" + str(int(TLVersion))
    # 不在白名单中，检查不通过
    if hostVersion not in versionWhiteList:
        # 白名单检查失败
        return constant.RESULT_FAILED, cliRet, HostResource.getMsg(lang, "Host.Not.In.WhiteList.error", hostVersion)

    return constant.RESULT_PASS, cliRet, errMsg


def _switchDevType4HPUX(devType):
    if 'B.11.1' in devType.upper():
        return 'HP-UX 11i v1'
    if 'B.11.2' in devType.upper():
        return 'HP-UX 11i v2'
    if 'B.11.3' in devType.upper():
        return 'HP-UX 11i v3'
    return devType
