# -*- coding: UTF-8 -*-

import common
import cliUtil

LOGGER = common.getLogger(PY_LOGGER, __file__)
LANG = common.getLang(py_java_env)

RESULT_CONTINUE = "continue"

REDHAT6 = "Red Hat 6"
REDHAT7 = "Red Hat 7"
SUSE11 = "SUSE 11"
SUSE12 ="SUSE 12"
ORACLELINUX6 = "Oracle Linux 6"
ORACLELINUX7 = "Oracle Linux 7"


def execute(ssh):
    """
    @summary: 华为自研多路径软件安装检查。检查多路径是否安装且安装正常。
    """
    allCliRet = ""
    
    try:
        cliUtil.executePS1(ssh, LOGGER)
        
        common.mark_host_upadmin_hyper_metro_luns(py_java_env, ssh, LOGGER)
        # 检查主机操作系统。
        hostType = common.getDevTypeFromContext(py_java_env)
        flag  = checkHostOS(hostType)
        if flag != RESULT_CONTINUE:
            return (True, allCliRet, "")
        
        # 检查是否安装多路径。
        isInstalled, cliRet = isInstalledMultipath(ssh)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not isInstalled:
            return (True, allCliRet, "")
        
        # 检查多路径版本。
        flag, errMsg, cliRet = checkMultipathVersion(ssh)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != RESULT_CONTINUE:
            return (flag, allCliRet, errMsg)

        # 检查多路径内核模块。
        flag, errMsg, cliRet = checkMultipathKernel(ssh)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag == False:
            return (False, allCliRet, errMsg)

        # 检查多路径用户态服务进程。
        flag, errMsg, cliRet = checkMultipathUpservice(ssh)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag == False:
            return (False, allCliRet, errMsg)

        # 检查是否安装多路径脚本
        isInstalled, errMsg, cliRet = isMultipathScriptInstalled(ssh)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not isInstalled:
            return (False, allCliRet, errMsg)

        # 检查多路径服务
        flag, errMsg, cliRet = checkMultipathService(ssh, hostType)
        allCliRet = common.joinLines(allCliRet, cliRet)
        return (flag, allCliRet, errMsg)

    except cliUtil.UnCheckException, unCheckException:
        LOGGER.logException()
        errMsg = getErrMsg(unCheckException.errCode)
        allCliRet = common.joinLines(allCliRet, unCheckException.cliRet)
        return (unCheckException.flag, allCliRet, errMsg)
    
    except:
        LOGGER.logException()
        errMsg = getErrMsg("get.host.info.failure")
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)


def getInfoFromHost(ssh, cmd):
    """
    @summary: 到主机上执行cmd，获取信息。
    """
    return cliUtil.getInfoFromHost(ssh, cmd, LOGGER)


def getErrMsg(errCode, args = ""):
    """
    @summary: 获取国际化错误消息。
    """
    return common.getMsg(LANG, errCode, args)

   
def iskeyWordInEcho(ssh, cmd, keyWord):
    """
    @summary: 判断关键字是否在命令执行回显中。
    """
    echo = getInfoFromHost(ssh, cmd)
    
    if keyWord in "\n".join(echo.splitlines()[1:]):
        return True, echo
    
    return False, echo


def iskeyWordInEchoes(ssh, cmds, keyWord):
    """
    @summary: 判断关键字是否同时在多条命令执行回显中。
    """
    echoes = ""
    checkFlag = True
    
    for cmd in cmds:
        flag, echo = iskeyWordInEcho(ssh, cmd, keyWord)
        echoes = common.joinLines(echoes, echo)
        if flag == False:
            checkFlag = False
    
    return checkFlag, echoes
            

def getOSNameAndVersion(hostType):
    """
    @summary: 获取主机OS的名称与版本 。
    """
    items = hostType.split()
    
    if len(items) < 2:
        return "", ""
    
    OSName = " ".join(items[:-1])
    OSVersion = items[-1]
    
    return OSName, OSVersion


def isHostOSinList(hostOS, OSlist):
    """
    @summary: 判断主机OS是否在列表中
    """
    for OS in OSlist:
        if OS in hostOS:
            return True
    
    return False


def checkHostOS(hostType):
    """
    @summary: 检查主机是否为需要检查的系统 。
    """
    # 需要检查的主机OS
    checkHostOSList = [REDHAT6, REDHAT7, SUSE11, SUSE12, ORACLELINUX6, ORACLELINUX7]
    
    if isHostOSinList(hostType, checkHostOSList):
        return RESULT_CONTINUE
    
    LOGGER.logInfo("The current system(%s) does not need to check for Multipath." % hostType)
    return True


def isInstalledMultipath(ssh):
    """
    @summary: 检查华为自研多路径是否已经安装。
    """
    keyWord = "UltraPath"
    cmd = "rpm -qa |grep UltraPath"
    flag, cliRet = iskeyWordInEcho(ssh, cmd, keyWord)
    
    if flag == False:
        LOGGER.logInfo("Huawei Self-Test multipath software is not installed in the current system.")
    
    return flag, cliRet


def getMultipathVersion(ssh):
    """
    @summary:获取多路径版本。 
    """
    softwareVersion = ""
    driverVersion = ""
    
    cmd = "upadmin show version"
    cliRet = getInfoFromHost(ssh, cmd)
    
    for line in cliRet.splitlines():
        if ":" not in cliRet:
            continue
        items = line.split(":")
        if len(items) != 2:
            continue
        key = items[0].strip()
        value = items[1].strip()
        if key == "Software Version":
            softwareVersion = value.strip()
        if key == "Driver   Version":
            driverVersion = value.strip()
        if softwareVersion and driverVersion:
            break
    
    if not (softwareVersion and driverVersion):
        errCode = "get.multipath.version.failure"
        LOGGER.logError("get multipath versdion failed!")
        raise cliUtil.UnCheckException(errCode, cliRet)
    
    return softwareVersion, driverVersion, cliRet


def getBigSoftwareVersion(softwareVersion):
    """
    @summary:获取软件大版本（第一个版本号）。 
    """
    bigSoftwareVersion = softwareVersion.split(".")[0]
    if bigSoftwareVersion == "":
        return ""
    try:
        bigSoftwareVersion = int(bigSoftwareVersion)
    except:
        LOGGER.logException()
    
    return bigSoftwareVersion


def checkMultipathVersion(ssh):
    """
    @summary:检查多路径版本有效性。 
    """
    # 获取多路径版本。
    softwareVersion, driverVersion, cliRet = getMultipathVersion(ssh)
    
    # 多路径软件大版本
    bigSoftwareVersion = getBigSoftwareVersion(softwareVersion)
    
    # 多路径大版本小于等于6，则当前方案无法检查，退出检查，检查结果为通过。
    if bigSoftwareVersion <= 6:
        LOGGER.logInfo("Multipath version is less than or equal to 6, the current scenario cannot be checked, exits the check.")
        return True, "", cliRet
    
    # 多路径软件版本与驱动版本不一致，则表示多路径安装存在问题。
    if softwareVersion != driverVersion:
        LOGGER.logInfo("Multipath software version is inconsistent with the driver version and there is a problem with multipath.")
        errCode = "multipath.softwareVersion.not.equal.driverVersion"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet
    
    # 多路径版本有效性检查通过，继续后面的检查
    return RESULT_CONTINUE, "", cliRet
            
            
def checkMultipathKernel(ssh):
    """
    @summary: 检查多路径内核模块是否生效。
    """
    nxupFalg = False
    nxupextFlag = False
    
    cmd = "lsmod |grep nxup"
    cliRet = getInfoFromHost(ssh, cmd)
    
    # 命令执行结果包含关键字nxupext与nxup，且均在第一行，则表示内核模块安装生效。
    for line in cliRet.splitlines():
        items = line.split()
        if not items:
            continue
        if "nxupext" in items[0]:
            nxupextFlag = True
        elif "nxup" in items[0]:
            nxupFalg = True
        if nxupFalg and nxupextFlag:
            break
    
    if not (nxupFalg and nxupextFlag):
        LOGGER.logInfo("The UltraPath kernel module does not take effect.")
        errCode = "multipath.kernel.not.effect"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet
        
    return True, "", cliRet
             
    
def checkMultipathUpservice(ssh):
    """
    @summary: 检查多路径的用户态服务进程upService是否生效。
    """
    keyWord = "otherTools/upservice"
    cmd = "ps -ef |grep upservice"
    
    flag, cliRet = iskeyWordInEcho(ssh, cmd, keyWord)
    
    if flag == False:
        LOGGER.logInfo("The user-mode service process upService of UltraPath does not take effect.")
        errCode = "multipath.upService.not.effect"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet

    return True, "", cliRet


def isMultipathScriptInstalled(ssh):
    """
    @summary: 检查是否安装多路径脚本。 
    """
    keyWord = "nxup"
    cmd = "ls /etc/init.d/ |grep nxup" 
    
    flag, cliRet = iskeyWordInEcho(ssh, cmd, keyWord)
    
    if flag == False:
        LOGGER.logInfo("The UltraPath script is not installed.")
        errCode = "multipath.script.not.installed"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet

    return True, "", cliRet


def checkMultipathService4RedHat6(ssh):
    """
    @summary: 检查以RedHat6为代表的OS的多路径服务是否安装正常。 包含RedHat6，Suse11，Oracle Linux6。
    """
    keyWord = "nxup"
    cmds = ["ls /etc/rc.d/rc3.d/ |grep nxup", "ls /etc/rc.d/rc5.d/ |grep nxup"]
    
    flag, cliRet = iskeyWordInEchoes(ssh, cmds, keyWord)
    
    if flag == False:
        LOGGER.logInfo("The UltraPath service is not installed.")
        errCode = "multipath.service.not.installed.properly"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet
    
    return True, "", cliRet


def getMultipathInstallationMethod(ssh):
    """
    @summary: 获取多路径软件安装方式
    """
    cmd = "cat /etc/UltraPath.packages.info"
    cliRet = getInfoFromHost(ssh, cmd)
    
    for line in cliRet.splitlines():
        if "=" not in line:
            continue
        items = line.split("=")
        if len(items) != 2:
            continue
        key = items[0].strip()
        if key == "t140s_solutionType":
            return items[1].strip(), cliRet
    
    return "", cliRet


def checkMultipathService4RedHat7(ssh):
    """
    @summary: 检查以RedHat7为代表的OS的多路径服务是否安装正常。包含RedHat7，Oracle Linux7。
    """
    allCliRet = ""
    
    # 获取多路径安装方式
    installationMethod, cliRet = getMultipathInstallationMethod(ssh)
    LOGGER.logInfo("The multipath installation method is %s." % installationMethod)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if installationMethod == "mkinitrd":
        return True, "", allCliRet
    
    keyWord = "nxup"
    cmds = [ "ls -l /etc/systemd/system/sysinit.target.wants |grep systemd-nxup.service",
             "ls -l /usr/lib/systemd/system/ |grep systemd-nxup.service",
             "ls -l /usr/lib/systemd/system/sysinit.target.wants/ |grep systemd-nxup.service" ]
    
    flag, cliRet = iskeyWordInEchoes(ssh, cmds, keyWord)
    allCliRet = common.joinLines(allCliRet, cliRet)
    
    if flag == False:
        LOGGER.logInfo("The UltraPath service is not installed.")
        errCode = "multipath.service.not.installed.properly"
        errMsg = getErrMsg(errCode)
        return False, errMsg, allCliRet
    
    return True, "", allCliRet


def checkMultipathService4Suse12(ssh):
    """
    @summary: 检查Suse12的多路径服务是否安装正常。
    """
    keyWord = "nxup.service"
    cmds = [ "ls -l /usr/lib/systemd/system |grep nxup",
             "ls -l /etc/systemd/system/multi-user.target.wants/ |grep nxup"]
    
    flag, cliRet = iskeyWordInEchoes(ssh, cmds, keyWord)
    
    if flag == False:
        LOGGER.logInfo("The UltraPath service is not installed.")
        errCode = "multipath.service.not.installed.properly"
        errMsg = getErrMsg(errCode)
        return False, errMsg, cliRet
    
    return True, "", cliRet


def checkMultipathService(ssh, hostType):
    """
    @summary: 检查多路径服务是否安装正常。 
    """
    
    if isHostOSinList(hostType, [REDHAT6, SUSE11, ORACLELINUX6]):        
        return checkMultipathService4RedHat6(ssh)
    
    if isHostOSinList(hostType, [REDHAT7, ORACLELINUX7]):
        return checkMultipathService4RedHat7(ssh)
        
    if isHostOSinList(hostType, [SUSE12]):
        return checkMultipathService4Suse12(ssh)
    
    return True, "", ""
        
        