# -*- coding: UTF-8 -*-
import re
import resource
import os
import datetime
import logger
import cliUtil
import sys

script_path = os.path.dirname(os.path.abspath(__file__))
common_path = os.path.join(
    script_path, os.path.join("..", "..", "HOST_Common")
)
sys.path.append(common_path)
import host_common

CLI_RET_KEY = 'cliRet'
ERROR_MSG_KEY = 'errMsg'
FLAG_KEY = 'flag'
DIR_RELATIVE_CMD = "..{}..".format(os.sep)
HYPERMETRO_DATA = "{0}tools{0}inspector{0}hypermetroData".format(os.sep)
SLEEP_TIMES = 30
PORTS_BIT_ERROR_SPEC = 60
PORTS_BIT_ERROR_INTERVAL = 30
SERIAL_NUMBER_LENGTH = 20
THRESHOLD_BBU_REMAINING_LIFETIME = 90
THRESHOLD_CONTRL_CPU_USAGE = 80
ALUA_CONFIGURED_PROPERLY = 1
ALUA_CONFIGURED_FALSELY = -1
ALUA_CONFIGURED_NOT_CHECK = 0
HOST_CMD_TIMEOUT = 5 * 60
HOST_CMD_SHORT_TIMEOUT = 2 * 60
LINUX_UPSCAN_CMD_TIMEOUT = 10 * 60
FILE_SUFFIX = "."


def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")


def getCurDeviceInfo(py_java_env):
    """
    @summary: query the device info from java env context
    """
    dev = py_java_env.get("devInfo")

    return dev


def getMsg(lang, msg, args=""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"

    try:
        if not resource.MESSAGES_DICT.has_key(msg):
            return errMsg

        localeDict = resource.MESSAGES_DICT.get(msg)
        if not localeDict.has_key(lang):
            return errMsg

        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg


def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)


def getBaseName(file_path):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name


def getSnFromDevNode(devNode):
    """
    @summary: 通过DevNode.java对象获取设备SN。
    """
    return devNode.getDeviceSerialNumber()


def getIpFromDevNode(devNode):
    """
    @summary: 通过DevNode.java对象获取设备IP。
    """
    return devNode.getIp()


def getNameFromDevNode(devNode):
    """
    @summary: 通过DevNode.java对象获取设备名称。
    """
    return devNode.getDeviceName()


def getRemoteSNsFromDevNode(devNode):
    """
    @summary: 通过DevNode.java对象获取设备关联的远程设备SN。
    """
    return devNode.getRemoteSNs()


def getDevNodeFromContext(py_java_env):
    """
    @summary: 从工具箱的缓存中获取当前设备DevNode对象。
    """
    return py_java_env.get("devInfo")


def getDevTypeFromContext(py_java_env):
    """
    @summary: 从工具箱的缓存中获取当前设备的产品类型
    """
    devNode = getDevNodeFromContext(py_java_env)
    if not devNode:
        return ""
    return str(devNode.getDeviceType())


def mark_host_upadmin_hyper_metro_luns(context, sshCon, LOGGER):
    '''
    @summary: get hyperMetro luns on host whose been designated to upadmin
    '''
    used_utralpath = 1
    not_used_utralpath = 2
    echos = ''
    lun_wwn_dicts = context.get("allStrgHyprMtrLns")
    cmd = "upadmin show vlun type=hypermetro"

    objectForPy = context.get("objectForPy")
    host_dict = objectForPy.get("hostUseUtralPathInitiatorDict")
    if not host_dict:
        host_dict = {}
    devNode = context.get("devInfo")
    # 主机侧FC启动器信息
    hostFCLauncherWwns = devNode.getHostFCLauncherWwns()
    # 主机侧ISCSI启动器信息
    hostISCSILauncherWwns = devNode.getHostISCSILauncherWwns()
    if not hostFCLauncherWwns:
        hostFCLauncherWwns = []
    if not hostISCSILauncherWwns:
        hostISCSILauncherWwns = []
    fcList = list(hostFCLauncherWwns)
    iscsiList = list(hostISCSILauncherWwns)
    fcList.extend(iscsiList)
    nvme_wwns = devNode.getHostNvmeLauncherWwns()
    if nvme_wwns:
        fcList.extend(nvme_wwns)
    iniList = [initiator.lower() for initiator in fcList]
    host_type = str(devNode.getDeviceName())
    isSucess, echos, resultMsg = cliUtil.executeCmdWithTimeout(sshCon, cmd, LOGGER)
    if not isSucess:
        return False, echos
    if "lun wwn" not in echos.lower() and '-------' not in echos:
        LOGGER.logInfo("no such command or no vlun detected.")
        for ini in iniList:
            host_dict[ini] = {"isUseUtrapath": not_used_utralpath,
                              "hostType": host_type
                              }
        objectForPy.put("hostUseUtralPathInitiatorDict", host_dict)
        return False, echos
    host_common.update_lun_alua_from_echo(lun_wwn_dicts, echos)

    for ini in iniList:
        host_dict[ini] = {"isUseUtrapath": used_utralpath,
                          "hostType": host_type
                          }
    objectForPy.put("hostUseUtralPathInitiatorDict", host_dict)

    return True, echos


def joinLines(originLines, postLines):
    """
    @summary: 将postLines追加originLines后
    """
    if not (originLines or postLines):
        return ""

    if not originLines:
        return postLines

    if not postLines:
        return originLines

    return "\n".join([originLines, postLines])


class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """

    def __init__(self, errorMsg, cliRet, flag=None):
        self.errorMsg = errorMsg
        self.cliRet = cliRet
        self.flag = flag


CLI_RET_END_FLAG = ":/>"


def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            # 标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def get_horizontal_nostandard_cli_ret(cli_ret):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,
    此方法用来解析CLI回显未对其情况
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为
    键的字典集合,处理不正常时，返回空集合
    """
    head_line = ""
    i = 0
    cli_ret_list = cli_ret.encode("utf8").splitlines()
    for line in cli_ret_list:
        reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            head_line = match_headline.group()
            break
        i += 1
    if head_line == "" or i == 0 or i >= len(cli_ret_list) - 1:
        return []

    title = cli_ret_list[i - 1]
    field_words = cli_ret_list[(i + 1):]
    reg_split = re.compile(r"\s*-+\s*")
    tuple_idxs = []
    start_pos = 0
    end_pos = 0

    while (start_pos <= len(head_line)):
        match = reg_split.search(head_line, start_pos)
        if match:
            end_pos = match.end()
            tuple_idxs.append((start_pos, end_pos))
            start_pos = end_pos
        else:
            break

    keys = []
    for item in tuple_idxs:
        key = title[item[0]:item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key.decode("utf8"))

    lenkeys = len(keys)
    dict_list = []
    for line in field_words:
        if line.find(":/>") >= 0:
            break

        if re.search(r"^-+(\s+-+)*\s*$", line):
            continue

        if len(line.strip()) == 0:
            continue

        vals = []
        value_list = line.strip().split("  ")

        value_space_list = []
        for value in value_list:
            if value != "":
                vals.append(value.strip().decode("utf8"))
                value_space_list.append(value)
        lenvalue = len(value_space_list)
        if lenvalue == lenkeys:
            dict_list.append(dict(zip(keys, vals)))

    return dict_list


def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList


def get_time_difference(old_time, now_time):
    '''
    @summary: 获取时差，格式：（年 - 月 - 日），单位：天
    '''
    try:
        old_time_year, old_time_month, old_time_day = list(map(int, old_time.split("-")))
        now_time_year, now_time_month, now_time_day = list(map(int, now_time.split("-")))

        time_difference = (
                datetime.datetime(now_time_year, now_time_month, now_time_day) - datetime.datetime(old_time_year,
                                                                                                   old_time_month,
                                                                                                   old_time_day)).days
        return time_difference

    except Exception:
        return -1


def check_dpa_environment(ssh, dpa_logger):
    """
    判断DPA设备当前环境，返回True为单机环境，返回False为分布式环境，并返回命令执行回显
    """
    cmd = "id fsadmin"
    cli_ret = ssh.execCmd(cmd)
    if "no such user" in cli_ret:
        dpa_logger.logInfo("There is a only_node system")
        return True, cli_ret
    dpa_logger.logInfo("There is a distributed system")
    return False, cli_ret


def get_dpa_version(ssh, dpa_logger):
    '''
    @summary: 获取DPA设备当前版本
    '''
    cmd = 'cat /opt/CDMServer/VersionDetails |grep "Package Version"'
    cli_ret = cliUtil.getInfoFromHost(ssh, cmd, dpa_logger)
    cli_ret_list = cli_ret.encode("utf8").splitlines()
    if len(cli_ret_list) < 3:
        return "", cli_ret
    check_info = cli_ret_list[1].split("=")
    if len(check_info) < 2:
        return "", cli_ret
    return cli_ret_list[1].split("=")[1], cli_ret


def compare_dpa_version(old_version, new_version):
    '''
    @summary: 比较DPA设备版本，格式：（X.X.X.X）
    @param old_version: 较旧的版本号
    @param new_version: 较新的版本号
    @return: 返回版本比较结果，若检查结果版本号new_version大于old_version返回True，否则返回False
    '''
    old_version_numbers = old_version.split(".")
    new_version_numbers = new_version.split(".")
    for old_number, new_number in zip(old_version_numbers, new_version_numbers):
        if int(new_number) > int(old_number):
            return True
        elif int(new_number) < int(old_number):
            return False
    return False


def eoss_client_tool_pre_operation(ssh, dpa_version):
    '''
    @summary: eoss_client_tool工具使用的前置命令，包含进入工具目录，更改权限及设置环境变量。不同版本的工具路径有区别。
    @param dpa_version: dpa设备版本
    @return: 将执行结果记录及错误信息返回，若无执行异常，错误信息返回空字符串
    '''
    total_cli_ret = list()
    cmd = 'cd /opt/CDMServer/HWClientService/'
    total_cli_ret.append(ssh.execCmd(cmd))
    flag = compare_dpa_version("8.0.1.7", dpa_version)
    if flag:
        cmd = 'cd AggregateApp/'
        total_cli_ret.append(ssh.execCmd(cmd))
    cmd = 'chmod +x eoss_client_tool'
    total_cli_ret.append(ssh.execCmd(cmd))
    cmd = 'export LD_LIBRARY_PATH=`pwd`'
    total_cli_ret.append(ssh.execCmd(cmd))
    return "\n".join(total_cli_ret), ""
