# -*- coding: UTF-8 -*-
import re
import cliUtil
import common
import config
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
STANDARD_LUN_WWN_LENGTH = 32
VMWARE_VERSION = 'VMware ESXi %s %s'
POISTION_NOT_FETCHED = -1
HOST_CMD_TIMEOUT = 5 * 60
VMWARE_VERSION_5_0 = 5.0
VMWARE_VERSION_5_1 = 5.1
VMWARE_VERSION_5_5 = 5.5
VMWARE_VERSION_6_0 = 6.0

ISCSI_UP_MIN_SUPPORT_VER = '8.06.061'
FC_UP_MIN_SUPPORT_VER = '21.0.2'



def execute(ssh):
    """
    summary:Vmware双活PDL配置检测
    :param ssh: connection
    :return: result, echos, erro message
    """
    context = py_java_env
    remoteStrgArrayRisks = []
    localConfigRisks = []
    failedLocalQueryItems = []
    echos = ''
    errorMsg = ""
    try:
        exec_result, cli_flag, cli_ret = common.\
            mark_host_upadmin_hyper_metro_luns(context, ssh, LOGGER)
        echos += cli_ret
        if 'unknown command' in cli_ret.lower():
            LOGGER.logInfo("Host not used UltraPath")
            return True, echos, ''
        if 'not found' in cli_ret.lower() or 'find any vluns' in \
                cli_ret.lower():
            LOGGER.logInfo("no hyperMetroLuns "
                           "detected on current device,quit")
            return True, echos, ''
        if not exec_result:
            return cliUtil.RESULT_NOCHECK, echos, \
                   common.getMsg(LANG, "hyper.metro.pair.query.failure")
        vlun_list = analysis_vlun_list(cli_ret)
        if len(vlun_list) == 0:
            LOGGER.logInfo("no hyperMetroLuns "
                           "detected on current device,quit")
            return True, echos, ''
        # If it is a cross-network, the APD to PDL
        # must be off, otherwise the check fails.
        if all_lun_bilateral_mapping_and_normal(vlun_list):
            LOGGER.logInfo("all hyperMetroLuns bil"
                           "ateral mapping and normal on current device")
            is_apd_check_suc, apd_2_pdl_conf, cli_ret = \
                getApd2PdlModeConfig(context, ssh)
            echos += cli_ret
            if not is_apd_check_suc:
                return cliUtil.RESULT_NOCHECK, echos, \
                       common.getMsg(LANG,
                                     "hyper.metro.adptopdl.query.failure")
            LOGGER.logInfo('current host APD to PDl mode configuration: %s'
                           % str(apd_2_pdl_conf))
            if apd_2_pdl_conf.strip().lower() == 'off':
                return True, echos, ''
            else:
                return False, echos, common.\
                    getMsg(LANG, "hyper.crossnetwork.apdtopdl.status.error")
        LOGGER.logInfo("not all hyperMetroLuns bilateral mapping "
                       "and normal on current device,continue check")
        is_succ, storage_sn_list = getVmwareAALunStorageSNs(context,
                                                            vlun_list)
        if not is_succ:
            return cliUtil.RESULT_NOCHECK, echos, \
                   common.getMsg(LANG, "hyper.metro.pair.query.failure")
        if not storage_sn_list:
            LOGGER.logInfo('current device has no hyperMetro '
                           'lun mapped to it,quit.')
            return True, echos, ''
        finalResult, hostCheckParams, echo, failedLocalQueryItems = \
            getHostCheckParams(context, ssh)
        echos += echo
        if not finalResult:
            return parseCheckingResult(remoteStrgArrayRisks, localConfigRisks,
                                       failedLocalQueryItems, [], [], echos)
        isOsOld, apd_2_pdl_conf, upSoftwareVer, osVersion = hostCheckParams
        if osVersion < VMWARE_VERSION_5_0:
            LOGGER.logInfo('current host os version does not support UltraPath or hyperMetro lun mapping, quit.')
            return cliUtil.RESULT_NOSUPPORT, echos, ''

        isSuc, sn2IdDict, cli_ret = \
            getStorageArraySn2IdMap(context, ssh, storage_sn_list)
        echos += cli_ret
        if not isSuc or not sn2IdDict:
            return False, echos, ''
        check_result, check_cli, check_err_msg = \
            array_fc_and_iscsi_check(context, ssh, sn2IdDict,
                                     hostCheckParams)
        echos += check_cli
        return check_result, echos, check_err_msg
    except BaseException, exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, echos, common.getMsg(LANG, "query.result.abnormal")
    return True, echos, ''


def array_fc_and_iscsi_check(context, ssh, sn_2_id_dict, host_check_params):
    """
    @summary:If the host uses the ISCSI link to the array, go to Step 5.
    If the host uses the FC link to the array, go to Step 6.
    Check whether the host and the array meet the inspection rules in
    the ISCSI network and the FC network.
    :param context: py_java_env
    :param ssh: Vmware host SSH link
    :param sn_2_id_dict: Host connection dual live
    array SN-host mapping Array Id dictionary
    :param host_check_params:isOsOld, apd2PdlConf, upSoftwareVer, osVersion
    :return:total_result, err_msg, echos
    """
    total_result = True
    echos = ''
    err_msg = ''
    for sn in sn_2_id_dict.keys():
        iscsi_link_check = False
        fc_link_check = False
        disk_array_id = sn_2_id_dict[sn]
        # get iSCSI
        i_scsi_cmd = "esxcli upadm show path -a %s |grep iSCSI | wc -l"\
                     % disk_array_id.strip()
        excute_success, cli_ret, get_iscsi_err_msg = cliUtil.\
            executeHostCmd(context, ssh, i_scsi_cmd)
        echos += cli_ret
        if not excute_success:
            LOGGER.logInfo(
                "found remote storage array(SN:%s) "
                "failed to query up show path command" % sn)
            return cliUtil.RESULT_NOCHECK, common.\
                getMsg(LANG,
                       "hyper.crossnetwork.get.iscsi.error",
                       sn), echos
        for line in cli_ret.splitlines():
            if line.strip().isdigit() and int(line.strip()) > 0:
                iscsi_link_check = True
                break
        # get FC
        fc_cmd = "esxcli upadm show path -a %s |grep FC | wc -l" % \
                 disk_array_id.strip()
        excute_success, cli_ret, get_fc_err_msg = \
            cliUtil.executeHostCmd(context, ssh, fc_cmd)
        echos += cli_ret
        if not excute_success:
            LOGGER.logInfo(
                "found remote storage array(SN:%s) "
                "failed to query up show path command" % sn)
            return cliUtil.RESULT_NOCHECK, common.\
                getMsg(LANG,
                       "hyper.crossnetwork.get.fc.error", sn), echos
        for line in cli_ret.splitlines():
            if line.strip().isdigit() and int(line.strip()) > 0:
                LOGGER.logInfo(
                    "found remote storage array "
                    "using FC path to current device")
                fc_link_check = True
                break
        # check iSCSI network
        if iscsi_link_check:
            iscsi_check, iscsi_error_msg = \
                check_iscsi_net_work(sn, context, host_check_params)
            total_result = total_result and iscsi_check
            if iscsi_error_msg != '':
                err_msg += common.getMsg(LANG,
                                         "hyper.lun.network.iscsi.error",
                                         (sn, iscsi_error_msg))
        # check FC network
        if fc_link_check:
            fc_check, fc_error_msg = check_fc_net_work(sn, context,
                                                       host_check_params)
            total_result = total_result and fc_check
            if fc_error_msg != '':
                err_msg += common.getMsg(LANG, "hyper.lun.network.fc.error",
                                         (sn, fc_error_msg))
    return total_result, echos, err_msg


def check_iscsi_net_work(device_sn, context, host_check_params):
    """
    @summary:
    Step 5 In a Fibre Channel network environment, if the version is
    V300R006C00SPC100 or later or Dorado V300R001C01SPC100 or later and
    the value of APD to PDL Mode is on,
    the check result is Not passed. In this case, disable this function.
    :param device_sn: Double active array SN
    :param context: py_java_env
    :param host_check_params: is_os_old, apd_2_pdl_conf,
    up_software_ver, os_version
    :return: check_flag, err_msg
    """
    check_flag = True
    err_msg = ''
    # checking remote storage array software version
    is_os_old, apd_2_pdl_conf, up_software_ver, os_version = \
        host_check_params
    device_type, software_ver = getRemoteDeviceVersion(context, device_sn)
    is_risk_ver = isCurStrgArrayVerTooOldAtRisk(device_type, software_ver)
    if is_risk_ver:
        LOGGER.logInfo("current storage device[SN:%s]"
                       " has iscsi path and is risk version" % device_sn)
        check_flag = False
        err_msg += common.getMsg(LANG, "software.version.need.2.upgrade",
                                 (device_sn, software_ver))

    # checking UltraPath software version
    if isUpadmVersionLowerThanDest(up_software_ver,
                                   ISCSI_UP_MIN_SUPPORT_VER):
        LOGGER.logInfo("current host has iscsi path UP is risk version")
        check_flag = False
        err_msg += common.\
            getMsg(LANG,
                   common.getMsg(LANG,
                                 "upadmin.software.version.suggest.upgde",
                                 (up_software_ver,
                                  ISCSI_UP_MIN_SUPPORT_VER)))
    # checking APD 2PDL mode
    if apd_2_pdl_conf.strip().lower() == 'on':
        LOGGER.logInfo("current host's apd2pdl configuration is at risk")
        check_flag = False
        err_msg += common.getMsg(LANG, "apd2pdl.switch.suggest.upgde",
                                 (apd_2_pdl_conf, 'off'))
    return check_flag, err_msg


def check_fc_net_work(device_sn, context, host_check_params):
    '''
    @summary:If the version is V300R003C20SPC200 or earlier, or is earlier
     than Dorado V3R1C01SPC100, and the value of APD to PDL Mode is on,
     the ESXi version is 5.0.
    Check whether the UltraPath version is later than 21.0.2. If the
    UltraPath version is not later than 21.0.2, upgrade UltraPath to 21.1.0
    :param device_sn: Double active array SN
    :param context: py_java_env
    :param host_check_params: is_os_old, apd_2_pdl_conf,
    up_software_ver, os_version
    :return: check_flag, err_msg
    '''
    check_flag = True
    err_msg = ''
    is_os_old, apd_2_pdl_conf, up_software_ver, os_version = \
        host_check_params
    device_type, software_ver = \
        getRemoteDeviceVersion(context, device_sn)
    is_old_ver = isCurStrgArrayVerTooOldAtRisk(device_type, software_ver)
    is_new_ver = isCurStrgArrayVerNew(device_type, software_ver)
    if is_old_ver and is_os_old \
            and isUpadmVersionLowerThanDest(up_software_ver,
                                            FC_UP_MIN_SUPPORT_VER):
        LOGGER.logInfo("storage array(SN%s)  FC , "
                       "array, os version & UP version is too old"
                       % str(device_sn))
        check_flag = False
        err_msg += \
            common.getMsg(
                LANG, "array.sn.version.local.osver.upver.combine.not.pass",
                (device_sn, software_ver, os_version, up_software_ver))

    if (is_old_ver and apd_2_pdl_conf.strip().lower() == 'off') \
            or (is_new_ver and apd_2_pdl_conf.strip().lower() == 'on'):
        LOGGER.logInfo("storage array(SN%s) is "
                       "risk version and apd2pdl config is wrong." %
                       device_sn)
        check_flag = False
        err_msg += \
            common.getMsg(LANG,
                          "array.sn.version.local.apd2pdl.not.pass",
                          (device_sn, software_ver,
                           apd_2_pdl_conf.strip().lower()))
    return check_flag, err_msg


def parseCheckingResult(remoteStrgArrayRisks, localConfigRisks,
                        failedLocalQueryItems, failedIscsiSns, failedFcSns, allEchos):
    """
    summary:
    :param remoteStrgArrayRisks: the remote storage array's risk software version dict
    :param localConfigRisks: local configuration risk dict
    :param failedLocalQueryItems: local configuration items that failed to query information
    :param failedIscsiSns:
    :param failedFcSns:
    :param allEchos
    """
    errMsg = ''
    result = True
    if failedIscsiSns:
        result = cliUtil.RESULT_NOCHECK
        errMsg += common.getMsg(LANG,
                                "following.sn.failed.to.query.iSCSI.path", ",".join(failedIscsiSns))
    if failedFcSns:
        result = cliUtil.RESULT_NOCHECK
        errMsg += common.getMsg(LANG,
                                "following.sn.failed.to.query.FC.path", ",".join(failedFcSns))
    if failedLocalQueryItems:
        result = cliUtil.RESULT_NOCHECK
        errMsg += common.getMsg(LANG,
                                "following.local.config.failed.to.query", ",".join(failedLocalQueryItems))
    if localConfigRisks:
        result = False
        errMsg += common.getMsg(LANG,
                                "following.config.on.host.not.pass", "\n".join(localConfigRisks))
    if remoteStrgArrayRisks:
        result = False
        errMsg += common.getMsg(LANG,
                                "following.array.sn.version.not.pass", "\n".join(remoteStrgArrayRisks))
    return result, allEchos, errMsg


def getVmwareAALunStorageSNs(context, vlunList):
    """
    summary:  get current host's corrosponding storage arry's Serial number
               who's providing hyper-metro luns
    :param context: py_java_env
    :param ssh: ssh connection object
    """
    storageSnList = []
    hyperMetroLunWwns = []
    for line in vlunList:
        hyperMetroLunWwns.append(line.get('Lun WWN'))
    if not bool(hyperMetroLunWwns):
        LOGGER.logInfo("no hyperMetroLuns detected on current device,quit")
        return True, []
    remoteLunWwnDict = context.get("allStrgHyprMtrLns")
    for remoteLunWWn in remoteLunWwnDict:
        if remoteLunWWn.upper() in hyperMetroLunWwns:
            # remote dict structure: {lunWWn:{hostAluaStatus: ,lunId: ,deviceSn: }}
            remoteAASnList = remoteLunWwnDict.get(remoteLunWWn).get("deviceSn")
            storageSnList.extend(remoteAASnList.split(','))
    LOGGER.logInfo("storageSnList:%s;" % storageSnList)
    storageSnList = list(set(storageSnList))
    LOGGER.logInfo("found current host's mapping remote device(SN):%s;" % storageSnList)
    return True, storageSnList

def getStorageArraySn2IdMap(context, ssh, storageSnList):
    """
    summary: get current host's AA LUN mapped storage arrays' disk array Id
    :param context: py_java_env
    :param ssh: ssh connection object
    :param storageSnList hyper-metro lun provided storage array sn list
    :return sn2Id map
    """
    sn2IdDict = {}
    localsn2IdDict = {}
    cmd = "esxcli upadm show diskarray"
    excuteSuccess, cliRet, errMsg = cliUtil.executeHostCmd(context, ssh, cmd)
    if not excuteSuccess:
        return False, sn2IdDict, cliRet
    arrayIdPoi = POISTION_NOT_FETCHED
    ArraySnPoi = POISTION_NOT_FETCHED
    patten = re.compile("[\s][\s]+")
    for line in cliRet.splitlines()[1:]:
        params = patten.split(line.strip().lower())
        if '----' in line:
            continue
        if 'array id' in line.lower() and 'array sn' in line.lower():
            arrayIdPoi = params.index('array id')
            ArraySnPoi = params.index('array sn')
            continue

        if len(params) > arrayIdPoi and len(params) > ArraySnPoi:
            localsn2IdDict[str(params[ArraySnPoi]).upper()] = str(params[arrayIdPoi])
            continue
    LOGGER.logInfo("current device mapped storage array lists:" + str(localsn2IdDict))
    if not localsn2IdDict:
        LOGGER.logInfo("current device mapped no storage array, quit!")
        return True, sn2IdDict, cliRet
    for storageSn in storageSnList:
        if storageSn.upper() in localsn2IdDict:
            sn2IdDict[storageSn] = localsn2IdDict[storageSn.upper()]
        else:
            LOGGER.logInfo("found AA lun's belonging storageArray(SN:%s) not matched any array on host!" % storageSn)

    return True, sn2IdDict, cliRet

def getRemoteDeviceVersion(context, devSn):
    """
    @summary: get remote storage array device's version
    :param context: py_java_env
    :param devSn storage device 's sn
    @return: version, type
    """
    deviceType = ''
    softwareVer = ''
    selectDevs = context.get("selectDevs")
    for devNode in selectDevs:
        if devNode.getDeviceSerialNumber() == devSn:
            LOGGER.logInfo("fetching remote device's version and type:")
            deviceType = devNode.getDeviceType()
            softwareVer = devNode.getProductVersion()
            break
    LOGGER.logInfo("Remote device(SN:%s)'s version: %s; type %s:" % (devSn, softwareVer, deviceType))
    return deviceType, softwareVer


def getUpSoftwareVer(context, ssh):
    """
    @summary: get current device's UltraPath software's version
    :param context: py_java_env
    :param ssh: ssh connection object
    :return: isSuc, software version, echos
    """
    softwareVer = ''
    cmd = "esxcli upadm show version |grep 'Software Version'"
    excuteSuccess, cliRet, errMsg = cliUtil.executeHostCmd(context, ssh, cmd)
    if not excuteSuccess:
        return False, softwareVer, cliRet
    for line in cliRet.splitlines():
        if 'software version' in line.lower() and ":" in line.lower() \
                and len(line.split(':')) > 1:
            softwareVer = line.split(':')[1].strip()
    if len(softwareVer.split(".")) == 3 \
            and softwareVer.split(".")[0].isdigit() \
            and softwareVer.split(".")[1].isdigit() \
            and softwareVer.split(".")[2].isdigit():
        LOGGER.logInfo("current host's upadmin software version is valid: %s " % str(softwareVer))
        return True, softwareVer, cliRet
    LOGGER.logInfo("current host's upadmin software version is invalid: %s " % str(softwareVer))
    return False, softwareVer, cliRet


def getApd2PdlModeConfig(context, ssh):
    """
    @summary:
        :param context: py_java_env
    :param ssh: ssh connection object
    @return: 
    """
    apd2PdlConf = ''
    cmd = "esxcli upadm show upconfig |grep -i 'APD to PDL Mode'"
    excuteSuccess, cliRet, errMsg = cliUtil.executeHostCmd(context, ssh, cmd)
    if not excuteSuccess:
        return False, apd2PdlConf, cliRet
    for line in cliRet.splitlines():
        if 'apd to pdl mode' in line.lower() and ":" in line.lower() \
                and len(line.split(':')) > 1:
            apd2PdlConf = line.split(':')[1].strip()
    return True, apd2PdlConf, cliRet


def isCurStrgArrayVerTooOldAtRisk(deviceType, softwareVer):
    """
    @summary: 
    @param deviceType: 
    @param softwareVer: 
    @return: 
    """
    if deviceType.toString() in config.DORADO_DEVS:
        DORADO_MIN_SUPPORTED_VER = 'V300R001C01SPC100'
        return softwareVer < DORADO_MIN_SUPPORTED_VER
    else:
        V3_MIN_SUPPORTED_VER = 'V300R006C00SPC100'
        return softwareVer < V3_MIN_SUPPORTED_VER


def isCurStrgArrayVerNew(deviceType, softwareVer):
    """
    summary:
    :param deviceType:
    :param softwareVer:
    :return:
    """
    if  deviceType.toString() in config.DORADO_DEVS:
        DORADO_MIN_SUPPORTED_VER = 'V300R001C01SPC100'
        return softwareVer >= DORADO_MIN_SUPPORTED_VER
    else:
        V3_MIN_SUPPORTED_VER = 'V300R006C00SPC100'
        return softwareVer >= V3_MIN_SUPPORTED_VER


def isUpadmVersionLowerThanDest(softwareVer, comparedVer):
    """
    summary:
    :param softwareVer:
    :param comparedVer:
    """
    try:
        LOGGER.logInfo("upadmin risk compared verion: %s" % str(comparedVer))
        LOGGER.logInfo("current upadmin verion: %s" % str(softwareVer))
        cpVerList = map(int, comparedVer.strip().split("."))

        curVerList = map(int, softwareVer.strip().split("."))


        if cpVerList > curVerList :
            LOGGER.logInfo("current UP version is lower than risk baseline version")
            return True
    except BaseException, exp:
        LOGGER.logInfo("exception occurred while checking upadmin verion info" % str(exp))

    return False


def isEsxOsVerTooOld(context, ssh):
    """
    :summary:
    :param context: py_java_env
    :param ssh: ssh connection object
    
    """
    # 查询主机版本号
    isSuc, cliRet, versionNum, errorMsg = common.getOsVersion(context, ssh)
    if not isSuc:
        return False, '', '', cliRet
    if versionNum <= VMWARE_VERSION_5_0:
        return True, True, versionNum, cliRet
    else:
        return True, False, versionNum, cliRet


def getHostCheckParams(context, ssh):
    """
    summary:
    :param context: py_java_env
    :param ssh: ssh connection object
    """
    echos = ''
    failedLocalQueryItems = []
    isOsVerCheckSuc, isOsOld, osVersion, echo = isEsxOsVerTooOld(context, ssh)
    LOGGER.logInfo('current host os version: %s' % str(osVersion))
    echos += echo
    isApdCheckSuc, apd2PdlConf, cliRet = getApd2PdlModeConfig(context, ssh)
    LOGGER.logInfo('current host APD to PDl mode configuration: %s' % str(apd2PdlConf))
    echos += cliRet
    isUpVerCheckSucc, upSoftwareVer, cliRet = getUpSoftwareVer(context, ssh)
    echos += cliRet
    LOGGER.logInfo('current host UltraPath version: %s' % str(upSoftwareVer))
    if not isOsVerCheckSuc:
        failedLocalQueryItems.append(common.getMsg(LANG, "os.version"))
    if not isApdCheckSuc:
        failedLocalQueryItems.append(common.getMsg(LANG, "apd2pdl.switch"))
    if not isUpVerCheckSucc:
        failedLocalQueryItems.append(common.getMsg(LANG, "upadmin.software.version"))
    finalResult = isOsVerCheckSuc and isApdCheckSuc and isUpVerCheckSucc
    hostCheckParams = (isOsOld, apd2PdlConf, upSoftwareVer, osVersion)
    return finalResult, hostCheckParams, echos, failedLocalQueryItems


def all_lun_bilateral_mapping_and_normal(vlun_list):
    """
    @summary:Determine whether the active LUNs queried by the
     host are bilaterally mapped and the status is Normal.
    :param vlun_list:
    :return: True or False
    """
    for vlun_info in vlun_list:
        vlun_id = vlun_info.get('Vlun ID')
        wwn = vlun_info.get('Lun WWN')
        status = vlun_info.get('Status')
        if not bool(vlun_id) or not bool(wwn) or not bool(status):
            return False
        if status.lower() != 'normal':
            LOGGER.logInfo("vlun id:%s status is not Normal;" % vlun_id)
            return False
        if is_lun_bilateral_mapping_and_normal(vlun_list, vlun_id, wwn) \
                is not True:
            LOGGER.logInfo("vlun id:%s do not have bilateral mapping;"
                           % vlun_id)
            return False
    return True


def is_lun_bilateral_mapping_and_normal(vlun_list, vlun_id, wwn):
    """
    @summary:Determine whether there is a bilateral mapping between the
    active LUNs of the host
    (the Vlun ID has 2 duplicates, the Lun WWN has 2 identical,
    and the Status is normal)
    :param vlun_list:
    :param vlun_id:
    :param wwn:
    :return:True or False
    """
    bilateral = 0
    for vlun_info in vlun_list:
        curr_v_lun_id = vlun_info.get('Vlun ID')
        curr_wwn = vlun_info.get('Lun WWN')
        curr_status = vlun_info.get('Status')
        if not bool(curr_status):
            continue
        if curr_v_lun_id == vlun_id and curr_wwn == wwn \
                and curr_status.lower() == 'normal':
            bilateral += 1
        if bilateral > 1:
            return True
    return False


def analysis_vlun_list(cli_ret):
    """
    @summary:change cli to vlunList
    :param cli_ret:
    :return:vlunList
    """
    head_keys = ['Vlun ID', 'Disk', 'Lun WWN', 'Status']
    return get_vmware_horizontal_cli_ret(cli_ret, head_keys)


def get_vmware_horizontal_cli_ret(cli_ret, head_keys):
    """
    @summary: Get the cli echo collection in the form of a horizontal
    table in a row-by-row dictionary
     eg:
    XXX xx xx
    ---------
    xxx
    ---------
    [root@localhost:~]
    :type cli_ret: cli_ret
    :param head_keys: An echo header set that needs to be loaded
    into the parsing dictionary collection
    :return: dict_list
    """
    dict_list = []
    patten = re.compile(r"[\s][\s]+")
    vlun_infos = cli_ret.splitlines()
    vlun_lines = []
    find_head_flag = False
    head_line = ''
    head_key_dict = dict()
    # 分割符出现次数
    separator_count = 0
    for line in vlun_infos:
        if '----' in line:
            # 获取两个分隔符之间的数据作为解析正文
            if separator_count >= 1:
                break
            else:
                separator_count += 1
                continue
        if not find_head_flag and is_head_line(line, head_keys):
            head_line = line
            find_head_flag = True
            for key in head_keys:
                poi = patten.split(head_line.strip().lower()).index(
                    key.lower())
                head_key_dict.setdefault(key, poi)
        if find_head_flag:
            vlun_lines.append(line)

    if head_line == '':
        return []
    for line in vlun_lines:
        values = []
        if '----' in line:
            continue
        if line == head_line:
            continue
        cur_line_params = patten.split(line.strip().lower())
        for key in head_keys:
            value_poi = head_key_dict.get(key)
            if value_poi != POISTION_NOT_FETCHED and len(
                    cur_line_params) > value_poi:
                values.append(cur_line_params[value_poi].strip().upper())
            else:
                values.append(None)
        dict_list.append(dict(zip(head_keys, values)))
    return dict_list


def is_head_line(line, head_keys):
    """
    @summary:Query the header line by keyword
    :param line:
    :param head_keys:
    :return:True or False
    """
    for head_key in head_keys:
        if head_key not in line:
            return False
    return True
