# -*- coding: UTF-8 -*-
import traceback
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

VMWARE_VERSION = 'VMware ESXi %s %s'
HOST_CMD_TIMEOUT = 5 * 60
VMWARE_VERSION_5_0 = 5.0
VMWARE_VERSION_5_1 = 5.1
VMWARE_VERSION_5_5 = 5.5
VMWARE_VERSION_6_0 = 6.0
VMWARE_VERSION_6_7 = 6.7
VMWARE_VERSION_6_LIST = [6.0, 6.5, 6.7]

context = py_java_env


def execute(ssh):
    '''
    @summary:Vmware双活PDL配置检测
    1、版本为VMware EXSI 5.1、5.1U1、5.1U2、5.1U3，
    若字段“Disk.terminateVMOnPDLDefault”的值为“True”，
    则检查结果通过，否则检查结果为不通过。
    2、版本为VMware EXSI 5.5、5.5U1、5.5U2、5.5U3，
    若字段“Configured”和“Runtime”的值不全为“TRUE”，则检查结果为不通过。
    3、版本为VMware EXSI 5.5、5.5U1、5.5U2、5.5U3，
    若字段“Configured”和“Runtime”的值均为“TRUE”，
    并且“Int Value”为“0”，则检查结果为通过，否则检查结果为不通过。

    '''
    cliRet = ""
    allCliRet = ""
    errorMsg = ""
    try:

        common.mark_host_upadmin_hyper_metro_luns(py_java_env, ssh, LOGGER)

        # 查询主机版本号
        checkResult = common.getOsVersion(context, ssh)
        allCliRet += checkResult[1] + "\n"
        if checkResult[0] is not True:
            return cliUtil.RESULT_NOCHECK, allCliRet, checkResult[3]
        versionNum = checkResult[2]
        versionType = checkResult[3]

        # 判断主机是否为双活主机，是双活主机：
        # 继续检查，非双活主机：终止检查，巡检通过
        if not isHypermetroHost(context):
            return (True, allCliRet, "")

        # 若版本为5.0或5.1时，执行命令cat /etc/vmware/settings
        # 查看字段“Disk.terminateVMOnPDLDefault”的值
        if versionNum == VMWARE_VERSION_5_0 \
                or versionNum == VMWARE_VERSION_5_1:
            version = VMWARE_VERSION % (versionNum, versionType)
            flag, cliRet, errorMsg = checkVmware5(version, context, ssh)
            allCliRet += cliRet + "\n"
            return flag, allCliRet, errorMsg

        # VMware EXSI 6.7U3以后版本,工具无法检查，需要手动检查
        if (versionNum == VMWARE_VERSION_6_7 and versionType > "Update 3") \
                or versionNum > VMWARE_VERSION_6_7:
            version = VMWARE_VERSION % (versionNum, versionType)
            errorMsg = common.getMsg(LANG,
                                     "PDL.config.error.for."
                                     "after.version.6.0U2",
                                     version)
            return cliUtil.RESULT_WARNING, allCliRet, errorMsg

        if versionNum == VMWARE_VERSION_5_5 or \
                versionNum in VMWARE_VERSION_6_LIST:
            flag, cliRet, errorMsg = checkVmwareFrom55(versionNum, context,
                                                       ssh, versionType)
            allCliRet += cliRet + "\n"
            return (flag, allCliRet, errorMsg)

        return True, allCliRet, errorMsg
    except BaseException, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet,
                common.getMsg(LANG, "query.result.abnormal"))


def checkVmwareFrom55(version_num, context, ssh, version_type):
    '''
    @summary: version include 5.5 5.5U1 5.5U3,6.0,6.5,6.7
    '''
    try:
        allCliRet = ""
        errorMsg = ""
        terminateCmd = "esxcli system settings kernel " \
                       "list -o 'terminateVMOnPDL'"
        executeStatus, terminateCmdRet, errorMsg = cliUtil.executeHostCmd(
            context, ssh, terminateCmd)
        terminate_ret = common.deal_cli_ret_special_long_split_flag(
            terminateCmdRet
        )
        allCliRet = common.joinLines(allCliRet, terminate_ret)
        if executeStatus is not True:
            LOGGER.logInfo(
                "***[execute cmd: %s failure, errorMsg is: %s]***" % (
                    terminateCmd, errorMsg))
            return (cliUtil.RESULT_NOCHECK,
                    allCliRet,
                    common.getMsg(
                        LANG,
                        "hyper.metro.host.query.vmware.config.failure"))

        # 解析terminate信息
        settingDicts = cliUtil.getHorizontalCliRet(terminateCmdRet)
        for setting in settingDicts:
            configured = setting.get('Configured', '')
            runtime = setting.get('Runtime', '')
            if not configured or not runtime:
                return (cliUtil.RESULT_NOCHECK, allCliRet,
                        common.getMsg(LANG,
                                      "hyper.metro.host.query."
                                      "vmware.config.failure"))

        autoremoveCmd = "esxcli system settings advanced list" \
                        " -o '/Disk/AutoremoveOnPDL'"
        executeStatus, autoremoveCmdRet, errorMsg = cliUtil.executeHostCmd(
            context, ssh, autoremoveCmd)
        allCliRet += autoremoveCmdRet
        if executeStatus is not True:
            LOGGER.logInfo(
                "***[execute cmd: %s failure, errorMsg is: %s]***" % (
                    autoremoveCmd, errorMsg))
            return (cliUtil.RESULT_NOCHECK, allCliRet,
                    common.getMsg(
                        LANG,
                        "hyper.metro.host.query.vmware.config.failure"))

        # 解析autoremove信息
        fields = ""
        lineList = autoremoveCmdRet.splitlines()
        for line in lineList:
            if "Int Value:" in line:
                fields = line.split(":")[1].strip()
                if not fields:
                    return (cliUtil.RESULT_NOCHECK, allCliRet,
                            common.getMsg(
                                LANG,
                                "hyper.metro.host.query."
                                "vmware.config.failure"))
                break

        version = VMWARE_VERSION % (version_num, version_type)
        if version_num == VMWARE_VERSION_5_5 \
            or (version_num == VMWARE_VERSION_6_0
                and version_type <= "Update 1"):
            LOGGER.logInfo("get into before 6.0 u1 branch")
            flag = common.check_config_value_before_6U1(configured, runtime,
                                                        fields)
            key = "PDL.config.error.for.version.5.5"
            errorMsg = common.getMsg(LANG, key, (version, configured,
                                                 runtime, fields))
            return flag, allCliRet, errorMsg

        if version_num in VMWARE_VERSION_6_LIST:
            LOGGER.logInfo("get into 6.0 u2 later branch")
            flag = common.check_config_value_after_6U1(configured, runtime,
                                                       fields)
            if flag is False:
                key = "PDL.config.error.for.version.5.5"
                errorMsg = common.getMsg(LANG, key, (version, configured,
                                                     runtime, fields))
                return False, allCliRet, errorMsg

            errorMsg = common.getMsg(LANG, "PDL.config.ui.not.check.v6")
            return cliUtil.RESULT_WARNING, allCliRet, errorMsg

        LOGGER.logInfo("do not get into anly branch")
        return True, allCliRet, errorMsg
    except Exception as ex:
        msg = traceback.format_exc()
        LOGGER.logError("checkVmwareFrom55To6u3 ==> " + msg + ex)
        return (False, cliUtil.RESULT_NOCHECK,
                common.getMsg(
                    LANG,
                    "hyper.metro.host.query.vmware.config.failure"))


def checkVmware5(versionNum, context, ssh):
    """
     版本为5.0或5.1时,PDL配置是否正确
    :param versionNum:
    :param context:
    :param ssh:
    :return:
    """
    try:
        errorMsg = ""
        cmd = "cat /etc/vmware/settings"
        executeStatus, cliRet, errorMsg = cliUtil.executeHostCmd(context,
                                                                 ssh, cmd)
        if executeStatus is not True:
            LOGGER.logInfo(
                "***[execute cmd: %s failure, errorMsg is: %s]***" % (
                    cmd, errorMsg))
            errorMsg = common.getMsg(
                LANG,
                "hyper.metro.host.query.vmware.config.failure")
            return cliUtil.RESULT_NOCHECK, cliRet, errorMsg
        # 默认PDL配置为False
        PDLDefault = "False"
        for line in cliRet.splitlines():
            if 'Disk.terminateVMOnPDLDefault' not in line:
                continue

            settings = line.split('=')
            PDLDefault = settings[-1].strip()

        # 判断PDL配置是否正确
        if PDLDefault and PDLDefault.upper() != "TRUE":
            errorMsg = common.getMsg(LANG,
                                     "PDL.config.error.for."
                                     "version.5.0.or.5.1",
                                     versionNum)
            return False, cliRet, errorMsg

        return True, cliRet, ""
    except Exception as ex:
        msg = traceback.format_exc()
        LOGGER.logError("checkVmware5 ==> " + msg + ex)
        return (cliUtil.RESULT_NOCHECK, cliRet,
                common.getMsg(
                    LANG,
                    "hyper.metro.host.query.vmware.config.failure"))


def isHypermetroHost(context):
    '''
    @summary: 当前主机是双活主机
    @return: True:主机双活
             False:非主机双活
    '''
    try:
        devNode = context.get("devInfo")
        # 主机侧FC启动器信息
        hostFCLauncherWwns = devNode.getHostFCLauncherWwns()
        # 主机侧ISCSI启动器信息
        hostISCSILauncherWwns = devNode.getHostISCSILauncherWwns()
        # 阵列侧启动器信息
        hostLauncherWwnList = py_java_env.get("hostLauncherWwnList")
        LOGGER.logInfo("hostFCLauncherWwns = %s" % str(hostFCLauncherWwns))
        LOGGER.logInfo(
            "HostISCSILauncherWwns = %s" % str(hostISCSILauncherWwns))
        LOGGER.logInfo(
            "hostLauncherWwnList = %s" % str(hostLauncherWwnList))

        ini_list = []
        if hostFCLauncherWwns:
            ini_list.extend(hostFCLauncherWwns)
        if hostISCSILauncherWwns:
            ini_list.extend(hostISCSILauncherWwns)
        nvme_ini_wwns_list = devNode.getHostNvmeLauncherWwns()
        if nvme_ini_wwns_list:
            ini_list.extend(nvme_ini_wwns_list)

        # 判断是否为主机双活
        for ini in ini_list:
            if str(ini).lower() in str(
                    hostLauncherWwnList).lower():
                return True
        return False

    except Exception as ex:
        LOGGER.logInfo(str(traceback.format_exc()))
        LOGGER.logInfo("exception %s" % ex)
        # 异常情况视为双活主机，继续检查，规避漏报场景，可能存在误报
        return True
