# -*- coding: UTF-8 -*-
import os
import logger
import resource
import cliUtil
import re
import sys
script_path = os.path.dirname(os.path.abspath(__file__))
common_path = os.path.join(
    script_path, os.path.join("..", "..", "HOST_Common")
)
sys.path.append(common_path)
import host_common

ALUA_CONFIGURED_PROPERLY = 1
ALUA_CONFIGURED_FALSELY = -1
ALUA_CONFIGURED_NOT_CHECK = 0
FILE_SUFFIX = "."
HOST_CMD_TIMEOUT = 5 * 60
MAX_TILE_LEN = 150

def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")

def getBaseName(file_path):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name

def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)

def getMsg(lang, msg, args=""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    '''
    return cliUtil.getMsg(lang, msg, args, resource.MESSAGES_DICT)

def mark_host_upadmin_hyper_metro_luns(context, sshCon, LOGGER):
    '''
    @summary: get hyperMetro luns on host whose been designated to upadmin
    '''
    used_utralpath = 1
    not_used_utralpath = 2
    echos = ''
    exec_result = True
    lun_wwn_dicts = context.get("allStrgHyprMtrLns")
    cmd = "esxcli upadm show vlun -t hypermetro"

    objectForPy = context.get("objectForPy")
    host_dict = objectForPy.get("hostUseUtralPathInitiatorDict")
    if not host_dict:
        host_dict = {}
    devNode = context.get("devInfo")
    # 主机侧FC启动器信息
    hostFCLauncherWwns = devNode.getHostFCLauncherWwns()
    # 主机侧ISCSI启动器信息
    hostISCSILauncherWwns = devNode.getHostISCSILauncherWwns()
    if not hostFCLauncherWwns:
        hostFCLauncherWwns = []
    if not hostISCSILauncherWwns:
        hostISCSILauncherWwns = []
    fcList = list(hostFCLauncherWwns)
    iscsiList = list(hostISCSILauncherWwns)
    fcList.extend(iscsiList)
    nvme_wwns = devNode.getHostNvmeLauncherWwns()
    if nvme_wwns:
        fcList.extend(nvme_wwns)
    iniList = [initiator.lower() for initiator in fcList]
    host_type = str(devNode.getDeviceName())
    isSucess, echos, resultMsg = cliUtil.executeHostCmd(context, sshCon, cmd)
    if not isSucess:
        return False, False, echos
    if "lun wwn"  not in echos.lower() and '-------' not in echos:
        LOGGER.logInfo("no such command or no vlun detected.")

        for ini in iniList:
            host_dict[ini] = {"isUseUtrapath": not_used_utralpath,
                              "hostType": host_type
                              }
        objectForPy.put("hostUseUtralPathInitiatorDict", host_dict)

        return False, False, echos
    host_common.update_lun_alua_from_echo(lun_wwn_dicts, echos)

    for ini in iniList:
        host_dict[ini] = {"isUseUtrapath": used_utralpath,
                          "hostType": host_type
                          }
    objectForPy.put("hostUseUtralPathInitiatorDict", host_dict)

    return exec_result, True, echos

def getOsVersion(context, ssh):
    '''
    @summary: 检查主机版本
    @param context: 上下文
    @param ssh: 实时链接
    @return: 
            flag：是否查询成功
            versionNum:版本号
            versionType:版本类型
    '''
    INDEX_INT2 = 2
    cmd = 'vmware -l'
    executeStatus, cliRet, errorMsg = cliUtil.executeHostCmd(context, ssh, cmd)
    if executeStatus != True:
        return (False, cliRet, "", errorMsg)
    
    patternVersion = re.compile('vmware\sesxi\s(\d\.\d)\.\d\s(.*)', flags=re.IGNORECASE)
    for line in cliRet.splitlines():
        match = patternVersion.search(line.strip())
        if match:
            return (True, cliRet, float(match.group(1)), match.group(INDEX_INT2))
    
    return (False, cliRet, '', errorMsg)

def joinLines(originLines, postLines):
    """
    @summary: 将postLines追加originLines后
    """
    if not (originLines or postLines):
        return ""
    
    if not originLines:
        return postLines
    
    if not postLines:
        return originLines
    
    return "\n".join([originLines, postLines])

class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """
    def __init__(self, errorMsg, cliRet, flag=None):
        self.errorMsg = errorMsg
        self.cliRet = cliRet
        self.flag = flag
        


CLI_RET_END_FLAG = ":/>"
def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def get_horizontal_nostandard_cli_ret(cli_ret):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,
    此方法用来解析CLI回显未对其情况
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为
    键的字典集合,处理不正常时，返回空集合
    """
    head_line = ""
    i = 0
    cli_ret_list = cli_ret.encode("utf8").splitlines()
    for line in cli_ret_list:
        reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            head_line = match_headline.group()
            break
        i += 1
    if head_line == "" or i == 0 or i >= len(cli_ret_list) - 1:
        return []

    title = cli_ret_list[i - 1]
    field_words = cli_ret_list[(i + 1):]
    reg_split = re.compile(r"\s*-+\s*")
    tuple_idxs = []
    start_pos = 0
    end_pos = 0

    while (start_pos <= len(head_line)):
        match = reg_split.search(head_line, start_pos)
        if match:
            end_pos = match.end()
            tuple_idxs.append((start_pos, end_pos))
            start_pos = end_pos
        else:
            break

    keys = []
    for item in tuple_idxs:
        key = title[item[0]:item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key.decode("utf8"))

    lenkeys = len(keys)
    dict_list = []
    for line in field_words:
        if line.find(":/>") >= 0:
            break

        if re.search(r"^-+(\s+-+)*\s*$", line):
            continue

        if len(line.strip()) == 0:
            continue

        vals = []
        value_list = line.strip().split("  ")

        value_space_list = []
        for value in value_list:
            if value != "":
                vals.append(value.strip().decode("utf8"))
                value_space_list.append(value)
        lenvalue = len(value_space_list)
        if lenvalue == lenkeys:
            dict_list.append(dict(zip(keys, vals)))

    return dict_list


def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList


def check_config_value_before_6U1(configured, runtime, int_value):
    """
    检查6.0 u1 及其之前的PDL配置
    :param configured: configured的值
    :param runtime: runtime
    :param int_value: int_value
    :return: 配置是否正确
    """
    if configured.lower() != 'true' or runtime.lower() != 'true' \
            or int_value != '0':
        return False

    return True


def check_config_value_after_6U1(configured, runtime, int_value):
    """
    检查6.0 u1 之后的PDL配置
    :param configured: configured的值
    :param runtime: runtime
    :param int_value: int_value
    :return: 配置是否正确
    """
    if configured.lower() != 'false' or runtime.lower() != 'false' \
            or int_value != '1':
        return False

    return True


def deal_cli_ret_special_long_split_flag(cli_ret, max_len=MAX_TILE_LEN):
    """
    处理特殊原始信息，防止因兼容性问题导致报告无法打开
    注意：此方法只用于叠加回文前处理--过长的问题，不能在解析前使用此方法
    :param cli_ret: 原始信息
    :param max_len: 最大长度
    :return: 截取后的回文。
    """
    lines = cli_ret.splitlines()
    ret_list = []
    for line in lines:
        reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
        match_headline = reg_headline.search(line)
        if match_headline:
            headline = match_headline.group()
            if len(headline) > max_len:
                ret_list.append(headline[:max_len])
                continue
        ret_list.append(line)

    return "\n".join(ret_list)
