# -*- coding: UTF-8 -*-
import os
import sys
import common
import re
import cliUtil
import preCheck
import traceback
scriptpath = os.path.dirname(os.path.abspath(__file__))
hostCommonPath = os.path.join(scriptpath, "..\\..\\HOST_Common")
sys.path.append(hostCommonPath)
import host_common

LOGGER = common.getLogger(PY_LOGGER, __file__)
STANDARD_WWN_LEN = 32
ALUA_CONFIGURED_PROPERLY = 1
ALUA_CONFIGURED_FALSELY = -1
HEXADECIMAL_MAX = 16
remote_lun_cut_dict = {}
DEFECT_WINDOWS_VER = ['6.1.7601', '6.2.9200', '6.3.9200']
ALUA_NOT_SUPPORTED = "alua not supported"
IMPLICIT_ONLY = "implicit only"
ROUND_ROBIN_WITH_SUBSET = "round robin with subset"
ROUND_ROBIN = "round robin"


def execute(ssh_con):
    """
    Function name      : execute
    Input              : context
    Return             : cmd display
    """
    context = py_java_env
    LANG = context.get("lang")
    cli_rets = ""
    try:
        is_upadmin_designated, echos = common.\
            mark_host_upadmin_hyper_metro_luns(context, ssh_con, LOGGER)
        cli_rets += echos
        if is_upadmin_designated:
            LOGGER.logInfo("current host has takeover the hypermetro luns.")
            return cliUtil.RESULT_NOSUPPORT, echos, ''

        # 白名单检查
        context["logger"] = PY_LOGGER
        context["ssh"] = ssh_con
        check_flag, cli_ret, err_msg = preCheck.execute_doradov6(context)
        cli_rets += cli_ret + "\n\n"
        if check_flag is not True:
            return check_flag, cli_rets, err_msg

        is_succ, is_mpio_open, echos = is_mpio_feature_opened(ssh_con)
        cli_rets += echos
        if not is_succ:
            LOGGER.logInfo("current host's MPIO configuration "
                           "command executed failed.")
            return (cliUtil.RESULT_NOCHECK,
                    cli_rets,
                    common.getMsg(
                        LANG, "hyper.metro.host.query.nmp.failure"))
        if not is_mpio_open:
            LOGGER.logInfo("current host's MPIO feature is not opened .")
            return (False, echos,
                    common.getMsg(
                        LANG, "hyper.metro.host.query.multipath.not.open"))
        is_succ, is_config_ok, echos = is_mpio_switched_on(ssh_con)
        cli_rets += echos
        if not is_succ:
            return cliUtil.RESULT_NOCHECK, \
                   cli_rets, \
                   common.getMsg(LANG, "hyper.metro.host.query.nmp.failure")
        if not is_config_ok:
            return False, cli_rets,\
                   common.getMsg(LANG, "host.MPIO.switch.not.open")
        is_suc, disk_num_list, echos = get_all_disk_numbers(ssh_con)
        cli_rets += echos
        if not is_suc:
            return cliUtil.RESULT_NOCHECK, cli_rets,\
                   common.getMsg(LANG, "mpio.disk.number.not.found")
        if not disk_num_list:
            return False, cli_rets,\
                   common.getMsg(LANG, "hyper.metro.host.disks.empty")
        failed_disk_numbers = []
        wrong_sn_disks = {}
        disk_results = {}
        hyper_device_sn_list = host_common.get_access_mode(context).keys()
        disk_in_hyper_flag = False
        for disk_number in disk_num_list:
            is_succ, sn, policy, echos = \
                get_specify_disk_infos(context, ssh_con, disk_number)
            cli_rets += echos
            if sn in hyper_device_sn_list:
                disk_in_hyper_flag = True
            if not is_succ or not sn:
                LOGGER.logInfo("current disk query disk info failed ")
                failed_disk_numbers.append(disk_number)
            elif len(sn) < STANDARD_WWN_LEN:
                LOGGER.logInfo("current disk's SN is not standard length")
                wrong_sn_disks[sn.upper()] = {"disk_number": disk_number,
                                              "result": policy}
            else:
                disk_results[sn.upper()] = {"disk_number": disk_number,
                                            "result": policy}
        cli_rets += build_device_access_mode_cli_ret(context)
        if not disk_in_hyper_flag:
            return
        result = True
        err_msg = ""
        is_succ, has_matched, wrong_configured_luns = \
            seek_remote_hyperMetro_lun_status(context, disk_results)
        if not is_succ:
            return True, cli_rets, err_msg
        elif not has_matched and not failed_disk_numbers \
                and not wrong_sn_disks:
            return False, cli_rets, \
                   common.getMsg(
                       LANG,
                       "hyper.metro.host.query.multipath.not.take.lun")

        if len(failed_disk_numbers) > 0:
            result = cliUtil.RESULT_NOCHECK
            err_msg += common.getMsg(
                LANG, "hyper.metro.disk.wwn.query.failure",
                ",".join(failed_disk_numbers))

        if len(wrong_sn_disks) > 0 and is_cur_host_has_glitch(ssh_con):
            LOGGER.logInfo("current device has non-standard disks:" + str(
                wrong_sn_disks))
            is_succ, has_matched_wrong_sn, wrong_alua_luns_cut = \
                seek_remote_hyperMetro_lun_status(context, wrong_sn_disks)
            if not has_matched_wrong_sn and not has_matched:
                return False, cli_rets, \
                       err_msg + common.getMsg(
                           LANG,
                           "hyper.metro.host.query.multipath.not.take.lun")

            if wrong_alua_luns_cut:
                result = cliUtil.RESULT_WARNING
                err_msg += common.getMsg(
                    LANG, "device.sn.not.standard.length",
                    ",".join(wrong_alua_luns_cut))
        if len(wrong_configured_luns) > 0:
            result = False
            err_msg += common.getMsg(
                LANG, "hyper.metro.alua.evaluate.notpass",
                ",".join(wrong_configured_luns))

        return result, cli_rets, err_msg
    except BaseException, exception:
        LOGGER.logInfo(traceback.format_exc() + str(exception))
        return (cliUtil.RESULT_NOCHECK,
                cli_rets, common.getMsg(LANG, "query.result.abnormal"))


def is_cur_host_has_glitch(ssh_con):
    '''
    @summary: windows 2008R2 - 2012 R2 will cut lun's wwn
    '''
    cmd = "wmic os get version"
    is_sucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd, LOGGER,
                                      cliUtil.HOST_CMD_TIMEOUT, True)
    if not is_sucess:
        LOGGER.logInfo("current host faile to "
                       "fetch device's version, set as default.")
        return True
    for defect_ver in DEFECT_WINDOWS_VER:
        if defect_ver in echos:
            LOGGER.logInfo("current host's version is defective version!")
            return True
    return False


def seek_remote_hyperMetro_lun_status(context, disk_results):
    '''
    @summary: get remote storage's hyperMetro lun's wwn information
    @return: whether has matched remote lun,
    remote lun that is not_configured properly
    '''
    has_matched = False
    wrong_configured_luns = []
    is_succ = True
    lun_wwn_dicts = context.get("allStrgHyprMtrLns")
    if not lun_wwn_dicts:
        LOGGER.logInfo(
            "found no hyper metro lun wwn info in context, quit!")
        return False, has_matched, wrong_configured_luns
    for current_disk in disk_results:
        remote_lun_wwn = get_remote_matched_wwn(current_disk, lun_wwn_dicts)
        if remote_lun_wwn:
            has_matched = True
            is_policy_ok = disk_results.get(current_disk, {}).get("result")
            lun_wwn_dicts.get(remote_lun_wwn)[
                "hostAluaStatus"] = \
                is_policy_ok and ALUA_CONFIGURED_PROPERLY \
                or ALUA_CONFIGURED_FALSELY
            LOGGER.logInfo("found lun(WWN: %s) matched storage devs', "
                           "alua check result is %s."
                           % (str(current_disk), is_policy_ok))
            if not is_policy_ok:
                LOGGER.logInfo("lun(WWN: %s) matched remote"
                               " storage device but configured falsely"
                               % current_disk)
                wrong_configured_luns.append(
                    disk_results.get(current_disk, {}).get("disk_number"))

    return is_succ, has_matched, wrong_configured_luns


def get_remote_matched_wwn(host_disk_sn, remote_wwn_dicts):
    '''
    @summary checking whether the lun mapped to
    the windows host is from specified storage array
    @warning: windows 2008-2012 may cut the LUN's wwn
    and a patch is required.
    @bug: windows version varies from 2008 to 2012
    need patch KB2990170 to fire that bug
    '''
    if len(host_disk_sn) == STANDARD_WWN_LEN:
        for remote_wwn in remote_wwn_dicts:
            if remote_wwn and remote_wwn.upper() == host_disk_sn.upper():
                LOGGER.logInfo("found lun(WWN: %s) matched storage devs'"
                               % host_disk_sn)
                return remote_wwn
    else:
        if not remote_lun_cut_dict:
            for remote_lun in remote_wwn_dicts:
                remote_lun_cut_dict[get_cut_lun_wwn(remote_lun)] = \
                    remote_lun
            LOGGER.logInfo("current device's ")
        if host_disk_sn.upper() in remote_lun_cut_dict:
            LOGGER.logInfo("found cut wwn's lun(WWN: %s) "
                           "matched storage devs'" % host_disk_sn)
            return remote_lun_cut_dict[host_disk_sn]
    return ''


def get_cut_lun_wwn(remote_lun_wwn):
    """
    查看lun_wwn是否被裁剪
    :param remote_lun_wwn:
    :return:
    """
    cut_lun_wwn = ''
    LOGGER.logInfo("now start to simulate windows's "
                   "cutting lun scenario[wwn:%s].." % str(remote_lun_wwn))
    hex_len_group = len(remote_lun_wwn) / 2
    for poi in range(hex_len_group):
        one_byte_in_hex = remote_lun_wwn[poi * 2:poi * 2 + 2]
        if int(one_byte_in_hex, base=HEXADECIMAL_MAX) < HEXADECIMAL_MAX:
            cut_lun_wwn += one_byte_in_hex[1]
        else:
            cut_lun_wwn += one_byte_in_hex
    if len(remote_lun_wwn) > hex_len_group * 2:
        cut_lun_wwn += remote_lun_wwn[-1]
    cut_lun_wwn = cut_lun_wwn.upper()
    LOGGER.logInfo("after cutting wwn operation,"
                   " lun's [wwn:%s].." % str(cut_lun_wwn))
    return cut_lun_wwn


def is_mpio_switched_on(ssh_con):
    '''
    @summary: get current os's MPIO switch status
    '''
    cmd_new = "powershell  Get-MPIOSetting"
    cmd_old = 'powershell Get-ItemProperty ' \
              '"HKLM:\\System\\Current' \
              'ControlSet\\Services\\msdsm\\Parameters"'
    is_switch_on = False
    has_discovered = False
    is_sucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd_new, LOGGER,
                                      cliUtil.HOST_CMD_TIMEOUT, True)
    echos += '\n'
    LOGGER.logInfo("echos:" + echos)
    if not is_sucess or "<<<<" in echos:
        LOGGER.logInfo("current device does not support "
                       "Get-MPIOSetting command , searching regedit...")
        is_sucess, echo, err_msg = \
            cliUtil.executeCmdWithTimeout(ssh_con, cmd_old, LOGGER,
                                          cliUtil.HOST_CMD_TIMEOUT, True)
        echo += '\n'
        if not is_sucess:
            LOGGER.logInfo("failed executed "
                           "'powershell Get-ItemProperty' command.")
            return False, is_switch_on, echos + echo
        for line in echo.splitlines():
            if re.search("PathVerifyEnabled(\\s)*[=:](\\s)*(\\S)+(\\s)*",
                         line.strip(), re.I):
                LOGGER.logInfo("found windows MPIO "
                               "configuration: %s" % line.strip())
                has_discovered = True
                is_switch_on = "1" in line.lower()
        return has_discovered, is_switch_on, echo
    else:
        for line in echos.splitlines():
            if re.search(
                    "pathverificationstate(\\s)*[=:](\\s)*(\\S)+(\\s)*",
                    line.strip(), re.I):
                LOGGER.logInfo("found windows MPIO configuration: %s"
                               % line.strip())
                has_discovered = True
                is_switch_on = "enabled" in line.lower()
        return has_discovered, is_switch_on, echos


def get_specify_disk_infos(context, ssh_con, disk_number):
    '''
    @summary: get specified disk information
    @return: is_succ, sn, policy_ok, echos, err_msg
    '''
    cmd = 'mpclaim -s -d "' + disk_number + '"'
    sn_num = ""
    alua_status = ''
    mpio_policy = ''
    policy_ok = False
    is_sucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd, LOGGER,
                                      cliUtil.HOST_CMD_TIMEOUT, True)
    echos += '\n'
    if not is_sucess:
        return False, sn_num, policy_ok, echos, ''
    for line in echos.splitlines():
        if re.search("^sn(\\s)*:(\\s)*[\\S]+", line.strip(), re.I):
            sre_match = re.search("^sn(\\s)*:(\\s)*[\\S]+", line.strip(),
                                  re.I).group()
            sn_num = sre_match.split(":")[1].strip()
            LOGGER.logInfo("current windows disk %s 's sn: %s"
                           % (disk_number, sn_num))
        if re.search('^MPIO Disk[0-9]+:[\\s\\S]+', line.strip(), re.I):
            policy = line.strip()
            LOGGER.logInfo("current windows disk %s 's policy: %s"
                           % (disk_number, policy))
            if ROUND_ROBIN_WITH_SUBSET in policy.lower():
                mpio_policy = "round robin with subset"
            elif ROUND_ROBIN in policy.lower():
                mpio_policy = "round robin"
            LOGGER.logInfo("current windows disk %s 's mpio_policy :  %s"
                           % (disk_number, mpio_policy))
    if not sn_num:
        LOGGER.logInfo("current windows disk %s matched no sn number."
                       % disk_number)
        is_sucess = False
    if ALUA_NOT_SUPPORTED in echos.lower() \
            or u"不支持 ALUA" in echos.lower():
        alua_status = ALUA_NOT_SUPPORTED
    elif "implicit only" in echos.lower():
        alua_status = IMPLICIT_ONLY
        LOGGER.logInfo("current windows disk %s 's alua %s." %
                       (disk_number, alua_status))
    policy_ok = check_hyper_device_access_mode(context,
                                               alua_status,
                                               mpio_policy)
    return is_sucess, sn_num, policy_ok, echos


def check_hyper_device_access_mode(context, alua_status, mpio_policy):
    sn_access_mode_dict = host_common.get_access_mode(context)
    for sn, access_mode in sn_access_mode_dict.items():
        access_mode_lower = access_mode.lower()
        if access_mode_lower != 'balanced' \
                and access_mode_lower != 'asymmetric':
            return False
        if 'balanced' == access_mode_lower:
            if alua_status != ALUA_NOT_SUPPORTED \
                    or mpio_policy != ROUND_ROBIN:
                return False
        if 'asymmetric' == access_mode_lower:
            if alua_status != IMPLICIT_ONLY \
                    or mpio_policy != ROUND_ROBIN_WITH_SUBSET:
                return False
    return True


def get_all_disk_numbers(ssh_con):
    '''
    @summary: get all mpio disk numbers on windows
    '''
    cmd = "mpclaim -s -d"
    disk_number_list = []
    result = False
    is_sucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd, LOGGER,
                                      cliUtil.HOST_CMD_TIMEOUT, True)
    echos += '\n'
    if not is_sucess:
        return False, [], echos
    for line in echos.splitlines():
        if re.search("^MPIO Disk[0-9]+", line.strip(), re.I):
            re_result = re.search("^MPIO Disk[0-9]+", line.strip(),
                                  re.I).group()
            disk_number_list.append(re_result[9:])
    LOGGER.logInfo("current windows's disk number lists: %s"
                   % ",".join(disk_number_list))
    if disk_number_list:
        result = True
    return result, disk_number_list, echos


def execute_cmd(ssh_con, cmd):
    """
    get command echos
    :param ssh_con:
    :param cmd:
    :return:
    """
    isSucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd, LOGGER)
    return isSucess, echos, err_msg


def is_mpio_feature_opened(ssh_con):
    """
    是否打开mpio
    :param ssh_con:
    :return:
    """
    is_switch_on = False
    cmd = "dism /online /get-features"
    is_sucess, echos, err_msg = \
        cliUtil.executeCmdWithTimeout(ssh_con, cmd, LOGGER,
                                      cliUtil.HOST_CMD_TIMEOUT, True)
    echos += '\n'
    if not is_sucess or "error:" in echos.lower():
        LOGGER.logInfo("current windows's dism command"
                       " executed failed or with error code, quit!")
        return False, is_switch_on, echos
    is_fetched = False
    for line in echos.splitlines():
        if "multipathio" in line.lower():
            is_fetched = True
            continue
        if is_fetched:
            if "enabled" in line.lower() or u"已启用" in line.lower():
                return is_sucess, True, echos
            else:
                return is_sucess, False, echos
    return is_sucess, False, echos


def build_device_access_mode_cli_ret(context):
    """
    组装查询双活阵列访问主机配置cli
    :param context:
    :return:
    """
    cli_rets = ''
    sn_cli_ret_dict = host_common.get_host_access_cli_ret(context)
    for sn, cli_ret in sn_cli_ret_dict.items():
        cli_rets += 'HyperMetro Device SN[%s]:\r\n%s\r\n' % (sn, cli_ret)
    return cli_rets
