# -*- coding: UTF-8 -*-
import common
import util
import traceback

LANG = py_java_env.get("lang")


def execute(cli):
    cliRet = ""
    errMsg = ""
    minTimingLength = 3
    maxTimingLength = 30
    
    try:
        #1. 判断版本是否是问题版本
        #18000 V1R1C00版本全部存在问题，直接进行后续检查
                    
        #2. 获取远程复制一致性组ID
        flag, consistencyGroupIdList, cliInfo, errInfo = getConsistencyGroupIdList(cli)
        cliRet += cliInfo
        if not flag:
            return (False, cliRet, errInfo)
        
        #3. 获取远程复制一致性组的Timing Length信息；原始信息只记录有问题的，避免信息过多
        for consistencyGroupId in consistencyGroupIdList:
            flag, timingLength, cliInfo, errInfo = getConsistencyGroupTimingLength(cli, consistencyGroupId)
            
            #无法获取Timing length，后续也无法获取，直接退出
            if not flag:
                cliRet += "\n" + cliInfo
                return (False, cliRet, errInfo)
                
            #任何一个在[3,30]之间，则不通过，可中断后续检查直接退出 
            if minTimingLength <= timingLength <= maxTimingLength:
                cliRet += "\n" + cliInfo
                errInfoDict = {
                               "zh":u"\n远程复制一致性组（ID:%s）的Timing Length为%s秒在3秒到30秒之间。",
                               "en":"\nFor remote replication consistency group (ID:%s), the value of Timing Length is %s seconds, which is between 3 seconds and 30 seconds.",
                               } 
                errMsg = errInfoDict.get(LANG) % (consistencyGroupId, timingLength)
                return (False, cliRet, errMsg)
        
        
        #4. 获取远程复制ID
        flag, remoteReplicationIdList, cliInfo, errInfo = getRemoteReplicationIdList(cli)
        cliRet += "\n" + cliInfo
        if not flag:
            return (False, cliRet, errInfo)
        
        #5. 获取远程复制的Consistency Group ID和Timing Length信息
        for remoteReplicationId in remoteReplicationIdList:  
            flag, consistencyGroupId, timingLength, cliInfo, errInfo = getRemoteReplicationInfo(cli, remoteReplicationId)
            
            #如果获取信息失败，后续也会高概率失败，直接退出
            if not flag:
                cliRet += "\n" + cliInfo
                return (False, cliRet, errInfo)
            
            #若存在远程复制的Consistency Group ID 为--且Timing Length在3秒到30秒之间，则检查结果为不通过,中断后续检查，直接退出
            if consistencyGroupId == "--" and minTimingLength <= timingLength <= maxTimingLength:
                cliRet += "\n" + cliInfo
                errInfoDict = {
                               "zh":u"\n远程复制（ID:%s）的Consistency Group ID为--且Timing Length为%s秒在3秒到30秒之间。",
                               "en":"\nFor remote replication (ID:%s), Consistency Group ID  is --, and Timing Length is %s seconds, which is between 3 seconds and 30 seconds.",
                               }
                errMsg = errInfoDict.get(LANG) % (remoteReplicationId, timingLength)
                return (False, cliRet, errMsg)
            
        return (True, cliRet, errMsg)
    
    except:
        PY_LOGGER.error("[exception: ]" + traceback.format_exc())
        errMsgDict = {
                      "zh":u"\n查询结果异常。",
                      "en":"\nQuery result is abnormal.",
                      }
        return (False, cliRet, errMsgDict.get(LANG))
    
    
def getConsistencyGroupIdList(cli):
    '''
    @summary: 获取远程一致性组ID
    @return: 
        True|False:成功|失败
        IdList:ID列表
    ''' 
    cmd = "show consistency_group general"
    return getIdListByCmd(cli, cmd)

def getConsistencyGroupTimingLength(cli, consistencyGroupId):
    '''
    @summary: 获取远程复制一致性组的Timing Length信息
    @param consistencyGroupId: 一致性组ID
    @return: Timing Length, 统一转换成秒为单位
    '''
    timingLength = ""
    cmd = "show consistency_group general consistency_group_id=%s" % consistencyGroupId
    errMsg = ""
    
    cliRet = cli.execCmd(cmd)
    if not util.checkCliInfoValid(cliRet, False):
        errMsgDict = {
                      "zh":u"\n无效CLI信息。",
                      "en":"\nInvalid Cli information.",
                      }
        errMsg = errMsgDict.get(LANG)
        return (False, timingLength, cliRet, errMsg)
    
    consistencyGroupInfoDictList = common.getVerticalCliRet(cliRet)
    for consistencyGroupInfo in consistencyGroupInfoDictList:
        flag, timingLength = getTimingLength(consistencyGroupInfo)
        if flag:
            return (True, timingLength, cliRet, errMsg)
        
    errMsgDict = {
                  "zh":u"\n无法获取远程复制一致性组[ID:%s]的Timing Length信息。",
                  "en":"\nFailed to obtain Timing Length of remote replication consistency group [ID:%s].",
                  }
    errMsg = errMsgDict.get(LANG) % consistencyGroupId
    return (False, timingLength, cliRet, errMsg)
   
      
def getRemoteReplicationIdList(cli):
    '''
    @summary: 获取远程复制ID列表
    @return: IdList:复制ID列表
    '''    
    cmd = "show remote_replication general"
    return getIdListByCmd(cli, cmd)
    
    
def getRemoteReplicationInfo(cli, remoteReplicationId):
    '''
    @summary: 获取远程复制的Consistency Group ID和Timing Length信息
    @param remoteReplicationId: 远程复制ID
    @return: Consistency Group ID & Timing Length, Timing Length统一转换成秒为单位
    '''
    consistencyGroupId = ""
    timingLength = ""
    errMsg = ""
    
    cmd = "show remote_replication general remote_replication_id=%s" % remoteReplicationId
    cliRet = cli.execCmd(cmd)
    if not util.checkCliInfoValid(cliRet, False):
        errMsgDict = {
                      "zh":u"\n无效CLI信息。",
                      "en":"\nInvalid Cli information.",
                      }
        errMsg = errMsgDict.get(LANG)
        return (False, consistencyGroupId, timingLength, cliRet, errMsg)
    
    remoteReplicationInfoDictList = common.getVerticalCliRet(cliRet)
    for remoteReplicationInfo in remoteReplicationInfoDictList:
        flag, consistencyGroupId = getConsistencyGroupId(remoteReplicationInfo)
        if not flag:
            errMsgDict = {
                          "zh":u"\n无法获取远程复制[ID:%s]的Consistency Group ID信息。",
                          "en":"\nFailed to obtain Consistency Group ID of remote replication [ID:%s].",
                          }
            errMsg = errMsgDict.get(LANG) % remoteReplicationId
            return (False, consistencyGroupId, timingLength, cliRet, errMsg)
        
        flag, timingLength = getTimingLength(remoteReplicationInfo)
        if not flag:
            errMsgDict = {
                          "zh":u"\n无法获取远程复制[ID:%s]的Timing Length信息。",
                          "en":"\nFailed to obtain Timing Length of remote replication [ID:%s].",
                          }
            errMsg = errMsgDict.get(LANG) % remoteReplicationId
            return (False, consistencyGroupId, timingLength, cliRet, errMsg)
        
        return (True, consistencyGroupId, timingLength, cliRet, errMsg)
    
    errMsgDict = {
                  "zh":u"\n无法获取远程复制[ID:%s]的Consistency Group ID和Timing Length信息。",
                  "en":"\nFailed to obtain Consistency Group ID and Timing Length of remote replication [ID:%s].",
                  }
    errMsg = errMsgDict.get(LANG) % remoteReplicationId
    return (False, consistencyGroupId, timingLength, cliRet, errMsg)

    
def getIdListByCmd(cli, cmd):
    '''
    @summary:执行给定的cmd，获取回显中的ID列表
    '''
    idList = []
    cliRet = cli.execCmd(cmd)
    errMsg = ""
    if not util.checkCliInfoValid(cliRet, False):
        errMsgDict = {
                      "zh":u"\n无效CLI信息。",
                      "en":"\nInvalid Cli information.",
                      }
        errMsg = errMsgDict.get(LANG)
        return (False, idList, cliRet, errMsg)
     
    consistencyGroupDictList = common.getHorizontalCliRet(cliRet)
    for consistencyGroupDict in consistencyGroupDictList:
        id = consistencyGroupDict.get("ID")
        idList.append(id)
        
    return (True, idList, cliRet, errMsg)


def getTimingLength(dictInfo):
    '''
    @summary: 根据处理后的字典信息，获取Timing Length
    '''
    ratio = 1
    timingLength = ""
    if "Timing Length" not in dictInfo:
        return (False, timingLength)
    
    #据反馈，Timing Length都是整数，存在两种单位：Seconds 和 Minute(s)
    timingLength = dictInfo.get("Timing Length")
    
    #如果当前Timing Length显示为“--”（手动同步），则认为不存在问题，直接返回一个-1
    if timingLength == "--":
        timingLength = -1
        return (True, timingLength)
    
    if "Minute" in timingLength:  #去掉（s）,避免1分钟时没有(s)
        ratio = 60
    try: 
        timingLength = int(timingLength.split()[0]) * ratio
        return (True, timingLength)
    except:
        return (False, timingLength)


def getConsistencyGroupId(dictInfo):
    '''
    @summary: 根据处理后的字典列表信息，获取Consistency Group ID
    '''
    consistencyGroupId = ""
    if "Consistency Group ID" not in dictInfo:
        return (False, consistencyGroupId)
    
    consistencyGroupId = dictInfo.get("Consistency Group ID")
    return (True, consistencyGroupId)
    
    