import common

LANG = common.getLang(py_java_env)

def execute(cli):
    '''
    @summary: 检查当前设备型号和版本号，且是否包含4K盘
    '''
    
    cliRet = ""
    errMsg = ""
    flag = True
    try:
        #定义风险型号和版本号
        bugProduct_18000 = ("HVS85T", "HVS88T", "18500", "18800", "18800F")
        bugVersion_18000 = "V100R001C10"
        
        #查询产品型号
        result, cliRet, productModel = common.getProductModel(cli, LANG)
        if not result:
            if "zh" == LANG:
                errMsg = u"\n获取产品型号失败。"
            else:
                errMsg = "\nFailed to query the product model."
            return (False, cliRet, errMsg)
        
        if productModel not in bugProduct_18000:
            return ("NOSUPPORT", cliRet, "")
        
         #获取版本号和补丁版本号，并判断是否为风险版本
        resultList, currentVersionDictList, hotPatchVersionDictList = common.parse_upgradePackage(cli, LANG)
        cliRet += resultList[1]
        if resultList[0] != True:
            return (resultList[0], cliRet, resultList[2])
        
        result, currentVersion, errInfo = common.getCurrentVersion(currentVersionDictList, LANG)
        if not result:
            errMsg += errInfo
            return (False, cliRet, errMsg)
         
        #检查产品版本是否有风险
        if currentVersion != bugVersion_18000:
             return (True, cliRet, errMsg)
        
        #获取当前cli显示模式
        checkRet, cmdRet, errInfo = getCliCapacityMode(cli, LANG)
        cliRet += cmdRet
        if not checkRet:
            return (checkRet, cliRet, errInfo)
        
        cliConfigInfoDictlist = common.getVerticalCliRet(cmdRet)
        capacityMode = ""
        for cliConfigInfoDict in cliConfigInfoDictlist:
            if "Capacity Mode" in cliConfigInfoDict:
                capacityMode = cliConfigInfoDict["Capacity Mode"]
        
        #无法获取CLI显示模式
        if capacityMode == "":
            errMsgDict = {"zh":u"\n无法获取Capacity Mode值",
                          "en":"\nFailed to obtain the value of Capacity Mode."}
            return ("NOCHECK", cliRet, errMsgDict[LANG])
        
        #根据当前模式确定是否需要切换模式
        if capacityMode.lower() != "precise":
            #切换后执行show lun general得到的所有lun的容量单位均为B
            cmdRet= changeCapacityMode(cli, "precise")
            cliRet += cmdRet
    
    except Exception, exception:
        errMsgDict = {"zh":u"\n获取信息失败",
                       "en":"\nThere is an error in getting information"}
        return (False, cliRet, errMsgDict[LANG])
    
    try:
        #获取lun信息
        checkRet, cmdRet, errInfo = getLunInfo(cli, LANG)
        cliRet += cmdRet
        if common.NOT_SUPPORT_CMD_FLAG in cmdRet:
            return ("NOCHECK", cliRet, "")
        
        if checkRet != True:
            return (checkRet, cliRet, errInfo)
        
        #检查是否存在lun,不存在lun则不存在问题
        if common.isExecSuccess(cmdRet):
            return (True, cliRet, errMsg)
        
        #获取每个lun的容量和Disk Location
        cliRetLinesList = getCapacityAndDiskLocation(cmdRet)
        if len(cliRetLinesList) == 0:
            errMsgDict = {"zh":u"\n无法获取LUN信息",
                          "en":"\nCannot get information about LUN"}
            errMsg = errMsgDict[LANG]
            return (False, cliRet, errMsg)
        
        #检查所有lun是否是4k对齐且是否为edevlun
        notEdevlunIdList = []
        not4kAlignLunIdList, not4kAligenAndEdevlunIdList = checkLunIs4kAlignAndNotEdevlun(cliRetLinesList)
        notEdevlunIdList += not4kAligenAndEdevlunIdList
        
        #没有需要进一步检查的lun 且 没有存在问题的lun
        if len(not4kAlignLunIdList) == 0 and len(notEdevlunIdList) == 0:
            return (True, cliRet, errMsg)
        
        #检查非容量显示不全的lun
        checkRet, notEdevlunIdListtemp, getInfoFailedLunIdList, cmdRet= checkLunAllIsEdevlun(cli, not4kAlignLunIdList, LANG)
        notEdevlunIdList += notEdevlunIdListtemp
        cliRet += cmdRet
       
        if len(notEdevlunIdList) != 0:
            flag = False
            errMsgDict = {"zh":u"\n当前设备中存在非4K对齐且Disk Location字段值不为External的LUN:\nID              最小扩容量(Block)\n%s",
                          "en":"\nNon-4 KB aligned LUNs whose values of Disk Location are not External exist in the device:\nID              Minimum Capacity Expansion (Block)\n%s"}
            errMsg += errMsgDict[LANG] % ('\n'.join(' '.join(notEdevlun) for notEdevlun in notEdevlunIdList))
        
        if len(getInfoFailedLunIdList) != 0:
            flag = False
            errMsgDict = {"zh":u"\n无法获取LUN[ID：%s]的信息",
                          "en":"\nCannot get information about LUN [ID:%s]."}
            errMsg += errMsgDict[LANG] % (', '.join(getInfoFailedLunIdList))
    
        return (flag, cliRet, errMsg)
    
    except Exception, exception:
        errMsgDict = {"zh":u"\n获取信息失败",
                       "en":"\nThere is an error in getting information"}
        return (False, cliRet, errMsgDict[LANG])
    
    finally:
        #切换cli显示模式到原模式
        changeCapacityMode(cli, capacityMode.lower())
        
def getCliCapacityMode(cli,lang):
    """
    @summary: 获取cli显示模式
    @param cli: cli链接
    @return: cmdRet 命令回文
    """    
    cmd = "show cli configuration"
    cmdRet = common.isCliExecRetInBlackList(cli, cmd, lang, True)
    flag = not cmdRet[0]
    cliRet = cmdRet[1]
    errMsg = cmdRet[2]
    return (flag, cliRet, errMsg)
    
def changeCapacityMode(cli, mode):
    """
    @summary: 切换cli显示模式
    @param cli: 
            lang:
    @return: cmdRet 命令回显
    """
    cmd = "change cli capacity_mode=%s" % mode
    cmdRet = cli.execCmd(cmd)
    return cmdRet
    
def getLunInfo(cli, lang):
    """
    @summary: 获取lun信息
    @param cli: 
            lang:
    @return: cmdRet 命令回显
    """
    cmd = "show lun general |filterColumn include columnList=ID,Capacity,Disk\sLocation"
    cmdRet = common.isCliExecRetInBlackList(cli, cmd, lang, True)
    #兼容cli命令columnList拼写错误场景
    if (not cmdRet[0]) and common.NOT_SUPPORT_CMD_FLAG in cmdRet[1]:
        cmd = "show lun general |filterColumn include colunmList=ID,Capacity,Disk\sLocation"
        cmdRet = common.isCliExecRetInBlackList(cli, cmd, lang, True)
        
    flag = not cmdRet[0]
    cliRet = cmdRet[1]
    errMsg = cmdRet[2]
    return (flag, cliRet, errMsg)

def checkLunIs4kAlignAndNotEdevlun(cliRetLinesList):
    """
    @summary: 检查lun的容量是否4K对齐
    @param cliRetLinesList: lun信息结果解析的字典列表,且所有lun的容量单位必须为B
    @return: not4kAlignLunIdList 非4k对齐的lun ID列表(容量显示不全的ID)
            not4kAligenAndEdevlunIdList 确定的非4K对齐lun
    """
    not4kAlignLunIdList = []
    not4kAligenAndEdevlunIdList = []
    for lunInfoDict in cliRetLinesList:
        lunId = lunInfoDict["ID"]
        lunCapacity = lunInfoDict["Capacity"]
        diskLocation = lunInfoDict["DiskLocation"]
        #External类型的lun不存在问题
        if diskLocation == "External":
            continue
        
        #容量显示不全且不为External，记录其ID，在后续使用其他命令查询
        if '..' in lunCapacity:
            not4kAlignLunIdList.append(lunId)
            continue
        
        lunCapacityValue = lunCapacity.replace("B", "")
        lunCapacityNum = int(lunCapacityValue)
        if lunCapacityNum % (4 * 1024) != 0:
            #扩容以block为单位，需要计算扩到4KB整数倍需要多少个Block。 1Block = 512B, lun容量一定是512B的倍数
            expansionBlock = str((4096 - lunCapacityNum % 4096) / 512)
            problemLun = (lunId.ljust(15), expansionBlock)
            not4kAligenAndEdevlunIdList.append(problemLun)
            
    return (not4kAlignLunIdList, not4kAligenAndEdevlunIdList)
    
def checkLunAllIsEdevlun(cli, lunIdList, lang):
    """
    @summary: 检查lun是否全部为edevlun
    @param lunIdList: 需要检查的LUN
    @return:True/False: 存在lun为非edevlun 
            notEdevlunIdList： 非edevlun的ID列表
            getInfoFailedLunIdList:无法获取详细信息或Disk Loaction的lun
            cliRet：所有lun详细信息
    """
    flag = True
    cliRet = ""
    notEdevlunIdList = []
    getInfoFailedLunIdList = []
    
    for lunId in lunIdList:
        cmd = "show lun general lun_id=%s" % lunId
        cmdRet, cmdInfo, errInfo = common.isCliExecRetInBlackList(cli, cmd, lang, True)
        cliRet += cmdInfo
        #无法获取原始信息的lun做记录上报
        if cmdRet:
            flag = False
            getInfoFailedLunIdList.append(lunId)
            continue
        
        #获取capacity和Disk Location,并判断是否为edevlun
        checkRet, capacity, diskLocation = getLunDiskCapacityAndLocation(cmdInfo)
        
        #无法获取Disk Location的lun也记录上报
        if not checkRet:
            flag = False
            getInfoFailedLunIdList.append(lunId)
            continue
        
        #如果容量不为4K整数倍且不是edevlun，则存在问题
        lunCapacityValue = capacity.replace("B", "")
        lunCapacityNum = int(lunCapacityValue)
        if lunCapacityNum % (4 * 1024) != 0 and diskLocation != "External":
            flag = False
            #扩容以block为单位，需要计算扩到4KB整数倍需要多少个Block。 1Block = 512B, lun容量一定是512B的倍数
            expansionBlock = str((4096 - lunCapacityNum % 4096) / 512)
            problemLun = (lunId.ljust(15), expansionBlock)
            notEdevlunIdList.append(problemLun)
            
    return (flag, notEdevlunIdList, getInfoFailedLunIdList, cliRet)
      
def getLunDiskCapacityAndLocation(cmdRet):
    """
    @summary: 根据LUN信息获取Disk Location字段值和容量
    @param cmdRet: 单个lun的信息
    @return: （True/False, capacity, diskLocation）
    """ 
    lunInfoDictList = common.getVerticalCliRet(cmdRet)    
    for lunInfoDict in lunInfoDictList:
        if "Disk Location" in lunInfoDict and "Capacity" in lunInfoDict:
            capacity = lunInfoDict["Capacity"] 
            diskLocation = lunInfoDict["Disk Location"] 
            return (True, capacity, diskLocation)
        
    return (False, "", "")

def getCapacityAndDiskLocation(cmdRet):
    """
    @summary: 解析show lun general |filterColumn include columnList=ID,Capacity回文，得到每个lun的容量值
    @param cmdRet:show lun general |filterColumn include columnList=ID,Capacity回文 
    @return: idAndCapacityDictList:ID为key，Capacity为值的字典列表
    """
    
    idAndCapacityDictList=[]
    for line in cmdRet.splitlines():
        idAndCapacityDict={}
        #cli结束
        if common.CLI_RET_END_FLAG in line:
            break
        
        #非数据信息
        lineInfo = line.split()
        if len(lineInfo) != 3:
            continue
        id = lineInfo[0].strip()
        capacity = lineInfo[1].strip()
        diskLocation = lineInfo[2].strip()
        if id == "ID":
            continue
        if id.startswith("-"):
            continue
        
        #添加lun信息
        idAndCapacityDict["ID"] = id
        idAndCapacityDict["Capacity"] = capacity
        idAndCapacityDict["DiskLocation"] = diskLocation
        idAndCapacityDictList.append(idAndCapacityDict)
    
    return idAndCapacityDictList
