# -*- coding: UTF-8 -*-
import traceback
import re
import os
import util

FILE_SUFFIX = "."
NOCHECK = 'NOCHECK'
#########################################
#      异常回显判断脚本
# 检查回显是否异常
#########################################
#CLI回显结果异常，直接判断巡检不通过
CLI_RET_FAULT = -1
#CLI回显结果成功，直接判断巡检通过
CLI_RET_SUCCESS = 0
#CLI回显包含状态值，需要通过后续逻辑判断是否通过
CLI_RET_OK =1

PRODUCT_VESION = "V100R001C10"

#结束标志
CLI_EXEC_SUCCESS = "Command executed successfully"

NOT_SUPPORT_CMD_FLAG = "^"


#模式标志
SUPER_USER_FLAG = "admin:/>"

DEVELOPER_MODEL_FLAG = "developer:/>"

CLI_RET_END_FLAG = ":/>"

KEY_WORD_VALID_LINES = 4

#重试次数，防止多次重试死循环
MAX_RETRYS = 5


#异常回显黑名单，针对捕捉到回显信息，判断回显是否正常，若是不正常，直接巡检不通过
blackDict = [
{"key_word":"-bash:",
 "msg_zh":u"\n统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},

{"key_word":"Storage:",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},

{"key_word":"minisystem>",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},

{"key_word":"upgrade:/>",
 "msg_zh":u"\n系统处于升级模式",
 "msg_en":"\nSystem is in upgrade Mode"},

{"key_word":"The system is powering off",
 "msg_zh":u"\n系统不正常，处于下电模式",
 "msg_en":"\nSystem is abnormal, powering off"},

{"key_word":"system is busy",
 "msg_zh":u"\n系统忙,请稍后重试",
 "msg_en":"\nSystem is busy.Please try again later"},

{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},

{"key_word":"Processing...",
 "msg_zh":u"\n正在处理",
 "msg_en":"\nSystem is busy"},

{"key_word":"Error: Operation timeout.",
    "msg_zh":u"\n执行超时",
    "msg_en":"\nOperation timeout"},

{"key_word":"Error:",
 "msg_zh":u"\n命令执行出错",
 "msg_en":"\nCmd execute Error"},

{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"}]


#白名单，回显中捕捉到这列关键字时，巡检通过，巡检通过时无法添加详细信息，只需要列出关键字即可
whiteList = [NOT_SUPPORT_CMD_FLAG,CLI_EXEC_SUCCESS]

#行黑名单，针对捕捉到某行回显信息，判断当前行是否显示异常
blackLineDict = [
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},

{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"}]

# **************************************************************************** #
# 函数名称: isChinese
# 功能说明: 判断当前语言是否为中文
# 输入参数: lang
# 输出参数: True or False
# **************************************************************************** # 
def isChinese(lang):
    if "zh"==lang:
        return True
    return False

# **************************************************************************** #
# 函数名称: isExecSuccess
# 功能说明: 为了加速判定，根据回显结果判定命令是否执行成功（一般针对无记录的情况）
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False:
# **************************************************************************** # 
def isExecSuccess(cliRet):
    if re.search(CLI_EXEC_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserRet
# 功能说明: 根据CLI回显判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserRet(cliRet):

    if re.search(SUPER_USER_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserCli
# 功能说明: 根据传入的SSH连接执行CLI命令后判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserCli(cli):
    cliRet = cli.execCmdNoLog("show system general")
    lineList = cliRet.splitlines()

    #用户名一定包含在最后一行中
    if SUPER_USER_FLAG == lineList[-1].replace(" ",""):
        return True

    return False

# **************************************************************************** #
# 函数名称: isSuperUserObj
# 功能说明: 根据根据框架传入的参数判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserObj(devObj):
    if "admin" ==  devObj.get("devInfoMap").get("userName"):
        return True
    return False

# **************************************************************************** #
# 函数名称: isDeveloperModle
# 功能说明: 判断当前捕捉到的CLI回显是否在Developer模式下
# 输入参数: cliRet
# 输出参数: True:Developer模式，False:非Developer模式
# **************************************************************************** # 
def isDeveloperModle(cliRet):
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: getCurrentVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC
# 输入参数: cli
# 输出参数: currentVesion
# **************************************************************************** # 
def getCurrentVesion(cli):
    currentVesion = ""
    cliRet = cli.execCmdNoLog("show upgrade package")
    lineList = cliRet.splitlines()

    #直接从包含版本号的行开始
    for line in lineList[5:-1]:
        columnList = line.split()
        #找到关键字，直接返回当前版本号
        for column in columnList[2:-1]:
            if -1 != column.find(PRODUCT_VESION):
                currentVesion = column[0:17]
                return currentVesion
    return currentVesion


def getCurrentVesionWithCliRet(cli):
    currentVesion = ""
    cliRet = cli.execCmdNoLog("show upgrade package")
    lineList = cliRet.splitlines()

    # 直接从包含版本号的行开始
    for line in lineList[5:-1]:
        columnList = line.split()
        # 找到关键字，直接返回当前版本号
        for column in columnList[2:-1]:
            if -1 != column.find(PRODUCT_VESION):
                currentVesion = column[0:17]
                return currentVesion, cliRet
    return currentVesion, cliRet


def getProductFullVersion(cli, lang):
    """
    Function name      : getUserLevel
    Function describe  : get user level
    Input              : (cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, productVersion)
    """
    result, cliRet, errMsg = isCliExecRetInBlackList(cli, "show upgrade package", lang, True)
    if result:
        return (False, cliRet, errMsg)

    endIndex = cliRet.find('HotPatch Version')
    if endIndex == -1:
        return (False, cliRet, '')

    try:
        dictList = util.formatList(cliRet[:endIndex])
        fullVersion = dictList[0]['Current Version']
        if fullVersion.endswith('T'):
            fullVersion = fullVersion[:-1]
    except:
        return (False, cliRet, '')
    else:
        return (True, cliRet, fullVersion)

# **************************************************************************** #
# 函数名称: isLessThanSupportVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC
# 输入参数: cli,supportVesion(命令开始支持的版本)
# 输出参数: currentVesion
# **************************************************************************** # 
def isLessThanSupportVesion(cli, supportVesion):
    currentVesion =  getCurrentVesion(cli)
    if ""==currentVesion:
        return False

    if currentVesion<supportVesion:
        return True
    return False

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取当前用户名
# 输入参数: cli
# 输出参数: userName
# **************************************************************************** # 
def getUserName(cli):
    userName = ""
    cliRet = cli.execCmdNoLog("show system general")
    lineList = cliRet.splitlines()

    #用户名一定包含在最后一行中
    if lineList[-1].find(CLI_RET_END_FLAG):
        userName = lineList[-1].replace(" ","").replace(CLI_RET_END_FLAG,"")

    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取登录用户名的权限级别
# 输入参数: cli
# 输出参数: super_admin,guest或admin
# **************************************************************************** # 
def getUserPrivilege(cli):

    userName = getUserName(cli)
    if ""== userName:
        return ""

    cliRet = cli.execCmdNoLog("show user user_name="+userName)
    lineList = cliRet.splitlines()

    #用户名一定包含在最后一行中
    for line in lineList[4:-1]:
        columnList = line.split()
        if len(columnList)>2:
            if userName == columnList[0]:
                return columnList[1]
    return ""

# **************************************************************************** #
# 函数名称: change2cli
# 功能说明: 任何模式尝试切换到CLI模式        
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def anyModel2Cli(cli):
    index =0
    cliRet = cli.execCmdNoLog("show system general")

    #密码输入错误时及正常情况从developer模式下退出
    if -1 != cliRet.find("Password") or -1 != cliRet.find(DEVELOPER_MODEL_FLAG):
        cliRet = cli.execCmd("exit")
        while(-1 == cliRet.find(SUPER_USER_FLAG)):
            index+=1
            cliRet = cli.execCmd("exit")
            if -1!=cliRet.find("Are you sure to exit?(y/n):"):
                cliRet = cli.execCmd("n")
            if index>MAX_RETRYS:
                break

# **************************************************************************** #
# 函数名称: change2Developer
# 功能说明: 切换到developer模式
# 输入参数: cli,py_java_env
# 输出参数: True:，False:
# **************************************************************************** # 
def change2Developer(cli, py_java_env):
    errMsg = ""
    flag = False
    lang = py_java_env.get("lang")
    if None == cli:
        return (flag,"",errMsg)

    #guest用户无法切换到developer模式
    userLevel = getUserPrivilege(cli)
    if "guest" ==userLevel or ""==userLevel:
        if isChinese(lang):
            errMsg = u"\n系统没有运行在admin模式，无法进入调试模式"
        else:
            errMsg = "\nSystem is not in admin model, can't enter to debug model"
        return (flag,"",errMsg)

    cliRet = cli.execCmd("change user_mode current_mode user_mode=developer")

    password = str(py_java_env.get("devPwd").get("developer"))
    if(None == password or ""== password):
        if isChinese(lang):
            errMsg = u"\n获取developer密码为空，请确定是否已配置密码"
        else:
            errMsg = "\nThe obtained developer password is blank. Check whether the password has been configured."
        return (flag, cliRet, errMsg)

    cliRet = cli.execCmdNoLog(password)

    #已经成功进入Developer模式
    if isDeveloperModle(cliRet):
        flag = True
    else:
        anyModel2Cli(cli)
        if isChinese(lang):
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):
                errMsg = u"\ndeveloper密码配置错误"
            else:
                errMsg = u"\n系统不能进入developer模式"
        else:
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):
                errMsg = "\nThe password of developer is wrong"
            else:
                errMsg = "\nIt failed to access the developer mode"
    return (flag, cliRet, errMsg)


# **************************************************************************** #
# 函数名称: developer2Cli
# 功能说明: developer模式下退出到CLI模式
# 输入参数: cli
# 输出参数: 无
# **************************************************************************** # 
def developer2Cli(cli):
    cli.execCmd("exit")

# **************************************************************************** #
# 函数名称: isEndLine
# 功能说明: 判断此行是否是回显结束行
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isEndLine(cliLine):

    #空行，无效行 非结束标志
    if "" == cliLine or None == cliLine:
        return False

    if -1 != cliLine.find(CLI_RET_END_FLAG):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithStart
# 功能说明: 跳过以某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithStart(cliLine, startWordList):

    #未知对象，跳过
    if None == cliLine:
        return True

    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True

    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True

    #没传入任何关键字，不跳过
    if 0 == len(startWordList):
        return False


    for startWord in startWordList:
        #是以XXX起头的行，需要跳过
        if line.startswith(startWord):
            return True

    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithContain
# 功能说明: 跳过包含某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithContain(cliLine, containWordList):

    #未知对象，跳过
    if None == cliLine:
        return True

    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True

    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True

    #没传入任何关键字，不跳过
    if 0 == len(containWordList):
        return False


    for containWord in containWordList:
        #包含XXX关键字的行，需要跳过
        if -1 != line.find(containWord):
            return True

    return False

# **************************************************************************** #
# 函数名称: isLineInBlackDict
# 功能说明: 判断某行回显是否在黑名单中，在黑名单中表示CLI回显存在异常，巡检不通过，
# 不用逻辑判断，最终应返回CLI_RET_FAULT
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInBlackDict(cliLine, lang):

    errMsg = ""

    #空行，无效行，回显正常时会包含，不属于黑名单
    if None == cliLine or "" == cliLine:
        return (False, errMsg)

    for dictItems in blackDict:
        #黑名单中找到，表示回显无效，直接退出
        if -1 != cliLine.find(dictItems.get("key_word")):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)


# **************************************************************************** #
# 函数名称: isLineInWhiteList
# 功能说明: 判断某行回显是否在白名单中，在白名单中表示CLI执行完成，巡检通过，
# 不用逻辑判断，最终应返回CLI_RET_SUCCESS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInWhiteList(cliLine):

    keyWordMsg = ""

    #空行，无效行，回显正常时会包含，不属于白名单
    if None == cliLine or "" == cliLine:
        return (False, keyWordMsg)

    for keyWords in whiteList:
        #在白名单中找到关键字，表示命令执行成功，直接退出
        if re.search(keyWords, cliLine, re.IGNORECASE):
            keyWordMsg = keyWords
            return (True, keyWordMsg)
        #无license情况通过
        if re.search("license", cliLine, re.IGNORECASE):
            if -1 != cliLine.find("Error:") or -1 != cliLine.find("Suggestion:"):
                keyWordMsg = "license"
                return (True, keyWordMsg)
    return (False,keyWordMsg)

# **************************************************************************** #
# 函数名称: isNormalLine
# 功能说明: 判断某行回显是否正常
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isNormalLine(cliLine, lang):
    errMsg = ""

    #空行，无效行正常
    if None == cliLine or "" == cliLine:
        return (True, errMsg)

    #异常的关键字存在于前几行，防止全部判断时误判
    #某行是否因为超时或是其它原因，导致端口信息查询不到的情况，返回不正常行
    for dictItems in blackLineDict:
        if -1 != cliLine.find(dictItems.get("key_word")):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (False, errMsg)
    return (True, errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInBlackList
# 功能说明: 传入待执行CLI命令，判断回显是否在黑名单中，带回结果供各个巡检项调用
#           1）黑名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回异常回显对应的中英文提示信息
# **************************************************************************** # 
def isCliExecRetInBlackList(cli, cmd, lang, isHasLog):

    errMsg = ""

    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)

    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:
            errMsg = "\nThe display of cli result is empty"
        return (True, cliRet, errMsg)


    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()

    #行数大于5行，则回显正常，不在黑名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, errMsg)

    #回显在白名单中直接返回，不带回key值
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            return (False, cliRet, errMsg)

    #是否在黑名单中，若是获取具体错误原因
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (True, cliRet, errMsg)

    #不在黑名单中
    return (False,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInWriteList
# 功能说明: 传入待执行CLI命令，判断回显是否在白名单中，带回结果供各个巡检项调用
#           1）白名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回捕捉到的白名单关键字信息
# **************************************************************************** # 
def isCliExecRetInWriteList(cli, cmd, lang, isHasLog):

    writeKeyWords = ""

    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)

    if None == cliRet or ""==cliRet:
        return (True, cliRet, writeKeyWords)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:

        #是否在白名单中
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            writeKeyWords = checkRet[1]
            return (True, cliRet, writeKeyWords)

    #不在白名单中
    return (False,cliRet,writeKeyWords)

# **************************************************************************** #
# 函数名称: cliExecRetJudgeWithDoubleList
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CLI_RET_FAULT，巡检不通过
#           2）白名单回显，返回：CLI_RET_SUCCESS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CLI_RET_OK，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CLI_RET_FAULT，CLI_RET_SUCCESS or CLI_RET_OK
# **************************************************************************** # 
def cliExecRetJudgeWithDoubleList(cli, cmd, lang, isHasLog):

    errMsg = ""

    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)

    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:
            errMsg = "\nThe display of cli result is empty"
        return (CLI_RET_FAULT, cliRet, errMsg)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:

        #在白名单中，巡检可以直接通过
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CLI_RET_SUCCESS, cliRet,whiteKeyWord)

    for line in lineList[0:KEY_WORD_VALID_LINES]:

        #在黑名单中，巡检直接不通过
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CLI_RET_FAULT, cliRet, errMsg)

    #其它情况，说明需要通过回显来判断此项巡检是否通过
    return (CLI_RET_OK,cliRet,errMsg)


# **************************************************************************** #
# 函数名称: cliExecRetWithTimeout
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项，本函数带超时处理机制
#           1）黑名单回显，返回：CLI_RET_FAULT，巡检不通过
#           2）白名单回显，返回：CLI_RET_SUCCESS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CLI_RET_OK，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CLI_RET_FAULT，CLI_RET_SUCCESS or CLI_RET_OK
# **************************************************************************** # 

def cliExecRetWithTimeout(cli, cmd, timeOut, lang):

    #执行带超时设置的CLI命令
    cliRet = cli.execCmdWithTimout(cmd,timeOut)

    errMsg = ""
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:
            errMsg = "\nThe display of cli result is empty"
        return (CLI_RET_FAULT, cliRet, errMsg)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:

        #在黑名单中，巡检直接不通过
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CLI_RET_FAULT, cliRet, errMsg)

        #在白名单中，巡检可以直接通过
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CLI_RET_SUCCESS, cliRet,whiteKeyWord)

    #其它情况，说明需要通过回显来判断此项巡检是否通过
    return (CLI_RET_OK,cliRet,errMsg)

def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, productmodel)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        productmodel：产品型号
    '''
    product_model = ""
    cmd = "show system general"
    result, cliRet, errMsg = isCliExecRetInBlackList(cli, cmd, lang, True)
    if result:
        return (False, cliRet, errMsg)

    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Product Model":
            product_model = fieldValue

        if len(product_model) != 0 and product_model != "--":
            return (True, cliRet, product_model)

    return (False, cliRet, "")

def parse_upgradePackage(cli, lang):
    '''
    @summary: 执行show upgrade packge命令，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型,并返回CLI回显
    @param cli: cli对象
    @param lang: 语言lang
    '''
    cmd = "show upgrade package"
    softwareVersionList = []
    hotPatchVersionList = []
    flag = True
    (result, cliRet, errMsg) = isCliExecRetInBlackList(cli, cmd, lang, True)
    if result:
        return ((False, cliRet, errMsg),softwareVersionList,hotPatchVersionList)

    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = getHorizontalCliRet(cliRet[(softwareVersionIndex):hotPatchVersionIndex])
    hotPatchVersionList = getHorizontalCliRet(cliRet[(hotPatchVersionIndex):])

    if len(softwareVersionList) == 0 or len(hotPatchVersionList) == 0:
        if "zh" == lang:
            errMsg = u"\n无法获取升级包信息。"
        else:
            errMsg = "\nFailed to obtain the upgrade package."
        return ((False, cliRet, errMsg) , softwareVersionList , hotPatchVersionList)

    for softwareVersion in softwareVersionList:
        if softwareVersion["Current Version"] == "--":
            flag = False
            if "zh" == lang:
                errMsg += u"\n控制器[%s]:无法获取控制器软件版本。" % softwareVersion["Name"]
            else:
                errMsg += "\nController[%s]: Failed to obtain the controller software version." % softwareVersion["Name"]

    if not isControllerVersionSame(softwareVersionList):
        if "zh" == lang:
            errMsg += u"\n控制器软件版本不一致，请修复后再进行巡检检查。"
        else:
            errMsg += "\nController software versions are inconsistent, rectify the fault before performing the inspection."
        return ((False, cliRet, errMsg),softwareVersionList,hotPatchVersionList)

    return ((flag, cliRet, errMsg),softwareVersionList,hotPatchVersionList)

def isControllerVersionSame(softwareVersionList):
    '''
    @summary:校验控制器版本是否一致
    @param softwareVersionList: 控制器版本列表
    @return: 
        True:控制器版本一致
        False:控制器软件版本不一致
    '''
    VersionSet = set(softwareVersion["Current Version"] for softwareVersion in softwareVersionList)
    if len(VersionSet) !=1:
        return False
    return True

def getCurrentVersion(softwareVersionList, lang):
    '''
    @summary: 通过software信息字典列表获取版本信息
    '''
    currentVersion = ""
    errMsg = ""
    for controller in softwareVersionList:
        currentVersion = controller["Current Version"]
        whereIsBVer = currentVersion.find("B")
        if whereIsBVer != -1:
            currentVersion = currentVersion[:whereIsBVer]
        return (True, currentVersion, "")

    #未获取到版本信息
    if "zh" == lang:
        errMsg = u"\n无法获取产品版本信息"
    else:
        errMsg += "\nCannot get information about product version"
    return (False, "", errMsg)

def getHotPatchVersion(hotPatchVersionList, lang):
    '''
    @summary: 通过hotPatch信息字典列表获取补丁信息
    @return: flag, hotPatch, errMsg
    flag:
        True:补丁信息一致，且获取成功
        False:补丁信息不一致，或未安装补丁
    hotPatch:补丁信息
    errMsg:错误信息
    '''
    hotPatchVersion = ""
    errMsg = ""
    flag = True

    if not isControllerVersionSame(hotPatchVersionList):
        if "zh" == lang:
            errMsg = u"\n控制器热补丁版本不一致，请修复后再进行巡检检查。"
        else:
            errMsg = "\nController hot patch versions are inconsistent, rectify the fault before performing the inspection. ",
        return (False, "", errMsg)

    for controller in hotPatchVersionList:
        hotPatchVersion = controller["Current Version"]
        return (True, hotPatchVersion, "")

    #未获取到信息
    if "zh" == lang:
        errMsg = u"\n无法获取热补丁版本信息"
    else:
        errMsg = "\nCannot get information about hot patch version"
    return (False, hotPatchVersion, errMsg)

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []

def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList


def getDiskSectorSize(cli, lang):
    '''
    @summary: 执行show disk general |filterColumn include columnList=ID,Sector\sSize命令，获取扇区大小
    '''
    sectorSizeDictList = []
    cmd = "show disk general |filterColumn include columnList=ID,Sector\sSize"
    checkRet = isCliExecRetInBlackList(cli, cmd, lang, True)

    #兼容产品命令Bug
    if not hasCliExecPrivilege(checkRet[1]):
        cmd = "show disk general |filterColumn include colunmList=ID,Sector\sSize"
        checkRet = isCliExecRetInBlackList(cli, cmd, lang, True)

    cliRet = checkRet[1]
    if checkRet[0]:
        return (False, cliRet, checkRet[2], sectorSizeDictList)

    sectorSizeDictList = getHorizontalCliRet(cliRet)
    return (True, cliRet, "", sectorSizeDictList)

def is4KDiskWithSectorSize(diskSectorSizeDictList, lang):
    '''
    @summary: 根据扇区大小判断是否为4K盘
    '''
    is4KDiskList = []
    for diskSectorSizeDict in diskSectorSizeDictList:
        diskId = diskSectorSizeDict["ID"]
        sectorSizeStr = diskSectorSizeDict["Sector Size"]

        #4196B=4.0625KB，显示为4.062KB，需要处理
        if sectorSizeStr == "4.062KB":
            sectorSizeStr = "4.0625KB"
        result, sectorSizeNum = changUnit2B(sectorSizeStr)
        if not result:
            if "zh" == lang:
                errMsg = u"\n转换单位失败。"
            else:
                errMsg = "\nFailed to convert units."
            return (True, [], errMsg)

        if sectorSizeNum == 4096.0 or sectorSizeNum == 4160.0:
            is4KDiskList.append(diskId)

    if is4KDiskList:
        if "zh" == lang:
            errMsg = u"\n设备中存在扇区大小为4096B或4160B的硬盘[ID: %s]" % ', '.join(is4KDiskList)
        else:
            errMsg = "\nDisks [ID: %s] whose sector size is 4096 bytes or 4160 bytes exist in the device." % ', '.join(is4KDiskList)
        return (True, is4KDiskList, errMsg)

    return(False, [], "")

def changUnit2B(strValue):
    '''
    @summary: 根据传入的值转换单位为B
    @return: True/False, sectorSizeNum
    '''
    floatValue = 0.0

    try:
        if not strValue:
            return(False, floatValue)

        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024 * 1024 * 1024 * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip()) * 1024 * 1024 * 1024
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) * 1024 * 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) * 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip())
        else:
            return(False, floatValue)

        return(True, floatValue)

    except Exception, exception:
        return (False, floatValue)


def changUnit2GB(strValue):
    '''
    @summary: 根据传入的值转换单位为GB
    @return: True/False, floatValue
    '''
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)

        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip())
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) / 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) / 1024 / 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip()) / 1024 / 1024 / 1024
        else:
            return(False, floatValue)

        return(True, floatValue)

    except Exception, exception:
        return (False, floatValue)


def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "does not have the permission" in line:
            return False
    return True

def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")

def get4kDiskInfo(cli, lang):
    '''
    @summary: 判断是否存在故障硬盘
    @param cli: cli对象
    @param lang: 语言lang
    '''

    cliRet = ""
    errMsgDict = {"zh":"", "en":""}
    msgLang = "zh"

    #定义语言标识，用于返回错误信息
    if lang != "zh":
        msgLang = "en"

    #判断硬盘是否存在于1TB的SAS盘
    diskSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=SAS " \
                "logicOp=and column=Capacity predict=greater_than value=1.000TB " \

    #判断硬盘是否存在大于5TB的NearLine SAS盘
    diskNearLineSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=NearLine\sSAS " \
                "logicOp=and column=Capacity predict=greater_than value=5.000TB " \

    all4kDiskList = []
    isCmdExeSucc = True
    for diskCmd in ( diskSASCmd, diskNearLineSASCmd):
        isCmdExeSucc, diskCliRet, fkDiskList = exeQry4kDiskCmd(diskCmd, cli, lang)
        #命令执行失败或存在故障盘或存在4K
        cliRet += diskCliRet
        all4kDiskList += fkDiskList
        if not isCmdExeSucc:
            errMsgDict["zh"] = u"\n执行命令失败。"
            errMsgDict["en"] = "\nexecute command failed."
            return (False, cliRet, errMsgDict[msgLang], [])

    #存在4K盘返回Flase
    if all4kDiskList:
        errMsgDict["zh"] = u"\n设备中存在大于1TB的SAS盘或大于5TB的NearLine SAS盘[ID: %s],当前系统版本不支持该类型的硬盘。" % ', '.join(all4kDiskList)
        errMsgDict["en"] = "\nSAS disks larger than 1 TB or NL-SAS disks larger than 5 TB [ID: %s] exist in the device. The current version does not support such disks." % ', '.join(all4kDiskList)

    return (True, cliRet, errMsgDict[msgLang], all4kDiskList)

def exeQry4kDiskCmd(cmd, cli, lang):

    cliRet = ""
    errMsg = ""
    #用isCliExecRetInBlackList执行命令会误判，所以直接用execCmd
    cliRet = cli.execCmd(cmd)

    #判断命令回显是否为Command executed successfully
    isCmdExeSucc = queryResultWithNoRecord(cliRet)
    if isCmdExeSucc:
        return (True, cliRet, [])

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()

    #检测前五行行数大于5行，则回显正常，不在黑名单中，是否在黑名单中，若是获取具体错误原因
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (False, cliRet, [])

    #按逐行字典的方式获取水平表格形式的cli回显集合
    diskInfoDictList = getHorizontalCliRet(cliRet)

    fkDiskList = []
    for diskInfoDict in diskInfoDictList:
        diskId = diskInfoDict.get('ID')
        #把大于1T的SAS盘或大于5T的NearLine SAS盘添加到bugDiskList
        fkDiskList.append(diskId)

    return (True, cliRet, fkDiskList)


def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    cliExecuteCmdSuccess = "Command executed successfully"
    if re.search(cliExecuteCmdSuccess, cliRet, re.IGNORECASE):
        return True
    return False


def getLogicEngineOfDisk(diskId, productModel):
    """
    @summary: 获取硬盘所属的逻辑引擎
    """
    #硬盘id的第一个数字即为该硬盘所属的逻辑引擎
    firstDigit = re.search('([A-Z]+)(\d)', diskId).group(2)
    if firstDigit is None:
        return 0

    firstDigit = int(firstDigit)

    logicEngine = firstDigit

    #6U4C大卡环境逻辑i引擎需特殊处理
    bigCardModelList = ["6800 V3", "6900 V3", "18500 V3", "18800 V3"]
    if productModel in bigCardModelList:
        secondDigit = re.search('([A-Z]+)(\d)(\d|[A-F])', diskId).group(3)
        if secondDigit is None:
            return firstDigit

        if secondDigit <= "7":
            logicEngine = firstDigit * 2 + 0
        else:
            logicEngine = firstDigit * 2 + 1

    return logicEngine


def isBugProductVersion(currentVersion, bugProductVersionList):
    '''
    @summary: 判断产品版本是否是某个已知产品问题的产品版本
    @param: bugProductVersionList
    eg:("V300R003C00", "V300R003C00SPC300]", "V300R003C10.*"]/V300R003C00到V300R003C00SPC300（不包含V300R003C00，包含V300R003C00SPC300），V300R003C10所有版本
    '''
    for bugProductVersion in bugProductVersionList:
        if bugProductVersion.find(",") > -1:
            startVersion = bugProductVersion.split(",")[0][1:].strip()
            endVersion = bugProductVersion.split(",")[1][:-1].strip()
            isContainStartVersion = bugProductVersion.startswith("[")
            isContainEndVersion = bugProductVersion.endswith("]")

            if isContainStartVersion and isContainEndVersion:
                if currentVersion >= startVersion and currentVersion <= endVersion:
                    return True

            if isContainStartVersion and (not isContainEndVersion):
                if currentVersion >= startVersion and currentVersion < endVersion:
                    return True

            if (not isContainStartVersion) and isContainEndVersion:
                if currentVersion > startVersion and currentVersion <= endVersion:
                    return True

            if (not isContainStartVersion) and (not isContainEndVersion):
                if currentVersion > startVersion and currentVersion < endVersion:
                    return True
        else:
            if re.match(bugProductVersion, currentVersion):
                return True

    return False


def checkProductModelAndVersion(cli, modelAndVersionListDict, LANG):
    """
    @summary: 检查设备型号和版本是否为指定的型号和版本
    @param modelAndVersionListDict: 问题型号与对应的版本，型号与版本为正则表达式，若问题版本为某个区间，用数学表达式区间表示，
            eg:{"(?!2800 V3)" : ["V200R002C20.*", "[V200R002C30,V200R002C30SPC200]"}，表示问题型号为非2800 V3，
                                        问题版本为所有V200R002C20版本，V200R002C30到V200R002C30SPC200版本，包含问题版本为所有V200R002C20版本与V200R002C30SPC200。
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLID回显
        errMsg：方法异常结束时的原因
        isPass：True/False，检查是否通过（设备型号与版本不是问题版本时检查通过）
        productModel：设备型号
        productVersion：设备版本
    """
    isSucc = True
    allCliRet = ""
    errMsg = ""

    isPass = True
    productModel = ""
    productVersion = ""

    #获取产品型号
    checkRet = getProductModel(cli, LANG)
    allCliRet += checkRet[1]
    if checkRet[0] != True:
        isSucc = False
        isPass = False
        errMsg = checkRet[4]
        return (isSucc, allCliRet, errMsg, isPass, productModel, productVersion)

    #设备型号
    productModel = checkRet[2]

    #获取产品软件版本
    checkRet, versionDictList, hotPatchVersionList = parse_upgradePackage(cli, LANG)
    allCliRet += checkRet[1]
    if checkRet[0] != True:
        isSucc = False
        isPass = False
        errMsg = checkRet[2]
        return (isSucc, allCliRet, errMsg, isPass, productModel, productVersion)

    #产品版本（版本全称）
    productVersion = versionDictList[0]["Current Version"]

    for model in modelAndVersionListDict:
        if not re.match(model, productModel):
            #设备型号不是问题型号
            continue

        versionList = modelAndVersionListDict.get(model)

        if isBugProductVersion(productVersion, versionList):
            #问题型号与版本均匹配
            isSucc = True
            isPass = False
            return (isSucc, allCliRet, errMsg, isPass, productModel, productVersion)
        else:
            return (isSucc, allCliRet, errMsg, isPass, productModel, productVersion)

    return (isSucc, allCliRet, errMsg, isPass, productModel, productVersion)