# -*- coding: UTF-8 -*-
import traceback
import re

#########################################
#      公共判断脚本
#########################################

PRODUCT_VERSION = "V100R001C99"

#非超级用户CLI命令开放的SPC版本号
CLI_SUPPORT_VERSION = "V100R001C99SPC500"

#非超级用户CLI命令没开放的SPC版本号
NOT_SUPPORT_VERSION = ["V100R001C99SPC100","V100R001C99SPC200","V100R001C99SPC300","V100R001C99SPC400"]

###*****   巡检结果标志,不支持（命令因权限等原因不支持情况），通过，不通过   *****###
CHECK_NOSUPPORT = "NOSUPPORT"
CHECK_PASS = "TRUE"
CHECK_NOPASS = "FLASE"
CHECK_NOCHECK = 'NOCHECK'
#返回此种结果，需要根据后续的逻辑判断来确实是否通过
CHECK_UNKOWN = "UNKOWN"

###*****   模式标志   *****###
SUPER_USER_FLAG = "admin:/>"
DEVELOPER_MODEL_FLAG = "developer:/>"

CLI_RET_END_FLAG = ":/>"

#重试次数，防止多次重试死循环
MAX_RETRYS = 5

#黑名单中字符串一定存在于前5行
KEY_WORD_VALID_LINES = 5

###*****   异常回显黑名单，前四行包含下面关键字，巡检不通过，此名单中返回CHECK_NOPASS   *****###
nopassDict = [
{"key_word":"Safe mode",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},
              
{"key_word":"minisystem",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"}, 
                            
{"key_word":"-bash:",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},

{"key_word":"Storage:",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},
           
{"key_word":"System is upgrading",
 "msg_zh":u"\n系统处于升级模式",
 "msg_en":"\nSystem is in upgrade Mode"},
           
{"key_word":"The system is powering off",
 "msg_zh":u"\n系统不正常，处于下电模式",
 "msg_en":"\nSystem is abnormal, powering off"},
           
{"key_word":"The user is offline",
 "msg_zh":u"\n系统不正常，用户离线",
 "msg_en":"\nThe user is offline"}, 
 
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},
             
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},
              
{"key_word":"Receive message failed",
 "msg_zh":u"\n接受信息失败",
 "msg_en":"\nReceive message failed"},
 
{"key_word":"Cmd execute except",
 "msg_zh":u"\n命令执行异常",
 "msg_en":"\nCmd execute except"},
             
{"key_word":"The system has an internal error",
 "msg_zh":u"\n设备内部错误",
 "msg_en":"\nThe system has an internal error"},
              
{"key_word":"The user account has been locked",
 "msg_zh":u"\n用户账号被锁定",
 "msg_en":"\nThe user account has been locked"},
    
{"key_word":"Unknown error",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"},
              
{"key_word":"error:",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"}                    
             ]

###*****   行黑名单，针对捕捉到某行回显信息，判断当前行是否显示异常，巡检不通过，此名单中返回CHECK_NOPASS   *****###
nopassLineDict = [
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},
           
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},
                 
{"key_word":"The system has an internal error",
 "msg_zh":u"\n设备内部错误",
 "msg_en":"\nThe system has an internal error"},
 
{"key_word":"Cmd execute except",
 "msg_zh":u"\n命令执行异常",
 "msg_en":"\nCmd execute except"},
                  
{"key_word":"Unknown error",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"}
             ]

###*****   不支持名单，针对捕捉到某行回显信息，判断当前回显是否支持，此名单中返回CHECK_NOSUPPORT   *****###
#命令当前版本不支持时
nosupportDict = [
                               
{"key_word":"\^",
 "msg_zh":u"\n当前版本不支持此命令",
 "msg_en":"\nCommand not support at current version"} 
              ]

###*****   白名单可能存在的情况，含有以下标记表示直接巡检通过   *****###
#命令执行成功
CLI_EXEC_SUCCESS = "Command executed successfully"
#命令当前版本不支持，需要使用超时接口cliExecRetWithTimeout
NOT_SUPPORT_CMD_FLAG = "\^"
CLI_EXEC_NO_LICENSE = "Suggestion: Import a valid license file."
CLI_EXEC_LICENSE_VALID = "license file is unavailable."

###*****   白名单，前四行包含下面关键字，直接巡检通过，此名单中返回CHECK_PASS  *****###
#白名单，回显中捕捉到这列关键字时，巡检通过，巡检通过时无法添加详细信息，只需要列出关键字即可
passList = [CLI_EXEC_SUCCESS,NOT_SUPPORT_CMD_FLAG,CLI_EXEC_NO_LICENSE,CLI_EXEC_LICENSE_VALID]


# **************************************************************************** #
# 函数名称: isChinese
# 功能说明: 判断当前语言是否为中文
# 输入参数: lang
# 输出参数: True or False
# **************************************************************************** # 
def isChinese(lang):
    if "zh"==lang:
        return True
    return False

# **************************************************************************** #
# 函数名称: isExecSuccess
# 功能说明: 为了加速判定，根据回显结果判定命令是否执行成功（一般针对无记录的情况）
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False:
# **************************************************************************** # 
def isExecSuccess(cliRet):
    if re.search(CLI_EXEC_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserRet
# 功能说明: 根据CLI回显判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserRet(cliRet):
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if SUPER_USER_FLAG == lineList[-1].replace(" ",""):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserCli
# 功能说明: 根据传入的SSH连接执行CLI命令后判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserCli(cli):
    cliRet = cli.execCmdNoLog("show system general")
    return isSuperUserRet(cliRet)

# **************************************************************************** #
# 函数名称: isSuperUserJava
# 功能说明: 根据根据框架传入的参数判断是否为超级管理员用户
# 输入参数: cliRet,py_java_env
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserJava(py_java_env):
    userName =  py_java_env.get("devInfoMap").get("userName")
    if "admin" ==  userName:
        return True
    return False

# **************************************************************************** #
# 函数名称: isDeveloperModle
# 功能说明: 判断当前捕捉到的CLI回显是否在Developer模式下
# 输入参数: cliRet
# 输出参数: True:Developer模式，False:非Developer模式
# **************************************************************************** # 
def isDeveloperModle(cliRet):
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: getCurrentVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC，为了加快获取速率，
#           一次获取成功后保存到框架提供的全局变量中
# 输入参数: cli, py_java_env
# 输出参数: currentVesion
# **************************************************************************** # 
def getCurrentVesion(cli, py_java_env):  
    currentVesion = ""
    currentVesion = py_java_env.get("currentVesion")
    
    #未在全局标记中获取到有效的版本号，则通过执行CLI命令查看
    if None == currentVesion or "" == currentVesion:        
        cliRet = cli.execCmdNoLog("show upgrade package")
        lineList = cliRet.splitlines()
        
        #直接从包含版本号的行开始
        for line in lineList[5:-1]:
            columnList = line.split()
            #找到关键字，直接返回当前版本号
            for column in columnList[2:-1]:
                if -1 != column.find(PRODUCT_VERSION):
                    currentVesion = column[0:17]
                    #在框架提供的全局变量里设置当前的版本号,精确到SPC
                    py_java_env.put("currentVesion", currentVesion)
                    return currentVesion           
    return currentVesion


def getCurrentVesionWithCliRet(cli, py_java_env):
    currentVesion = ""
    currentVesion = py_java_env.get("currentVesion")
    cliRet = ''
    # 未在全局标记中获取到有效的版本号，则通过执行CLI命令查看
    if None == currentVesion or "" == currentVesion:
        cliRet = cli.execCmdNoLog("show upgrade package")
        lineList = cliRet.splitlines()

        # 直接从包含版本号的行开始
        for line in lineList[5:-1]:
            columnList = line.split()
            # 找到关键字，直接返回当前版本号
            for column in columnList[2:-1]:
                if -1 != column.find(PRODUCT_VERSION):
                    currentVesion = column[0:17]
                    # 在框架提供的全局变量里设置当前的版本号,精确到SPC
                    py_java_env.put("currentVesion", currentVesion)
                    return currentVesion, cliRet
    return currentVesion, cliRet


# **************************************************************************** #
# 函数名称: isSupportVesion
# 功能说明: 判断当前版本是否支持
# 输入参数: cli,py_java_env
# 输出参数: True or False
# **************************************************************************** # 
def isSupportVersion(cli, py_java_env):
    currentVesion =  getCurrentVesion(cli,py_java_env)
    #没有查询到版本号，最少还可以切入developer模式执行命令，不至于卡主，因此返回为False
    if ""==currentVesion:
        return False
    
    if currentVesion not in NOT_SUPPORT_VERSION: 
        return True
    return False

# **************************************************************************** #
# 函数名称: getUserNameCliRet
# 功能说明: 根据CLI回显获取用户名
# 输入参数: cli,py_java_env
# 输出参数: userName
# **************************************************************************** # 
def getUserNameCliRet(cliRet):
    userName = ""
    
    #输入CLI回显是否有效
    if None == cliRet or "" == cliRet: 
        return userName
    
    #执行命令获取当前登录的用户名      
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if lineList[-1].find(CLI_RET_END_FLAG):
        userName = lineList[-1].replace(" ","").replace(CLI_RET_END_FLAG,"")
    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取当前用户名
# 输入参数: cli,py_java_env
# 输出参数: userName
# **************************************************************************** # 
def getUserName(cli,py_java_env):
    userName = ""
    
    #从框架提供的接口获取
    userName = py_java_env.get("devInfoMap").get("userName")
    if None != userName and "" != userName: 
        #在框架提供的全局变量里设置当前的用户名
        py_java_env.put("userName", userName)
        return userName
    
    #从脚本保存好的全局变量中获取，防止框架提供的接口中无法获取的情况
    userName = py_java_env.get("userName")
    if None != userName and "" != userName: 
        return userName
    
    #执行命令获取当前登录的用户名      
    cliRet = cli.execCmdNoLog("show system general")
    
    userName = getUserNameCliRet(cliRet)
    py_java_env.put("userName", userName)
     
    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取登录用户名的权限级别
# 输入参数: cli,py_java_env
# 输出参数: super_admin,guest或admin
# **************************************************************************** # 
def getUserPrivilege(cli,py_java_env):
    
    userLevel = ""
    userName = getUserName(cli,py_java_env) 
    if None == userName or "" == userName: 
        return ""
    
    userLevel = py_java_env.get("userLevel")
    #未在全局标记中获取到当前用户登录的权限级别，则通过执行CLI命令查看
    if None == userLevel or "" == userLevel:
        cliRet = cli.execCmdNoLog("show user "+userName)
        lineList = cliRet.splitlines()
        
        #用户名一定包含在最后一行中
        for line in lineList[4:-1]:
            columnList = line.split()
            if len(columnList)>2:
                if userName == columnList[0]:
                    userLevel = columnList[1]
                    py_java_env.put("userLevel", userLevel)
                    return userLevel
    return userLevel

# **************************************************************************** #
# 函数名称: change2cli
# 功能说明: 任何模式尝试切换到CLI模式        
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def anyModel2Cli(cli, py_java_env):
    index =0 
    cliRet = cli.execCmdNoLog("show system general")
    
    userName = getUserName(cli,py_java_env)
    if ""== userName:
        userName = "admin"
    
    #已经在CLI模式下，无需退出
    if userName == getUserNameCliRet(cliRet):
        return
    
    #密码输入错误时及正常情况从developer模式下退出
    if -1 != cliRet.find("Password") or -1 != cliRet.find(DEVELOPER_MODEL_FLAG):
        cliRet = cli.execCmd("exit")
        while(-1 == cliRet.find(userName+CLI_RET_END_FLAG)):
            index+=1
            cliRet = cli.execCmd("exit")
            if -1!=cliRet.find("Are you sure to exit?(y/n):"):
                cliRet = cli.execCmd("n")
            if index>MAX_RETRYS:
                break
    return
            
# **************************************************************************** #
# 函数名称: change2Developer
# 功能说明: 切换到developer模式
# 输入参数: cli,py_java_env
# 输出参数: True:，False:
# **************************************************************************** # 
def change2Developer(cli, py_java_env):
    errMsg = ""
    flag = False
    lang = py_java_env.get("lang")
    if None == cli:    
        return (flag,"",errMsg)
    
    #guest用户无法切换到developer模式
    userLevel = getUserPrivilege(cli,py_java_env)
    if "guest" ==userLevel or ""==userLevel:
        if isChinese(lang):
            errMsg = u"\n系统没有运行在admin模式，无法进入调试模式"
        else:
            errMsg = "\nSystem is not in admin model, can't enter to debug model"
        return (flag,"",errMsg)

    checkRet = isCliExecRetInBlackList(cli,"developer",lang,True)
    cliRet = checkRet[1]
    errMsg = checkRet[2]
    if checkRet[0]:
        flag = False
        anyModel2Cli(cli, py_java_env)
        return (flag, cliRet, errMsg)
    
    password = str(py_java_env.get("devPwd").get("developer"))
    if(None == password or ""== password):
        anyModel2Cli(cli, py_java_env)
        if isChinese(lang):
            errMsg = u"\n获取developer密码为空，请确定是否已配置密码"
        else:
            errMsg = "\nGet the password of developer is null, make sure to configure it right"
        return (flag, cliRet, errMsg)
    
    checkRet = isCliExecRetInBlackList(cli,password,lang,False)
    cliRet = checkRet[1]
    errMsg = checkRet[2]
    if checkRet[0]:
        flag = False
        anyModel2Cli(cli, py_java_env)
        return (flag, cliRet, errMsg)
    
    #已经成功进入Developer模式
    if isDeveloperModle(cliRet):
        flag = True
    else:
        anyModel2Cli(cli, py_java_env)
        
        #再次判断CLI回显是否在黑名单中，由于进入失败，需要全部判断
        lineList = cliRet.splitlines()
        for line in lineList:
            checkRet = isLineInBlackDict(line, lang)
            if checkRet[0]:
                flag = False
                errMsg = checkRet[1]     
                return (flag, cliRet, errMsg)   
                
        if isChinese(lang):
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):
                errMsg = u"\ndeveloper密码配置错误"
            else:
                errMsg = u"\n系统不能进入developer模式"
        else:
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):     
                errMsg = "\nThe password of developer is wrong"
            else:
                errMsg = "\nIt failed to access the developer mode" 
    return (flag, cliRet, errMsg)


# **************************************************************************** #
# 函数名称: developer2Cli
# 功能说明: developer模式下退出到CLI模式
# 输入参数: cli
# 输出参数: 无
# **************************************************************************** # 
def developer2Cli(cli):
    cli.execCmd("exit")
   
# **************************************************************************** #
# 函数名称: isEndLine
# 功能说明: 判断此行是否是回显结束行
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isEndLine(cliLine):
    
    #空行，无效行 非结束标志
    if "" == cliLine or None == cliLine:
        return False
    
    if -1 != cliLine.find(CLI_RET_END_FLAG):
        return True     
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithStart
# 功能说明: 跳过以某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithStart(cliLine, startWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(startWordList):
        return False
    

    for startWord in startWordList:
        #是以XXX起头的行，需要跳过
        if line.startswith(startWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithContain
# 功能说明: 跳过包含某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithContain(cliLine, containWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(containWordList):
        return False
    

    for containWord in containWordList:
        #包含XXX关键字的行，需要跳过
        if -1 != line.find(containWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isLineInBlackDict
# 功能说明: 判断某行回显是否在黑名单中，在黑名单中表示CLI回显存在异常，巡检不通过，
# 不用逻辑判断，最终应返回CHECK_NOPASS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInBlackDict(cliLine, lang):
    
    errMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于黑名单
    if None == cliLine or "" == cliLine:
        return (False, errMsg)
    
    for dictItems in nopassDict:
        #黑名单中找到，表示回显无效，直接退出
        if re.search(dictItems.get("key_word"), cliLine, re.IGNORECASE):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

# **************************************************************************** #
# 函数名称: isLineInWhiteList
# 功能说明: 判断某行回显是否在白名单中，在白名单中表示CLI执行完成，巡检通过，
# 不用逻辑判断，最终应返回CHECK_PASS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInWhiteList(cliLine):
       
    keyWordMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于白名单
    if None == cliLine or "" == cliLine:
        return (False, keyWordMsg)

    for keyWords in passList:      
        #在白名单中找到关键字，表示命令执行成功，直接退出
        if re.search(keyWords, cliLine, re.IGNORECASE):
            keyWordMsg = keyWords
            return (True, keyWordMsg)
        
        #无license情况通过
        if re.search("license", cliLine, re.IGNORECASE):
            if -1 != cliLine.find("Error:") or -1 != cliLine.find("Suggestion:"):
                keyWordMsg = "license"
                return (True, keyWordMsg)
        
    return (False,keyWordMsg)

# **************************************************************************** #
# 函数名称: isNormalLine
# 功能说明: 判断某行回显是否正常
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isNormalLine(cliLine, lang):
    errMsg = ""
    
    #空行，无效行正常
    if None == cliLine or "" == cliLine:
        return (True, errMsg)
    
    #异常的关键字存在于前几行，防止全部判断时误判
    #某行是否因为超时或是其它原因，导致端口信息查询不到的情况，返回不正常行
    for dictItems in nopassLineDict:
        if re.search(dictItems.get("key_word"), cliLine, re.IGNORECASE):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (False, errMsg)
    return (True, errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInBlackList
# 功能说明: 传入待执行CLI命令，判断回显是否在黑名单中，带回结果供各个巡检项调用
#           1）黑名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回异常回显对应的中英文提示信息
# **************************************************************************** # 
def isCliExecRetInBlackList(cli, cmd, lang, isHasLog):
    
    errMsg = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (True, cliRet, errMsg)
    
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，则回显正常，不在黑名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, errMsg)
    
    #回显在白名单中直接返回，不带回key值
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            return (False, cliRet, errMsg) 
    
    #是否在黑名单中，若是获取具体错误原因
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (True, cliRet, errMsg)  
    
    #不在黑名单中 
    return (False,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInWriteList
# 功能说明: 传入待执行CLI命令，判断回显是否在白名单中，带回结果供各个巡检项调用
#           1）白名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回捕捉到的白名单关键字信息
# **************************************************************************** # 
def isCliExecRetInWriteList(cli, cmd, lang, isHasLog):
    
    writeKeyWords = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        return (False, cliRet, writeKeyWords)
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，不在白名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, writeKeyWords)
    
    for line in lineList[0:KEY_WORD_VALID_LINES]:
               
        #是否在白名单中
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            writeKeyWords = checkRet[1]
            return (True, cliRet, writeKeyWords)
    
    #不在白名单中
    return (False,cliRet,writeKeyWords)


# **************************************************************************** #
# 函数名称: cliRetPreCheck（供公共接口cliExecRetJudgeWithDoubleList及
#           cliExecRetWithTimeout调用）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cliRet（CLI回显内容），lang
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 
def cliRetPreCheck(cliRet, lang):
    errMsg = ""
    
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (CHECK_NOPASS, cliRet, errMsg)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，说明回显执行一定有结果，需要通过后续逻辑去判断
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (CHECK_UNKOWN, cliRet, errMsg)
    
    #在白名单中，巡检可以直接通过
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CHECK_PASS, cliRet,"")
        
    #在黑名单中，巡检直接不通过    
    for line in lineList[0:KEY_WORD_VALID_LINES]:   
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CHECK_NOPASS, cliRet, errMsg) 
        
    #其它情况，说明需要通过回显来判断此项巡检是否通过   
    return (CHECK_UNKOWN,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: cliExecRetJudgeWithDoubleList（新增接口）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 
def cliExecRetJudgeWithDoubleList(cli, cmd, lang, isHasLog):
    
    #CLI命令执行是否打印日志 
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    return cliRetPreCheck(cliRet, lang)


# **************************************************************************** #
# 函数名称: cliExecRetWithTimeout（公共调用函数）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项，本函数带超时处理机制
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 

def cliExecRetWithTimeout(cli, cmd, timeOut, lang):
    
    #执行带超时设置的CLI命令
    cliRet = cli.execCmdWithTimout(cmd,timeOut)
    
    return cliRetPreCheck(cliRet, lang)
