# -*- coding: UTF-8 -*-
import cliUtil
import common
import re
import traceback

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


# 配置涉及该预警的产品型号和版本
bugProductModelList = [
    "18500",
    "18800",
    "2600 V3",
    "5500 V3",
    "5800 V3",
    "6800 V3",
    "18800 V3",
    "5500 V5",
    "5600 V5",
    "6800 V5",
    "18800 V5",
    "Dorado5000 V3",
    "Dorado6000 V3",
]

RISK_VERSION_DICT = {
    "V100R001C30SPC300": "V100R001C30SPH302",
    "V100R001C30SPC200": "V100R001C30SPH212",
    "V300R001C20SPC200": "V300R001C20SPH211",
    "V300R002C10SPC200": "V300R002C10SPH212",
    "V300R003C00SPC100": "V300R003C00SPH111",
    "V300R003C10SPC100": "V300R003C10SPH119",
    "V300R003C20": "V300R003C20SPH003",
    "V300R003C20SPC200": "V300R003C20SPH218",
    "V300R005C00SPC300": "V300R005C00SPH310",
    "V300R006C00SPC100": "V300R006C00SPH116",
    "V300R006C10SPC100": "V300R006C10SPH106",
    "V300R006C20": "V300R006C20SPH018",
    "V300R006C50SPC100": "V300R006C50SPH109",
    "V500R007C00SPC100": "V500R007C00SPH106",
    "V500R007C10": "V500R007C10SPH023",
    "V500R007C30SPC100": "V500R007C30SPH109",
}

DORADO_VERSION_DIST = {
    "V300R001C01SPC100": "V300R001C01SPH106",
    "V300R001C21SPC100": "V300R001C21SPH117",
    "V300R001C30SPC100": "V300R001C30SPH108",
    "V300R002C00SPC100": "V300R002C00SPH105",
    "V300R002C10SPC100": "V300R002C10SPH110",
    "V300R002C20": "V300R002C20SPH002",
}


def execute(cli):
    ckeck = CheckAS32420200706(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = ckeck.execute_check()
    return flag, "\n".join(ckeck.all_ret_list), msg


class CheckAS32420200706:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = list()

    def execute_check(self):
        """
        @summary: 	A&S324-20200706-ALL关于集中式存储的若干重大问题整改公告
        @param cli: CLI连接
        @return: (通过标识, CLI回显信息, 错误信息) as (boolean, string, string)
        """

        # 判断当前系统型号和版本是否存在风险
        try:
            # 查询产品型号
            flag, cli_ret, product_model, err_msg = cliUtil.getProductModel(
                self.cli, LANG
            )
            self.all_ret_list.append(cli_ret)
            if flag is not True:
                return flag, err_msg

            # 判断当前产品型号是否是风险产品型号
            if product_model not in bugProductModelList:
                return True, err_msg

            # 查询当前产品软件版本号
            (
                flag,
                soft_version,
                patch_version,
                cli_ret,
                err_msg,
            ) = common.get_soft_and_patch_version(self.cli, LOGGER, LANG)
            self.all_ret_list.append(cli_ret)
            if flag is not True:
                return flag, err_msg
            risk_dict = (
                DORADO_VERSION_DIST
                if "Dorado" in product_model
                else RISK_VERSION_DICT
            )
            # 检查当前软件版本是否在属于风险版本
            flag = self.check_risk_version(
                soft_version, patch_version, risk_dict
            )
            if flag is True:
                return (
                    False,
                    common.getMsg(
                        self.lang,
                        "soft.ware.ad.domain.access.cifs.suggestion",
                        (soft_version, patch_version)))
            return True, ""
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def check_risk_version(self, soft_version, patch_version, risk_dict):
        """
        检查是否为风险版本
        :return: 有风险的为True
        """
        if soft_version not in risk_dict:
            return False
        if re.search(r"SPH\d[6-9]\d", patch_version):
            return True
        risk_hot_patch = risk_dict.get(soft_version)
        if risk_hot_patch and patch_version >= risk_hot_patch:
            return False
        return True
