# -*- coding: UTF-8 -*-
import re
import traceback

import cliUtil
import common
from config import RISK_AD_DOMAIN_ACCESS_CIFS_VERSION_AND_PATCH

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    AD域用户CIFS访问检查
    :param cli:
    :return:
    """
    access_check = AdDomainAccessCifsCheck(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = access_check.execute_check()
    return flag, "\n".join(access_check.all_ret_list), msg


class AdDomainAccessCifsCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []

    def execute_check(self):
        try:
            return self.cifs_ad_domain_patch_check()
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def cifs_ad_domain_patch_check(self):
        """
        如果步骤2中Running Status是Start状态且步骤3中Domain Status是Joined状态则继续检查；否则检查通过；
        如果步骤4中查询的补丁版本高于或等于应打补丁版本则检查通过
        如果当前版本没有推荐补丁，则检查结果为建议优化。
        :return:
        """

        p_version = str(self.env.get("devInfo").getProductVersion())
        flag, __, p_patch, ret, msg = common.get_soft_and_patch_version(
            self.cli, self.logger, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)

        sugg_patch = RISK_AD_DOMAIN_ACCESS_CIFS_VERSION_AND_PATCH.get(
            p_version, ""
        )
        if (
            p_version in RISK_AD_DOMAIN_ACCESS_CIFS_VERSION_AND_PATCH
            and p_patch >= sugg_patch
            and not common.is_special_patch(p_patch)
        ):
            return True, ""

        if not self.check_all_vstore_has_access_cifs():
            return True, ""

        if p_version not in RISK_AD_DOMAIN_ACCESS_CIFS_VERSION_AND_PATCH:
            msg = common.getMsg(
                self.lang,
                "soft.ware.ad.domain.access.cifs.suggestion",
                (p_version, p_patch),
            )
            return cliUtil.RESULT_WARNING, msg

        msg = common.getMsg(
            self.lang,
            "soft.ware.ad.domain.access.cifs.not.pass",
            (p_version, p_patch, sugg_patch),
        )
        return False, msg

    def check_all_vstore_has_access_cifs(self):
        """
        任意一个租户下的 存在cifs start和状态joined 就不通过
        :param vstore_ids:
        :return:
        """
        if self.has_cifs_service() and self.has_domain_ad_info():
            return True

        all_vstore_ids = self.get_all_vstore()
        if not all_vstore_ids:
            return False

        param_dict = {}
        param_dict["cli"] = self.cli
        param_dict["logger"] = self.logger
        param_dict["lang"] = self.lang
        param_dict["cli_ret_all"] = self.all_ret_list
        for vstore_id in all_vstore_ids:
            param_dict["vstore_id"] = vstore_id
            # 在当前租户下检查控制器IP
            if self.check_vstore_status(param_dict):
                self.logger.logInfo(
                    "vstore is joined and used cifs[{}]".format(vstore_id)
                )
                return True

        return False

    def has_cifs_service(self):
        """
        是否Running Status是Start状态
        :return: True: Start, False: 非Start
        """
        cmd = "show service cifs"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if not cliUtil.hasCliExecPrivilege(ret):
            return False
        if flag is not True:
            raise common.UnCheckException(msg, ret)

        regx = re.compile(r"Running Status\s+:\s*Start", re.IGNORECASE)
        if regx.search(ret):
            return True

        return False

    def has_domain_ad_info(self):
        """
        Domain Status是Joined状态
        :return: True: Joined, False: 非Joined
        """
        cmd = "show domain ad"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if not cliUtil.hasCliExecPrivilege(ret):
            return False
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        regx = re.compile(r"domain status\s+:\s*joined", re.IGNORECASE)
        if regx.search(ret):
            return True

        return False

    @common.check_in_vstore_mode_wrap
    def check_vstore_status(self, param_dict, *args, **kargs):
        """
        :param param_dict: 参数字典
        :param args: 可变参数
        :param kargs:
        :return:
        """
        return self.has_cifs_service() and self.has_domain_ad_info()

    def get_all_vstore(self):
        """
        获取所有的vstore
        :return: id 列表
        """
        cmd = "show vstore |filterColumn include columnList=ID"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if not cliUtil.hasCliExecPrivilege(
            ret
        ) or cliUtil.queryResultWithNoRecord(ret):
            return []

        if flag is not True:
            if flag == cliUtil.RESULT_NOSUPPORT:
                return []
            raise common.UnCheckException(msg, ret)

        res_dict_list = cliUtil.getHorizontalCliRet(ret)
        return [
            res_dict.get("ID")
            for res_dict in res_dict_list
            if res_dict.get("ID")
        ]
