# -*- coding: UTF-8 -*-
import re
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
DEVPWD = py_java_env.get("devPwd").get("developer")

def execute(cli):
    '''
    @summary: 客户端查询CIFS共享导致控制器复位
    @param cli: CLI连接
    @return: (通过标识, CLI回显信息, 错误信息) as (boolean, string, string)
    '''
    
    #返回值定义
    flag = False
    cliRet = ""
    errMsg = ""
    
    #配置涉及该预警的产品型号和版本
    bugProductModelList = ['S2600T', 'S5500T', 'S5600T', 'S5800T', 'S6800T']
    bugVersionList = ['V200R002C00','V200R002C00SPC100','V200R002C00SPC200','V200R002C00SPC300','V200R002C00SPC400','V200R002C10','V200R002C20']

    #判断当前系统型号和版本是否存在风险
    try:
        #查询产品型号
        result, cliRet, productModel, errMsg = cliUtil.getProductModel(cli, LANG)
        if not result or len(productModel) == 0:
            LOGGER.logSysAbnormal()
            return (flag, cliRet, errMsg)
        
        #判断当前产品型号是否是风险产品型号
        if productModel not in bugProductModelList:
            flag = cliUtil.RESULT_NOSUPPORT
            LOGGER.logPass()
            return (flag, cliRet, errMsg)
            
        #查询当前产品软件版本号
        checkRet, currentVersionDictList, hotPatchVersionList = common.parse_upgradePackage(cli, LANG)
        cliRet += '\n' + checkRet[1]
        if checkRet[0] != True:
            LOGGER.logSysAbnormal()
            return (checkRet[0], cliRet , checkRet[2])
        
        #检查当前软件版本是否在属于风险版本
        result = common.checkRiskVersion(currentVersionDictList, bugVersionList, LANG)
        if result != True:
            flag = True
            return (flag, cliRet, errMsg)
        
        #获取当前环境soft version
        SoftVersionDict = currentVersionDictList[0]
        softCurrentVersion = SoftVersionDict.get("Current Version")
        sotfStandardVersion = common.switchValidVersion(softCurrentVersion)
        errMsg = common.getMsg(LANG, "current.version", sotfStandardVersion)

        patchVersion = "V200R002C00SPH402"
        softVersion = "V200R002C00SPC400"
        if sotfStandardVersion == softVersion:
        #预设每个控制器的补丁都是ok的
            flag = True

            #循环获取每个控制器的补丁信息       
            for hotPatchVersion in hotPatchVersionList:
                hotPatchCurrentVersion = hotPatchVersion.get("Current Version")
                contrllerID = hotPatchVersion.get("Name")

                if hotPatchCurrentVersion == "--" or hotPatchCurrentVersion == "":
                        errMsg += common.getMsg(LANG, "contrller.not.install.pacth", contrllerID)
                        flag = False
                elif hotPatchCurrentVersion >= patchVersion:
                        continue
                else:
                    errMsg += common.getMsg(LANG, "contrller.pacth.too.low", (contrllerID, hotPatchCurrentVersion, patchVersion))
                    flag = False

            if flag == True:
                return (flag, cliRet, errMsg)


        #获取thin LUN信息
        result = getThinLunInfo(cli)
        
        #获取thin LUN信息时发生异常
        if result[0] != True:
            LOGGER.logNoPass("get thin LUN information exception")
            return (False, cliRet, common.getMsg(LANG, "get.thin.LUN.info.exception"))
        
        thinLunInfoDictList = result[2]
        cliRet += '\n' + result[1]
        
        #未配置thin LUN
        if thinLunInfoDictList == []:
            flag = True
            errMsg = ""
            return (flag, cliRet, errMsg)
        
        #比较每个thin LUN的Subscribed Capacity值和Capacity值大小
        result = compareCapacityAndSubscribedCapacity(thinLunInfoDictList)
        if result[0] != True:
            errMsg += result[1]
            return(False, cliRet, errMsg)
        
        #建立所有控制器的连接，然后各个thin LUN匹配各自控制器对应的连接执行命令。最后统一关闭链接
        allControlConnectDict = buildAllControlConnect(currentVersionDictList)
        if allControlConnectDict == {}:
            return(False, cliRet, common.getMsg(LANG, "cannot.connect.controller"))
        
        #在所有thin lun对应的控制器上执行命令，查看Is write protected是否为TRUE
        notSetIpList = []
        cannotConnectList = []
        hasIsWriteProtected = False
        for thinLunInfoDict in thinLunInfoDictList:
            thinLunId = thinLunInfoDict.get("ID")
            workController = thinLunInfoDict.get("Work Controller")
            
            #该thin LUN对应的控制器没有IP，故没有对应的链接
            if not allControlConnectDict.has_key(workController):
                if not workController in notSetIpList:
                    errMsg += common.getMsg(LANG, "not.set.up.ip", workController)
                    notSetIpList.append(workController)
                continue
           
            #查看当前控制器的连接是否创建成功,不成功给出未连接成功的提示
            connectCli = allControlConnectDict[workController]
            if connectCli == None:
                for currentVersionDict in currentVersionDictList:
                    if workController == currentVersionDict.get("Name"):
                        controlIp = currentVersionDict.get("IP")
                        if not controlIp in cannotConnectList:
                            errMsg += common.getMsg(LANG, "cannot.connect.ip", controlIp)
                            cannotConnectList.append(controlIp)
                            break
                continue
            
            #执行命令volume showctrl ID查看是否写保护  
            result = checkIsWriteProtected(connectCli, LANG, thinLunId) 
            cliRet += '\n' + result[1]
            errMsg += result[2]
            if result[0] == True: 
                hasIsWriteProtected = True
        
        closeAllConnect(currentVersionDictList, allControlConnectDict)
        
        if hasIsWriteProtected == True:
            flag = False
            return(flag, cliRet, errMsg)
        
        
        
        #配置了thin lun，则报的级别最低为建议优化
        flag = cliUtil.RESULT_WARNING
        errMsg += common.getMsg(LANG, "set.up.thin.lun")
        return(flag, cliRet, errMsg)  
       
        
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cliRet, common.getMsg(LANG, "query.result.abnormal"))
        
 
def compareCapacityAndSubscribedCapacity(thinLunInfoDictList):
    '''
    @summary: 比较Capacity和Subscribed Capacity两者值的大小、
    @param 
        thinLunInfoDictList：所有thin LUN的字典形式非空列表
    @return:
        True/False:Capacity > Subscribed Capacity / Capacity <= Subscribed Capacity
        errMsg:错误信息
    '''
    Capacity = "Capacity"
    SubscribedCapacity = "Subscribed Capacity"
    CapacityValue = ""
    CapacityValueNum = 0
    SubscribedCapacityValue = ""
    SubscribedCapacityValueNum = 0
    errMsg = ""
    flag = True
    
    try:
        for thinLunInfoDict in thinLunInfoDictList:
            thinLunId = thinLunInfoDict.get("ID")
            CapacityValue = thinLunInfoDict[Capacity]
            SubscribedCapacityValue = thinLunInfoDict[SubscribedCapacity]
        
            #转换成浮点数，统一使用GB作单位
            result = common.changeUnit2GB(CapacityValue)
            if result[0] != True:
                errMsg = common.getMsg(LANG, "change.unit.failed")
                return(False, errMsg)
            CapacityValueNum = result[1]
            
            result = common.changeUnit2GB(SubscribedCapacityValue)
            if result[0] != True:
                errMsg = common.getMsg(LANG, "change.unit.failed")
                return(False, errMsg)
            SubscribedCapacityValueNum = result[1]
        
            #已分配容量小于总容量时，不存在问题
            if SubscribedCapacityValueNum < CapacityValueNum:
                continue
            else:
                errMsg += common.getMsg(LANG, "Capacity.less.than.Subscribed.Capacity", (thinLunId, SubscribedCapacityValue, CapacityValue))
                flag = False
        
        return(flag, errMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, common.getMsg(LANG, "query.result.abnormal"))
    
    
    
def checkIsWriteProtected(connectCli, lang, thinLunId):
    '''
    @summary: 检查对应Thin LUN ID 的pool是否写保护
    @param 
        connectCli: CLI
        lang:语言
        thinLUNID：Thin LUN ID
    @return: 
        flag: True/False  写保护/不是写保护
        cliRet:cli回显
        errMsg:错误信息
    '''
    flag = False
    cliRet = ""
    errMsg = ""
    keyWord = "Is write protected"
    cmd = "volume showctrl " + thinLunId
    status = "TRUE"
   
    try:
        result = cliUtil.excuteCmdInDiagnoseMode(connectCli, cmd, True, lang, LOGGER, DEVPWD)
        cliRet += result[1]
        
        if result[0] != True:
            errMsg += result[2] 
            return(flag, cliRet, errMsg)
        
        thinLunInfo = result[1].splitlines()
        for line in thinLunInfo:
            if len(line.split(':')) < 2:
                continue
            if line.split(':')[0].strip() == keyWord and line.split(':')[1].strip() == status:
                flag = True         
                errMsg += common.getMsg(LANG, "Is.write.protected", thinLunId)
                return(flag, cliRet, errMsg)
            
        return(flag, cliRet, errMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cliRet, common.getMsg(LANG, "query.result.abnormal"))        

def buildAllControlConnect(currentVersionDictList):
    '''
    @summary: 建立到所有控制器的连接
    @param currentVersionDictList: 所有控制器信息字典列表
    @return: allControlConnectDictList 所有控制器名为key，连接的CLI为value的字典
    '''
    allConnectDict = {}
    
    try:
        if currentVersionDictList == []:
            return allConnectDict
        
        #获取所有控制器名列表
        for currentVersion in currentVersionDictList:
            controlName = currentVersion.get("Name")
            controlIp = currentVersion.get("IP")
            if controlIp == "--":
                continue
            
            connectCli = common.createSshConnection(py_java_env, controlIp, LOGGER)
            allConnectDict[controlName] = connectCli
            
        return allConnectDict
   
    except Exception, exception:
        allConnectDict = {}
        return allConnectDict 
    
def closeAllConnect(currentVersionDictList, allControlConnectDict):
    '''
    @summary: 关闭所有已建立的链接
    @param currentVersionDictList: 所有控制器信息字典列表
    '''
    except_counter = 0
    for currentVersion in currentVersionDictList:
        controlName = currentVersion.get("Name")
        if not allControlConnectDict.has_key(controlName):
            continue
       
        connectCli = allControlConnectDict[controlName]
        if connectCli == None:
            continue
        try:
            connectCli.close()     
        except:
            except_counter += 1
            
def getThinLunInfo(cli):
    '''
    @summary: 获取thin LUN信息
    @param cli: CLI链接
    @return: 
        flag:True/False 执行正确/错误
        cmdRet:命令回文
        thinLunInfoDictList：所有字段转换成字典形式的列表
    '''
    cmdRet = ""
    thinLunInfoDictList = []
    cmd = "show lun general |filterColumn include colunmList=ID,Capacity,Subscribed\sCapacity,Type,Work\sController|filterRow column=Type predict=equal_to value=Thin"
    
    try:
        cmdRet = cli.execCmd(cmd)
        thinLunInfoDictList = handleCliRet(cmdRet)
        return (True, cmdRet, thinLunInfoDictList)

    except Exception, exception:
        thinLunInfoDictList = []
        return (False, cmdRet, thinLunInfoDictList)
    
def handleCliRet(cliRet):
    '''
    @summary: 处理cli回显，将其处理成已表头为key的字典集合
    @param cliRet: CLI回显
    @return: 字典列表
    '''
    CLI_RET_END_FLAG = ":/>"
    
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
       
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            valueList = line.strip().split()
            for value in valueList:
                vals.append(value.strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
    