# -*- coding: UTF-8 -*-
import re
import cliResource
from com.huawei.ism.exception import IsmException

CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"
DEBUG_MODEL_FLAG = ":/diagnose>"
STORAGE_MODEL_FLAG = "Storage:~ #"
MINISYSTEM_MODEL_FLAG = "minisystem>"
DIAGNOSE_MODE_FLAG = "diagnose>"

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"

SERIES_T = "T_SERIES"
SERIES_V3 = "V3_SERIES"
SERIES_18000 = "18000_SERIES"

STOR_DEV_INFO_DICT = {
"S2600T":SERIES_T,
"S5500T":SERIES_T,
"S5600T":SERIES_T,
"S5800T":SERIES_T,
"S6800T":SERIES_T,
"2800 V3":SERIES_V3,
"5300 V3":SERIES_V3,
"5500 V3":SERIES_V3,
"5600 V3":SERIES_V3,
"5800 V3":SERIES_V3,
"6800 V3":SERIES_V3,
"6900 V3":SERIES_V3,
"HVS85T":SERIES_18000,
"HVS88T":SERIES_18000,
"18500":SERIES_18000,
"18800":SERIES_18000,
"18800F":SERIES_18000,
"18500 V3":SERIES_18000,
"18800 V3":SERIES_18000,
}

def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg
    
def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False

def isNoneLicense(cliRet):
    '''
    @summary: 判断回显结果里面是否包含需要license
    @param cliRet: cli回显
    @return: 
        True: cli回显包含需要license
        False: cli回显不包含需要license
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True
    
    return False

def isNotSupport(cliRet):
    '''
    @summary: 判断回显结果里面是否包含 does not support the VM service
    @param cliRet: cli回显
    @return: 
        True: cli回显包含 does not support the VM service
        False: cli回显不包含does not support the VM service
    '''
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "does not support" in lowerLine:
            return True
    
    return False

def isVMNotStarted(cliRet):
    '''
    @summary: 判断回显结果里面是否包含 The VM service is not started
    @param cliRet: cli回显
    @return: 
        True: cli回显包含 The VM service is not started
        False: cli回显不包含The VM service is not started
    '''
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "the vm service is not started" in lowerLine:
            return True
    
    return False

def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "does not have the permission" in line:
            return False
    return True

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    if re.search(CLI_EXECUTE_CMD_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

def isInternalError(cliRet):
    '''
    @summary: 判断回显结果是否包含内部错误信息
    @param cliRet: cli回显
    @return: 
        True: cli回显包含内部错误信息
        False: cli回显不包含内部错误信息
    '''
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False
    
def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    userName = ""

    try:
        cmd = "show system general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        lineList = cliRet.splitlines()
        
        if len(lineList) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)
        
        lastLine = lineList[-1]
        if CLI_RET_END_FLAG in lastLine:
            userName = lastLine.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return (True, userName, errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def getUserPrivilege(cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang) 
        if checkRet[0] != True:
            return checkRet
        
        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            name = line.get("Name")
            if name == userName:
                userPrivilege = line.get("Level")
                return (True, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def hasAdminOrSuperAdminPrivilege(cli, lang):
    '''
    @summary: 判断用户是否具有管理员或超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时
                True: 具有管理员或超级管理员权限
                False： 不具有管理员或超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    checkRet = getUserPrivilege(cli, lang)
    if checkRet[0] != True:
        return checkRet
    
    userLevel = checkRet[1]
    if userLevel is None or len(userLevel) == 0 or userLevel.lower() not in ["super_admin", "admin"]:
        return (True, False, "")
    else:
        return (True, True, "")
            
def enterDeveloperMode(cli, lang, devPasswd = ""):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    cmd = "change user_mode current_mode user_mode=developer"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    ctrl_C_ASCII = 0x03
    
    if checkRet[0] != True: 
        return checkRet
    cliRet = checkRet[1]
    
    #查看是否需要输入调试密码
    if re.search("Password", cliRet, re.IGNORECASE):
        #调试密码为空
        
        if devPasswd == "" or devPasswd == None:
            errMsg = getMsg(lang, "debugPasswd.is.empty")
            outCliRet = cli.execCmd(chr(ctrl_C_ASCII))
            return (False, cliRet, errMsg)
        
        #输入调试密码，不带日志
        enterCliRet = cli.execCmdNoLog(devPasswd)
        cliRet += enterCliRet
        #再次检查是否需要输入调试密码，如需要表示当前输入的密码错误，需要下发Ctrl C结束命令，回到CLI模式
        if re.search("Password", enterCliRet, re.IGNORECASE):
            errMsg = getMsg(lang, "cannot.access.developer.mode")
            outCliRet = cli.execCmd(chr(ctrl_C_ASCII))
            return (False, cliRet, errMsg)

    if isInDeveloperMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = getMsg(lang, "has.not.cli.privilege") 
        return (RESULT_NOCHECK, cliRet, errMsg)
    
def enterMinisystemMode(cli, lang, devPasswd = ""):
    '''
    @summary: 进入minisystem
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入minisystem模式成功
            False: 进入minisystem模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    cmd = "minisystem"
    checkRet = ""
    errMsg = ""
    checkRet = enterDeveloperMode(cli, lang, devPasswd)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    
    execCmdEnterMinisysRet = cli.execCmd(cmd)
    
    cliRet += '\n' + execCmdEnterMinisysRet
    
    if isInMinisystemMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = "Enter minisystem failed"
        return (RESULT_NOCHECK, cliRet, errMsg)
    
def enterDiagnoseMode(cli, lang, devPasswd = ""):
    '''
    @summary: 进入diagnose
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入diagnose模式成功
            False: 进入diagnose模式失败
        cliRet: cli回显
        errMsg: 进入diagnose模式失败时的错误消息
    '''
    cmd = "debug"
    checkRet = ""
    errMsg = ""
    
    checkRet = enterDeveloperMode(cli, lang, devPasswd)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    
    enterDiagnoseRet = cli.execCmd(cmd)
    
    cliRet += '\n' + enterDiagnoseRet
    
    if isInDiagnoseMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = enterDiagnoseRet
        return (RESULT_NOCHECK, cliRet, errMsg)

def developerMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    #此处输出“n”，目的是为了防止意外退出cli模式
    if "y/n" in cliRet:
        cli.execCmd("n")
        
    return None

def minisystemMode2DeveloperMode(cli):
    '''
    @summary: minisystem模式下退回到developer模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cli.execCmd("y")
        
    return None

def diagnoseMode2DeveloperMode(cli):
    '''
    @summary: diagnose模式下退回到developer模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    #此处输出“n”，目的是为了防止意外退出cli模式
    if "y/n" in cliRet:
        cli.execCmd("n")
        
    return None
        
def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

def isInMinisystemMode(cliRet):
    '''
    @summary: 判断当前是否在minisystem模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在minisystem模式下
        False: 当前不在minisystem模式下
    '''
    if re.search(MINISYSTEM_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

def isInDiagnoseMode(cliRet):
    '''
    @summary: 判断当前是否在diagnose模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在diagnose模式下
        False: 当前不在diagnoses模式下
    '''
    if re.search(DIAGNOSE_MODE_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, isHasLog, lang):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (RESULT_NOCHECK, "", errMsg)
    
    try:
        if isHasLog:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except IsmException:
        #出现异常时，由框架上层处理
        raise
    
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (RESULT_NOCHECK, cliRet, errMsg)   # 修改备注：回显长度为零，说明命令执行失败，由False改为未检查
    
    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            return (RESULT_NOCHECK, cliRet, checkRet[1]) # 修改备注：命令在黑名单中为未检查
    
    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)
    
    if isVMNotStarted(cliRet):
        errMsg = getMsg(lang, r"vm.is.not.started")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if isNotSupport(cliRet):
        errMsg = getMsg(lang, r"does.not.support.vm")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if isNoneLicense(cliRet):
        errMsg = getMsg(lang, "has.not.license")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    if isInternalError(cliRet):
        return (True, cliRet, errMsg)
    
    return (True, cliRet, errMsg)

def excuteCmdInDeveloperMode(cli, cmd, isHasLog, lang, devPasswd = ""):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    try:
        #兼容部分命令视图切换，先尝试在普通用户模式下执行，再尝试在研发模式下执行
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        if checkRet[0] == True:
            return checkRet
        
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang, devPasswd)
        if enterDeveloperCheckRet[0] != True:
            return enterDeveloperCheckRet
    
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        return checkRet
    except IsmException:
        #出现异常时，由框架上层处理
        raise

def excuteCmdInMinisystemMode(cli, cmd, isHasLog, lang, logger, devPasswd = ""):
    '''
    @summary: 获取minisystem模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param logger: 日志对象
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    flag = True
    errMsg = ""
    cliRet = ""
    
    try:
        enterMinisystemCheckRet = enterMinisystemMode(cli, lang, devPasswd)
        if enterMinisystemCheckRet[0] != True:
            return enterMinisystemCheckRet
        
        cliRet += enterMinisystemCheckRet[1]
        
        psCmdRet = cli.execCmd(cmd)
        
        if isInMinisystemMode(cliRet):
            minisystemMode2DeveloperMode(cli)
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        
        return (flag, psCmdRet, errMsg)
    
    except IsmException:
        #出现异常时，由框架上层处理
        if isInMinisystemMode(cliRet):
            minisystemMode2DeveloperMode(cli)
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        raise
   
def excuteCmdInDiagnoseMode(cli, cmd, isHasLog, lang, logger, devPasswd = ""):
    '''
    @summary: 获取diagnose模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param logger: 日志对象
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    flag = True
    errMsg = ""
    cliRet = ""
    
    try:
        enterDiagnoseRet = enterDiagnoseMode(cli, lang, devPasswd)
        if enterDiagnoseRet[0] != True:
            return enterDiagnoseRet
        
        cliRet += enterDiagnoseRet[1]
        
        cliRet += '\n' + cli.execCmd(cmd)
        
        if isInDiagnoseMode(cliRet):
            diagnoseMode2DeveloperMode(cli)
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        
        return (flag, cliRet, errMsg)
    
    except IsmException:
        #出现异常时，由框架上层处理
        if isInMinisystemMode(cliRet):
            minisystemMode2DeveloperMode(cli)
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        raise


def getProductModelVersion(cli, lang):
    '''
    @summary: 获取产品类型和产品版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型 产品版本 
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return (checkRet[0],checkRet[1],[],checkRet[2])
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    product_version = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
        elif fieldName == "Product Version":
            product_version = fieldValue
            
        if len(product_model) != 0 and len(product_version) != 0 and product_model != "--" and product_version != "--":
            return (True , cliRet, [product_model,product_version], "")
    
    return (False, cliRet, [], getMsg(lang, "cannot.get.product.version.info"))

def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return (checkRet[0],checkRet[1],"",checkRet[2])
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, cliRet, product_model, "")
        
    return (False, cliRet, "", getMsg(lang, "cannot.get.product.model.info"))

def getStorageType(cli, lang):
    '''
    @summary: 获取存储设备类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，存储设备类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = getProductModel(cli, lang)
    if getProductModelRet[0] != True:
        return getProductModelRet
    
    pdtModel = getProductModelRet[1]
    
    if not STOR_DEV_INFO_DICT.has_key(pdtModel):
        errMsg = getMsg(lang, "cannot.get.storage.type.info")
        return (False, "", errMsg)
    
    storageType = STOR_DEV_INFO_DICT.get(pdtModel)
    return (True, storageType, "")

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        
        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())
    
    return dictList

def getHorizontalCliRetFilterElabel(cliRet, getDetail=True):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合（只处理水平表格中过滤显示电子标签场景）
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    
    if not ("Electronic\sLabel" in cliRet and "Electronic Label" in cliRet):
        return None
    
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            #处理电子标签属性行
            if not line.startswith(" "):
                if len(dictList) == 0:
                    continue
                lineDict = dictList[-1]
                eLabelValue = lineDict.get("Electronic Label")
                newELabelValue = eLabelValue + "\n" + line
                lineDict["Electronic Label"] = newELabelValue
                dictList[-1] = lineDict.copy()
                continue
                
            #处理新行    
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
        
        #处理电子标签详细属性值
        if getDetail:
            for lineDict in dictList:
                eLabelValue = lineDict.get("Electronic Label")
                eLableLines = eLabelValue.splitlines()
                #没有电子标签场景
                if len(eLableLines) <=1:
                    lineDict["BoardType"] = ""
                    lineDict["BarCode"] = ""
                    lineDict["Item"] = ""
                    lineDict["Description"] = ""
                    lineDict["Manufactured"] = ""
                    lineDict["VendorName"] = ""
                    lineDict["IssueNumber"] = ""
                #存在电子标签属性
                else:
                    propertyList = []
                    for elabelProperty in eLableLines[1:]:
                        if "=" in elabelProperty:
                            propertyList.append(elabelProperty)
                        else:
                            if len(propertyList) == 0:
                                continue
                            propertyList[-1] += elabelProperty
                    for elabelProperty in propertyList:
                        props = elabelProperty.split("=")
                        propKey = props[0].strip().decode("utf8")
                        propValue = "=".join(props[1:]).strip().decode("utf8")
                        lineDict[propKey] = propValue
        return dictList
    except:
        return []


def getVerticalCliRetFilterElabel(cliRet, isParseElcLabel=True):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合（只处理表格中过滤显示电子标签场景）
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    elcLabel = ""
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            if not isParseElcLabel and lineDict.has_key("Electronic Label"):
                lineDict["Electronic Label"] = lineDict.get("Electronic Label") + elcLabel
                elcLabel = ""

            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            if isParseElcLabel and "=" in line:
                fields = line.split("=")
            elif "=" in line and re.search("^\s{16,}", line):
                elcLabel += line
                continue
            else:
                continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        if not isParseElcLabel and lineDict.has_key("Electronic Label"):
            lineDict["Electronic Label"] = lineDict.get("Electronic Label") + elcLabel
        dictList.append(lineDict.copy())

    return dictList


def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记 
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    
    splitedCliRet = cliRetlist[1]
    match = re.search("^.*:(/>)*\r*\n*$", splitedCliRet, re.MULTILINE)
    if match:
        splitedCliRet = splitedCliRet[0:match.start()]
    
    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet

def isEngine0(value):
    '''
    @summary: 根据id判断是否为引擎0
    @param value: 回显中的ID
    @return: 
        True: 是引擎0
        False: 不是引擎0
    '''
    if "ENG0" in value or "CTE0" in value:
        return True
    else:
        return False

def isEngineId(value):
    '''
    @summary: 根据id判断是否为引擎ID
    @param value: 回显中的ID
    @return: 
        True: 是引擎ID
        False: 不是引擎ID
    '''
    if "ENG" in value or "CTE" in value:
        return True
    else:
        return False

def isDiskId(value):
    '''
    @summary: 根据id判断是否为硬盘ID
    @param value: 回显中的ID
    @return: 
        True: 是硬盘ID
        False: 不是硬盘ID
    '''
    if "DAE" in value:
        return True
    else:
        return False

def isDataSwitchId(value):
    '''
    @summary: 根据id判断是否为交换机ID
    @param value: 回显中的ID
    @return: 
        True: 是交换机ID
        False: 不是交换机ID
    '''
    if "DSW" in value:
        return True
    else:
        return False

def isHighDensityDisk(enclosureId, cli, lang):
    '''
    @summary: 判断硬盘框是否为高密框
    @param enclosureId: 框ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，是高密框
            flag为False时，不是高密框
        errMsg: 错误时的消息
    '''
    if not isDiskId(enclosureId):
        return (True, False, "")
    
    checkRet = getHighDensityDiskEnclosureIdList(cli, lang)
    if checkRet[0] != True: 
        return checkRet
    
    highDensityDiskEnclosureIdList = checkRet[1]
    if enclosureId in highDensityDiskEnclosureIdList:
        return (True, True, "")
    else:
        return (True, False, "")

def getEnclosureHeight(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框高度(U)
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的高度(U)
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Height(U)" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Height(U)":
            enclosureHeight = fieldValue
            return (True, enclosureHeight, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getEnclosureType(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框类型
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Type" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Type":
            enclosureType = fieldValue
            return (True, enclosureType, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getControllerBarCode(cli, lang):
    '''
    @summary: 获取所有控制器的信息
    @param cli : cli对象
    @param lang: 语言
    @return: (falg, BarCodeDictList, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        cliRet:cli回显
        BarCodeDictList: 
                                以控制器为key，BarCode为值的字典列表
        errMsg:
                                 错误消息
    '''
    barCodeDictList = []
    cmd = "show controller general"
    barCode = "BarCode="
    controller = "Controller"
    needBarCode = False
    
    #执行命令获取所有控制器信息
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    cliRet = checkRet[1]
    if checkRet[0] != True: 
        return (checkRet[0],cliRet, [], checkRet[2]) 
    
    #解析控制器信息，获取BarCode
    cliRetLinesList = cliRet.splitlines()
    barCodeDict = {}
    for line in cliRetLinesList:

        #获取控制器名称，并置needBarCode为True，说明需要一个BarCode来匹配
        if line.strip().startswith(controller):
            fields = line.split(":")
            controllerID = fields[1].strip()
            needBarCode = True 
        
        #在控制器需要对应的BarCode时，为该控制器匹配BarCode
        if needBarCode == True and line.strip().startswith(barCode):  
            barCodeSplit = line.strip().split("=")
            if len(barCodeSplit) < 2:
                barCodeValue = ""
            else:
                barCodeValue = barCodeSplit[1]
            
            barCodeDict[controllerID] = barCodeValue
            needBarCode = False
        
        #开始解析下一个控制器的信息，需要清理上一个控制器的标识
        if line.strip().startswith("----"):
            needBarCode = False
    
    return (True, cliRet, barCodeDict, "")


def getControllerIdList(cli, lang):
    '''
    @summary: 获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，设备所有控制器ID列表
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show controller general|filterColumn include columnList=Controller"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = cliRet.splitlines()
    controlList = []
    
    for line in cliRetLinesList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Controller":
            ctrlId = fieldValue
            controlList.append(ctrlId)
            
    if len(controlList) == 0:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return (False, cliRet, errMsg)
    
    return (True, controlList, "")


def getEng0Id(cli, lang):
    '''
    @summary: 获取引擎0ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，引擎0ID
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = getHorizontalCliRet(cliRet)
    
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)
    
    for line in cliRetLinesList:
        enclosureId = line.get("ID")
        if isEngine0(enclosureId):
            return (True, enclosureId, "")
    
    return (True, "", "")

def getHighDensityDiskEnclosureIdList(cli, lang):
    '''
    @summary: 获取高密硬盘框ID集合
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，高密硬盘框ID集合
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    highDensityDiskEnclosureIdList = []
    cliRetLinesList = getHorizontalCliRet(cliRet)
    
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)
    
    for line in cliRetLinesList:
        enclosureType = line.get("Type")
        enclosureId = line.get("ID")
        if "4U 75" in enclosureType:
            highDensityDiskEnclosureIdList.append(enclosureId)
    
    return (True, highDensityDiskEnclosureIdList, "")


def excuteCmdCommon(cli, cmd, lang):
    '''
    @summary: 执行cli命令，只判断是否拿到回显信息
    '''
    cliRet = ""
    errMsg = ""

    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)

    try:
        cliRet = cli.execCmd(cmd)
    except Exception:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    return (True, cliRet, errMsg)


def isInDebugMode(cliRet):
    '''
    @summary: 判断当前是否在debug模式下
    '''
    if re.search(DEBUG_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def isInStorageMode(cliRet):
    '''
    @summary: 判断当前是否在Storage模式下，debug 版本.
    @param cliRet: cli回显
    @return:
        True: 当前在Storage模式下
        False: 当前不在Storage模式下
    '''
    if re.search(STORAGE_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def enterCliModeFromSomeModel(cli, lang):
    '''
    @summary: 从其他模式进入cli模式
    '''
    excuRet = excuteCmdCommon(cli, "show system general", lang)
    if excuRet[0] is False:
        return excuRet

    if isInDebugMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if isInMinisystemMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if isInDeveloperMode(excuRet[1]) or isInStorageMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if excuRet[1].endswith(":/>"):
        return (True, excuRet[1], "")

    return (False, excuRet[1], excuRet[2])
