# -*- coding: UTF-8 -*-
import re
import time
import traceback
import resource
import cliUtil
import config
import os
import logger
from defusedxml import ElementTree
from toolutilities import sshutils
from com.huawei.ism.tool.obase.entity import DevNode

STATUS_NORMAL = "Normal"
STATUS_FAULT = "Fault"
STATUS_ONLINE = "Online"
STATUS_RUNNING = "Running"
STATUS_BALANCING = "Balancing"
STATUS_CHARGING = "Charging"
STATUS_DISCHARGING = "Discharging"
STATUS_ENABLED = "Enabled"
STATUS_YES = "Yes"
STATUS_NO = "No"
STATUS_LINK_UP = "Link Up"
STATUS_LINK_DOWN = "Link Down"
STATUS_ON = "On"
STATUS_OFF = "Off"
STATUS_START = "Start"
STATUS_SYNCHRONIZING = "Synchronizing"
STATUS_PAUSE = "Paused"
STATUS_FORCE_STARTED = "Force Started"

SLEEP_TIMES = 30
PORTS_BIT_ERROR_SPEC = 60
PORTS_BIT_ERROR_INTERVAL = 10
SERIAL_NUMBER_LENGTH = 20
THRESHOLD_BBU_REMAINING_LIFETIME = 90
THRESHOLD_CONTRL_CPU_USAGE = 80

FILE_SUFFIX = "."

def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")
    
def getMsg(lang, msg, args = ""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    '''
    return cliUtil.getMsg(lang, msg, args, resource.MESSAGES_DICT)


def getStandardVersion(VersionDicList, lang):
    '''
    @summary: 通过software信息字典列表获取版本信息
    '''
    #补丁信息需要校验是否一致，版本信息在此之前已经校验过
    if not isControllerVersionSame(VersionDicList):
        errMsg = getMsg(lang, "controller.hot.patch.version.not.consistence")
        return (False, "", errMsg)  
    
    #获取版本或热补丁版本
    VersionDict = VersionDicList[0]
    Version = VersionDict.get("Current Version")
    
    #系统版本需要转换为标准版本形式
    if  re.search("SPC", Version, re.IGNORECASE):
        Version = switchValidVersion(Version)
    
    errMsg = getMsg(lang, "current.version", Version)
    return (True, Version, errMsg)

def getSymmetricalPort(port):
    '''
    @summary: 获取端口的对称端口
    @param port: 端口
    '''
    for item in [(".A", ".B"), (".B", ".A"), (".L", ".R"), (".R", ".L")]:
        item0 = item[0]
        item1 = item[1]
        if item0 in port:
            symmetricalPort = port.replace(item0, item1)
            return symmetricalPort
    return ""

def getDiskGeneralList(cli, lang):
    '''
    @summary: 获取所有硬盘信息集合
    @param cli: cli对象
    @param lang: 语言lang
    '''
    cmd = "show disk general"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    cliRet = checkRet[1]
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, [], "")
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.disk.info")
        return (False, cliRet, errMsg)
    
    return (True, cliRetLinesList, "")

def hasSSDDisks(enclosureId, diskGeneralList):
    '''
    @summary: 判断框中是否下挂SSD盘
    @param enclosureId: 框id
    @param diskGeneralList: 所有硬盘信息集合
    @return: 
        True: 框中下挂SSD盘
        False: 框中没有下挂SSD盘
    '''
    for line in diskGeneralList:
        diskID = line.get("ID")
        diskType = line.get("Type")
        if enclosureId in diskID and diskType == "SSD":
            return True
    return False

def getEngineSet(cli, lang):
    '''
    @summary: 获取引擎集合
    @param cli: cli对象
    @param lang: 语言lang
    @return:
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，引擎集合
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show storage_engine |filterColumn include columnList=ID"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.engine.info")
        return (False, cliRet, errMsg)
    
    engineSet = set()
    for retDict in cliRetLinesList:
        ctrlId = retDict["ID"]
        engineSet.add(ctrlId)
    
    return (True, engineSet, cliRet)

def getPCIeSwitchNum(cli, lang):
    '''
    @summary: 获取PCIe交换机个数
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，PCIe交换机个数
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        cmd = "show enclosure"
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        dataSwitchList = []
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.enclosure.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            logicType = line.get("Logic Type")
            frameId = line.get("ID")
            if not "Data Switch" in logicType:
                continue
            else:
                dataSwitchList.append(frameId)
                
        return (True, len(dataSwitchList), "")
    except:
        errMsg = getMsg(lang, "cannot.get.pcie.switch.info")
        return (False, cliRet, errMsg)
            
def checkPCIePortExists(cli, lang):
    '''
    @summary: 判断是否需要检查PCIe端口
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，需要检查PCIe端口
            flag为False时，不需要检查PCIe端口
        errMsg: 错误消息
    '''
    getControllerIdListRet = cliUtil.getControllerIdList(cli, lang)
    if getControllerIdListRet[0] != True:
        return getControllerIdListRet
    
    engSet = set([ctrlId[0] for ctrlId in getControllerIdListRet[1]])
    if len(engSet) <= 1:
        return (True, False, "")
    return (True, True, "")

        
def getBBUConfig(cli, lang):
    '''
    @summary: 获取单个引擎的BBU规格配置
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，BBU规格
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = cliUtil.getProductModel(cli, lang)
    if getProductModelRet[0] != True:
        return getProductModelRet
    
    pdtModel = getProductModelRet[1]
    
    if config.BBU_OF_ENGINE_CONFIG_DICT.has_key(pdtModel):
        return (True, config.BBU_OF_ENGINE_CONFIG_DICT.get(pdtModel), "")
    else:
        errMsg = getMsg(lang, "cannot.get.bbu.config")
        return (False, "", errMsg)

def getEngCtrlNum(enclosureID, controllerIdList):
    '''
    @summary: 获取引擎下控制器数量
    @param cli: cli对象
    @param controllerIdList: 控制器ID列表
    @return: 特定引擎下控制器数量
    '''
    
    eng = enclosureID[-1]
    engCtrlNum = len([ctrlId for ctrlId in controllerIdList if eng in ctrlId])
    if engCtrlNum % 2 != 0:
        engCtrlNum += 1
    return engCtrlNum
    
def getEnclosureFanConfig(cli, enclosureInfo, controllerIdList, highDensityDiskEnclosureIdList, lang):
    '''
    @summary: 获取风扇的规格配置
    @param cli: cli对象
    @param enclosureInfo: 机框信息
    @param highDensityDiskEnclosureIdList: 高密框集合
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，风扇的规格
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    storageTypeRet = cliUtil.getStorageType(cli, lang)
    if storageTypeRet[0] != True:
        return storageTypeRet
    product_series = storageTypeRet[1]
    
    enclosureID = enclosureInfo.get("ID")
    enclosureHeight = enclosureInfo.get("Height(U)")
    
    for enclosureType in ["DSW", "DAE", "ENG", "CTE"]:
        if enclosureType in enclosureID:
            if enclosureHeight == "6" and enclosureType in ["ENG", "CTE"]:
                #6U高端2C和4C场景，都需要2个辅助散热模块，风扇个数一样
                if product_series == cliUtil.SERIES_18000:
                    return (True, 12, "")
                #6U中低端2C和4C场景，风扇个数不一样
                else:
                    engCtrlNum = getEngCtrlNum(enclosureID, controllerIdList)
                    return (True, engCtrlNum * 3, "")
            else:
                daeType = "--"
                if enclosureID in highDensityDiskEnclosureIdList:
                    daeType = "high-density"
                key = (enclosureType, enclosureHeight, daeType)
                if config.FAN_CONFIG_DICT.has_key(key):
                    return (True, config.FAN_CONFIG_DICT.get(key), "")
                    
    return (False, "", getMsg(lang, "cannot.get.fan.config"))

def checkSystemStatus(cliRet, lang):
    '''
    @summary: 根据回显判断设备是否开工正常，Health Status及Running Status都要为Normal，版本号要正常
    @param cliRet: cli回显
    @param lang: 语言lang
    @return: (flag, errMsg)
        flag:
            True: 系统状态正常
            False： 系统状态异常
        errMsg: 错误消息
    '''
    healthStatus = ""
    runningStatus = ""
    productModel = ""
    productVersion = ""
    
    lineList = cliRet.splitlines()
    for line in lineList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Health Status":
            healthStatus = fieldValue
        elif fieldName == "Running Status":
            runningStatus = fieldValue
        elif fieldName == "Product Model":
            productModel = fieldValue
        elif fieldName == "Product Version":
            productVersion = fieldValue
    
    if len(healthStatus) == 0 or len(runningStatus) == 0:
        return (False, getMsg(lang, "cannot.get.system.info"))
    
    if len(productModel) == 0 or len(productVersion) == 0 or productModel == "--" or productVersion == "--":
        return (False, getMsg(lang, "cannot.get.product.version.info"))
    
    if healthStatus != STATUS_NORMAL:
        return (False, getMsg(lang, "system.health.status.abnormal", healthStatus))
    
    if runningStatus != STATUS_NORMAL:
        return (False, getMsg(lang, "system.running.status.abnormal", runningStatus))
    
    return (True, "")

def getRealCapacity(capacity):
    '''
    @summary: 获取数值
    '''
    matchCapacity = re.search("\d+\.?\d*", capacity)
    if matchCapacity:
        return matchCapacity.group()
    else:
        return ""

def getBaseName(file_path):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name
    
def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)

def checkPortsBitError(cli, cmd, ports, lang):
    '''
    @summary: 检查所有端口在特定时间段内是否有持续增加的误码
    @param cli: cli对象
    @param cmd: cli命令
    @param ports: 端口列表
    @param lang: 语言对象
    @return:
        flag:
            True: 检查通过
            False: 检查不通过
        cliRetAll: cli回显
        errMsg: 错误时的消息
    '''
    flag = True
    errMsg = ""
    cliRetAll = ""
        
    firstResultDictRet = getPortsBitErrorResultDict(cli, cmd, ports, lang)
    firstCliRet = firstResultDictRet[1]
    if firstResultDictRet[0] != True:
        return (firstResultDictRet[0], firstCliRet, firstResultDictRet[2])
    firstResultDict = firstResultDictRet[3]
    
    retriedTimes = int(PORTS_BIT_ERROR_SPEC / PORTS_BIT_ERROR_INTERVAL)
    for i in range(0, retriedTimes):
        time.sleep(PORTS_BIT_ERROR_INTERVAL)
        nextResultDictRet = getPortsBitErrorResultDict(cli, cmd, ports, lang)
        nextCliRet = nextResultDictRet[1]
        cliRetAll = firstCliRet + nextCliRet
        if nextResultDictRet[0] != True:
            return (nextResultDictRet[0], cliRetAll, nextResultDictRet[2])
        nextResultDict = nextResultDictRet[3]
        
        for portName in ports:
            firstBitErrorList = firstResultDict.get(portName)
            nextbitErrorList = nextResultDict.get(portName)
            
            if len(firstBitErrorList) != len(nextbitErrorList):
                continue
            
            cmpResult = cmpDictList(firstBitErrorList, nextbitErrorList)
            if cmpResult == -1:
                flag = False
                errMsg += getMsg(lang, "port.exists.error.bits", portName)
        
        if not flag:
            return (False, cliRetAll, errMsg)
            
    return (True, cliRetAll, "")
        
def getPortsBitErrorResultDict(cli, cmd, ports, lang):
    '''
    @summary: 获取指定端口列表对应的误码列表
    @param cli: cli对象
    @param cmd: cli命令
    @param ports: 端口列表
    @param lang: 语言对象
    @return: 
        flag: 
            True: 获取成功
            False: 获取失败
        cliRet: cli回显
        errMsg: 错误时的消息
        resultDict: 以字典的形式返回对应端口的误码列表。键对应端口，值对应该端口的误码列表
    '''
    resultDict = {}
    
    checkRet = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, lang)
    cliRet = checkRet[1]
    if checkRet[0] != True:
        return (checkRet[0], cliRet, checkRet[2], None)
    
    for portName in ports:
        getBitErrorResultListRet = getPortBitErrorResultList(cliRet, portName, lang)
        if getBitErrorResultListRet[0] != True:
            return (getBitErrorResultListRet[0], cliRet, getBitErrorResultListRet[2], None)
        
        resultDict.setdefault(portName, getBitErrorResultListRet[1])
    return (True, cliRet, "", resultDict)

def getPortBitErrorResultList(cliRet, portName, lang):
    '''
    @summary: 获取每个端口的误码列表
    @param cliRet: cli回显
    @param portName: 端口名
    @param lang: 语言对象
    @return: 
        flag: 
            True: 获取成功
            False: 获取失败
        resultList: 误码列表
        errMsg: 错误时的消息
    '''
    portInfoRet = cliUtil.getSplitedCliRet(cliRet, getSplitedPortWords(portName))
    if len(portInfoRet) == 0:
        return (True, [], "")
        
    cliRetLinesList = cliUtil.getHorizontalCliRet(portInfoRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.port.info", portName)
        return (False, None, errMsg)
    
    return (True, cliRetLinesList, "")
    
def cmpDictList(firstList, secondList):
    '''
    @summary: 判断误码数是否有增加
    @param firstList: 第一个列表（列表中的内容为字典）
    @param secondList: 第二个列表（列表中的内容为字典，且字典的键与第一个列表中字典的键一致，同时第二个列表的长度与第一个列表的长度一致）
    @return: 
        -1: 第一个列表中的元素小于第二个列表中的元素
         0: 第一个列表与第二个列表相等
         1: 第一个列表中的元素大于第二个列表中的元素
    '''
    for i in range(0, len(firstList)):
        firstDict = firstList[i]
        secondDict = secondList[i]
        
        for key in firstDict.keys():
            firstValue = str(firstDict.get(key))
            secondValue = str(secondDict.get(key))
            
            if firstValue.isdigit() and secondValue.isdigit():
                ret = cmp(long(firstValue), long(secondValue))
                if ret != 0:
                    return ret
    return 0

def getSplitedPortWords(portName):
    '''
    @summary: 根据端口名获取对应的分割字
    @param portName: 端口名
    @return: 分割字
    '''
    if portName == "PCIe":
        portName = "PCIE" 
    return portName + " port:"

def downloadErrCodeFile():
    '''
    @summary: 使用SFTP功能将错误码文件下载到本地
    '''
    return

def getErrCodeInfoDict(lang, fileName = "event_xve"):
    '''
    @summary: 获取对应语言的错误码信息，将错误码信息转换为以错误码ID为key，以含错误码详细信息和修复建议的字典为value的字典
    @param lang: 语言对象
    @param fileName: 错误码文件默认为error_{zh|en}.xml
    '''
    try:
        #1.下载错误码文件到本地
        downloadErrCodeFile()
        #2.获取xml中根节点对象
        root = ElementTree.parse(os.path.join(os.path.abspath(
            os.path.join(os.path.dirname(__file__), "..")), "res", "%s_%s.xml" % (fileName, lang)))
        #3.找到error子节点
        errorNodes = root.findall("eventDefinition/param")

        errCodeInfoDict = {}
        for errorNode in errorNodes:
            #4.获取错误码ID
            eventId = errorNode.attrib.get("eventID", "")
            #5.获取错误码修复建议
            suggesions = errorNode.attrib.get("suggestion", "")
            #6.获取误码对象名称
            objName = errorNode.attrib.get("objName", "")
            errCodeInfoDict[eventId] = {"objName":objName, "suggesions":suggesions}
        
        return errCodeInfoDict
    except Exception, exp:
        #解析失败时，使用默认的修复建议策略
        return {}

def getErrSuggestions(lang, errCodeInfoDict, errId):
    '''
    @summary: 获取单条错误码对应的修复建议
    @param lang: 语言对象
    @param errCodeInfoDict: 错误码信息字典
    @param errId: 错误码ID
    '''
    
    suggestions = ""
    
    errCodeInfo = errCodeInfoDict.get(errId, {})
    suggestions += errCodeInfo.get("suggesions", "")

    return suggestions

def getSuggestions(lang, errCodeInfoDict, alarmIds, des = None):
    '''
    @summary: 获取多条错误码对应的修复建议
    @param lang: 语言对象
    @param errCodeInfoDict: 错误码信息字典
    @param errIds: 错误码ID列表
    '''
    sugs = "\n".join([getErrSuggestions(lang, errCodeInfoDict, alarmId) for alarmId in alarmIds])
    
    if des is None:
        return sugs
    else:
        return des + "\n" + sugs

def getObjAlarmIds(lang, cli, obj):
    '''
    @summary: 根据指定对象查询该对象下的告警ID列表
    @param lang: 语言对象
    @param cli: cli对象
    @param obj: 指定对象
    '''
    alarmIds = set()
    cmd = "show alarm object_type=%s|filterColumn include columnList=ID" % (obj)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    
    cliRet = checkRet[1]
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    for line in cliRetLinesList:
        alarmId = line.get("ID")
        alarmId = str(int(alarmId, 16))
        
        alarmIds.add(alarmId)

    return alarmIds

def getSuggestionResults(lang, errCodeInfoDict, cli, obj, des = None):
    '''
    @summary: 根据指定对象，返回该指定对象下的所有错误码修复建议
    @param lang: 语言对象
    @param errCodeInfoDict: 错误码信息字典
    @param cli: cli对象
    @param obj: 指定对象
    '''
    try:
        alarmIds = getObjAlarmIds(lang, cli, obj)
        result = getSuggestions(lang, errCodeInfoDict, alarmIds, des)
        
        return result
    except:
        return ""

def isSigleModel(cli, lang):
    '''
          控制器模式检查：
                  执行show system config_model命令，查看返回结果：Configuration Model为Multi-Controller时多多控，为Single-Controller时为单控
    '''
    
    cliRet = ""
    errMsg = ""
    
    cmd = "show system config_model"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    cliRet = checkRet[1]
    if "single" in cliRet.lower():
        return True
    else:
        return False


def parse_upgradePackage(cli, lang):
    '''
    @summary: 执行show upgrade packge命令回显，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型
    '''    
    cmd = "show upgrade package"
    softwareVersionList = []
    hotPatchVersionList = []

    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return ((flag, cliRet, errMsg), softwareVersionList, hotPatchVersionList)
    
    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = cliUtil.getHorizontalCliRet(cliRet[(softwareVersionIndex):hotPatchVersionIndex])
    hotPatchVersionList = cliUtil.getHorizontalCliRet(cliRet[(hotPatchVersionIndex):])
    
    if len(softwareVersionList) == 0 or len(hotPatchVersionList) == 0:
        errMsg += getMsg(lang, "cannot.get.upgrade.package.info")
        return ((False, cliRet, errMsg) , softwareVersionList , hotPatchVersionList)
        
    for softwareVersion in softwareVersionList:
        if softwareVersion["Current Version"] == "--":   
            flag = False
            errMsg += getMsg(lang, "cannot.get.contrller.version.info", softwareVersion.get("Name"))
    
    if not isControllerVersionSame(softwareVersionList):
        errMsg += getMsg(lang, "controller.version.not.consistence")
        return ((False, cliRet, errMsg), softwareVersionList, hotPatchVersionList)
    
    return ((flag, cliRet, errMsg), softwareVersionList, hotPatchVersionList)

    
def hotPatchVersionCompare(contorllerID, hotPatchVersionList, patchVersion, lang):
    '''
    @summary: 判断控制器contorllerID的热补丁版本是否低于installPatch
    @param contorllerID: 控制器名称
    @param hotPatchVersionList: 补丁版本列表
    @param installPatch: 问题解决的版本
    @param lang: 语言lang
    @return: (flag, errMsg)
        flag:
            True:热补丁版本高于等于问题解决的版本
            False: 热补丁版本低于问题解决的版本
        errMsg: 错误消息
    ''' 
    errMsg = ""  
    flag = True
      
    for hotPatchVersion in hotPatchVersionList:   
        if contorllerID != hotPatchVersion.get("Name"):
            continue
        installedPatch = hotPatchVersion.get("Current Version")
        installedPatchSPH = getHotpacthSPH(installedPatch)
        patchVersionSPH = getHotpacthSPH(patchVersion)
        
        if installedPatchSPH == "--":
            errMsg += getMsg(lang, "contrller.not.install.pacth", contorllerID)
            flag = False
        elif installedPatchSPH >= patchVersionSPH:
            continue
        else:
            errMsg += getMsg(lang, "contrller.pacth.too.low", (contorllerID , installedPatch , patchVersion))
            flag = False
        
    return (flag, errMsg)


             
def CurrentVersionCmp(softwareVersionList, versionType, lang):
    '''
    @summary: 判断当前阵列版本是否低于指定版本
    @param softwareVersionList: 当前版本列表
    @param versionType: 指定版本
    @return: (flag, errMsg)
        flag:
            True:当前版本不低于指定版本
            False: 当前版本低于指定版本
        errMsg: 错误消息
    '''
    flag = True
    errMsg = ""
    for softwareVersion in softwareVersionList:
        contorllerID = softwareVersion.get("Name")
        originalContorllerVersion = softwareVersion.get("Current Version")
        standardContorllerVersion = switchValidVersion(originalContorllerVersion)

        if standardContorllerVersion < versionType:
            flag = False
            errMsg += getMsg(lang, "contrller.software.version.too.low", (contorllerID, originalContorllerVersion, versionType))
                
    return (flag, errMsg)

def checkRiskVersion(currentVersionDickList, riskVersionList, lang):
    '''
    @summary: 判断当前阵列版本是否在指定的风险版本中
    @param softwareVersionList: 当前版本列表
    @param riskVersionList: 指定的风险版本
    @return: flag:
            Ture: 当前版本不在指定风险版本中
            False:当前版本在指定风险版本中
    '''
    for currentVersionDict in currentVersionDickList:
        currentVersion = currentVersionDict.get("Current Version")
        standardVersion = switchValidVersion(currentVersion)

        #识别当前版本是否是在风险版本列表中
        if standardVersion in riskVersionList:
            return True
            
    return False
    
def switchValidVersion(version):
    '''
    @summary: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
    @param version: 输入版本
    @return: 有效的版本号
    '''
    temp = ""
    
    startIndex = version.index("V")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  #格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  #格式：V100R005C02
    
    #提取有效的版本号信息
    temp = version[startIndex: endIndex]
    
    return temp

def getHotpacthSPH(hotpatch):
    '''
    @summary: 获得SPH版本加上热补丁号，转换为小写
    '''
    start = hotpatch.lower().find("sph")
    if start < 0:
        return "--"
    standardHotpatch = hotpatch[start:start + 6]
    return standardHotpatch.lower()
    
def isControllerVersionSame(softwareVersionList): 
    '''
    @summary:校验控制器版本是否一致
    @param softwareVersionList: 控制器版本列表
    @return: 
        True:控制器版本一致
        False:控制器软件版本不一致
    '''  
    VersionSet = set(softwareVersion.get("Current Version") for softwareVersion in softwareVersionList)
    if len(VersionSet) != 1:
        return False
    return True

def getThinLUNIDList(cli, lang): 
    '''
    @summary:获取Thin LUN的ID列表
    @param 
        cli:CLI
        lang:语言环境
    @return: 
        thinLUNIDList：Thin LUN的ID列表
        cliRet: cli回文
    '''  
    cmd = "show lun general"
    thinKey = "Thin"
    thinLUNIDList = []
    cliRet = ""
    
    #获取所有LUN的ID
    cmdRet = cli.execCmd(cmd)
    cliRet += cmdRet
    lunDictList = cliUtil.getHorizontalCliRet(cmdRet)
    for lunInfo in lunDictList:
        if lunInfo.get("Type") == thinKey:
            thinLUNIDList.append(lunInfo.get("ID"))
    
    return(thinLUNIDList, cliRet)

def changeUnit2GB(strValue):
    '''
    @summary: 将传进来的字符串值转换为以GB为单位的浮点数
    @param strValue: 字符串形式的值
    @return:
        flag:转换是否成功
        floatValue：转换后的值
    '''    
    flag = False
    floatValue = 0
    
    try:
        if not strValue:
            return(flag, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip())
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) / 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) / (1024 * 1024)
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip()) / (1024 * 1024 * 1024)
        else:
            return(False, floatValue)
        
        flag = True
        return(flag, floatValue)
    
    except Exception, exception:
        return (False, floatValue)

def copyDevNode(devNode):
    '''
            函数名称: copyDevNode
            功能说明: 复制DevNode对象的部分参数，创建新的DevNode对象，除IP外请不要修改其它成员变量，否则可能影响原DevNode对象的值
            输入参数: devNode
            输出参数: 无
            返 回 值: devNode    
    '''
    devTmp = DevNode()
    devTmp.setSocks5Proxy(devNode.getSocks5Proxy());
    devTmp.setIp(devNode.getIp())   
    devTmp.setPort(devNode.getPort())
    devTmp.setTlvPort(devNode.getTlvPort())
    devTmp.setLoginUser(devNode.getLoginUser())
    devTmp.setSshForwardList(devNode.getSshForwardList())
    devTmp.setDeveloperPwd(devNode.getDeveloperPwd())
    return devTmp

def createSshConnection(py_java_env, ipAddr, logger):
    """
    @summary: 获取到其他节点的链接
    Input   : cli--Telnet access proxy 
    Return  : cli链接
    """
    lang = py_java_env.get("lang")
    errMsg = ""
    devNode = py_java_env.get("devInfo")
    myDevNode = copyDevNode(devNode)        
    myDevNode.setIp(ipAddr)
    try:
        cliCon = sshutils.createSSHConnectionByDevNode(myDevNode)
        cliCon.connect()
    except:
        logger.logExecCmd("Exception: Create SSH [" + ipAddr + "] connection catch exception")  
        return None
    else:
        logger.logExecCmd("Create SSH [" + ipAddr + "] connection successfully!")
        return cliCon


def getFaultDiskInfo(cli, lang):
    '''
    @summary: 判断是否存在故障硬盘
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg，list)
        flag:
            True： 判断时正常
            False： 判断时异常
        ret: 命令回显
        errMsg: 错误时的消息
        list: 存放故障硬盘的ID号的列表
    '''

    faultDiskCmd = "show disk general |filterRow " \
                    "column=Health\sStatus predict=not predict2=equal_to value=Normal " \
                    "logicOp=or column=Running\sStatus predict=not predict2=equal_to value=Online "

    cliRet = ""
    errMsg = ""
    faultDiskList = []
    diskStatusInfoFlag, diskStatusInfoCliRet, diskStatusInfoErrMsg = cliUtil.excuteCmdInCliMode(cli, faultDiskCmd, True, lang)
    cliRet = diskStatusInfoCliRet

    if diskStatusInfoFlag != True:
        errMsg = diskStatusInfoErrMsg
        return (diskStatusInfoFlag, cliRet, errMsg, [])

    #判断命令回显是否为Command executed successfully
    isCmdExeSucc = cliUtil.queryResultWithNoRecord(diskStatusInfoCliRet)
    if isCmdExeSucc:
        return (True, cliRet, "", [])

    #按逐行字典的方式获取水平表格形式的cli回显集合
    diskInfoDictList = cliUtil.getHorizontalCliRet(diskStatusInfoCliRet)

    for diskInfoDict in diskInfoDictList:
        diskId = diskInfoDict.get('ID')
        healthStatus = diskInfoDict.get('Health Status')
        runningStatus = diskInfoDict.get('Running Status')
        #存在信息就表示有故障盘
        errMsg += getMsg(lang, "disk.status.abnormal", (diskId, healthStatus, runningStatus))
        faultDiskList.append(diskId)

    #存在信息就表示有故障盘
    if faultDiskList:
        return (True, cliRet, errMsg, faultDiskList)

    return (True, cliRet, "", [])

def get4kDiskInfo(cli, LANG):
    '''
    @summary: 判断是否存在故障硬盘
    @param cli: cli对象
    @param lang: 语言lang
    '''

    cliRet = ""
    errMsg = ""

    #判断硬盘是否存在于1TB的SAS盘
    diskSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=SAS " \
                "logicOp=and column=Capacity predict=greater_than value=1.000TB " \

    #判断硬盘是否存在大于5TB的NearLine SAS盘
    diskNearLineSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=NearLine\sSAS " \
                "logicOp=and column=Capacity predict=greater_than value=5.000TB " \

    all4kDiskList = []
    isCmdExeSucc = True
    for diskCmd in (diskSASCmd, diskNearLineSASCmd):
        isCmdExeSucc, diskCliRet, fkDiskErrMsg, fkDiskList = exeQry4kDiskCmd(diskCmd, cli, LANG)
        #命令执行失败或存在故障盘或存在4K
        cliRet += diskCliRet
        errMsg += fkDiskErrMsg
        all4kDiskList += fkDiskList
        if isCmdExeSucc != True:
            return (isCmdExeSucc, cliRet, errMsg, [])

    #存在4K盘返回Flase
    if all4kDiskList:
        errMsg += getMsg(LANG, "exist.4K.disk.capacity", ','.join(all4kDiskList))
        return (True, cliRet, errMsg, all4kDiskList)

    return (True, cliRet, errMsg, [])


def exeQry4kDiskCmd(cmd, cli, LANG):

    cliRet = ""
    errMsg = ""
    isCmdExeSucc, infoCliRet, infoErrMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cliRet += infoCliRet

    if isCmdExeSucc != True:
        errMsg += infoErrMsg
        return (isCmdExeSucc, cliRet, errMsg, [])

    #判断命令回显是否为Command executed successfully
    isCmdExeSucc = cliUtil.queryResultWithNoRecord(infoCliRet)
    if isCmdExeSucc:
        return (True, cliRet, "", [])

    #按逐行字典的方式获取水平表格形式的cli回显集合
    diskInfoDictList = cliUtil.getHorizontalCliRet(infoCliRet)

    fkDiskList = []
    for diskInfoDict in diskInfoDictList:
        diskId = diskInfoDict.get('ID')
        #把大于1T的SAS盘或大于5T的NearLine SAS盘添加到bugDiskList
        fkDiskList.append(diskId)

    if fkDiskList:
        return (True, cliRet, errMsg, fkDiskList)

    return (True, cliRet, errMsg, [])


def getDiskSectorSize(cli, lang):
    '''
    @summary: 执行show disk general |filterColumn include columnList=ID,Sector\sSize命令，获取扇区大小
    '''
    sectorSizeDictList = []
    cmd = "show disk general |filterColumn include columnList=ID,Sector\sSize"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    
    #兼容产品命令Bug
    if not cliUtil.hasCliExecPrivilege(checkRet[1]):
        cmd = "show disk general |filterColumn include colunmList=ID,Sector\sSize"
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    cliRet = checkRet[1]   
    if checkRet[0] != True: 
        return (checkRet[0], cliRet, checkRet[2], sectorSizeDictList)
    
    sectorSizeDictList = cliUtil.getHorizontalCliRet(cliRet)
    return (True, cliRet, "", sectorSizeDictList)


def isRiskVersion(lang, currentVersion, bugVersion, hotPatchVersion, repairHotPatchDict):
    '''
    @summary: 检查当前版本是否存在风险
    @return: True:存在风险
            False:不存在风险
    '''
    #需要打补丁的版本
    
    if currentVersion in bugVersion and currentVersion in repairHotPatchDict:
        needhotPatchVersion = repairHotPatchDict.get(currentVersion, "")
        if hotPatchVersion == "--" or hotPatchVersion < needhotPatchVersion:
            errMsg = getMsg(lang, "current.version", currentVersion)
            errMsg += getMsg(lang, "hot.patch.version.too.low", needhotPatchVersion)
            return (True, errMsg)
        else:
            return (False, "")
    #不需要打补丁的风险版本
    elif currentVersion in bugVersion:
        errMsg = getMsg(lang, "current.version", currentVersion)
        return (True, errMsg)
    #不是风险版本
    else:
        return (False, "")  

def changUnit2B(strValue):
    '''
    @summary: 根据传入的值转换单位为B
    @return: True/False, floatValue
    '''
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024 * 1024 * 1024 * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip()) * 1024 * 1024 * 1024
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) * 1024 * 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) * 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip())
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)
    
def getCurrentVersion(softwareVersionList, lang):
    '''
    @summary: 通过software信息字典列表获取版本信息
    '''
    currentVersion = ""
    errMsg = ""
    for controller in softwareVersionList:
        currentVersion = controller.get("Current Version")
        return (True, currentVersion, "")
    
    errMsg = getMsg(lang, "cannot.get.product.version.info")
    return (False, "", errMsg)

def getProductModeFromContext(py_java_env):
    """
    @summary: 从工具箱的缓存中获取当前设备的产品类型
    """
    return py_java_env.get("devInfo").getDeviceType()

def joinLines(originLines, postLines):
    """
    @summary: 将postLines追加originLines后
    """
    if not (originLines or postLines):
        return ""
    
    if not originLines:
        return postLines
    
    if not postLines:
        return originLines
    
    return "\n".join([originLines, postLines])


def getVstoreNameById(cli, vstoreId, LANG):
    """
    获取租户名称
    :param cli:
    :param vstoreId:
    :return:
    """
    vstoreName = ""
    cmd = "show vstore id=%s" % vstoreId
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        return (flag, vstoreName, cliRet, errMsg)
    vstoreInfoDictList = cliUtil.getVerticalCliRet(cliRet)
    if len(vstoreInfoDictList) == 0:
        return (
            False,
            vstoreName,
            cliRet,
            getMsg(LANG, "get.vstore.name.fail", vstoreId),
        )
    vstoreInfoDict = vstoreInfoDictList[0]
    vstoreName = vstoreInfoDict.get("Name", "")
    if len(vstoreName) == 0:
        return (
            False,
            vstoreName,
            cliRet,
            getMsg(LANG, "get.vstore.name.fail", vstoreId),
        )
    return (True, vstoreName, cliRet, "")


class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """

    def __init__(self, errorMsg, cliRet, flag=cliUtil.RESULT_NOCHECK):
        self.errorMsg = errorMsg
        self.cliRet = cliRet
        self.flag = flag


def isInVstoreMode(cliRet, vstoreName):
    """
    根据回显判断是否在vstoreName租户模式下
    :param cli:
    :param vstoreId:
    :param vstoreName:
    :return:
    """
    if re.search("@%s" % vstoreName, cliRet, re.IGNORECASE):
        return True
    return False


def enterVstoreMode(
    cli, vstoreId, vstoreName, lang, LOGGER,
):
    """
    进入vstoreId的租户模式
    :param cli:
    :param vstoreId:
    :param lang:
    :param vstoreName:用来判断是否进入了对应的租户模式
    :return:
    """
    cliRet = ""
    try:
        cmd = "change vstore view id=%s" % vstoreId
        flag, cliRet, errMsg = cliUtil.excuteCmdCommon(cli, cmd, lang)
        if flag is not True:
            return (False, cliRet, errMsg)
        if isInVstoreMode(cliRet, vstoreName):
            return (True, cliRet, "")
        else:
            return (
                False,
                cliRet,
                getMsg(lang, "enter.vstore.mode.fail", vstoreId),
            )
    except Exception:
        LOGGER.logInfo(str(traceback.format_exc()))
        return (
            False,
            cliRet,
            getMsg(lang, "enter.vstore.mode.fail", vstoreId),
        )


def exitVstoreMode(cli, lang, LOGGER, vstoreName):
    """
    退出vstoreId租户模式到CLI模式
    :param cli:
    :param vstoreId:
    :param vstoreName:
    :return:
    """
    allCliRet = ""
    try:
        flag, cliRet, errMsg = cliUtil.excuteCmdCommon(
            cli, "show system general", lang
        )
        if flag is False:
            return (False, allCliRet, errMsg)
        if isInVstoreMode(cliRet, vstoreName):
            flag, cliRet, errMsg = cliUtil.excuteCmdCommon(cli, "exit", lang)
            allCliRet = joinLines(allCliRet, cliRet)
            if flag is False:
                return (False, allCliRet, errMsg)
        if isInVstoreMode(cliRet, vstoreName):
            return (False, allCliRet, errMsg)
        return (True, allCliRet, "")
    except Exception:
        LOGGER.logInfo(str(traceback.format_exc()))
        return (
            False,
            allCliRet,
            getMsg(lang, "exit.vstore.mode.fail", vstoreName),
        )


def reConnectionCli(cli, LOGGER):
    """
    @summary: 重新获取SSH通道
    """
    if cli is None:
        return False
    try:
        cli.reConnect()
        LOGGER.logInfo("it is afresh to connect to device by ssh gallery.")
        return True
    except Exception:
        LOGGER.logError(str(traceback.format_exc()))
        return False


def check_in_vstore_mode_wrap(func):
    """
    vstore 租户模式装饰器
    :param func:
    :return:
    """

    def wrap(cls, param_dict, *args, **kargs):
        cli = param_dict.get("cli")
        vstore_id = param_dict.get("vstore_id")
        logger = param_dict.get("logger")
        lang = param_dict.get("lang")
        all_ret_list = param_dict["cli_ret_all"]
        enter_ret = ""
        # 获取Vstore Name
        flag, vstore_name, cli_ret, err_msg = getVstoreNameById(
            cli, vstore_id, lang
        )
        if flag is not True:
            enter_ret = joinLines(enter_ret, cli_ret)
            all_ret_list.append(enter_ret)
            raise UnCheckException(err_msg, enter_ret)
        try:
            flag, cli_ret, err_msg = enterVstoreMode(
                cli, vstore_id, vstore_name, lang, logger
            )
            enter_ret = joinLines(enter_ret, cli_ret)
            all_ret_list.append(enter_ret)
            if flag is not True:
                raise UnCheckException(err_msg, enter_ret)
            func_ret = func(cls, param_dict)
            return func_ret
        except Exception:
            logger.logError(str(traceback.format_exc()))
            raise UnCheckException(
                getMsg(lang, "query.result.abnormal"), enter_ret
            )
        finally:
            _, cli_ret, _ = exitVstoreMode(cli, lang, logger, vstore_name)
            all_ret_list.append(cli_ret)
            logger.logInfo("exit vstore mode success.")
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(cli, lang)
            logger.logInfo("enter cli mode from some model  %s" % str(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                reConnectionCli(cli, logger)

    return wrap


def getHotPatchVersion(hotPatchVersionList, lang):
    '''
    @summary: 通过hotPatch信息字典列表获取补丁信息
    @return: flag, hotPatch, errMsg
    flag:
        True:补丁信息一致，且获取成功
        False:补丁信息不一致，或未安装补丁
    hotPatch:补丁信息
    errMsg:错误信息
    '''
    hotPatchVersion = ""
    errMsg = ""
    flag = True

    if not isControllerVersionSame(hotPatchVersionList):
        errMsg += getMsg(lang, "controller.hot.patch.version.not.consistence")
        return (False, "", errMsg)

    for controller in hotPatchVersionList:
        hotPatchVersion = controller.get("Current Version")
        return (True, hotPatchVersion, "")

    errMsg = getMsg(lang, "cannot.get.hot.patch.version.info")
    return (False, hotPatchVersion, errMsg)


def get_soft_and_patch_version(cli, LOGGER, LANG):
    """
    获取阵列版本和热补丁版本
    :return: flag 阵列版本，热补丁版本，回显，错误提示
    """
    (ver_flag, cliRet, err_msg), software_version_list, \
        hot_patch_version_list = parse_upgradePackage(cli, LANG)

    if ver_flag is not True:
        LOGGER.logSysAbnormal()
        LOGGER.logInfo(
            "get package info fail! software_version_list:%s,"
            " hot_patch_version_list%s" %
            (software_version_list, hot_patch_version_list))
        return cliUtil.RESULT_NOCHECK, '', '', cliRet, err_msg

    flag, soft_version, err_msg = getCurrentVersion(
        software_version_list,
        LANG)
    if flag is not True:
        LOGGER.logInfo(
            "get current product version error. "
            "err_msg: %s" % err_msg)
        return cliUtil.RESULT_NOCHECK, '', '', cliRet, err_msg

    flag, patch_version, err_msg = getHotPatchVersion(
        hot_patch_version_list,
        LANG)
    if flag is not True:
        LOGGER.logInfo("get hotPatchVersion error. "
                       "err_msg: %s" % err_msg)
        return cliUtil.RESULT_NOCHECK, '', '', cliRet, err_msg

    return True, soft_version, patch_version, cliRet, err_msg


def is_special_patch(patch_version):
    """
    是否特殊补丁
    :param patch_version:
    :return: True 是，False 不是
    """
    return re.search(r"SPH\d[6-9]\d", patch_version) is not None
