# -*- coding: UTF-8 -*-
import os
import time
import re
import shutil
import traceback

from com.huawei.ism.tool.obase.exception import ToolException

import commons

# 风险产品型号
RISK_DEV_LIST = ["S3900-M200", "S3900-M300"]

# BBU 过期告警
BBU_EXP_ALARM_LIST = ["0xE01F40008", "0xE01F40003"]
NOT_SUPPORT = "not supported"
NO_ALARM_RET = "\nNo bbu expiration alarm."
WARNING = "WARNING"
NO_SUPPORT = "NOSUPPORT"
PY_JAVA_ENV = py_java_env
LOGGER = PY_LOGGER
LANG = PY_JAVA_ENV.get("lang")


def execute(cli):
    """
    A&S317-20200117-关于IT外购件配套的OceanStor S3900磁阵BBU寿命到期
    问题的预警公告
    """
    cli_ret_list = []
    try:
        resource = commons.getResource(PY_JAVA_ENV)

        # 获取回文和设备型号
        cli_ret, device_odel = commons.getCurDeviceType(cli)
        if not device_odel:
            err_msg = resource.getString("invalid.cli.msg")
            return False, cli_ret, err_msg

        # 当前产品型号不在该预警检查项中
        if device_odel not in RISK_DEV_LIST:
            err_msg = resource.getString("no.support.model.or.version")
            return NO_SUPPORT, device_odel, err_msg

        # 检查告警是否存在
        risk_flag, cli_ret = check_alarm_exist(cli)
        LOGGER.info("risk flag is {}".format(risk_flag))
        cli_ret_list.append(cli_ret)

        # 如果命令支持，且有告警则返回检查结果
        if risk_flag is True:
            err_msg = resource.getString("prewarning.no.implement")
            return False, "\n".join(cli_ret_list), err_msg

        # 如果命令不支持，则需要下载日志文件
        elif risk_flag == NOT_SUPPORT:
            flag, err_msg = check_alarm_in_event(cli, cli_ret_list, resource)
            if flag:
                return False, "\n".join(cli_ret_list), err_msg

        # 如果没有告警则继续检查日期是否超期
        warning_bbu, not_pass_bbu = check_bbu_product_time(cli, cli_ret_list)
        LOGGER.info(
            "warning_bbu:{0}, not_pass_bbu:{1}".format(
                warning_bbu, not_pass_bbu
            )
        )

        if not_pass_bbu:
            err_msg = resource.getString("prewarning.no.implement")
            return False, "\n".join(cli_ret_list), err_msg

        if warning_bbu:
            err_msg = resource.getString("prewarning.no.implement")
            return WARNING, "\n".join(cli_ret_list), err_msg

        return True, "\n".join(cli_ret_list), ""
    except (ToolException, Exception) as e:
        LOGGER.error(str(e))
        LOGGER.error(traceback.format_exc())
        err_msg = resource.getString("invalid.cli.msg")
        return False, cli_ret, err_msg
    finally:
        commons.enter_cli_mode_from_other_mode(cli, LOGGER)


def check_alarm_in_event(cli, cli_ret_list, resource):
    """
    检查日志文件中是否有告警
    :param cli:
    :param cli_ret_list:
    :param resource:
    :return:True 有告警，False 无告警
    """
    # 判断是否为安全红线设备
    secure_flag = commons.check_secure(cli)
    LOGGER.info("secure flag is {}".format(secure_flag))
    local_file_path, saved_dir = get_event_file(
        secure_flag, cli, cli_ret_list
    )
    LOGGER.info(
        "local_file_path:{0}, saved_dir:{1}".format(
            local_file_path, saved_dir
        )
    )
    if not local_file_path:
        err_msg = resource.getString("get.bbu.alarm.from.file.error")
        return True, err_msg

    alarm_ids, cli_ret = check_bbu_alarm(local_file_path, saved_dir)
    LOGGER.info("alarm_ids:{0}".format(alarm_ids))
    if alarm_ids:
        cli_ret_list.append(cli_ret)
        err_msg = resource.getString("prewarning.no.implement")
        return True, err_msg
    else:
        cli_ret += NO_ALARM_RET
        cli_ret_list.append(cli_ret)

    return False, ''


def get_event_file(secure_flag, cli, cli_ret_list):
    """
    下载event文件
    :param secure_flag:
    :param cli:
    :param cli_ret_list:
    :return:
    """
    if secure_flag:
        remote_path, cli_ret = commons.secureGetEventFile(cli)
    else:
        remote_path, cli_ret = commons.noSecureGetEventFile(cli, PY_JAVA_ENV)
    cli_ret_list.append(cli_ret)
    if not remote_path:
        return "", cli_ret
    sftp = PY_JAVA_ENV.get("sftp")
    dev_node = PY_JAVA_ENV.get("devInfo")
    save_dir = (
        "pytemp"
        + os.sep
        + dev_node.getDeviceSerialNumber()
        + "_"
        + time.strftime("%Y%m%d%H%M%S", time.localtime(time.time()))
        + os.sep
    )
    local_file_path = commons.download_event_file(
        save_dir, remote_path, sftp, LOGGER, PY_JAVA_ENV
    )
    return local_file_path, save_dir


def check_bbu_alarm(local_file_path, save_dir):
    """
    检查bbu告警
    :param local_file_path:
    :param save_dir:
    :return:
    """
    file_name = "local_alm_file.txt"
    commons.decompress_pkg(local_file_path, save_dir)
    alarm_file = open(save_dir + os.sep + file_name, "r")
    alarm_ids = []
    cliRet = (
        "Raised  Local Time    Alarm ID    Alarm Level    "
        "Recovered Local Time    Description\n"
    )
    for line in alarm_file:
        line = line.strip()
        if line.find("Event list") >= 0:
            break

        line_list = [e for e in line.split("  ") if e]
        if len(line_list) < 4:
            continue

        if line_list[3] != "None":
            continue

        tmp_alarm_ids = [
            bbu_alarm
            for bbu_alarm in BBU_EXP_ALARM_LIST
            if bbu_alarm.lower() in line.lower()
        ]
        if tmp_alarm_ids:
            # 将Suggestion字段之前的字符串添加到原始信息cliRet中。
            index = line.find("Step 1")
            cliRet += line[0:index]
            cliRet += "\n"
            alarm_ids.extend(tmp_alarm_ids)
    alarm_file.close()

    # 删除临时文件
    shutil.rmtree(save_dir, True)
    return alarm_ids, cliRet


def check_bbu_product_time(cli, cli_ret_list):
    """
    检查是否存在BBU过期告警
    :param cli:
    :return:
    """
    sys_time = get_sys_time(cli, cli_ret_list)
    bbu_ret = cli.execCmd("showbbu")
    cli_ret_list.append(bbu_ret)
    bbu_split_line = bbu_ret.splitlines()
    warning_bbu = []
    not_pass_bbu = []
    bbu_id = ""
    for bbu_line in bbu_split_line:
        if "BBU ID" in bbu_line:
            tmp_list = bbu_line.split("|")
            if len(tmp_list) == 2:
                bbu_id = tmp_list[1]
                continue

        if "Manufactured" in bbu_line:
            if not bbu_id:
                continue

            tmp_list = bbu_line.split("=")
            if len(tmp_list) == 2:
                bbu_p_time = tmp_list[1].strip()
                if not bbu_p_time:
                    continue
                diff_years, diff_month = get_year_and_month_diff(
                    sys_time, bbu_p_time
                )
                if diff_years == -1 or diff_month == -1:
                    continue

                flag = check_year_and_month(diff_years, diff_month)
                if flag is False:
                    not_pass_bbu.append(bbu_id)
                elif flag == WARNING:
                    warning_bbu.append(bbu_id)
            # 置空bbu_id
            bbu_id = ""

    return warning_bbu, not_pass_bbu


def check_year_and_month(diff_years, diff_month):
    """
    2.1如果运行时间小于5年6个月则检查通过；
   2.2 如果运行时间小于7年6个月则检查结果为建议优化；
   2.3如果运行时间大于7年6个月则检查结果为不通过。
    :param diff_years:
    :param diff_month:
    :return:
    """
    # 小于5年6个月就通过
    if diff_years < 5:
        return True
    if diff_years == 5 and diff_month < 6:
        return True
    # 如果运行时间小于7年6个月则检查结果为建议优化；
    if diff_years < 7:
        return WARNING
    if diff_years == 7 and diff_month < 6:
        return WARNING

    # 如果运行时间大于7年6个月则检查结果为不通过。
    return False


def get_year_and_month_diff(sys_time, bbu_time):
    """
    获取年月数字
    :param sys_time:
    :param bbu_time:
    :return:
    """
    reg = re.compile(r"\d{4}-\d{2}-\d{2}")
    res = reg.match(sys_time)
    if not res:
        return -1, -1
    res = reg.match(bbu_time)
    if not res:
        return -1, -1

    tmp_list_sys = sys_time.split("-")
    tmp_list_bbu = bbu_time.split("-")
    if len(tmp_list_sys) != len(tmp_list_bbu):
        return -1, -1

    return (
        int(tmp_list_sys[0]) - int(tmp_list_bbu[0]),
        int(tmp_list_sys[1]) - int(tmp_list_bbu[1]),
    )


def check_alarm_exist(cli):
    """
    检查是否存在BBU过期告警
    :param cli:
    :return:
    """
    ret = cli.execCmd("showalarm")
    if '"showalarm" not exist.' in ret:
        return NOT_SUPPORT, ret

    for alarm in BBU_EXP_ALARM_LIST:
        if alarm.lower() in ret.lower():
            return True, ret

    return False, ret


def get_sys_time(cli, cli_ret_list):
    """
    获取系统时间
    :param cli:
    :param cli_ret_list:
    :return:
    """
    ret = cli.execCmd("showsys")
    cli_ret_list.append(ret)
    tmp_line = ret.splitlines()
    for line in tmp_line:
        line = line.strip()
        value_line = line.split("|")
        if len(value_line) != 2:
            continue

        if "Time" in value_line[0]:
            sys_time = value_line[1].strip()
            return sys_time.split()[0]

    return None
