# -*- coding: UTF-8 -*-
import re
import os
import shutil
from cbb.frame.util.tar_util import decompress_tar_all_file

from com.huawei.ism.tool.obase.exception import ToolException

R2C00_SECURE_VERSION = "V100R002C00SPC010"
R2C01_SECURE_VERSION = "V100R002C01SPC007"


def getCurSysVer(cli, resource):
    """
    @summary: 
    @param cli: 
    @param resource: 
    @return: 
    """
    # 查询系统版本号
    sysVer = ""
    errMsg = ""
    cliRet = cli.execCmd("showupgradepkginfo -t 1")
    lines = cliRet.splitlines()

    # 命令执行结果小于7行，说明执行该命令后并没有输出有效信息
    if len(lines) < 7:
        errMsg = resource.getString("invalid.cli.msg")
        return (cliRet, False, errMsg, sysVer)

    # 从第7行开始，取出有效信息
    sysVer = lines[6].split()[1]
    return (cliRet, True, errMsg, sysVer)


# *************************************************************#
# 函数名称: getCurDeviceType
# 功能说明: 获取当前系统型号
# 其 他   :  无
# *************************************************************#
def getCurDeviceType(cli):

    # 查询设备类型
    DeviceType = ""

    sysInfo = cli.execCmd("showsys")
    lineList = sysInfo.splitlines()
    if re.search("System Information", sysInfo, re.IGNORECASE):
        for line in lineList:
            if line.find("Device Type") != -1:
                field = line.split()
                if len(field) == 5:
                    DeviceType = field[-1]
                break

    return sysInfo, DeviceType


def getSysVer(cli):
    """
    Function name      : getSysVer(cliConnection)
    Function describe  : get system version
    Input              : cli--Telnet access proxy 
                         e.g: cliRet = cli.execCmd('showctrlinfo')
    Return             : cliRet, system version
    """

    cliRet = cli.execCmd("showctrlinfo")
    lines = cliRet.splitlines()
    for field in lines:
        if field.strip().startswith("Software Version"):  # 查询系统版本号提取成单独的方法
            return cliRet, field[field.find("|") + 1:].strip()

    return cliRet, None


# *************************************************************#
# 函数名称: getCurHotPatchVersion
# 功能说明: 获取当前系统补丁版本号
# 其 他   :  无
# *************************************************************#
def getCurHotPatchVersion(cli, resource):

    # 查询当前设备已安装热补丁
    curHotPatchVer = ""
    errMsg = ""
    hotPatchVerInfo = cli.execCmd("showupgradepkginfo -t 3")

    # 没有安装任何热补丁，返回空
    if re.search(
        "command operates successfully", hotPatchVerInfo, re.IGNORECASE
    ):
        return (hotPatchVerInfo, True, errMsg, curHotPatchVer)

    # 导入了热补丁包但是没有激活
    if not re.search("Running", hotPatchVerInfo, re.IGNORECASE):
        return (hotPatchVerInfo, True, errMsg, curHotPatchVer)
    lineList = hotPatchVerInfo.splitlines()

    # 信息少于7行，查询信息失败
    if len(lineList) < 7:
        errMsg = resource.getString("invalid.cli.msg")
        return (hotPatchVerInfo, False, errMsg, curHotPatchVer)

    # 正常情况下获取Running的热补丁版本
    for line in lineList:
        if re.search("A|B", line) and re.search(
            "Running", line, re.IGNORECASE
        ):
            field = line.split()
            if len(field) >= 3:
                curHotPatchVer = field[1]
                break
        else:
            continue
    return (hotPatchVerInfo, True, errMsg, curHotPatchVer)


# *************************************************************#
# 函数名称: getCurColdPatchVersion
# 功能说明: 获取当前系统冷补丁版本号
# 其 他   :  无
# *************************************************************#
def getCurColdPatchVersion(cli, resource):

    # 查询当前设备已安装热补丁
    curColdPatchVer = ""
    errMsg = ""
    coldPatchVerInfo = cli.execCmd("showupgradepkginfo -t 2")

    # 没有安装任何热补丁，返回空
    if re.search(
        "command operates successfully", coldPatchVerInfo, re.IGNORECASE
    ):
        errMsg = resource.getString("no.cold.patch.version")
        return (coldPatchVerInfo, True, errMsg, curColdPatchVer)

    # 导入了冷补丁包但是没有激活
    if not re.search("Running", coldPatchVerInfo, re.IGNORECASE):
        errMsg = resource.getString("cold.patch.not.work")
        return (coldPatchVerInfo, True, errMsg, curColdPatchVer)
    lineList = coldPatchVerInfo.splitlines()

    # 信息少于7行，查询信息失败
    if len(lineList) < 7:
        errMsg = resource.getString("invalid.cli.msg")
        return (coldPatchVerInfo, False, errMsg, curColdPatchVer)

    # 正常情况下获取Running的热补丁版本
    for line in lineList:
        if re.search("A|B", line) and re.search(
            "Running", line, re.IGNORECASE
        ):
            field = line.split()
            if len(field) >= 3:
                curColdPatchVer = field[1]
                if len(curColdPatchVer) > 17:
                    # 部分设备获取冷不丁前有设备版本号，需去除。
                    curColdPatchVer = curColdPatchVer[-17:]
                errMsg = (
                    resource.getString("current.cold.patch.version")
                    % curColdPatchVer
                )
                break
        else:
            continue
    return (coldPatchVerInfo, True, errMsg, curColdPatchVer)


def getValByKey(key, keyLine, valueLine):
    """
    Function name      : getValByKey(key, keyLine, valueLine)
    Function describe  : get the value by key
    Input              : key--key value
                         keyLine -- that string include key
                         valueLine -- that string include value
    Return             : value
    """

    keys = re.findall(
        str(key).replace("(", r"\(").replace(")", r"\)") + r"\s*",
        keyLine,
        re.IGNORECASE,
    )

    if not keys:
        return ""
    else:
        beginIndex = keyLine.find(keys[0])
        return valueLine[beginIndex: beginIndex + len(keys[0])].strip()


def getResource(py_java_env):
    """
    Function name      : getResource(py_java_env)
    Function describe  : get Resource object
    Input              : py_java_env--context
    Return             : Resource object
    """

    if not py_java_env.get("resouce"):
        py_java_env["resouce"] = Resource(py_java_env)

    return py_java_env["resouce"]


class Resource:
    """
    Class name      : Resource()
    Class describe  : Resource file to get error massage
    """

    def __init__(self, py_java_env):
        """
        Function name      : __init__(filename)
        Function describe  : initialize Class 
        """

        self.lang = py_java_env.get("lang")
        filepath = py_java_env.get("path")

        if self.lang == "en":
            self.resource = (
                filepath + os.sep + "res" + os.sep + "resource_en.ini"
            )
        else:
            self.resource = (
                filepath + os.sep + "res" + os.sep + "resource_zh.ini"
            )

        self.dict = {}
        self.genDict()

    def genDict(self):
        """
        Function name      : genDict()
        Function describe  : read resource file and create dict
        Return             : dict
        """

        fileBuff = open(self.resource)
        for line in fileBuff:
            if line.startswith("[") or line.startswith("#") or "=" not in line:
                continue
            else:
                key = line.split("=")[0].strip()
                val = line.split("=")[1].strip().decode("utf-8", "ignore")
                self.dict[key] = val

    def getString(self, key):
        """
        Function name      : getString(key)
        Function describe  : get Value by key from resource
        Input              : key
        Return             : value
        """

        return self.dict.get(key, "")


def getHorizontalCliRet(cliRet):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    try:
        headline = ""
        CLI_RET_END_FLAG = ":/>"
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while start_pos <= len(headline):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]: item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            # 标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]: item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def secureGetEventFile(cli):
    """
    功能说明: 安全红线后的版本：获取告警文件
    :param cli:
    :return:
    """
    remoteFile = ""

    # 通过Cli命令导出告警文件
    cliRet = cli.execCmd("exportsysevent -t alarm")
    # Cli回文异常，直接返回不通过
    if not bool(re.search("Path", cliRet, re.IGNORECASE)):
        return "", cliRet

    lineList = cliRet.splitlines()
    for line in lineList:
        if re.search("Path :", line, re.IGNORECASE):
            remoteFile = line.replace("Path :", "").replace(" ", "")
            break

    return remoteFile, cliRet


def noSecureGetEventFile(cli, py_java_env):
    """
    功能说明: 安全红线前的版本：获取告警文件
    :param cli:
    :return:
    """
    remoteFile = ""

    iRet = changeCliToMml(cli, py_java_env)
    if not iRet:
        return remoteFile, iRet

    # 通过mml命令生成告警文件
    mmlRet = cli.execCmd("alarm exportevent")

    # 退回到Cli模式
    cli.execCmd("exit")
    cli.execCmd("exit")
    cli.execCmd("exit")

    if not bool(re.search("path", mmlRet, re.IGNORECASE)):
        return remoteFile, mmlRet

    # 解析命令回文
    lineList = mmlRet.splitlines()
    for line in lineList:
        if re.search("path", line, re.IGNORECASE):
            remoteFile = line[line.index(":/") + 1: -2]
            break

    return remoteFile, mmlRet


def deleteRemoteFile(py_java_env, logger, collectRemotePath):
    """
    功能说明: 清理设备临时文件
    :param collectRemotePath:
    :return:
    """
    try:
        sftp = py_java_env.get("sftp")
        # 使用sftp自带接口删除远端临时文件
        if not collectRemotePath:
            logger.error(
                "[deleteRemoteFile] delete file is:"
                + unicode(collectRemotePath)
            )
            return False
        else:
            sftp.deleteFile(collectRemotePath)
            return True
    except (ToolException, Exception) as e:
        logger.error(e)
        return False


def download_event_file(save_dir, remote_file_path, sftp, logger, py_java_env):
    # 创建文件保存路径
    if not os.path.exists(save_dir):
        os.makedirs(save_dir)
    file_name = save_dir + "event_export.tar"
    retry_time = 3
    while retry_time > 0:
        try:
            sftp.getFile(remote_file_path, file_name, None)
            deleteRemoteFile(py_java_env, logger, remote_file_path)
            return file_name
        except (ToolException, Exception) as e:
            logger.error(e)
            sftp.reConnect()
        retry_time -= 1
    shutil.rmtree(save_dir, True)
    return ""


def changeCliToMml(ssh, py_java_env):
    iRet = changeCliToDebug(py_java_env, ssh)
    if iRet is True:
        iRet1 = ssh.execCmd("mml")
        if bool(re.search("MML>", iRet1, re.IGNORECASE)):
            return True

    return False


def changeCliToDebug(py_java_env, ssh):
    """
    功能说明: Cli模式转换成Debug模式
    :param py_java_env:
    :param ssh:
    :return:
    """
    password = unicode(py_java_env.get("devInfo").getDeveloperPwd())
    ret = ssh.execCmd("developer")
    if "(y/n)" in ret:
        ssh.execCmd("y")
    ssh.execCmdNoLog(password)
    retDeveloper = ssh.execCmdNoLog("debug")
    if -1 != retDeveloper.find("Password:"):
        ssh.execCmdNoLog(password)
    return True


def switchValidVersion(version):
    """
    功能说明: 将版本号转化成有效的版本号（17位：V100R002C00SPC001
    或者11位：V100R005C02）
    :param version:
    :return:
    """
    temp = ""

    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  # 格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  # 格式：V100R002C00

    # 提取有效的版本号信息
    temp = version[startIndex:endIndex]

    return temp


def check_secure(ssh):
    """
    判断是否为安全红线设备
    :param ssh:
    :return:
    """
    isSecure = False

    cliRet = ssh.execCmd("showupgradepkginfo -t 1")
    rowList = cliRet.splitlines()

    count = len(rowList)
    for i in range(count):
        if bool(re.search("Controller ID", rowList[i], re.IGNORECASE)):

            columnList = rowList[i + 2].split()
            devVersion = switchValidVersion(columnList[1])
            # 根据设备版本号是否安全红线设备
            if bool(re.search("V100R001", devVersion, re.IGNORECASE)):
                isSecure = False
            elif bool(re.search("V100R002C00", devVersion, re.IGNORECASE)):
                if devVersion < R2C00_SECURE_VERSION:
                    isSecure = False
                else:
                    isSecure = True
            elif bool(re.search("V100R002C01", devVersion, re.IGNORECASE)):
                if devVersion < R2C01_SECURE_VERSION:
                    isSecure = False
                else:
                    isSecure = True
            elif bool(re.search("V100R005", devVersion, re.IGNORECASE)):
                isSecure = True
            else:
                isSecure = False
    # 返回判断结果
    return isSecure


def decompress_pkg(file_path, depress_path):
    """
    解压文件
    :param file_path:
    :param depress_path:
    :return:
    """
    decompress_tar_all_file(file_path, depress_path)


def enter_cli_mode_from_other_mode(cli, logger):
    """
    从其他模式进入cli模式
    :param cli:
    :param logger:
    :return:
    """
    try:
        ret = cli.execCmd("showsys")
        if is_in_mml_mode(ret):
            ret = cli.execCmd("exit")
            if "(y/n)" in ret:
                ret = cli.execCmd("y")

        if is_in_debug_mode(ret):
            ret = cli.execCmd("exit")
            if "(y/n)" in ret:
                ret = cli.execCmd("y")

        if is_in_developer_mode(ret):
            ret = cli.execCmd("exit")
            if "(y/n)" in ret:
                ret = cli.execCmd("y")

        if ret.endswith(":/>"):
            return True, ret

        return False, ret
    except (ToolException, Exception) as e:
        logger.error(str(e))
        cli.reConnect()
        return True, ""


def is_in_debug_mode(ret):
    """
    判断当前是否在debug模式下
    :param ret:
    :return:
    """
    return "Storage:/OSM #".lower() in ret.lower()


def is_in_mml_mode(ret):
    """
    判断当前是否在MML模式下
    :param ret:
    :return:
    """
    return "MML>".lower() in ret.lower()


def is_in_developer_mode(ret):
    """
    判断当前是否在developer模式下
    :param ret:
    :return:
    """
    return "developer:".lower() in ret.lower()
