# -*- coding: UTF-8 -*-
from common import constant, utils, modeMgt, cliMgt, cTV1R1
import traceback, os, re
import ast

java_env = py_java_env
logger = PY_LOGGER

def execute(cli):
    """
    # *****************************************************************************************#
    # 函数名称: execute(cli)
    # 功能说明: 查看 针对包含SATA盘的RAID组，同一个RAID组内所有的LUN是否归属于同一个控制器
    # 输入参数: cli
    # 返 回 值: flag(检查结果), allCliRet（命令回显）, errMsg（错误消息）
    # *****************************************************************************************#
    """
    lang = java_env.get("lang")   
    flag = True
    allCliRet = "" 
    errMsg = ""
    
    try:
        #检查当前设备是否为问题型号与问题版本
        isPass, cliRet = checkProductTypeAndVersion(cli)
        allCliRet += cliRet
        if isPass:
            return constant.NO_SUPPORT, allCliRet, errMsg
        
        #获取所有SATA盘位置列表
        sataDiskLocationList, cliRet, errMsg = getSataDiskLocationList(cli, lang)
        allCliRet += cliRet
        if sataDiskLocationList == None: 
            logger.error("Inspect[Dual-end_SATADiskAccess] not pass(cliRet invalid)")                                    
            return False, allCliRet, errMsg
        if not sataDiskLocationList:
            logger.info("Inspect[Dual-end_SATADiskAccess] pass!(No SATA disk infomation)")
            return True, allCliRet, errMsg
        
        #获取RAID组ID及其对应的成员盘位置列表
        rgDiskLocationListDict, cliRet, errMsg = getRgDiskLocationListDict(cli, lang)
        allCliRet += cliRet
        if rgDiskLocationListDict == None:
            logger.error("Inspect[Dual-end_SATADiskAccess] not pass(cliRet invalid)")
            return False, allCliRet, errMsg
        if not rgDiskLocationListDict:
            logger.info("Inspect[Dual-end_SATADiskAccess] pass!(No raid group infomation)")
            return True, allCliRet, errMsg
        
        #获取包含SATA盘的各个RAID组ID
        rgIdList = getRgIdIncludeSataDisk(sataDiskLocationList, rgDiskLocationListDict)
        if not rgIdList:
            logger.info("Inspect[Dual-end_SATADiskAccess] pass!(Raid groups do not include SATA disk.)")
            return True, allCliRet, errMsg
        
        #记录包含SATA盘的同一个RAID组下的所有LUNID
        sataLunDict, cliRet, errMsg = getLunInfo(rgIdList, cli, lang)
        allCliRet += cliRet
        if sataLunDict == False:
            logger.error("Inspect[Dual-end_SATADiskAccess] not pass(Execute command time out.)")
            return False, allCliRet, errMsg
        if sataLunDict == None:
            logger.error("Inspect[Dual-end_SATADiskAccess] not pass(cliRet invalid)")
            return False, allCliRet, errMsg
        if len(sataLunDict) == 0:
            logger.info("Inspect[Dual-end_SATADiskAccess] pass!(No lun infomation)")
            return True, allCliRet, errMsg
        
        #查看同一个RAID组下的LUN是否归属于同一个控制器。
        flag, cliRet, errMsg = checkAscriptionCtrl(sataLunDict, cli, lang)
        allCliRet += cliRet
        return flag, allCliRet, errMsg
    except:
        logger.error("Inspect[Dual-end_SATADiskAccess] catch except of trace back:" + str(traceback.format_exc()))
        return False, allCliRet, utils.getExceptionMsg(lang)  
                                    
def checkAscriptionCtrl(sataLunDict, cli, lang):
    '''
    @summary: 针对包含SATA盘的RAID组，检查同一个RAID组下的LUN是否归属于同一个控制器
    @param : sataLunDict, cli, lang
    @return: checkFlag(True：归属于同一个控制器，False：不归属于同一控制器), allCliRet(回显信息), errMsg(错误信息)
    '''
    
    allCliRet = ""
    errMsg = ""
    checkFlag = True
    
    for rgId, lunIdList in sataLunDict.items():
        ctrlIDList = []
        for lunId in lunIdList:
            cmd = "showlun -i %s" % lunId
            cliRet = cliMgt.execCmd(cli, cmd)
            preRet, errInfo = utils.preCheckCliRet(cmd, cliRet, lang, False)
            allCliRet += cliRet
            if constant.RET_FAIL == preRet:
                errMsg += errInfo
                checkFlag = False
                return checkFlag, allCliRet, errMsg
            else:       
                lunCtrlDict = getIdAndAscriptionCtrl(cliRet)
                ctrlIDList.append(lunCtrlDict.get(lunId))
        ctrlIDList = list(set(ctrlIDList))  
            
        #若同一个RAID组下的LUN不归属于同一个控制器，则巡检不通过           
        if len(ctrlIDList) != 1:
            checkFlag = False
            if lang == "zh":
                errMsg += u"\nRAID(ID:%s)下的LUN不归属于同一控制器" % rgId
            else:
                errMsg += u"\nLUNS in RAID(ID:%s) do not belong to the same controller." %rgId        
            
    return checkFlag, allCliRet, errMsg
                                                    
def getIdAndAscriptionCtrl(cliRet):
    '''
    @summary: 获取LUNID和归属控制器
    @param : cliRet回显
    @return: lunContrlDict(LUN归属控制器字典列表)
    '''
    lunId = ''
    ctrlId = ''
    lunContrlDict = {}
    
    cliRetList = cliRet.encode("utf8").splitlines()
    for line in cliRetList: 
        splitMarkColon ="|"
        key = ""
        val = ""
        if splitMarkColon in line:
            list = line.split(splitMarkColon)
            key = list[0].strip().upper()
            val = list[1].strip().upper()
            if key == "ID" :
                lunId = val
            elif key == "ASCRIPTION CONTROLLER":
                ctrlId = val
    if lunId and ctrlId:
        lunContrlDict.setdefault(lunId, ctrlId)    
            
    return lunContrlDict
                       
def getLunInfo(rgIdList, cli, lang):
    '''
           
    @summary: 获取包含SATA盘的Raid组的LUNID
    @param :  rgIdList, cli, lang
    @return: 返回sataLunDict(包含SATA盘的Raid组的LUNID字典),cliRet(回显信息),errMsg(错误信息)
    '''

    sataLunDict = {}
    errMsg = ""
    #执行命令showlun，获取包含SATA盘的各Raid组的LUN信息
    cmd = "showlun"
    cliRet = cliMgt.execCmdNoLog(cli, cmd)
    #命令执行超时处理
    if re.search('TOOLKIT_SEND_CMD_TIME_OUT',cliRet,re.IGNORECASE):
        if lang == "zh":
            errMsg +=u"\n执行命令超时。"
        else:
            errMsg +="\nExecute command time out."
        return False, cliRet, errMsg
    
    preRet, errMsg = utils.preCheckCliRet(cmd, cliRet, lang, False)
    if constant.RET_FAIL == preRet:
        return None, cliRet, errMsg
    if constant.RET_SUC == preRet:
        return sataLunDict, cliRet, errMsg
    
    #获取硬盘信息字典列表
    formatFunction = cTV1R1.cHandleTypeList(cliRet)
    lunDictList = formatFunction.handle()
    for lunInfoDict in lunDictList:
        rgId = lunInfoDict.get("RAID ID","").strip()
        lunId = lunInfoDict.get("ID","").strip()
        if rgId in rgIdList:
            if sataLunDict.has_key(rgId):
                lunIds = sataLunDict.get(rgId).append(lunId)
                sataLunDict.setdefault(rgId,lunIds)
            else:
                sataLunDict.setdefault(rgId,[lunId])
    return sataLunDict, cliRet, errMsg


    
def getRgDiskLocationListDict(cli, lang):
    '''
    @summary: 获取RAID组信息
    @param :  cli, lang
    @return: rgDiskListDict(RAID组信息字典),cliRet(回显信息),errMsg(错误信息)
    '''
    
    rgDiskListDict = {}
    errMsg = ''
    
    cmd = "showrg"
    cliRet = cliMgt.execCmd(cli, cmd)
    preRet, errMsg = utils.preCheckCliRet(cmd, cliRet, lang, False)
    if constant.RET_FAIL == preRet:
        return None, cliRet, errMsg
    if constant.RET_SUC == preRet:
        return rgDiskListDict, cliRet, errMsg
    
    formatFunction = cTV1R1.cHandleTypeList(cliRet)
    rgInfoDictList = formatFunction.handle()   
    for rgInfoDict in rgInfoDictList:
        diskList = rgInfoDict.get("Disk list","").strip()
        raidId = rgInfoDict.get("ID","").strip()
                    
        if raidId and diskList and diskList != "--":
            rgDiskListDict.setdefault(raidId, diskList)

    return rgDiskListDict, cliRet, errMsg
                  
def getSataDiskLocationList(cli, lang):
    '''
    @summary: 获取SATA盘的盘符信息
    @param :  cli, lang
    @return: sataDiskLocationList(SATA盘盘符信息列表),cliRet(回显信息),errMsg(错误信息)
    '''
      
    sataDiskLocationList = []
    
    cmd = "showdisk -p"
    cliRet = cliMgt.execCmd(cli, cmd)
    preRet, errMsg = utils.preCheckCliRet(cmd, cliRet, lang, False)
    if constant.RET_FAIL == preRet:
        return None, cliRet, errMsg
    if constant.RET_SUC == preRet:
        return sataDiskLocationList, cliRet, errMsg    
    #获取硬盘信息字典列表   
    formatFunction = cTV1R1.cHandleTypeList(cliRet)
    SATADictList = formatFunction.handle()
    for diskInfo in SATADictList:
        diskInfoUp = ast.literal_eval(str(diskInfo).upper())
        diskLocation = diskInfoUp.get("DISK LOCATION", "").strip()
        diskType = diskInfoUp.get("PHYSICAL TYPE", "").strip()
        if diskLocation and diskLocation != "--" and diskType == "SATA":
            diskLocation = diskLocation.replace(" ","")
            sataDiskLocationList.append(diskLocation)  
                      
    return sataDiskLocationList, cliRet, errMsg 

def getRgIdIncludeSataDisk(sataDiskLocationList,rgDiskLocationListDict):
    '''
    @summary: 查看各个RAID组的成员盘是是否包含SATA盘
    @param : sataDiskLocationList,rgDiskLocationListDict
    @return: rgIdList(包含SATA盘的RAIDID列表)
    '''    
    rgIdList = [] 
    for raidId, diskIdList in rgDiskLocationListDict.items():
        diskIdList = re.sub(r"(\d+,\d+);", r"(\1)", diskIdList)
        for diskId in re.findall(r"(\(\d+,\d+\))",diskIdList):
            if diskId in sataDiskLocationList:
                rgIdList.append(raidId)
                break
    logger.info("The rgIdList is:" + str(rgIdList))
    return rgIdList

def checkProductTypeAndVersion(cli):
    """
    @summary: 检查产品型号与版本号是否是问题型号
    @return: True(非问题版本与型号，检查项不涉及), cliRet(回显信息)
    @return: False(问题版本与型号，继续后续的检查), cliRet(回显信息)
    """
    
    cliRet = ""
    #S6800 V100R005C02SPC005（1.03.03.525.T02）及以上版本和S2600 V100R001C02SPC012（1.04.01.208.T10）及以上版本为问题型号与问题版本         
    typeVersionDict = {"S6800E": "1.03.03.525.T02", "S2600": "1.04.01.208.T10"}    
    deviceType = str(java_env.get("devInfo").getDeviceType())
    logger.info("The type of device is %s" % deviceType)
    if deviceType not in typeVersionDict:
        return True , cliRet
        
    productVersion, cliRet = utils.getDevVer(cli)
    logger.info("The version of product is %s" % productVersion)
    bugVersion = typeVersionDict.get(deviceType.upper(),"") 
    if productVersion < bugVersion:           
        return True, cliRet
    
    return False, cliRet

    
    
    
    
        
        
    
    
    
    
    

    
   

