#--*-- coding:utf-8 --*--
import traceback
import re
import time
import decimal
from common import utils
from common import cliMgt
from common import constant

DEF_SPEED_DICT = {"GE":100, "10GE":600, "8GFC":750, "4GFC":360, "2GFC":180, "SAS":2048, "TOE":700,}
INTERFACE_TYPE_LIST = ["4*iSCSI", "COMBO", "4*FC"]
CONTROLLER_ID_LIST = ["A", "a", "B", "b"]

def execute(cli):
    '''
    @summary: 检查端口利用率
    @param cli: cli链接对象
    '''
    cliRet = ""
    errMsg = ""
    needCloseSwitch = False
     
    try:
        lang = py_java_env.get("lang")
        #判断型号版本是否符合
        deviceType = str(py_java_env.get("devInfo").getDeviceType())
        sysVer,cmdRet = getDevVer(cli)
        PY_LOGGER.info("deviceType: " + str(deviceType) + ", Version: " + str(sysVer))
        #S2600  1.04.01.208.T10 V100R001C02SPC012
        if deviceType.upper() != "S2600" or sysVer < "1.04.01.208.T10":
            return (constant.NO_SUPPORT,cliRet,errMsg)
        
        #1. 执行showstatswitch命令查询系统性能统计状态并记录，未打开需要打开
        getFlag, cliInfo, status, errInfo = getPortStat(cli, lang)
        cliRet += cliInfo
        if not getFlag:
            return getFlag, cliInfo, errInfo
        
        #2. 状态为关闭，则打开
        if re.match("close", status, re.IGNORECASE):
            cliInfo = openSwitch(cli)
            needCloseSwitch = True
            cliRet += "\n" + cliInfo
            
        #3. 执行showinterface命令查询系统接口卡类型
        getFlag, cliInfo, typeDict, errInfo = getInterfaceType(cli, lang)
        PY_LOGGER.info("typeDict:" + str(typeDict))
        cliRet += "\n" + cliInfo
        if not getFlag:
            return False, cliRet, errInfo
        
        #4. 获取各端口对应的最大速率
        maxSpeedDict = {}
        firstExe4Combo = True
        firstExe4Fc = True
        for typeInfo in typeDict:
            type = typeDict.get(typeInfo)
            controllerId = typeInfo[0]
            interFacePort = typeInfo[1]
            
            #4*iSCSI （GE） 所有端口速率体现在00口
            if type == INTERFACE_TYPE_LIST[0]:
                zeroPort = interFacePort + "0"
                maxSpeedDict[(controllerId, zeroPort)] = 4 * DEF_SPEED_DICT.get("GE")
            
            #COMB0（2*GE+2*FC) 
            if type == INTERFACE_TYPE_LIST[1]:
                #2个FC需要获取协商速率;命令回显中端口有0->1, 1->2的关系;为0G未接入,不计算;只需要执行一次就可以获取所有控制器的FC信息
                if firstExe4Combo:
                    firstExe4Combo = False
                    getFlag, errInfo = getFcNegotiationSpeed(cli, lang, maxSpeedDict, interFacePort, isCombo=True)
                    if not getFlag:
                        return False, cliRet, errInfo
                     
                #2*GE 2个GE绑定在00端口
                zeroPort = interFacePort + "0"
                maxSpeedDict[(controllerId, zeroPort)] = 2 * DEF_SPEED_DICT.get("GE")
            
            #4*FC ,只需要执行一次就可以获取所有控制器的FC信息
            if type == INTERFACE_TYPE_LIST[2] and firstExe4Fc:
                firstExe4Fc = False
                getFlag, errInfo = getFcNegotiationSpeed(cli, lang, maxSpeedDict, interFacePort)
                if not getFlag:
                    return False, cliRet, errInfo
                           
        PY_LOGGER.info("maxSpeedDict:" + str(maxSpeedDict))
        
        #5. 执行statperf -p命令，获取各端口的使用带宽,最多需要查询3次
        lastOverflowAllDict = {}
        for loop in range(3):
            getFlag, cliInfo, bandwidthsDict, errInfo = getPortBandwidths(cli, lang)
            cliRet += "\n" + cliInfo
            if not getFlag:
                return False, cliRet, errInfo
            PY_LOGGER.info("bandwidthsDict:" + str(bandwidthsDict))
                
            #计算当前带宽利用率，并判断是否超过阈值
            overflowDict = getBandwidthsUsage(maxSpeedDict, bandwidthsDict)
            
            #任何一次的结果为空，则达不到3次超标，可直接返回
            if not overflowDict:
                return True, cliRet, ""
            
            #合并超标信息
            overflowAllDict = mergeOverflowDict(lastOverflowAllDict, overflowDict)
            lastOverflowAllDict = overflowAllDict
            #合并后，不存在信息，则表示没有连续超标的端口
            if not overflowAllDict:
                return True, cliRet,""
            
            #每次检查间隔10S, 最后一次不用休眠
            if loop != 2:
                time.sleep(10)
        
        PY_LOGGER.info("overflowAllDict:" + str(overflowAllDict)) 
           
        #6. 判断超标结果是警告还是不通过
        judgeRet, errInfo = judgeCheckRet(overflowAllDict, lang)
        return judgeRet, cliRet, errInfo
        
    except:
        PY_LOGGER.error("[except]" + traceback.format_exc())
        return False, cliRet, errMsg
    
    finally:
        #之前打开过开关，则需要关闭
        if needCloseSwitch:
            closeSwitch(cli)
    
    
def getPortStat(cli, lang):
    '''
    @summary: 获取端口性能状态
    '''
    cmd = "showstatswitch"
    cliInfo = cliMgt.execCmd(cli, cmd)
    

    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        return False, cliInfo, "", errInfo
    
    #获取端口状态
    getRet, portStatus = getPortStatByCliRet(cliInfo)
    PY_LOGGER.info("portStatus:" + str(portStatus))
    if not getRet:
        errMsg = {"zh":u"\n无法获取系统性能统计状态",
               "en":"\nFailed to obtain the performance statistical status of the system."}
        return False, cliInfo, portStatus, errMsg.get(lang)
    
    return True, cliInfo, portStatus, ""


def getPortStatByCliRet(cliInfo):
    '''
    @summary: 获取端口性能统计状态
    '''
    status = ""
    
    keyWords = "Statistic status based on PORT:"
    for line in cliInfo.splitlines():
        if re.search(keyWords, line, re.IGNORECASE):
            statInfo = line.split(":")
            if len(statInfo) == 2:
                status = statInfo[1].strip()
                return True, status
            
    return False, status
                

def openSwitch(cli):
    '''
    @summary: 执行命令打开系统端口性能统计开关
    '''         
    cmd = "chgstatswitch -o -p"
    cliInfo = cliMgt.execCmd(cli, cmd)
    return cliInfo


def closeSwitch(cli):
    '''
    @summary: 执行命令关闭系统端口性能统计开关
    '''         
    cmd = "chgstatswitch -c -p"
    cliInfo = cliMgt.execCmd(cli, cmd)
    return cliInfo


def getInterfaceType(cli, lang):
    '''
    @summary: 获取接口卡类型
    '''
    cmd = "showinterface"
    cliInfo = cliMgt.execCmd(cli, cmd)
    typeDict = {}
    
    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        errInfo = {"zh":u"\n无法获取系统接口卡信息",
                       "en":"\nFailed to obtain information about interface modules of the system.",}
        return False, cliInfo, typeDict, errInfo.get(lang)
    
    typeDict = getInterfaceTypeByCliRet(cliInfo)
    return True, cliInfo, typeDict, ""


def getInterfaceTypeByCliRet(cliInfo):
    '''
    @summary: 解析命令回文，获取端口类型
    @return: typeDict: {(controller, interface):type}
    '''
    
    typeDict = {}
    for line in cliInfo.splitlines():
        lineInfo = line.split()
        #无用数据行
        if len(lineInfo) < 3:
            continue
        
        contr = lineInfo[0]
        interface = lineInfo[1]
        type = lineInfo[2]

        #控制器和type同时满足条件才算有效信息
        if contr in CONTROLLER_ID_LIST and\
           type in INTERFACE_TYPE_LIST:
            typeDict[(contr, interface)] = type
            continue
        
    return typeDict
            
            
def getFcNegotiationSpeed(cli, lang, maxSpeedDict, interFacePort, isCombo=False):
    '''
    @summary: 执行showfps获取协商速率，不为0G的才需要进行后续计算. 调用此函数，一定要有FC卡. 函数主要是更改传入的maxSpeedDict字典
    '''            
    cmd = "showfps"
    cliInfo = cliMgt.execCmd(cli, cmd)
    
    notCountDict = {}
    
    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        errInfo = {"zh":u"\n无法获取端口总带宽",
                   "en":"\nFailed to obtain the total bandwidth of the port."}
        return False, errInfo.get(lang)
 
    getFcNegotiationSpeedByCliRet(cliInfo, maxSpeedDict, notCountDict, interFacePort, isCombo)
    
    #记录未接入的端口信息
    if notCountDict:
        PY_LOGGER.info("Not connect Port:" + str(notCountDict))
    
    return True, ""
    
     
def getFcNegotiationSpeedByCliRet(cliInfo, maxSpeedDict, notCountDict, interFacePort, isCombo):
    '''
    @summary: 解析cli回显，获取各控制器各端口的协商速率
    '''
    for line in cliInfo.splitlines():
        #开始结束行不处理
        if "showfps" in line:
            continue
        if ">" in line:
            break
        
        lineInfo = line.split()
        if len(lineInfo) != 3:
            continue
        
        #根据协商速率换算实际总带宽
        ctrlID = lineInfo[0]
        
        #非数据项（标题行）
        if ctrlID not in CONTROLLER_ID_LIST:
            continue
        
        #对于COMBO卡，showfps的到的端口号与实际端口号存在0->1, 1->2的关系
        fcPort = lineInfo[1]
        if isCombo:
            fcPort = str(int(fcPort) + 1)
            
        #实际portID第一位为interface ID，第二位为fcportID     
        portID = interFacePort + fcPort
        speed = lineInfo[2]
        if speed == "0G":
            notCountDict[(ctrlID, portID)] = speed
            continue
        
        if speed == "2G":
            speed = DEF_SPEED_DICT.get("2GFC")
        elif speed == "4G":
            speed = DEF_SPEED_DICT.get("4GFC")
        elif speed == "8G":
            speed = DEF_SPEED_DICT.get("8GFC")
        else:
            notCountDict[(ctrlID, portID)] = speed
            continue
        
        maxSpeedDict[(ctrlID, portID)] = speed
        
    return maxSpeedDict, notCountDict
    
    
def getPortBandwidths(cli, lang):
    '''
    @summary: 获取各端口的使用带宽
    '''
    bandwidthsDict = {}
    
    cmd = "statperf -p"
    cliInfo = cliMgt.execCmd(cli, cmd)
    
    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        return False, cliInfo, bandwidthsDict, errInfo
    
    #获取端口的当前带宽信息
    getRet, bandwidthsDict = getPortBandwidthsByCliRet(cliInfo)
    if not getRet:
        errMsg = {"zh":u"\n无法获取端口的当前带宽",
               "en":"\nFailed to obtain the current bandwidth of the port."}
        return False, cliInfo, bandwidthsDict, errMsg.get(lang)
    
    return True, cliInfo, bandwidthsDict, ""
    
   
def getPortBandwidthsByCliRet(cliInfo):
    '''
    @summary: 根据cli回显获取各个端口的信息
    @return: bandwidthsDict: bandwidthsDict: {(controller, portId): speed, }
    '''
    bandwidthsDict = {}
    cliInfoList = handleCliInfo(cliInfo)
    #解析回显失败
    if not cliInfoList:
        return False, bandwidthsDict
    
    for lineDict in cliInfoList:
        ctrlID = lineDict.get("CtrlID", "")
        portID = lineDict.get("PortID", "")
        bandwidthsDict[(ctrlID, portID)] = int(lineDict.get("CBW(MB/s)", "0"))
        
    return True, bandwidthsDict
        
    
def handleCliInfo(cliInfo):
    '''
    @summary: 解析回文，获取各字段对应的值
    '''
    keyPosDict = {}
    cliInfoList = []
    
    for line in cliInfo.splitlines():
        cliInfoDict = {}
        #非数据信息
        if "statperf -p" in line:
            continue
        
        #标题行,确定各个字段的位置
        nextStart = 0
        if re.search("CtrlID", line, re.IGNORECASE):
            keys = line.split()
            for key in keys:
                keyStart = line.find(key)
                keyEnd = keyStart + len(key)
                keyPosDict[key] = (nextStart, keyEnd)
                nextStart = keyEnd
            continue
        
        #结束行
        if ">" in line:
            break
        
        #处理数据行,需要在解析完key后才能解析数据
        if keyPosDict:
            for key in keyPosDict:
                posInfo = keyPosDict.get(key)
                startPos = posInfo[0]
                endPos = posInfo[1]
                valueOfKey = line[startPos:endPos].strip()
                cliInfoDict[key] = valueOfKey
            
            #将解析后的信息添加到列表中
            cliInfoList.append(cliInfoDict)
        
    return cliInfoList

def getBandwidthsUsage(maxSpeedDict, bandwidthsDict):
    '''
    @summary: 根据当前带宽，和总带宽计算利用率
    @return: overflowDict={(contr, portID):[Threshold]}
    '''
    warningThreshold = 0.7
    notpassThreshold = 0.8
    overflowDict = {}
    
    for contrAndPort in maxSpeedDict:
        if contrAndPort not in bandwidthsDict:
            continue
        
        currentSpeed = bandwidthsDict.get(contrAndPort)
        maxSpeed = maxSpeedDict.get(contrAndPort)
        warningThresholdValue = decimal.Decimal(str(maxSpeed)) * decimal.Decimal(str(warningThreshold))
        notpassThresholdValue = decimal.Decimal(str(maxSpeed)) * decimal.Decimal(str(notpassThreshold))
        
        if currentSpeed > notpassThresholdValue:
            overflowDict[contrAndPort] = [notpassThreshold]
        elif currentSpeed > warningThresholdValue:
            overflowDict[contrAndPort] = [warningThreshold]
          
    return overflowDict
    
 
def mergeOverflowDict(lastOverflowAllDict, overflowDict):
    '''
    @summary: 合并本次与上次合并后的结果，得到连续超标的端口
    ''' 
    overflowAllDict = {}
    #首次合并，返回本次结果
    if not lastOverflowAllDict:
        return overflowDict
    
    #合并多次超标的端口
    for overflowPort in overflowDict:
        if overflowPort in lastOverflowAllDict:
            lastUsage = lastOverflowAllDict.get(overflowPort)
            usage = overflowDict.get(overflowPort)
            overflowAllDict[overflowPort] = lastUsage + usage 
    
    return overflowAllDict
    
    
def judgeCheckRet(overflowAllDict, lang):
    '''
    @summary: 根据最终的超标结果，判断检查项是建议优化还是不通过
    '''
    #最低级别应该是建议优化了
    judgeRet = constant.WARNING
    errMsg = ""
    errInfo = {"zh":u"\n控制器%s的端口%s的利用率连续三次超过阈值%i%%",
               "en":"\nOn controller %s, the usage of port %s exceed %i%% consecutively for three times.",}
    for overflowPort in overflowAllDict:
        contrId = overflowPort[0]
        portId =  overflowPort[1] 
        Threshold = 70
        usageList = overflowAllDict.get(overflowPort)
        usageList.sort()
        if usageList[0] >= 0.8:
            judgeRet =  constant.NOT_PASS
            Threshold = 80
           
        errMsg += errInfo.get(lang) % (contrId, portId, Threshold)
    
    return judgeRet, errMsg
    
    
    