#--*-- coding:utf-8 --*--
import traceback
import re
import time
import decimal
from common import utils
from common import cliMgt
from common import constant
from common.cBase import cFastTypeDict

DEF_SPEED_DICT = {"GE":100, "10GE":600, "8GFC":750, "4GFC":360, "2GFC":180, "SAS":2048, "TOE":700,}
INTERFACE_TYPE_LIST = ["4x4GFC", "2x4GFC", "4x8GFC", "2xGE", "2x10GE"]
CONTROLLER_ID_LIST = ["A", "a", "B", "b"]

def execute(cli):
    '''
    @summary: 检查端口利用率
    @param cli: cli链接对象
    '''
    cliRet = ""
    errMsg = ""
    needCloseSwitch = False
     
    try:
        lang = py_java_env.get("lang")
        #判断型号版本是否符合
        deviceType = str(py_java_env.get("devInfo").getDeviceType())
        sysVer,cmdRet = getDevVer(cli)
        PY_LOGGER.info("deviceType: " + str(deviceType) + ", Version: " + str(sysVer))
        #S6800E 1.03.03.530.T06 V100R005C02SPC010
        if deviceType.upper() != "S6800E" or sysVer < "1.03.03.530.T06":
            return (constant.NO_SUPPORT,cliRet,errMsg)
        
        #1. 执行showstatswitch命令查询系统性能统计状态并记录，未打开需要打开
        getFlag, cliInfo, stats, errInfo = getPortStat(cli, lang)
        cliRet += cliInfo
        if not getFlag:
            return getFlag, cliInfo, errInfo
        
        #2. 状态为关闭，则打开
        if re.match("OFF", stats, re.IGNORECASE):
            cliInfo = openSwitch(cli)
            needCloseSwitch = True
            cliRet += "\n" + cliInfo
            
        #3. 执行showinterface命令查询系统接口卡类型
        getFlag, cliInfo, typeDict, errInfo = getInterfaceType(cli, lang)
        PY_LOGGER.info("typeDict:" + str(typeDict))
        cliRet += "\n" + cliInfo
        if not getFlag:
            return False, cliRet, errInfo
        
        #4. 获取各端口对应的最大速率
        maxSpeedDict = getMaxSpeed(typeDict)                  
        PY_LOGGER.info("maxSpeedDict:" + str(maxSpeedDict))
        
        #5. statperf -p -c controller ID -t type -i Port查询每个端口的当前带宽Current Bandwidth(MB/s),最多需要查询3次
        lastOverflowAllDict = {}
        for loop in range(3):
            getFlag, cliInfo, bandwidthsDict, errInfo = getPortBandwidths(cli, typeDict, maxSpeedDict, lang)
            cliRet += "\n" + cliInfo
            if not getFlag:
                return False, cliRet, errInfo
            PY_LOGGER.info("bandwidthsDict:" + str(bandwidthsDict))
                
            #计算当前带宽利用率，并判断是否超过阈值
            overflowDict = getBandwidthsUsage(maxSpeedDict, bandwidthsDict)
            
            #任何一次的结果为空，则达不到3次超标，可直接返回
            if not overflowDict:
                return True, cliRet, ""
            
            #合并超标信息
            overflowAllDict = mergeOverflowDict(lastOverflowAllDict, overflowDict)
            lastOverflowAllDict = overflowAllDict
            #合并后，不存在信息，则表示没有连续超标的端口
            if not overflowAllDict:
                return True, cliRet,""
            
            #每次检查间隔10S, 最后一次不用休眠
            if loop != 2:
                time.sleep(10)
        
        PY_LOGGER.info("overflowAllDict:" + str(overflowAllDict)) 
           
        #6. 判断超标结果是警告还是不通过
        judgeRet, errInfo = judgeCheckRet(overflowAllDict, lang)
        return judgeRet, cliRet, errInfo
        
    except:
        PY_LOGGER.error("[except]" + traceback.format_exc())
        return False, cliRet, errMsg
    
    finally:
        #之前打开过开关，则需要关闭
        if needCloseSwitch:
            closeSwitch(cli)
    
    
def getPortStat(cli, lang):
    '''
    @summary: 获取端口性能状态
    '''
    cmd = "showstatswitch"
    cliInfo = cliMgt.execCmd(cli, cmd)
    

    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        return False, cliInfo, "", errInfo
    
    #获取端口状态
    getRet, portStats = getPortStatByCliRet(cliInfo)
    PY_LOGGER.info("portStats:" + str(portStats))
    if not getRet:
        errMsg = {"zh":u"\n无法获取系统性能统计状态",
               "en":"\nFailed to obtain the performance statistical status of the system."}
        return False, cliInfo, portStats, errMsg.get(lang)
    
    return True, cliInfo, portStats, ""


def getPortStatByCliRet(cliInfo):
    '''
    @summary: 获取端口性能统计状态
    '''
    stats = ""
    
    keyWords = "Statistic status |"
    for line in cliInfo.splitlines():
        if re.search(keyWords, line, re.IGNORECASE):
            statInfo = line.split("|")
            if len(statInfo) == 2:
                stats = statInfo[1].strip()
                return True, stats
            
    return False, stats
                

def openSwitch(cli):
    '''
    @summary: 执行命令打开系统端口性能统计开关
    '''         
    cmd = "chgstatswitch -s 1"
    cliInfo = cliMgt.execCmd(cli, cmd)
    return cliInfo


def closeSwitch(cli):
    '''
    @summary: 执行命令关闭系统端口性能统计开关
    '''         
    cmd = "chgstatswitch -s 0"
    cliInfo = cliMgt.execCmd(cli, cmd)
    return cliInfo


def getInterfaceType(cli, lang):
    '''
    @summary: 获取接口卡类型
    '''
    cmd = "showinterface"
    cliInfo = cliMgt.execCmd(cli, cmd)
    typeDict = {}
    
    #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
    preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
    if constant.RET_OK != preRet:
        errInfo = {"zh":u"\n无法获取系统接口卡信息",
                       "en":"\nFailed to obtain information about interface modules of the system.",}
        return False, cliInfo, typeDict, errInfo.get(lang)
    
    typeDict = getInterfaceTypeByCliRet(cliInfo)
    return True, cliInfo, typeDict, ""


def getInterfaceTypeByCliRet(cliInfo):
    '''
    @summary: 解析命令回文，获取端口类型
    @return: typeDict: {(controller, interface):type}
    '''
    '''
    admin:/>showinterface
  Controller ID    Interface ID    Type                       Status    PCB Version       Electronic Label  
  A                0               4 x 4G FC Port Module      Normal    ST91HF44 VER.A    --                
  A                1               2 x 4G FC Port Module      Normal    ST91HF42 VER.A    --                
  B                0               4 x 4G FC Port Module      Normal    ST91HF44 VER.A    --                
  B                1               2 x GE RJ45 Port Module    Normal    ST91HI12 VER.A    --                

admin:/>
    '''
    typePosStart = -1
    typePosEnd = -1
    typeKey = "Type"
    typeRe = re.compile(r"(.*?(FC|GE))", re.IGNORECASE)
    
    typeDict = {}
    for line in cliInfo.splitlines():
        lineInfo = line.split()
        #无用数据行
        if len(lineInfo) < 3:
            continue
        
        #标题行,找到type的位置信息
        if "Controller ID" in line:
            typePosStart = line.find(typeKey)
            typePosEnd = typePosStart + len(typeKey)
            while line[typePosEnd] == " ":
                typePosEnd += 1
            continue
            
        contr = lineInfo[0]
        interface = lineInfo[1]
        if typePosStart != -1 and typePosEnd != -1:
            typeInfo = line[typePosStart:typePosEnd]
            typeInfo = typeInfo.replace(" ", "")
            interfaceType = typeRe.search(typeInfo)
            if interfaceType:
                interfaceType = interfaceType.group(1)
                if interfaceType == "10GE":
                    interfaceType = "2x10GE"
            else:
                continue
            
        #控制器和type同时满足条件才算有效信息
        if contr in CONTROLLER_ID_LIST and\
           interfaceType in INTERFACE_TYPE_LIST:
            typeDict[(contr, interface)] = interfaceType
            continue
        
    return typeDict
            

def getMaxSpeed(typeDict):
    '''
    @summary: 根据类型，获取端口数和对应的总带宽
    '''
    maxSpeedDict = {}
    for contrAndPort in typeDict:
        controllerId = contrAndPort[0]
        interFacePort = contrAndPort[1]
        
        typeInfo = typeDict.get(contrAndPort)
        typeInfoSplit = typeInfo.split("x")
        portNumber = int(typeInfoSplit[0])
        interFaceType = typeInfoSplit[1]
        maxSpeed = DEF_SPEED_DICT.get(interFaceType)
        
        #设置各端口的最大速率
        for port in range(portNumber):
            portId = interFacePort + str(port)
            key = (controllerId, portId)
            maxSpeedDict[key] = maxSpeed
        
    return maxSpeedDict

          
def getPortBandwidths(cli, typeDict, maxSpeedDict, lang):
    '''
    @summary: 获取各端口的使用带宽
    @return: bandwidthsDict = {(contr, portID):bandwidths}
    '''
    bandwidthsDict = {}
    cliRet = ""
    #命令格式：statperf -p -c controller ID -t type -i Port
    cmdForm = "statperf -p -c %s -t %s -i %s" 
    
    for contrAndPort in maxSpeedDict:
        controllerId = contrAndPort[0]
        portId = contrAndPort[1]
        interFaceId = portId[0]
        keyWords = (controllerId, interFaceId)
        interFaceType = typeDict.get(keyWords)
        numberOfType = getTypeForNumber(interFaceType)
        
        cmd = cmdForm % (controllerId, numberOfType, portId)
        cliInfo = cliMgt.execCmd(cli, cmd)
        cliRet += cliInfo
        
        #判断CLI命令执行结果是否有效，有效才进行后续的逻辑判断
        preRet, errInfo = utils.preCheckCliRet(cmd, cliInfo, lang)
        if constant.RET_OK != preRet:
            return False, cliInfo, bandwidthsDict, errInfo

        #获取端口的当前带宽信息
        getFlag = getPortBandwidthsByCliRet(cliInfo, bandwidthsDict, contrAndPort)
        if not getFlag:
            errMsg = {"zh":u"\n无法获取端口的当前带宽",
                      "en":"\nFailed to obtain the current bandwidth of the port."}
            return False, cliInfo, bandwidthsDict, errMs.get(lang)
    
    return True, cliRet, bandwidthsDict, ""
    

def getTypeForNumber(interFaceType):
    '''
    @summary: 根据类型获取命令中type对应的数字
    '''
    numberOfTypeDict = {"SAS":"0", "FC":"1", "GE":"2"}
    
    if "FC" in interFaceType:
        numberOfType = numberOfTypeDict.get("FC")
    elif "GE" in interFaceType:
        numberOfType = numberOfTypeDict.get("GE")
        
    return numberOfType

   
def getPortBandwidthsByCliRet(cliInfo, bandwidthsDict, contrAndPort):
    '''
    @summary: 根据cli回显获取各个端口的信息
    @return: bandwidthsDict: bandwidthsDict: {(controller, portId): speed, }
    '''
    keyWords = "Current Bandwidth(MB/s)"
    cFTDObj = cFastTypeDict(cliInfo)
    cliInfoList = cFTDObj.handle()
    #解析回显失败
    if not cliInfoList:
        return False, bandwidthsDict
    
    
    for lineDict in cliInfoList:
        curSpeed = int(lineDict.get(keyWords))
        bandwidthsDict[contrAndPort] = curSpeed
        
    return True, bandwidthsDict
        

def getBandwidthsUsage(maxSpeedDict, bandwidthsDict):
    '''
    @summary: 根据当前带宽，和总带宽计算利用率是否超过阈值
    @return: overflowDict={(contr, portID):[Threshold]}
    '''
    warningThreshold = 0.7
    notpassThreshold = 0.8
    overflowDict = {}
    
    for contrAndPort in bandwidthsDict:
        if contrAndPort not in maxSpeedDict:
            continue
        
        notpassThresholdList = []
        warningThresholdList = []
        currentSpeed = bandwidthsDict.get(contrAndPort)
        maxSpeed = maxSpeedDict.get(contrAndPort)
        warningThresholdValue = decimal.Decimal(str(maxSpeed)) * decimal.Decimal(str(warningThreshold))
        notpassThresholdValue = decimal.Decimal(str(maxSpeed)) * decimal.Decimal(str(notpassThreshold))
        
        if currentSpeed > notpassThresholdValue:
            notpassThresholdList.append(notpassThreshold)
            overflowDict[contrAndPort] = notpassThresholdList
        elif currentSpeed > warningThresholdValue:
            warningThresholdList.append(warningThreshold)
            overflowDict[contrAndPort] = warningThresholdList
          
    return overflowDict
    
 
def mergeOverflowDict(lastOverflowAllDict, overflowDict):
    '''
    @summary: 合并本次与上次合并后的结果，得到连续超标的端口
    ''' 
    overflowAllDict = {}
    #首次合并，返回本次结果
    if not lastOverflowAllDict:
        return overflowDict
    
    #合并多次超标的端口
    for overflowPort in overflowDict:
        if overflowPort in lastOverflowAllDict:
            lastUsage = lastOverflowAllDict.get(overflowPort)
            usage = overflowDict.get(overflowPort)
            overflowAllDict[overflowPort] = lastUsage + usage 
    
    return overflowAllDict
    
    
def judgeCheckRet(overflowAllDict, lang):
    '''
    @summary: 根据最终的超标结果，判断检查项是建议优化还是不通过
    '''
    #最低级别应该是建议优化
    judgeRet = constant.WARNING
    errMsg = ""
    errInfo = {"zh":u"\n控制器%s的端口%s的利用率连续三次超过阈值%i%%",
               "en":"\nOn controller %s, the usage of port %s exceed %i%% consecutively for three times.",}
    for overflowPort in overflowAllDict:
        contrId = overflowPort[0]
        portId =  overflowPort[1] 
        Threshold = 70
        usageList = overflowAllDict.get(overflowPort)
        usageList.sort()
        if usageList[0] >= 0.8:
            judgeRet =  constant.NOT_PASS
            Threshold = 80
           
        errMsg += errInfo.get(lang) % (contrId, portId, Threshold)
    
    return judgeRet, errMsg
    