# -*- coding: UTF-8 -*-
import time
import traceback
from common import constant
from common.utils import *

def execute(cli):
    cliRet = ""
    errMsg = ""
    lang = py_java_env.get("lang")
    
    try:
        #判断型号版本是否符合
        deviceType = str(py_java_env.get("devInfo").getDeviceType())
        sysVer,cmdRet = getDevVer(cli)
        PY_LOGGER.info("deviceType: " + str(deviceType) + ", Version: " + str(sysVer))
        #S6800E 1.03.03.530.T06 V100R005C02SPC010
        #S2600  1.04.01.208.T10 V100R001C02SPC012
        if not (deviceType.upper() == "S6800E" and sysVer >= "1.03.03.525.T02") and\
            not (deviceType.upper() == "S2600" and sysVer >= "1.04.01.208.T10"):
            return (constant.NO_SUPPORT,cliRet,errMsg)
    
        #判断当前系统单双控 
        sysMode, cmdRet, errInfo = getSysModeState(cli, py_java_env)
        if sysMode == constant.MODE_UNKOWN:
            return (False, cmdRet, errInfo)
        
        isDouble = False
        if sysMode == constant.MODE_DOUBLE:
            isDouble = True
            
        #获取当前登录的控制器ID
        peerId = "B"
        checkRet, cmdRet, currentID, errInfo = getLocalCtrlID(cli,py_java_env)
        if not checkRet:
            return (checkRet, cmdRet, errInfo)
        
        if currentID == "B":
            peerId = "A"
        
        #1.执行showcache命令，获取High Water Level(%)的值。
        showcacheInfo = getShowCacheInfo(cli)
        cliRet += showcacheInfo
        if deviceType.upper() == "S6800E":
            highWaterLevelDict = getHighWaterLever(showcacheInfo)
        elif deviceType.upper() == "S2600":
            highWaterLevelDict = getHighWaterLever2600(showcacheInfo)
              
        #切换到mml，执行命令bp p,cm show,一共需要执行7次，每10秒一次
        curNodebppCmdInfoList = []
        peerNodebppCmdInfoList = []
        mmlcmd_bp_p = "bp p"
        mmlcmd_cm_show = "cm show"
        cmdList = [mmlcmd_bp_p, mmlcmd_cm_show]
        for times in range(7):
            checkRet = execMmlCmdLocalBefore(cli, cmdList, py_java_env, PY_LOGGER, isDouble)
            #中途执行命令失败，直接退出
            if checkRet[0] != True:
                cliRet += checkRet[1]
                errMsg += checkRet[2]
                return (False, cliRet, errMsg)
            
            cliRet += "\n" + checkRet[1]
            cliRet += "\n" + checkRet[2]
            curNodebppCmdInfoList.append(checkRet[1])
            peerNodebppCmdInfoList.append(checkRet[2])
            
            #每次检查睡10秒，最后一次检查了之后不用睡
            if times != 6:
                time.sleep(10)
                
        #将bp p回文组成字典，以便与showcache回文一一对应
        if isDouble:
            mmlInfoDict = {currentID:curNodebppCmdInfoList, peerId:peerNodebppCmdInfoList}
        else:
            mmlInfoDict = {currentID:curNodebppCmdInfoList}
                  
        #写页面阈值比较
        writeThreshold = 0.5
        thresholdTimesLimit = 5
        writeCheckRet, errInfo = judgeWriteAndQuota(highWaterLevelDict, mmlInfoDict, writeThreshold, thresholdTimesLimit)
        errMsg += errInfo[lang]
        
        #刷盘并发请求数阈值比较
        destagingReqThreshold = 0.9
        thresholdTimesLimit = 5
        destagingReqCheckRet, errInfo = judgeDestagingReq(mmlInfoDict, destagingReqThreshold, thresholdTimesLimit)
        errMsg += errInfo[lang]
        
        #刷盘并发页面数阈值比较
        destagingPageThreshold = 0.9
        thresholdTimesLimit = 5
        destagingPageCheckRet, errInfo = judgeDestagingPage(mmlInfoDict, destagingPageThreshold, thresholdTimesLimit)
        errMsg += errInfo[lang]
        
        #根据3个检查项的结果确定最终检查结果
        checkRet = writeCheckRet and destagingReqCheckRet and destagingPageCheckRet
        return(checkRet, cliRet, errMsg)
    
    except:
        PY_LOGGER.error("Inspect[AbormalReboot] catch except of trace back:" + str(traceback.format_exc()))
        return False, cliRet, getExceptionMsg(lang)
    
def judgeWriteAndQuota(highWaterLevelDict, bpPInfoDict, writeThreshold, thresholdTimes):
    """
    @summary: 解析bp p回文，获取write 和 write quota值,并判断是否超过阈值
    @param highWaterLevelDict: 高水位与控制器的关系字典
        bpPInfoDict：各控制器的多次bp p回文字典
        writeThreshold：写页面阈值
        ThresholdTimes： 判断次数
    @return: 
    """     
    flag = True
    errMsg = {"zh":"", "en":""}
    
    #遍历各个控制器
    for conctrlId in bpPInfoDict:
        
        #无法获取对应控制器的高水位值
        if conctrlId not in highWaterLevelDict:
            errMsg["zh"] += u"\n无法获取控制器[%s]的高水位值。" % conctrlId
            errMsg["en"] += "\nFailed to obtain the high watermark of controller [%s]. " % conctrlId
            flag = False
            continue
        
        highWaterLevel = highWaterLevelDict[conctrlId]
        
        #获取该控制器的bp p信息集合
        exceedTimes = 0
        bppCmdInfoList = bpPInfoDict[conctrlId]
        for bppInfo in bppCmdInfoList:
            checkRet = judgeWritePage(bppInfo, highWaterLevel, writeThreshold)
            if not checkRet:
                exceedTimes += 1
            #若超过阈值次数达到预设值，则记录信息，并退出该节点的检查
            if exceedTimes >= thresholdTimes:
                flag = False
                errMsg["zh"] += u"\n[控制器：%s]Cache 写页面超过阈值。" % conctrlId
                errMsg["en"] += "\nFor controller [%s], the number of cache write pages exceeds the threshold." % conctrlId
                break
    return (flag, errMsg)


def judgeWritePage(bppInfo, highWaterLevel, writeThreshold):
    """
    @summary: 判断每次bp p信息是否超过阈值
    @param bppInfo: 含bp p回显的回文
        highWaterLevel：高水位值
        writeThreshold：阈值
    """
    writeQuota = ""
    write = ""
    for line in  bppInfo.splitlines():
        lineSplit = line.split(",")
        for element in lineSplit:
            if element.strip().startswith("write quota("):
                writeQuota = element.strip().replace("write quota(", "").replace(")", "")
            #2600R1和6800ER5quota拼写有误
            if element.strip().startswith("write qouta("):
                writeQuota = element.strip().replace("write qouta(", "").replace(")", "")   
            if element.strip().startswith("write("):
                write = element.strip().replace("write(", "").replace(")", "")
            
        if writeQuota != "" and write != "":
            break
            
    #带入公式与阈值作比较
    highWaterLevelRatio = float(highWaterLevel) / 100.0
    writeRatio = float(write) / float(writeQuota if float(writeQuota) != 0.0 else '1')
    currentRatio = (writeRatio - highWaterLevelRatio) / (
        1.0 - highWaterLevelRatio if highWaterLevelRatio != 1.0 else 1.0)
    if currentRatio > writeThreshold:
        return False
    
    return True

def getShowCacheInfo(cli):
    """
    @summary: 执行命令shwocache获取回显
    """
    cmd = "showcache"
    cliRet = cliMgt.execCmd(cli, cmd)
    return cliRet

def getHighWaterLever(showcacheInfo):
    """
    @summary: 解析showcache回文，获取High Water Level(%)的值,与控制器对应
    """
    #用于返回以控制器为key，对应High Water Level(%)值为value的字典
    highWaterLevelDict={}
    
    #回文转换成字典列表
    splitChar = "|"
    showcacheInfoDictList = getVerticalCliRet(showcacheInfo, splitChar)
    
    #解析每个控制器的信息
    highWaterLevelKey = "High Water Level(%)"
    contrlIdKey = "Controller Id"
    for showcacheInfoDict in showcacheInfoDictList:
        if highWaterLevelKey in showcacheInfoDict and contrlIdKey in showcacheInfoDict:
            highWaterLevel = showcacheInfoDict[highWaterLevelKey]
            contrlId = showcacheInfoDict[contrlIdKey]
            highWaterLevelDict.setdefault(contrlId, highWaterLevel)
            
    return highWaterLevelDict 

def getHighWaterLever2600(showcacheInfo):
    """
    @summary: 解析showcache回文，获取High Water Level(%)的值,与控制器对应  ，s2600版本适用
    """
    #用于返回以控制器为key，对应High Water Level(%)值为value的字典
    highWaterLevelDict={}
    highWaterLevelKey = "High Water Level(%)"
    neetHighWaterLevel = False
    for line in showcacheInfo.splitlines():
        if line.strip().startswith("Controller"):
            controllerId = line.strip().split()[1]
            neetHighWaterLevel = True
        if line.strip().startswith(highWaterLevelKey) and neetHighWaterLevel:
            highWaterLevel = line.split("|")[1].strip()
            highWaterLevelDict[controllerId] = highWaterLevel
            neetHighWaterLevel = False
    
    return highWaterLevelDict

def judgeDestagingReq(mmlInfoDict, destagingReqThreshold, thresholdTimes):
    '''
    @summary: 判断当前刷盘并发请求数是否超过阈值
    @param mmlInfoDict: 包含cm show回文的信息、
    destagingReqThreshold：阈值
    thresholdTimes：超过次数限制
    ''' 
    flag = True
    errMsg = {"zh":"", "en":""}
    
    for conctrlId in mmlInfoDict:
        #获取该控制器的bp p信息集合
        exceedTimes = 0
        mmlCmdInfoList = mmlInfoDict[conctrlId]
        for cmshowInfo in mmlCmdInfoList:
            checkRet = judgeDestagingReqIsOk(cmshowInfo, destagingReqThreshold)
            if not checkRet:
                exceedTimes += 1
            #若超过阈值次数达到预设值，则记录信息，并退出该节点的检查
            if exceedTimes >= thresholdTimes:
                flag = False
                errMsg["zh"] += u"\n[控制器：%s]Cache 刷盘并发请求数超过阈值。" % conctrlId
                errMsg["en"] += "\nFor controller [%s], the number of Cache concurrent disk flushing requests exceeds the threshold." % conctrlId
                break
            
    return (flag, errMsg)

def judgeDestagingReqIsOk(cmshowInfo, destagingReqThreshold):
    '''
    @summary: 判断单次回文是否超过阈值
    @param cmshowInfo: 包含cm show回文的信息
    destagingReqThreshold：阈值
    '''       
    destagingReq = ""
    destagingReqMax = ""
    for line in  cmshowInfo.splitlines():
        if "current destaging req is" in line.strip().lower():
            destagingReq = line.strip().split()[-1]
        
        if "default max destaging req is set to" in line.strip().lower():
            destagingReqMax = line.strip().split()[-1]
        
        if destagingReq != "" and destagingReqMax != "":
            break
        
    #带入公式与阈值作比较
    currentRatio =  float(destagingReq) / float(destagingReqMax if float(destagingReqMax) != 0.0 else '1')
    if currentRatio > destagingReqThreshold:
        return False
    
    return True
    
def judgeDestagingPage(mmlInfoDict, destagingPageThreshold, thresholdTimes):
    '''
    @summary: 判断当前刷盘并发请求数是否超过阈值
    @param mmlInfoDict: 包含cm show回文的信息
    destagingPageThreshold：阈值
    thresholdTimes：次数限制值
    ''' 
    flag = True
    errMsg = {"zh":"", "en":""}
    
    for conctrlId in mmlInfoDict:
        #获取该控制器的bp p信息集合
        exceedTimes = 0
        mmlCmdInfoList = mmlInfoDict[conctrlId]
        for cmshowInfo in mmlCmdInfoList:
            checkRet = judgeDestagingPageIsOk(cmshowInfo, destagingPageThreshold)
            if not checkRet:
                exceedTimes += 1
            #若超过阈值次数达到预设值，则记录信息，并退出该节点的检查
            if exceedTimes >= thresholdTimes:
                flag = False
                errMsg["zh"] += u"\n[控制器：%s]Cache 刷盘并发页面数超过阈值。" % conctrlId
                errMsg["en"] += "\nFor controller [%s], the number of Cache concurrent disk flushing pages exceeds the threshold." % conctrlId
                break
            
    return (flag, errMsg)

def judgeDestagingPageIsOk(cmshowInfo, destagingPageThreshold):
    '''
    @summary: 判断单次回文是否超过阈值
    '''       
    destagingPage = ""
    destagingPageMax = ""
    for line in  cmshowInfo.splitlines():
        if "current destaging page nr is" in line.strip().lower():
            destagingPage = line.strip().split()[-1]
        
        if "default max destaging page nr is set to" in line.strip().lower():
            destagingPageMax = line.strip().split()[-1]
        
        if destagingPage != "" and destagingPageMax != "":
            break
        
    #带入公式与阈值作比较
    currentRatio =  float(destagingPage) / float(destagingPageMax if float(destagingPageMax) != 0.0 else '1')
    if currentRatio > destagingPageThreshold:
        return False
    
    return True   
        