# -*- coding: UTF-8 -*-
import os
import re
import time
import datetime
import traceback
from utils import *
import checkS26Alarm
from resource import Resource 
from defusedxml import ElementTree as ET

IS_CLEAR_5608_ALM = False

#界面显示项目间的间隔
UI_ITEMS_ALIGN = 5
UI_ID = 0
UI_TIME = 1
UI_NAME = 2
UI_DETAIL = 3

##告警级别定义
CRITICAL_ALM = "Critical"
MAJOR_ALM = "Major"
MINOR_ALM = "Minor"
WARNING_ALM = "Warning"
OTHER_ALM = "Other"

#获取没有恢复告警的字典参数
ALM_ID = 1
ALM_CSN = 2
ALM_TYPE = 3
ALM_LEVEL = 4
ALM_PARAM = 5
ALM_TIME = 6

#OSM版本curalarm.txt关键信息的位置
ALM_INFO_ID = 0
ALM_INFO_NAME = 1
ALM_INFO_DETAIL = 7

#xml关键信息键值 
ALM_XML_ID = 0
ALM_XML_NAME = 1
ALM_XML_DETAIL = 2
ALM_XML_DESC = 3
ALM_XML_SUGGEST = 4
ALM_XML_PARAM = 5
#以下为ISM版本告警文件对应的全局变量

#告警记录文件中的一条告警是568字节
ALM_RECORD_SIZE = 568

ALIGN_SIZE = 4

ALM_TYPE_SIZE = 4
ALM_ID_SIZE = 8
ALM_LEVEL_SIZE = 4
ALM_CSN_SIZE = 4
ALM_OCCUR_TIME_SIZE = 8
ALM_CLEAR_TIME_SIZE = 8
ALM_RECOVER_TIME_SIZE = 8
ALM_CLEAR_USER_SIZE = 256
ALM_PARAM_SIZE = 256
ALM_STATUS_SIZE = 4
ALM_FLAG_SIZE = 4

ALM_TYPE_OFFSET = 0
ALM_ID_OFFSET = ALM_TYPE_SIZE + ALIGN_SIZE
ALM_LEVEL_OFFSET = ALM_ID_SIZE + ALM_ID_OFFSET
ALM_CSN_OFFSET = ALM_LEVEL_SIZE + ALM_LEVEL_OFFSET
ALM_OCCUR_TIME_OFFSET = ALM_CSN_SIZE + ALM_CSN_OFFSET
ALM_CLEAR_TIME_OFFSET = ALM_OCCUR_TIME_SIZE + ALM_OCCUR_TIME_OFFSET
ALM_RECOVER_TIME_OFFSET = ALM_CLEAR_TIME_SIZE + ALM_CLEAR_TIME_OFFSET 
ALM_CLEAR_USER_OFFSET = ALM_RECOVER_TIME_SIZE + ALM_RECOVER_TIME_OFFSET
ALM_PARAM_OFFSET = ALM_CLEAR_USER_SIZE + ALM_CLEAR_USER_OFFSET
ALM_STATUS_OFFSET = ALM_PARAM_SIZE + ALM_PARAM_OFFSET
ALM_FLAG_OFFSET = ALM_STATUS_SIZE + ALM_STATUS_OFFSET

ALM_TYPE_IS_EVENT = 0 #事件
ALM_TYPE_IS_FAULT = 1 #故障
ALM_TYPE_IS_RESUME = 2 #恢复
ALM_TYPE_IS_INVALID = 3 #无效

ALM_UNRECOVER_STATUS = 1 #未清除未恢复
ALM_CLEAR_STATUS = 2 #已清除未恢复
ALM_RECOVERY_STATUS = 3 #未清除已恢复
ALM_CLEAR_RECOERY_STATUS = 4 #已清除已恢复
ALM_INVALID_STATUS = 5 #无效状态


DEVICE_OSM_ALM_PATH = "/OSM/log_conf_local/log/almlog/curalm/curalarm.txt"
DEVICE_ISM_ALM_PATH = "/OSM/log/cur_debug/alarm/alarmrecord.dat"

DEVICE_ISM_ALM_OLD_PATH = "/OSM/log_conf_local/log/cur_debug/alarm/alarmrecord.dat"

DEVICE_R5_ALM_PATH = "/OSM/alarm/alarmrecord.dat"

def _getAlmInfoDir(devModel, lang, isR5Version):
    
    fileName = ""
    
    if not devModel or DevModel.UnKnown == devModel:
        return ""
  
    sriptPath = os.path.split(os.path.realpath(__file__))[0]
    ALM_INFO_PATH = sriptPath.replace("common", "alminfo") + os.path.sep
    if isR5Version:
        fileName = "event_en_R5.xml"
    else:
        if "zh" == lang:
            fileName = "alminfo_en_" + devModel[0:5] + ".ini"
        else:
            fileName = "alminfo_en_" + devModel[0:5] + ".ini"
    
    return ALM_INFO_PATH + fileName

def _isDatFile(localCurAlmDir):
    """
    # *****************************************************************************************#
    # 函数名称: _isDatFile(localCurAlmDir)
    # 功能说明: 根据当前传入的当警文件，判断为dat还是txt文件
    # 输入参数: localCurAlmDir
    # 输出参数: True or False
    # *****************************************************************************************#
    """
    return localCurAlmDir.endswith(".dat")

def _isS2000Ver(devModel):
    """
    # *****************************************************************************************#
    # 函数名称: _isS2000Ver(devModel)
    # 功能说明: 判断设备型号是否为S2000
    # 输入参数: devModel（自定义DevModel类型）
    # 输出参数: True or False
    # *****************************************************************************************#
    """
    return devModel in [DevModel.S2000OSM, DevModel.S2000ISM]

def _isS2600Ver(devModel):
    """
    # *****************************************************************************************#
    # 函数名称: _isS2600Ver(devModel)
    # 功能说明: 判断设备型号是否为S2600
    # 输入参数: devModel（自定义DevModel类型）
    # 输出参数: True or False
    # *****************************************************************************************#
    """
    return devModel in [DevModel.S2600OLD, DevModel.S2600NEW]

def _isS5000Ver(devModel):
    """
    # *****************************************************************************************#
    # 函数名称: _isS5000Ver(devModel)
    # 功能说明: 判断设备型号是否为S5000
    # 输入参数: devModel（自定义DevModel类型）
    # 输出参数: True or False
    # *****************************************************************************************#
    """
    return devModel in [DevModel.S5000OSM, DevModel.S5000ISM]

def _isFileExist(py_java_env, devFilePath):
    """
    # *****************************************************************************************#
    # 函数名称: _isFileExist(py_java_env, devFilePath)
    # 功能说明: 判断阵列上面devFilePath文件是否存在
    # 输入参数: py_java_env, devFilePath
    # 输出参数: True or False
    # *****************************************************************************************#
    """
    return isDevFileExsit(py_java_env, os.path.dirname(devFilePath), os.path.basename(devFilePath)) 
   
def _trans2AlmLevel(almLevel):
    """
    # *****************************************************************************************#
    # 函数名称: _trans2AlmLevel(almLevel)
    # 功能说明: OSM版本级别转换为可识别的级别字符串
    # 输入参数: almLevel（整数类型）
    # 输出参数: curAlmLevel（字符串类型）
    # *****************************************************************************************#
    """
    if 1 == almLevel:
        curAlmLevel = CRITICAL_ALM
    elif 2 == almLevel:
        curAlmLevel = MAJOR_ALM 
    elif 3 == almLevel:
        curAlmLevel = MINOR_ALM 
    elif 4 == almLevel:
        curAlmLevel = WARNING_ALM 
    else:
        curAlmLevel = OTHER_ALM 
    return curAlmLevel
    
def _getUnClearFaultRecord(localFileDir, almLevel):
    """
    # *****************************************************************************************#
    # 函数名称: _getUnClearFaultRecord(localFileDir, almLevel)
    # 功能说明: OSM版本获取告警文件中相应级别的未恢复的告警信息
    # 输入参数: localFileDir, almLevel
    # 输出参数: almLstDict
    # *****************************************************************************************#
    """
    almLstDict = []
    almDict = {}
    
    f = open(localFileDir, "rb")
    for lineInfo in f:
        almDict = {}
        splitList = lineInfo.split("\t")
        if len(splitList) < 7:
            continue
        
        curAlmLevel = _trans2AlmLevel(int(splitList[ALM_LEVEL]))
        
        #取相应级别的告警信息 
        if str(almLevel) == curAlmLevel:
            almDict[ALM_ID] = splitList[ALM_ID]
            almDict[ALM_CSN] = splitList[ALM_CSN] 
            almDict[ALM_TYPE] = splitList[ALM_TYPE] 
            almDict[ALM_LEVEL] = curAlmLevel
            almDict[ALM_PARAM] = splitList[ALM_PARAM] 
            almDict[ALM_TIME] = splitList[ALM_TIME]
            almLstDict.append(almDict)   
            
    f.close()
    return almLstDict


def _getAlmDetail(almParm, detailMsg):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmDetail(almParm, detailMsg)
    # 功能说明: 根据上报的告警参数及告警配置文件的描述信息生成具体带参数的告警描述
    # 输入参数: almParm, detailMsg
    # 输出参数: detailMsg
    # *****************************************************************************************#
    """
    detailMsg = str(detailMsg)
    parmsList = almParm.split(",")
    if re.search("##\d+", detailMsg):
        for i in range(len(parmsList)):
            detailMsg = detailMsg.replace("##" + str(i).zfill(2), str(parmsList[i]).strip())
    else:
        for parm in parmsList:
            detailMsg = detailMsg.replace("#???", str(parm).strip(), 1)
    if not detailMsg.endswith("."):
        detailMsg += "."
    return detailMsg

def _getItemUnknownDetailMsg(almDictInfo):
    """
    # *****************************************************************************************#
    # 函数名称: _getItemUnknownDetailMsg(almDictInfo)
    # 功能说明: 无法识别出告警描述信息时，即ALM_ID无法匹配时记录name与description为None
    # 输入参数: almDictInfo
    # 输出参数: itemMsg
    # *****************************************************************************************#
    """
    itemMsg = []
    itemMsg.append(almDictInfo.get(ALM_ID).strip())
    itemMsg.append(almDictInfo.get(ALM_TIME).strip())
    itemMsg.append("None")
    itemMsg.append("None.")
    
    return itemMsg   
    
def _getItemDetailMsg(detailLine, almDictInfo):
    """
    # *****************************************************************************************#
    # 函数名称: _getItemDetailMsg(detailLine, almDictInfo)
    # 功能说明: 根据告警文件中某条记录对比alminfo_en.ini文档，生成具体的告警描述信息
    # 输入参数: detailLine, almDictInfo
    # 输出参数: itemMsg
    # *****************************************************************************************#
    """
    itemMsg = []
    lineSplit = detailLine.split("\t")
    itemMsg.append(almDictInfo.get(ALM_ID).strip())
    itemMsg.append(almDictInfo.get(ALM_TIME).strip())
    itemMsg.append(lineSplit[ALM_INFO_NAME].strip())
    itemMsg.append(_getAlmDetail(almDictInfo.get(ALM_PARAM), lineSplit[ALM_INFO_DETAIL].strip()))
    
    return itemMsg       

def _getItemXmlDetailMsg(almXmlDetail, almDictInfo, py_java_env):
    """
    # *****************************************************************************************#
    # 函数名称: _getItemXmlDetailMsg(detailLine, almDictInfo)
    # 功能说明: 根据告警文件中某条记录对比event_en.xml文档，生成具体的告警描述信息
    # 输入参数: almXmlDetail, almDictInfo
    # 输出参数: itemMsg
    # *****************************************************************************************#
    """
    itemMsg = []
    itemMsg.append(almDictInfo.get(ALM_ID).strip())
    itemMsg.append(almDictInfo.get(ALM_TIME).strip())
    itemMsg.append(almXmlDetail.get(ALM_XML_NAME).strip())
    # ，将参数中需要转换的值转换成实际值，2014/10/9 modified Begin
    xmlParamDict = almXmlDetail.get(ALM_XML_PARAM)
    alarmParam = almDictInfo.get(ALM_PARAM)
    
    parmsList = alarmParam.split(",")
    resource = py_java_env.get("resource")
    if not resource:
        resource = Resource(py_java_env)
        py_java_env["resource"] = resource
        
    for i in range(len(parmsList)):
        paramType = xmlParamDict.get(str(i))
        if paramType:
            val = resource.getString("EventParamDesc_" + paramType + "_" + parmsList[i])
            parmsList[i] = val
    alarmParam = ",".join(parmsList)
    itemMsg.append(_getAlmDetail(alarmParam, almXmlDetail.get(ALM_XML_DESC).strip()))
    # 将参数中需要转换的值转换成实际值，2014/10/9 modified End
    return itemMsg  

def _parserXmlInfo(xmlfilePath):
    """
    # *****************************************************************************************#
    # 函数名称: _parserXmlInfo(xmlfilePath)
    # 功能说明: 解析event_en文档，生成列表结构(##01,##02替换成#???)
    # 输入参数: xmlfilePath
    # 输出参数: nodeList
    # *****************************************************************************************#
    """
    root = ET.parse(xmlfilePath)
    param = root.findall('eventDefinition/param')
    nodeList = []
    for node in param: 
        nodeDict = {}
        nodeDict[ALM_XML_ID] = node.attrib.get("eventID")
        nodeDict[ALM_XML_NAME] = node.attrib.get("name")
        nodeDict[ALM_XML_DETAIL] = node.attrib.get("detail")
        nodeDict[ALM_XML_DESC] = node.attrib.get("description")
        nodeDict[ALM_XML_SUGGEST] = node.attrib.get("suggestion")
        # 解析配置文件中的paramDesc，2014/10/9 modified Begin
        paramDesc = node.attrib.get("paramDesc")
        paramDescDict = {}
        if paramDesc:
            items = paramDesc.split(",")
            for item in items:
                paramDescDict[item.split("=")[0]] = item.split("=")[1]
        nodeDict[ALM_XML_PARAM] = paramDescDict
        # ，解析配置文件中的paramDesc，2014/10/9 modified End
        nodeList.append(nodeDict)
    
    return nodeList
            
def _getAlmDetailMsg(curAlmDir, almInfoDir, almLevel, py_java_env):
    """
    # *****************************************************************************************#
    # 函数名称:  _getAlmDetailMsg(curAlmDir, almInfoDir, almLevel)
    # 功能说明: 对比alminfo_en.ini文档，生成所有告警的具体描述信息列表
    # 输入参数: almFileDir, almLevel
    # 输出参数: detailMsgList
    # *****************************************************************************************#
    """
    
    detailMsgList = []
    
    if _isDatFile(curAlmDir):
        almLstDict = _getUnClearFaultRecordFromDat(curAlmDir, almLevel)
    else:
        almLstDict = _getUnClearFaultRecord(curAlmDir, almLevel)
    
    if almLstDict:
        if not os.path.exists(almInfoDir):
            return []
        #xml解析告警文件版本
        if almInfoDir.endswith(".xml"):
            almXmlDetailLst = _parserXmlInfo(almInfoDir)
            for almDictInfo in almLstDict:
                for almXmlDetail in almXmlDetailLst:
                    if(almDictInfo.get(ALM_ID) == almXmlDetail.get(ALM_XML_ID)):
                        detailMsgList.append(_getItemXmlDetailMsg(almXmlDetail, almDictInfo, py_java_env))
                        break
                else:
                    detailMsgList.append(_getItemUnknownDetailMsg(almDictInfo))

        #ini解析告警文件版本
        else:
            for almDictInfo in almLstDict:
                #S2000R1，S5000R1及S5000R2版本需要清除此告警信息
                if IS_CLEAR_5608_ALM and "5608" == str(almDictInfo.get(ALM_ID)):
                    continue
                else:
                    for detailLine in open(almInfoDir, "rb"):
                        lineSplit = detailLine.split("\t")
                        if(almDictInfo.get(ALM_ID) == lineSplit[ALM_INFO_ID]):
                            detailMsgList.append(_getItemDetailMsg(detailLine, almDictInfo))
                            break
                    else:
                        detailMsgList.append(_getItemUnknownDetailMsg(almDictInfo))
    return detailMsgList

#ISM版本获取告警信息处理    
def _reverseStrOfHex2Dec(hexStr):
    """
    # *****************************************************************************************#
    # 函数名称: _reverseStrOfHex2Dec(hexStr)
    # 功能说明: 以两个字节为单位反转字符串（例如c717000000000000转换为00000000000017c7）
    # 输入参数: hexStr
    # 输出参数: rvsHexStr
    # *****************************************************************************************#
    """
    rvsHexStr = ""
    while len(hexStr) != len(rvsHexStr):
        rvsHexStr += hexStr[len(hexStr) - len(rvsHexStr) - 2:len(hexStr) - len(rvsHexStr)]
    return int("0x" + rvsHexStr, 16)

def _getAlmType(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmType(buffer)
    # 功能说明: ISM版本获取告警类型
    # 输入参数: buffer
    # 输出参数: 告警类型
    # *****************************************************************************************#
    """
    almType = buffer[ALM_TYPE_OFFSET:ALM_TYPE_OFFSET + ALM_TYPE_SIZE].encode("hex")
    return _reverseStrOfHex2Dec(almType)

def _getAlmId(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmId(buffer)
    # 功能说明: ISM版本获取告警ID
    # 输入参数: buffer
    # 输出参数: 告警ID
    # *****************************************************************************************#
    """
    almId = buffer[ALM_ID_OFFSET:ALM_ID_OFFSET + ALM_ID_SIZE].encode("hex")
    return _reverseStrOfHex2Dec(almId) 

def _getAlmCsn(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmCsn(buffer)
    # 功能说明: ISM版本获取告警流水号
    # 输入参数: buffer
    # 输出参数: 告警流水号
    # *****************************************************************************************#
    """
    almCsn = buffer[ALM_CSN_OFFSET:ALM_CSN_OFFSET + ALM_CSN_SIZE].encode("hex")
    return _reverseStrOfHex2Dec(almCsn) 
 
def _getAlmLevel(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmLevel(buffer)
    # 功能说明: ISM版本获取告警级别
    # 输入参数: buffer
    # 输出参数: 告警级别
    # *****************************************************************************************#
    """
    hexStrLevel = buffer[ALM_LEVEL_OFFSET:ALM_LEVEL_OFFSET + ALM_LEVEL_SIZE].encode("hex")
    level = _reverseStrOfHex2Dec(hexStrLevel)
    if 6 == level:
        almLevel = CRITICAL_ALM
    elif 5 == level:
        almLevel = MAJOR_ALM 
    elif 4 == level:
        almLevel = MINOR_ALM 
    elif 3 == level:
        almLevel = WARNING_ALM 
    else:
        almLevel = OTHER_ALM 
    return almLevel
 
def _getAlmParams(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmParams(buffer)
    # 功能说明: ISM版本获取告警参数
    # 输入参数: buffer
    # 输出参数: 告警参数
    # *****************************************************************************************#
    """
    return (str(buffer[ALM_PARAM_OFFSET:ALM_PARAM_OFFSET + ALM_PARAM_SIZE])).strip(chr(0).strip())

def _getAlmOccurTime(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmOccurTime(buffer)
    # 功能说明: ISM版本获取告警产生时间
    # 输入参数: buffer
    # 输出参数: 告警产生时间
    # *****************************************************************************************#
    """
    almOccurTime = buffer[ALM_OCCUR_TIME_OFFSET:ALM_OCCUR_TIME_OFFSET + ALM_OCCUR_TIME_SIZE].encode("hex")
    localTime = time.localtime(_reverseStrOfHex2Dec(almOccurTime))
    return time.strftime("%Y-%m-%d %H:%M:%S", localTime)

def _getAlmStatus(buffer):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmOccurTime(buffer)
    # 功能说明: ISM版本获取告警状态
    # 输入参数: buffer
    # 输出参数: 告警状态
    # *****************************************************************************************#
    """
    almStatus = buffer[ALM_STATUS_OFFSET:ALM_STATUS_OFFSET + ALM_STATUS_SIZE].encode("hex")
    return _reverseStrOfHex2Dec(almStatus) 

#为了统一后续的获取详细信息接口，这里返回值和_getUnClearFaultRecord保持一致
def _getUnClearFaultRecordFromDat(localFileDir, almLevel):
    """
    # *****************************************************************************************#
    # 函数名称: _getUnClearFaultRecord(localFileDir, almLevel)
    # 功能说明: ISM版本获取告警文件中相应级别的未恢复的告警信息
    # 输入参数: localFileDir, almLevel
    # 输出参数: almLstDict
    # *****************************************************************************************#
    """
    almLstDict = []
    almDict = {}
    
    f = open(localFileDir, "rb")
    n = 0
    
    while(True):
        
        buffer = f.read(ALM_RECORD_SIZE)
        if len(buffer) < ALM_RECORD_SIZE:
            break
        
        almType = _getAlmType(buffer)
        
        almCsn = _getAlmCsn(buffer)
        
        curAlmLevel = _getAlmLevel(buffer)
        
        almId = _getAlmId(buffer)
        
        almParm = _getAlmParams(buffer) 
        
        occurTime = _getAlmOccurTime(buffer)
        
        almStatus = _getAlmStatus(buffer)  
        
        almDict = {}
        
        if ALM_TYPE_IS_FAULT == almType and ALM_UNRECOVER_STATUS == almStatus\
            and almLevel == curAlmLevel:
            almDict[ALM_ID] = str(almId)
            almDict[ALM_CSN] = str(almCsn)
            almDict[ALM_TYPE] = str(almType)
            almDict[ALM_LEVEL] = str(curAlmLevel)
            almDict[ALM_PARAM] = str(almParm)
            almDict[ALM_TIME] = occurTime
            almLstDict.append(almDict)       
    f.close()    
    return almLstDict

def _createUiDisplayHead(maxLen):
    """
    # *****************************************************************************************#
    # 函数名称: _createUiDisplayHead(maxLen)
    # 功能说明: 创建界面显示的头部信息
    # 输入参数: maxLen
    # 输出参数: headInfo
    # *****************************************************************************************#
    """
    headInfo = ""
    headInfo += "ID".ljust(maxLen[UI_ID] + UI_ITEMS_ALIGN)
    headInfo += "Time".ljust(maxLen[UI_TIME] + UI_ITEMS_ALIGN)
    headInfo += "Name".ljust(maxLen[UI_NAME] + UI_ITEMS_ALIGN)
    headInfo += "Description".ljust(maxLen[UI_DETAIL] + UI_ITEMS_ALIGN)
    return headInfo
    
    
def _formatDetailMsgList(detailMsgList):
    """
    # *****************************************************************************************#
    # 函数名称: _formatDetailMsgList(detailMsgList)
    # 功能说明: 根据告警列表参数生成一行具体的显示
    # 输入参数: detailMsgList
    # 输出参数: uiDisplay（返回空表示无相应级别的信息）
    # *****************************************************************************************#
    """
    uiDisplay = ""
    maxLen = [0] * (len(detailMsgList[0]))
    
    for msgList in detailMsgList:
        for i in xrange(0, len(msgList)):
            if maxLen[i] < len(msgList[i]):
                maxLen[i] = len(msgList[i])

    for msgList in detailMsgList:
        for i in xrange(0, len(msgList)):
            uiDisplay += msgList[i].ljust(maxLen[i] + 5)
        uiDisplay += "\n"
    
    if uiDisplay:
        uiDisplay = _createUiDisplayHead(maxLen) + "\n" + uiDisplay
    
    return uiDisplay

def _getAlmUiDisplay(curAlmDir, almInfoDir, almLevel, py_java_env):
    """
    # *****************************************************************************************#
    # 函数名称: _getAlmUiDisplay(almFileDir, almLevel)
    # 功能说明: 生成相应级别未恢复告警的界面显示
    # 输入参数: almFileDir, almLevel
    # 输出参数: uiMsg（返回空表示无相应级别的信息）
    # *****************************************************************************************#
    """
    uiMsg = ""
    
    detailMsgList = _getAlmDetailMsg(curAlmDir, almInfoDir, almLevel, py_java_env)
    # ，过滤超过半年的告警，2014/10/9 modified Begin
    validDetailMsgList = []
    for detailMsg in detailMsgList:
        curAlarmDevTime = datetime.datetime.strptime(py_java_env.get("curAlarmDevTime")[:8], "%Y%m%d")
        alarmTime = datetime.datetime.strptime(detailMsg[1][:10], "%Y-%m-%d")
        if (curAlarmDevTime - alarmTime).days <= VALID_ALM_DAYS:
            validDetailMsgList.append(detailMsg)
    # ，过滤超过半年的告警，2014/10/9 modified End
    if not validDetailMsgList:
        return uiMsg
    
    return _formatDetailMsgList(validDetailMsgList)

def checkAlarmInfo(cli, py_java_env, almLevel, PY_LOGGER, isR5Version=False, isClear5608Alm=False):
    flag = True 
    global IS_CLEAR_5608_ALM
    IS_CLEAR_5608_ALM = isClear5608Alm
    cliRet = ""
    errMsg = ""
    lang = py_java_env.get("lang")
    
    try:
        # ，获取当前阵列时间，2014/10/9 modified Begin
        if not py_java_env.get("curAlarmDevTime"):
            #模式切换
            isSuccess, cliRet, errMsg = changeAnyMode2Debug(cli, py_java_env)
            if not isSuccess:
                PY_LOGGER.info("[checkAlarmInfo] not pass(change to debug failed)")
                return False, cliRet, errMsg
            curAlarmDevTime = getCurDevTime(cli)
            py_java_env["curAlarmDevTime"] = curAlarmDevTime
            if not changeAnyMode2Cli(cli):
                PY_LOGGER.error("Failed to change to CLI mode")
                if "zh" == lang:
                    errMsg = u"\n切换到cli模式失败。"
                else:
                    errMsg = u"\nFailed to switch the cli mode."
                return False, cliRet, errMsg
        # ，获取当前阵列时间，2014/10/9 modified End
        #CLI模式下获取当前为ISM还是OSM版本
        devModel, cliRet = getDevModel(cli, py_java_env)
        PY_LOGGER.info("Current devcive model is:" + str(devModel))
        if isR5Version:
            deviceAlmDir = DEVICE_R5_ALM_PATH 
        #S2600版本单独处理
        elif _isS2600Ver(devModel):
            return checkS26Alarm.checkS26AlarmInfo(cli, py_java_env, almLevel, PY_LOGGER)
        #S2000告警文件位置
        elif _isS2000Ver(devModel):
            deviceAlmDir = DEVICE_OSM_ALM_PATH
        #S5000版本告警文件ISM和OSM版本不同
        elif _isS5000Ver(devModel):
            deviceAlmDir = DEVICE_ISM_ALM_PATH if DevModel.S5000ISM == devModel else DEVICE_OSM_ALM_PATH
            if DEVICE_ISM_ALM_PATH == deviceAlmDir and not _isFileExist(py_java_env, deviceAlmDir):
                deviceAlmDir = DEVICE_ISM_ALM_OLD_PATH
        else:
            if "zh" == lang:
                errMsg = u"\n获取设备型号失败。"
            else:
                errMsg = u"\nFailed to obtain the device model."
            return False, cliRet, errMsg  
        
        #模式切换
        isSuccess, cliRet, errMsg = changeAnyMode2Debug(cli, py_java_env)
        if not isSuccess:
            PY_LOGGER.info("[checkAlarmInfo] not pass(change to debug failed)")
            return False, cliRet, errMsg
         
        #创建本地临时存放文件
        localTmpPath = getLocaTmplFilePath(py_java_env)
        mkLocalTmpPath(localTmpPath)
        
        PY_LOGGER.info("[checkAlarmInfo] localTmpPath is:" + str(localTmpPath)) 
        PY_LOGGER.info("[checkAlarmInfo] deviceAlmDir is:" + str(deviceAlmDir))
        
        #拷贝阵列文件到本地
        cliRet = ""
        localIp = getLocalIpAddr(py_java_env)
        mvSucc, curAlmDir = mvLocalDevFile2Local(cli, py_java_env, localIp , deviceAlmDir)
        PY_LOGGER.info("[checkAlarmInfo] get local curAlmDir is:" + str(curAlmDir))
        #本端文件获取成功开始分析
        if not mvSucc or (mvSucc and not os.path.exists(curAlmDir)):
            PY_LOGGER.info("[checkAlarmInfo] not pass(Get file failed)")
            if "zh" == lang:
                errMsg = u"\n获取告警文件（" + os.path.basename(deviceAlmDir) + u"）失败。"
            else:
                errMsg = u"\nFailed to obtain alarm files(" + os.path.basename(deviceAlmDir) + ")."
            return False, cliRet, errMsg
        
        almInfoDir = _getAlmInfoDir(devModel, lang, isR5Version)
        PY_LOGGER.info("[checkAlarmInfo] get alminfo.ini path is:" + str(almInfoDir))
        if not almInfoDir or not os.path.exists(almInfoDir):
            if isR5Version:
                almFile = "event.xml"
            else:
                almFile = "alminfo.ini"
            PY_LOGGER.info("[checkAlarmInfo] not pass(alminfo.ini not exist)")
            if "zh" == lang:
                errMsg = u"\n告警资源解析文件" + almFile + u"不存在。"
            else:
                errMsg = u"\nThe file of " + almFile + u" does not exist."
            return False, cliRet, errMsg
        
        uiDisplay = _getAlmUiDisplay(curAlmDir, almInfoDir, almLevel, py_java_env)
        if uiDisplay:
            return False, uiDisplay, errMsg 
        else:
            return True, "No " + almLevel + " alarm information.", errMsg 
    except:
        PY_LOGGER.error("[checkAlarmInfo] catch except of trace back:" + str(traceback.format_exc())) 
        if "zh" == lang:
            errMsg = u"\n告警文件解析异常。"
        else:
            errMsg = u"\nFailed to parse alarm files."
        return False, cliRet, errMsg
    finally:
        changeAnyMode2Cli(cli)
