# -*- coding: UTF-8 -*-
import os
import re
import time
import traceback
from utils import *
from sysStatus import *

RAISED_TIME = 0
ALM_NAM = 1
ALM_DESCRIPT = 2
UI_DISPLAY_KEY = ["Raised Time", "Alarm Name", "Description"]

UI_ITEMS_ALIGN = 5

ALM_FILE_SUFFIX = "_alarm.txt"


def _produceUiDict(lineInfo):
    """
    # *****************************************************************************************#
    # 函数名称: _produceUiDict(lineInfo)
    # 功能说明: 根据捕获到的告警记录，生成一个字典，字典包含的元素将会显示在界面上
    # 输入参数: lineInfo
    # 输出参数: lineDict（返回字典，元素信息包含UI_DISPLAY_KEY）
    # *****************************************************************************************#
    """
    lineDict = {}
    lineSplit = lineInfo.strip().split(";")
    for keyVal in lineSplit:
        index = keyVal.find(":")
        if - 1 != index:
            key = keyVal[0:index].strip() 
            val = keyVal[index + 1:].strip() 
            if key and key in UI_DISPLAY_KEY:
                lineDict[key] = val
    return lineDict
    
def _parseAlmFile(curAlmDir, almLevel, py_java_env):
    """
    # *****************************************************************************************#
    # 函数名称: _parseAlmFile(curAlmDir, almLevel)
    # 功能说明: 解析S26告警文件，生成一个list，其中每一个元素为一个字典
    # 输入参数: curAlmDir, almLevel
    # 输出参数: dictList（返回list，其中每一个元素为一个字典）
    # *****************************************************************************************#
    """
    almDictList = []
    
    f = open(curAlmDir, "rb")
    
    for lineInfo in f:
        if(-1 != lineInfo.find("Recovered Time:--") and - 1 != lineInfo.find("Cleared Time:--")\
            and - 1 != lineInfo.find("Alarm Level:" + almLevel)):
            lineDict = _produceUiDict(lineInfo) 
            if lineDict:
                #过滤超过半年的告警，2014/10/11 modified Begin
                items = lineDict.get("Raised Time").split()
                #Raised Time格式 为:%a %b %d %H:%M:%S %Y
                almYear = int(items[4])
                almMonth = int(MONTH_NAME_ABBRE.get(items[1]))
                almDay = int(items[2])
                alarmTime = datetime.datetime(almYear, almMonth, almDay)
                curAlarmDevTime = datetime.datetime.strptime(py_java_env.get("curAlarmDevTime")[:8], "%Y%m%d")
                if (curAlarmDevTime - alarmTime).days <= VALID_ALM_DAYS:
                #过滤超过半年的告警，2014/10/11 modified End
                    almDictList.append(lineDict)
    
    f.close()
    return almDictList

def _createUiDisplayHead(maxLen):
    """
    # *****************************************************************************************#
    # 函数名称: _createUiDisplayHead(maxLen)
    # 功能说明: 创建界面显示的头部信息
    # 输入参数: maxLen
    # 输出参数: headInfo
    # *****************************************************************************************#
    """
    headInfo = ""
    headInfo += "Time".ljust(maxLen[RAISED_TIME] + UI_ITEMS_ALIGN)
    headInfo += "Name".ljust(maxLen[ALM_NAM] + UI_ITEMS_ALIGN)
    headInfo += "Description".ljust(maxLen[ALM_DESCRIPT] + UI_ITEMS_ALIGN)
    return headInfo

def _formatalmDictList(almDictList):
    """
    # *****************************************************************************************#
    # 函数名称: _formatalmDictList(almDictList)
    # 功能说明: 根据告警列表参数生成一行具体的显示
    # 输入参数: almDictList
    # 输出参数: uiDisplay（返回空表示无相应级别的信息）
    # *****************************************************************************************#
    """
    uiDisplay = ""
    maxLen = [0] * (len(UI_DISPLAY_KEY))
    
    for lineDict in almDictList:
        for i in xrange(0, len(UI_DISPLAY_KEY)):
            if maxLen[i] < len(str(lineDict.get(UI_DISPLAY_KEY[i]))):
                maxLen[i] = len(str(lineDict.get(UI_DISPLAY_KEY[i])))
      
                
    for lineDict in almDictList:
        for i in xrange(0, len(UI_DISPLAY_KEY)):
            uiDisplay += str(lineDict.get(UI_DISPLAY_KEY[i])).ljust(maxLen[i] + 5)
        uiDisplay += "\n"
        
    if uiDisplay:
        uiDisplay = _createUiDisplayHead(maxLen) + "\n" + uiDisplay
    
    return uiDisplay

def checkS26AlarmInfo(cli, py_java_env, almLevel, PY_LOGGER):
    flag = True 
    cliRet = ""
    errMsg = ""
    lang = py_java_env.get("lang")
    sftp = py_java_env.get("sftp")
    
    try:
        #获取阵列临时文件目录信息
        timeStamp = formatSec2TimeStamp(int(py_java_env.get("startTime")))
        fileName = timeStamp + "_alarm.tar"
        almPath = "/OSM/"
        
        #获取本地告警文件绝对路径信息
        localIp = getLocalIpAddr(py_java_env)
        localTmpPath = getLocaTmplFilePath(py_java_env)
        curAlmDir = localTmpPath + localIp + "_" + fileName
        
        #告警文件不存在，需要下发命令重新下载获取
        if not os.path.exists(curAlmDir):
            pwd = str(py_java_env.get("devInfo").getLoginUser().getPassword())

            #下发命令生成告警文件
            cliRet = cliMgt.execCmdNoLog(cli, "exportalm -i 127.0.0.1 -u admin -p " + pwd + " -f " + fileName + " -t 1")
            
            if pwd:
                cliRet = cliRet.replace("-p " + pwd, "-p ******")
                
            #登录备控无法导出告警文件，2014/09/04 modified Begin
            if re.search("failed|error", cliRet, re.IGNORECASE):
                if "zh" == lang:
                    errMsg = u"\nexportalm命令执行失败。"
                else:
                    errMsg = u"\nThe command exportalm execution failed."
            
                #单用户产品无法在本端收集对端信息      
                if re.search("not.*master|not.*primary", cliRet, re.IGNORECASE):
                    if "zh" == lang:
                        errMsg += u"\n请尝试连接主控收集告警信息。"
                    else:
                        errMsg += u"\nPlease try to connect to the master controller to collect alarm information."
                return False, cliRet, errMsg
            #登录备控无法导出告警文件，2014/09/04 modified End
            
            isSuccess, cliRet, errMsg = changeAnyMode2Debug(cli, py_java_env)
            if not isSuccess:
                PY_LOGGER.info("[checkS26AlarmInfo] not pass(change to debug failed)")
                return False, cliRet, errMsg
             
            #创建本地临时存放文件
            mkLocalTmpPath(localTmpPath)
            
            #拷贝阵列文件到本地
            mvSucc, curAlmDir = mvLocalDevFile2Local(cli, py_java_env, localIp , almPath + fileName)
            PY_LOGGER.info("[checkS26AlarmInfo] Get new local curAlmDir is:" + str(curAlmDir))
            
            #本端文件获取成功开始分析
            if not mvSucc or (mvSucc and not os.path.exists(curAlmDir)):
                flag = False
                PY_LOGGER.error("[checkS26AlarmInfo] not pass(Get file failed)")
                if "zh" == lang:
                    errMsg = u"\n获取告警文件失败。"
                else:
                    errMsg = u"\nFailed to obtain alarm files."
                return False, cliRet, errMsg
            
            decompressTar(curAlmDir, localTmpPath)
            os.renames(localTmpPath + "local_alm_file.txt", curAlmDir.replace('.tar', '.txt'))
            curAlmDir = curAlmDir.replace('.tar', '.txt')
            time.sleep(20)
            
        PY_LOGGER.info("[checkS26AlarmInfo] Local curAlmDir is:" + str(curAlmDir))
        
        #解析本地文件
        almDictList = _parseAlmFile(curAlmDir, almLevel, py_java_env)
        if not almDictList:
            PY_LOGGER.info("[checkS26AlarmInfo] No alarm information.")
            return True, "No " + almLevel + " alarm information.", errMsg
        
        #获取告警界面显示
        uiDisplay = _formatalmDictList(almDictList)
        if uiDisplay:
            return False, uiDisplay, errMsg 
        else:
            return True, "No " + almLevel + " alarm information.", errMsg 
        
    except:
        PY_LOGGER.error("[checkS26AlarmInfo] catch except of trace back:" + str(traceback.format_exc())) 
        raise
    finally:
        changeAnyMode2Cli(cli)
        
        #删除阵列上本次导出生成的临时文件
        if isDevFileExsit(py_java_env, almPath, fileName):
            PY_LOGGER.info("[checkS26AlarmInfo] Delete device file: " + str(almPath + fileName))
            initSftp(sftp)
            sftp.deleteFile(almPath + fileName) 
