# -*- coding: UTF-8 -*-
import os
import re
import shutil

from com.huawei.ism.tool.obase.exception import ToolException

from common import getDeviceType
from common import getCurSystemVersion
from common import selectErrorMsg
import common
from comm.cTV1R1 import parseHorizontal
from comm import compressionManager

java_env = py_java_env
logger = PY_LOGGER

CTRL_BOARD_BUTTON_BATTERY_ALM_ID_STR = '0xE02040063'
CTRL_BOARD_BUTTON_BATTERY_TAG_DICT = {"S5500T":15,
                                      "S3900-M200":15,
                                      "S3900-M300":15,
                                      }


# **************************************************************************** #
# 函数名称: execute
# 功能说明: 纽扣电池告警
# 输入参数: ssh连接
# **************************************************************************** # 
def execute(cli):
    flag = True
    lang = java_env.get("lang")
    errMsg = ''

    try:
        (isQryOk, deviceType, cliRet, errMsg) = getDeviceType(cli, lang)
        if not isQryOk:
            logger.error('Query device type failed!')
            return False, cliRet, errMsg
        # 判断产品型号
        if deviceType not in CTRL_BOARD_BUTTON_BATTERY_TAG_DICT:
            return True, cliRet, errMsg

        isQrySucc, sysSpcVersion, sysRet, errMsg = getCurSystemVersion(
            cli, lang)
        cliRet += sysRet
        if not isQrySucc:
            logger.error('Query system version failed!')
            return False, cliRet, errMsg

        # 版本大于等于V100R005C02SPC300时执行CLI命令showalarm
        if sysSpcVersion >= 'V100R005C02SPC300':
            flag, almRet, almErrMsg = checkButtonBatteryAlm(cli)
            cliRet += '\n' + almRet
            return flag, cliRet, almErrMsg
        # 版本小于V100R005C02SPC300时收集日志检查
        else:
            flag, almRet, almErrMsg = checkButtonBatteryBit(cli, deviceType)
            cliRet += '\n' + almRet
            return flag, cliRet, almErrMsg
    except (ToolException, Exception) as e:
        logger.error(str(e))
        return (common.RESULT_NOCHECK, cliRet,
                common.getMsg(lang, "query.result.abnormal"))
    finally:
        common.reconnection_cli(cli, logger)
# *************************************************************#
# 函数名称: checkButtonBatteryAlm
# 功能说明: 检查纽扣电池故障告警
# 返回  :  （isErrExist, almRet, almErrMsg）
# *************************************************************#
def checkButtonBatteryAlm(cli):
    
    almRet = cli.execCmd('showalarm')
    errMsg = ''
    lang = java_env.get("lang")
    
    if len(almRet.splitlines()) == 3:
        if 'success' in almRet:
            logger.info('No alarm queried!')
            return True, almRet, errMsg
        else:
            logger.error('Query alarm failed!')
            zhErrMsg = u"\n查询系统告警失败。"
            enErrMsg = "\nFailed to query system alarm!"
            errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
            return False, almRet, errMsg
        
    almInfoList = parseHorizontal(almRet).getResult() 
    if [almInfo for almInfo in almInfoList if almInfo.get("Alarm Id") == CTRL_BOARD_BUTTON_BATTERY_ALM_ID_STR]:
        logger.error('Coin battery alarm exist!')
        zhErrMsg = u"\n系统存在纽扣电池故障告警。"
        enErrMsg = "\nCoin battery alarm exist."
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        return False, almRet, errMsg
    else:
        logger.info('Coin battery alarm does not exist!')
        return True, almRet, errMsg
        
# *************************************************************#
# 函数名称: checkButtonBatteryBit
# 功能说明: 检查mml文件的标志位
# 返回  :  （isQryBitOk, isBitExist, almRet, almErrMsg）
# *************************************************************#
def checkButtonBatteryBit(ssh, deviceType):
    #获取一键收集日志解压文件夹
    cliRet = ""
    errMsg = ""
    flag = True
    
    #获取一键收集日志解压文件夹
    flag, dataCollectRet, errMsg, deCompressDestDir = compressionManager.getDataCollectDecompressDir(java_env, ssh, logger)
    logger.info("deCompressDestDir: " + unicode(deCompressDestDir))
    cliRet += "\n" + dataCollectRet
    if not flag:
        return (flag, cliRet, errMsg)
    
    for folderName in os.listdir(deCompressDestDir):
        if not folderName.startswith("DebugLog"):
            flagitem, checkStr, errMsgItem = checkFaultByParsePangeaMml(deCompressDestDir, folderName, deviceType, logger)
            cliRet += "\n" + checkStr
            if not flagitem:
                flag = False
                errMsg += errMsgItem
                
    localTmpPath = java_env.get('dataCollectTmpDir')
    if localTmpPath and os.path.exists(localTmpPath):
        shutil.rmtree(localTmpPath, ignore_errors=True)
        java_env['dataCollectTmpDir'] = ''
        
    #检查结果返回
    return (flag, cliRet, errMsg) 



# **************************************************************************** #
# 函数名称: checkFaultByParsePangeaMml
# 功能说明: 解析pangea的mml文件，查看纽扣电池告警
# 输入参数: deCompressDestDir, folderName,  logger
# 输出参数: flag
# **************************************************************************** # 
def checkFaultByParsePangeaMml(deCompressDestDir, folderName, deviceType, logger):
    
    flag = True
    errMsg = ""
    lang = java_env.get("lang")
    checkStr = ""
     
    #获取ip信息
    controllerIp = folderName.replace("_MAIN", "")

    #通过控制器Ip查找mml文件位置
    mmlFilePath = ""
    for root, dirs, files in os.walk(deCompressDestDir + os.sep + folderName, False):
        if 'pangea_mmlinfo' in files:
            mmlFilePath = root + os.sep + 'pangea_mmlinfo'
            break
    #解压后找不到pangea_mmlinfo文件认为日志解压失败   Begin
    else:
        flag = False
        if lang == "zh":
            errMsg += u"\n控制器" + controllerIp + u"：日志解压失败。"
        else:
            errMsg += "\nController " + controllerIp + ": Decompressing the log file failed."
        return (flag, checkStr, errMsg)
    #解压后找不到pangea_mmlinfo文件认为日志解压失败   End
    
    logger.info("Getting mml file path of controller( Ip: " + controllerIp + ") is: " + mmlFilePath)
    
    #打开pangea_mmlinfo文件失败   Begin
    #解析mml信息文件
    fileInfo = ""
    try:
        file = open(mmlFilePath)
        fileInfo = file.read()
        file.close()
    except:
        flag = False
        if lang == "zh":
            errMsg += u"\n控制器" + controllerIp + u"：打开pangea_mmlinfo文件失败。"
        else:
            errMsg += "\nController " + controllerIp + ": Opening the pangea_mmlinfo file failed."
        return (flag, checkStr, errMsg)
    #打开pangea_mmlinfo文件失败   End
    
    mmlCmdRetList = fileInfo.split("MML>")
    for mmlCmdRet in mmlCmdRetList:
        if "board voltage alarm" in mmlCmdRet:
            checkStr = "\n[Controller " + controllerIp + "]: pangea_mmlinfo file content:\n" + mmlCmdRet + "\nMML>"
            lines = mmlCmdRet.splitlines()
            for line in lines:
                if "board voltage alarm" in line:
                    buttonBatteryInfo = line.split(":")[-1].strip().split()
                    index = CTRL_BOARD_BUTTON_BATTERY_TAG_DICT.get(deviceType)

                    if (len(buttonBatteryInfo) >= index + 1 
                        and buttonBatteryInfo[index] == '1'):
                        flag = False
                        if lang == "zh":
                            errMsg += u"\n控制器" + controllerIp + u"：纽扣电池发生故障。"
                        else:
                            errMsg += "\nController " + controllerIp + ": The coin battery is faulty."
                    break
            return (flag, checkStr, errMsg)
    else:
        return True, "", ""
    
    return (flag, checkStr, errMsg)
