import traceback
import common
from common import selectErrorMsg
from common import getParseExceptErrMsg
from comm.cTV1R1 import parseHorizontal
from comm.cTV1R1 import ParseException

INFO_INVALID = 0

IS_WD_DISK = 1
NOT_WD_DISK = 2

IS_SATA_DISK = 1
NOT_SATA_DISK = 2

WARNING = 'WARNING'

WD_SATA_DISK = 0
WD_NOSATA_DISK = 1
NOWD_SATA_DISK = 2
NOWD_NOSATA_DISK = 3

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 西数SATA盘长连接问题
# 输入参数: ssh连接
# **************************************************************************** # 
def execute(ssh):
    errMsg = ""
    lang = py_java_env.get("lang")
    
    softPathRetDict = {"V100R002C00SPCh00":"V100R002C00SPHh03",
                       "V100R002C01SPCh00":"V100R002C01SPHh01",
                       "V100R005C00SPC700":"V100R005C00SPH701",
                       "V100R005C01SPC700":"V100R005C01SPH701"}
    
    PY_LOGGER.info("[WDSATADisk]Inspect begin...")
    
    #获取软件版本
    isQryOk, curSysVer, cliRetVer, errMsgVer = common.getCurSystemVersion(ssh, lang)
    cliRet = cliRetVer   
    if not isQryOk:
        PY_LOGGER.error("[WDSATADisk]Inspect result is not pass(curSysVer error)")
        return (False, cliRet, errMsgVer)
    
    #不存在此问题的版本，直接巡检通过
    if _isNoRiskVer(curSysVer) and curSysVer not in softPathRetDict:
        PY_LOGGER.info("[WDSATADisk]Inspect result is pass(%s)" % curSysVer)
        return (True, cliRet, "")
    
    #获取补丁版本号
    isQryOk, curHotPatchVer, cliRetPatch, errMsgPatch = common.getCurHotPatchVersion(ssh, lang)
    cliRet += '\n' + cliRetPatch
    if not isQryOk:
        PY_LOGGER.error("[WDSATADisk]Inspect result is not pass(curHotPatchVer error)")
        return (False, cliRet, errMsgPatch)
    
    PY_LOGGER.info("[WDSATADisk]curSysVer is:%s, curHotPatchVer is %s:" % (curSysVer, curHotPatchVer))
    
    #安装了热补丁，巡检通过
    if curSysVer in softPathRetDict and _isMoreThanPatch(curSysVer, curHotPatchVer, softPathRetDict.get(curSysVer)):
        PY_LOGGER.info("[WDSATADisk]Inspect result is pass(right patch)")
        return (True, cliRet, "")
    
    #查看系统中是否存在西数SATA硬盘
    diskCliRet = ssh.execCmd("showdisk -physic")
    cliRet += '\n' + diskCliRet
    
    #判断cli信息是否有效
    if not common.checkCliInfoValid(diskCliRet, True):
        if lang == "zh":
            errMsg = u"\nCLI信息无效。"
        else:
            errMsg = "\nInvalid CLI information."   
        PY_LOGGER.error("[WDSATADisk]Inspect result is not pass(showdisk invalid)")      
        return (False, cliRet, errMsg)
    
    wdSataList = []
    wdNoSataList = []
    noWdSataList = []
    noWdNoSataList = []
    
    try:
        listDict = parseHorizontal(diskCliRet).getResult()
    except ParseException:
        PY_LOGGER.error("[WDSATADisk]ParseException: " + unicode(traceback.format_exc()))
        return(False, cliRet, getParseExceptErrMsg(lang))
    
    for dictInfo in listDict:
        diskID = dictInfo.get('Disk Location')
        diskVendor = _isDiskVendorWD(dictInfo.get('Vendor', ' '))
        diskType = _isDiskTypeSata(dictInfo.get('Type', ' '))
        
        if IS_WD_DISK == diskVendor and IS_SATA_DISK == diskType:
            wdSataList.append(diskID)
            
        #是西数盘，但不确定是否为SATA硬盘
        elif diskVendor == IS_WD_DISK and diskType == INFO_INVALID:
            wdNoSataList.append(diskID)
            
        #是SATA盘，但不确定是否为西数盘
        elif diskVendor == INFO_INVALID and diskType == IS_SATA_DISK:
            noWdSataList.append(diskID)
        
        #既不确定是否为西数盘，也不确定是否为SATA盘
        elif diskVendor == INFO_INVALID and diskType == INFO_INVALID:
            noWdNoSataList.append(diskID)
            
        #不是西数SATA硬盘
        else:
            continue

    PY_LOGGER.info("[WDSATADisk]wdSataList is:" + unicode(wdSataList))
    PY_LOGGER.info("[WDSATADisk]wdNoSataList is:" + unicode(wdNoSataList))
    PY_LOGGER.info("[WDSATADisk]noWdSataList is:" + unicode(noWdSataList))
    PY_LOGGER.info("[WDSATADisk]noWdNoSataList is:" + unicode(noWdNoSataList))
    
    nosureWdSataErrMsg = _getErrMsg(wdNoSataList, WD_NOSATA_DISK, lang)
    nosureWdSataErrMsg += _getErrMsg(noWdSataList, NOWD_SATA_DISK, lang)
    nosureWdSataErrMsg += _getErrMsg(noWdNoSataList, NOWD_NOSATA_DISK, lang)
      
    #存在西数SATA盘的情况
    if wdSataList:
        errMsg = _getErrMsg(wdSataList, WD_SATA_DISK, lang)
        
        if curSysVer not in softPathRetDict:
            lowSoftErrMsg = selectErrorMsg(lang, u"\n系统软件版本过低。", \
                                                 "\nThe system software version is too low.")
            PY_LOGGER.warn("[WDSATADisk]Inspect result is warning(no patch)")
            return (WARNING, cliRet, lowSoftErrMsg + errMsg + nosureWdSataErrMsg)
        #没有安装热补丁或是热补丁版本过低的情况
        else:
            rightPatchVer = softPathRetDict.get(curSysVer)
            noOrLowPathErrMsg = selectErrorMsg(lang, u"\n系统未安装%s或之后版本的热补丁。" % (rightPatchVer), \
                                                    "\nHot patch %s or later has not been installed." % (rightPatchVer))
            PY_LOGGER.warn("[SATADisk]Inspect result is warning(low patch)")
            return (WARNING, cliRet, noOrLowPathErrMsg + errMsg + nosureWdSataErrMsg)
    
    #只存在疑似西数SATA盘的情况
    elif nosureWdSataErrMsg:
        PY_LOGGER.warn("[WDSATADisk]Inspect result is pass(maybe WD SATA disk)")
        return (WARNING, cliRet, nosureWdSataErrMsg)
    
    #不存在西数SATA盘的情况
    else:
        PY_LOGGER.info("[WDSATADisk]Inspect result is pass(no WD SATA disk)")
        return(True, cliRet, "")
    
# *************************************************************#
# 函数名称: _getErrMsg(diskLst, errType, lang)
# 功能说明: 获取提示信息
# 返回  :  提示信息
# *************************************************************#
def _getErrMsg(diskLst, errType, lang):
    if not diskLst:
        return ""
    
    if WD_SATA_DISK == errType:
        return selectErrorMsg(lang, u"\n西数SATA硬盘：%s在业务压力过大的时候，可能会导致IO响应慢。" % u'，'.join(diskLst), \
                                    "\nHeavy service loads may cause slow I/O responses on SATA disks:%s from Western Digital." % u','.join(diskLst),)
    elif WD_NOSATA_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s类型无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                     "\nType of disks:%s is invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    elif NOWD_SATA_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s厂商无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                    "\nVendor of disks:%s is invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    elif NOWD_NOSATA_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s类型与厂商无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                    "\nType and vendor of disks:%s are invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    return ""

# *************************************************************#
# 函数名称: isDiskVendorWD
# 功能说明: 检查硬盘厂商是否为西数
# 返回  :  0：信息无效      1：是西数硬盘         2：不是西数硬盘
# *************************************************************#
def _isDiskVendorWD(diskVendorInfo):
    if not re.findall('[a-zA-Z]', diskVendorInfo):
        return INFO_INVALID
    if 'WD' == diskVendorInfo.strip().upper():
        return IS_WD_DISK
    else:
        return NOT_WD_DISK
    
# *************************************************************#
# 函数名称: _isDiskTypeSata
# 功能说明: 检查硬盘类型是否为SATA
# 返回  :  0：信息无效      1：是SAS硬盘         2：不是SAS硬盘
# *************************************************************#
def _isDiskTypeSata(diskTypeInfo):
    if not re.findall('[a-zA-Z]', diskTypeInfo):
        return INFO_INVALID
    if 'SATA' == diskTypeInfo.strip().upper():
        return IS_SATA_DISK
    else:
        return NOT_SATA_DISK
            
# **************************************************************************** #
# 函数名称: _isMoreThanPatch(curSysVer, curHotPatchVer, baseHotPatchVer)
# 功能说明: 判断当前补丁版本curHotPatchVer是否高于或是等于baseHotPatchVer
# 输入参数: True  or False
# **************************************************************************** # 
def _isMoreThanPatch(curSysVer, curHotPatchVer, baseHotPatchVer):
    baseSphStr = curSysVer.replace("SPC", "SPH")
    
    #不是配套的补丁，直接返回
    if baseSphStr[:-1] not in curHotPatchVer:
        return False
    return curHotPatchVer >= baseHotPatchVer

# **************************************************************************** #
# 函数名称: _isNoRiskVer(curSysVer)
# 功能说明: 判断当前版本是否存在此风险问题
# 输入参数: True or False
# **************************************************************************** # 
def _isNoRiskVer(curSysVer): 
    #V100R001所有版本
    if "V100R001" in curSysVer:
        return False
    #V100R002C00SPCg00及之前的版本
    elif "V100R002C00" in curSysVer and curSysVer <= "V100R002C00SPCg00":
        return False
    #V100R002C01SPCg00及之前的版本
    elif "V100R002C01" in curSysVer and curSysVer <= "V100R002C01SPCg00":
        return False
    #V100R005C00SPC600及之前的版本
    elif "V100R005C00" in curSysVer and curSysVer <= "V100R005C00SPC600":
        return False
    #V100R005C01SPC600及之前的版本 
    elif "V100R005C01" in curSysVer and curSysVer <= "V100R005C01SPC600":
        return False
    else:
        return True
