# -*- coding: UTF-8 -*-

import traceback
import common
from comm import modelManager
from comm.cTV1R1 import cHandleCliRet
from comm.cTV1R1 import ParseException


CHECK_RESULT_NOSUPPORT = "NOSUPPORT"
CHECK_RESULT_WARNING = "WARNING"

EXECUTING = "Executing"

PERIOD_AND_SPEED_DICT = {
                      "300GB":{
                               "100P":(6,1),
                               "200P":(8,1),
                               "300P":(8,2),
                               "400P":(9,2),
                               "500P":(9,3),
                               "600P":(10,3),
                               "700P":(12,3),
                               },
                      
                      "600GB":{
                               "100P":(8,1),
                               "200P":(8,2),
                               "300P":(9,3),
                               "400P":(9,4),
                               "500P":(12,4),
                               "600P":(14,4),
                               "700P":(16,4),
                               },
                      
                      "900GB":{
                               "100P":(6,2),
                               "200P":(8,3),
                               "300P":(10,4),
                               "400P":(13,4),
                               "500P":(16,4),
                               "600P":(19,4),
                               "700P":(22,4),
                               },
                      
                      "1000GB":{
                               "100P":(13,1),
                               "200P":(13,2),
                               "300P":(14,3),
                               "400P":(14,4),
                               "500P":(17,4),
                               "600P":(21,4),
                               "700P":(24,4),
                               },
                      
                      "2000GB":{
                               "100P":(13,2),
                               "200P":(17,3),
                               "300P":(19,4),
                               "400P":(26,4),
                               "500P":(30,4),
                               "600P":(30,4),
                               "700P":(30,4),
                               },
                      
                      "3000GB":{
                               "100P":(13,3),
                               "200P":(19,4),
                               "300P":(28,4),
                               "400P":(30,4),
                               "500P":(30,4),
                               "600P":(30,4),
                               "700P":(30,4),
                               },
                      
                      "4000GB":{
                               "100P":(13,4),
                               "200P":(25,4),
                               "300P":(30,4),
                               "400P":(30,4),
                               "500P":(30,4),
                               "600P":(30,4),
                               "700P":(30,4),
                               },
                      }
# **************************************************************************** #
# 函数名称: execute
# 功能说明: 检查后台扫描策略
# 输入参数: cli
# 输出参数: 无
# 返 回 值: flag, cliRet, errMsg
# **************************************************************************** # 
def execute(cli):
    
    errMsg = ""
    cliRet = ""
    lang = py_java_env.get("lang")
    needCheckLowVersion = "V100R005C00SPC003"
    needCheckHighVersion = "V100R005C30SPC200"

    #1 查询当前设备版本
    checkRet, sysSpcVersion, systemVerInfo, errInfo = common.getCurSystemVersion(cli, lang)
    cliRet += systemVerInfo
    if not checkRet:
        errMsg = errInfo
        return (False, cliRet, errMsg)
    
    #V100R005C00SPC003(包含)到V100R005C30SPC200(包含)才查，其他版本不需要查询
    if sysSpcVersion < needCheckLowVersion or sysSpcVersion > needCheckHighVersion:
        return (True, cliRet, errMsg)
    
    #2 获取硬盘最大单盘容量值，及硬盘总数
    diskCliRet = getDiskinfo(cli)
    cliRet += "\n" + diskCliRet
    checkRet, maxCapacity, diskNumSum = getMaxCapacityAndDiskNumSum(diskCliRet)
    if not checkRet:
        if lang =="zh":
            errMsg = u"\n无法获取设备中硬盘数量及单块硬盘容量中的最大值。"
        else:
            errMsg = "\nThe number of disks on the device and the maximum capacity of a single disk cannot be obtained."
        return (False, cliRet, errMsg)
    
    #3 查询后台扫描状态
    checkRet, mediaScanInfo = getMediaScanInfo(cli, py_java_env)
    cliRet += "\n" + mediaScanInfo
    if not checkRet:
        if lang == "zh":
            errMsg += (u"\n进入developer模式失败，失败的原因可能为：" 
                       + u"\n（1）添加设备时未输入调试密码。"
                       + u"\n（2）添加设备时输入的调试密码无效。")
        else:
            errMsg += ("\nLog in to developer model failed, the reason of failure may be:" 
                       + "\n(1) Did not enter a debug password when adding the device."
                       + "\n(2) The debug password entered is incorrect.")
        return (False, cliRet, errMsg)
    
    #获取后台扫描状态
    checkRet, status = getMediaScanStatus(mediaScanInfo)
    if not checkRet:
        if lang == "zh":
            errMsg = u"\n无法获取后台扫描状态。"
        else:
            errMsg = "\nThe background scanning status cannot be obtained."
        return (False, cliRet, errMsg)
    
    #未开启，则通过
    if status != EXECUTING:
        return (True, cliRet, "")
    
    #开启状态，则获取后台扫描周期和速率
    checkRet, period, speed = getMediaScanPeriodAndSpeed(mediaScanInfo)
    if not checkRet:
        if lang == "zh":
            errMsg = u"\n获取后台扫描周期和扫描速率失败。"
        else:
            errMsg = "\nFailed to obtain the background disk scanning period and rate."
        return (False, cliRet, errMsg) 
    
    periodNum = int(float(period))
    speedNum = int(float(speed))
    
    #4 根据实际最大容量值和硬盘总数，获取应该设置的扫描周期和扫描速率
    needSetPeriod, needSetSpeed = getNeedSetPeriodAndSpeed(maxCapacity, diskNumSum)
    
    #5 检查周期和速率是否匹配
    if periodNum != needSetPeriod or speedNum != needSetSpeed:
        if lang == "zh":
            errMsg = (u"\n当前后台扫描状态为：%s" % status
                      + u"\n当前后台扫描周期为：%s，扫描速率为：%s，需要优化。" % (period, speed)
                      + u"\n请参考如下步骤进行后台扫描策略修改："
                      + u"\n1 在admin模式下执行命令：developer，进入developer模式。"
                      + u"\n2 若后台扫描是开启状态，则执行命令：stopmediascan，暂停后台扫描。"
                      + u"\n3 执行命令：startmediascan -period %i -speed %i，修改后台扫描策略并开启后台扫描。" % (needSetPeriod, needSetSpeed)
                      + u"\n4 执行命令：showmediascan，确认修改是否成功。"
                      + u"\n5 如果有任何疑问，请联系技术支持工程师协助解决。")
        else:
            errMsg = ("\nThe current background scanning status is %s. " % status
                      + u"\nThe scanning period is %s. The scanning rate is %s. An optimization is required." % (period, speed)
                      + u"\nChange the background scanning policy in accordance with the following steps: "
                      + u"\n1 Run the developer command in the admin mode to switch to the developer mode."
                      + u"\n2 If the background scanning function is enabled, run the stopmediascan command to stop the background disk scanning."
                      + u"\n3 Run the startmediascan -period %i -speed %i command to modify the background scanning policy and enable the background disk scanning function. " % (needSetPeriod, needSetSpeed)
                      + u"\n4 Run the showmediascan command to verify that the policy is successfully modified. "
                      + u"\n5 If you have any questions, please contact technical support engineers for further handling.")
        return (CHECK_RESULT_WARNING, cliRet, errMsg)
    
    return (True, cliRet, errMsg)
                      

def getDiskinfo(cli):
    """
    @summary: 获取硬盘信息
    """
    diskCliRet = cli.execCmd("showdisk -physic")    
    return diskCliRet
     
def getMaxCapacityAndDiskNumSum(diskCliRet):
    '''
    @summary: 获取最大单盘容量和硬盘总数
    '''
    maxCapacity = 0
    diskNumSum = 0
    transformedCapacity = 0
    
    try:
        listDict = cHandleCliRet(diskCliRet).getResult()
    except ParseException:
        PY_LOGGER.error("[getMaxCapacityAndDiskNumSum]ParseException: " + unicode(traceback.format_exc()))
        return(False, maxCapacity, diskNumSum)
    
    diskNumSum = len(listDict)
    
    for diskinfo in listDict:
        #经cHandleCliRet处理后各字段命令是驼峰格式，与实际的可能不一样
        Capacity = diskinfo.get('Raw Capacity(Gb)')
        transformedCapacity = transformCapacity(Capacity)
        if transformedCapacity >= maxCapacity:
            maxCapacity = transformedCapacity
            
    return (True, maxCapacity, diskNumSum)

def handleCapacity(Capacity):
    '''
    @summary: 根据当前容量值，取最近整百倍数值。
    @param Capacity: 最大单盘容量，单位GB
    @return targetCapacity:处理误差后的容量值Key。300GB，600GB，900GB，1000GB,  2000GB ,3000GB, 4000GB 
    '''
    
    targetCapacityKey = ""
    if Capacity < 450:
        targetCapacityKey = "300GB"
    if 450 <= Capacity < 750:
        targetCapacityKey = "600GB"
    if 750 <= Capacity < 950:   
        targetCapacityKey = "900GB"
    if 950 <= Capacity < 1500: 
        targetCapacityKey = "1000GB"
    if 1500 <= Capacity < 2500:
        targetCapacityKey = "2000GB"
    if 2500 <= Capacity < 3500:   
        targetCapacityKey = "3000GB"
    if 3500 <= Capacity:
        targetCapacityKey = "4000GB"
    return targetCapacityKey

def handleDiskNumSum(diskNumSum):
    '''
    @summary: 根据硬盘数量确定数量取值
    @param diskNumSum:硬盘总数,数量有限制不会超过700
    @return targetNum:处理后的硬盘数区间值 
    '''
    targetNumKey = ""
    if 0 < diskNumSum <= 100:
        targetNumKey = "100P"
    if 100 < diskNumSum <= 200:
        targetNumKey = "200P"
    if 200 < diskNumSum <= 300:
        targetNumKey = "300P"
    if 300 < diskNumSum <= 400:
        targetNumKey = "400P"
    if 400 < diskNumSum <= 500:
        targetNumKey = "500P"
    if 500 < diskNumSum <= 600:
        targetNumKey = "600P"
    if 600 < diskNumSum <= 700:
        targetNumKey = "700P"
    return targetNumKey
        
def transformCapacity(Capacity):
    '''
    @summary: 容量转换
    @param Capacity: 传入的字符型容量值，单位是GB
    @return transformedCapacity: 转换后的容量值，单位GB
    '''     
    #showdisk -pyhsic得到的容量需要经过转换，才能和表格容量匹配，先转换到B，在用另一个进制转换成GB
    capacity2B = float(Capacity) * 1024 * 1024 * 1024
    transformedCapacity = capacity2B / 1000 / 1000 / 1000
    return transformedCapacity
    
def getMediaScanInfo(cli, py_java_env):
    '''
    @summary: 获取后台扫描信息
    ''' 
    mediaScanInfo = ""
    
    #需要先切换到developer模式
    if not modelManager.changeCli2Developer(cli, py_java_env):
        return (False, mediaScanInfo)
    
    mediaScanInfo = cli.execCmd("showmediascan")    
    modelManager.changeAnyModel2Cli(cli)
    return (True, mediaScanInfo)  
 
def getMediaScanStatus(mediaScanInfo):   
    '''
    @summary: 获取后台扫描状态
    '''
    status = ""
    statusKey = "Status"
    
    try:
        listDict = cHandleCliRet(mediaScanInfo).getResult()
    except ParseException:
        PY_LOGGER.error("[getMaxCapacityAndDiskNumSum]ParseException: " + unicode(traceback.format_exc()))
        return(False, status)
    
    #后台扫描只有一条，获取到就可以退出了
    for mediaScanInfoDict in listDict:
        if statusKey in mediaScanInfoDict:
            status = mediaScanInfoDict.get(statusKey)
            return (True, status)
        else:
            return (False,status)
    
    #无信息
    return (False,status)
    
def getMediaScanPeriodAndSpeed(mediaScanInfo): 
    """
    @summary: 获取后台扫描周期和速率
    """
    period = ""
    speed = ""
    periodKey = "Period(Days)"
    speedKey = "Verify Speed"
    try:
        listDict = cHandleCliRet(mediaScanInfo).getResult()
    except ParseException:
        PY_LOGGER.error("[getMaxCapacityAndDiskNumSum]ParseException: " + unicode(traceback.format_exc()))
        return(False, period, speed)
    
    #后台扫描只有一条，获取到就可以退出了
    for mediaScanInfoDict in listDict:
        if periodKey in mediaScanInfoDict:
            period = mediaScanInfoDict.get(periodKey)
        else:
            return (False, period, speed)
        
        if speedKey in mediaScanInfoDict:
            speed = mediaScanInfoDict.get(speedKey)
        else:
            return (False, period, speed)
        
        return (True, period, speed)
    
    return (False, period, speed)
 
def getNeedSetPeriodAndSpeed(maxCapacity, diskNumSum):
    '''
    @summary: 根据最大容量和硬盘总数，计算需要配置的周期和速率
    '''   
    needSetPeriod = 0
    needSetSpeed = 0
    
    capacityKey = handleCapacity(maxCapacity)
    diskNumSumKey = handleDiskNumSum(diskNumSum)
    
    needSetParam = PERIOD_AND_SPEED_DICT.get(capacityKey).get(diskNumSumKey)
    needSetPeriod = needSetParam[0]
    needSetSpeed = needSetParam[1]
    return (needSetPeriod, needSetSpeed)


    