# -*- coding: UTF-8 -*-
import re
import common
from common import selectErrorMsg
from comm.cTV1R1 import parseHorizontal
from common import getCurHotPatchVersion

INFO_INVALID = 0

IS_TOSHIBA_DISK = 1
NOT_TOSHIBA_DISK = 2

IS_SAS_DISK = 1
NOT_SAS_DISK = 2

WARNING = 'WARNING'

TOSHIBA_SAS_DISK = 0
TOSHIBA_NOSAS_DISK = 1
NOTOSHIBA_SAS_DISK = 2
NOTOSHIBA_NOSAS_DISK = 3

# **************************************************************************** #
# 函数名称: switchValidVersion
# 功能说明: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
# 输入参数: version
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def switchValidVersion(version):
    temp = ""
    
    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  #格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  #格式：V100R005C02
    
    #提取有效的版本号信息
    temp = version[startIndex: endIndex]
    
    return temp

# *************************************************************#
# 函数名称: getCurSystemVersion
# 功能说明: 获取当前系统SPC版本号
# 其 他   :  无
# *************************************************************#
def getCurSystemVersion(ssh):
    
    #查询系统版本号
    sysSpcVersion = ""
    
    systemVerInfo = ssh.execCmd("showupgradepkginfo -t 1")
    lines = systemVerInfo.splitlines()
    # 命令执行结果小于7行，说明执行该命令后并没有输出有效信息
    if len(lines) < 7:
        return (sysSpcVersion, systemVerInfo)
    # 从第7行开始，取出有效信息
    field = lines[6].split()
    sysSpcVersion = switchValidVersion(field[1])
    return (sysSpcVersion, systemVerInfo)


#不存在东芝SAS硬盘检查不通过。 added 20150119 begin
# *************************************************************#
# 函数名称: isDiskVendorToshiba
# 功能说明: 检查硬盘厂商是否为东芝
# 返回  :  0：信息无效      1：是东芝硬盘         2：不是东芝硬盘
# *************************************************************#
def isDiskVendorToshiba(diskVendorInfo):
    if not re.findall('[a-zA-Z]', diskVendorInfo):
        return INFO_INVALID
    if re.search('Toshiba', diskVendorInfo, re.IGNORECASE):
        return IS_TOSHIBA_DISK
    else:
        return NOT_TOSHIBA_DISK
    
    
# *************************************************************#
# 函数名称: isDiskTypeSas
# 功能说明: 检查硬盘类型是否为SAS
# 返回  :  0：信息无效      1：是SAS硬盘         2：不是SAS硬盘
# *************************************************************#
def isDiskTypeSas(diskTypeInfo):
    if not re.findall('[a-zA-Z]', diskTypeInfo):
        return INFO_INVALID
    if re.search('SAS', diskTypeInfo, re.IGNORECASE) \
            and not re.search('NL SAS', diskTypeInfo, re.IGNORECASE):
        return IS_SAS_DISK
    else:
        return NOT_SAS_DISK
#不存在东芝SAS硬盘检查不通过。 added 20150119 begin


# *************************************************************#
# 函数名称: _getErrMsg(diskLst, errType, lang, sphVer="")
# 功能说明: 获取提示信息
# 返回  :  提示信息
# *************************************************************#
def _getErrMsg(diskLst, errType, lang, sphVer=""):
    if not diskLst:
        return ""
    
    if TOSHIBA_SAS_DISK == errType:
        return selectErrorMsg(lang, u"\n设备存在东芝SAS硬盘：%s，请安装%s或之后版本的热补丁。" % (u'，'.join(diskLst), sphVer), \
                                    "\nThe device has TOSHIBA SAS disks:%s, please install hot patch %s or later." % (u','.join(diskLst), sphVer))
    elif TOSHIBA_NOSAS_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s类型无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                     "\nType of disks:%s is invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    elif NOTOSHIBA_SAS_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s厂商无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                    "\nVendor of disks:%s is invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    elif NOTOSHIBA_NOSAS_DISK == errType:
        return selectErrorMsg(lang, u"\n硬盘：%s类型与厂商无效，请修复硬盘后，再重新检查。" % u'，'.join(diskLst), \
                                    "\nType and vendor of disks:%s are invalid, please repair the disk then perform a check again." % u','.join(diskLst),)
    return ""

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 东芝SAS硬盘检查
# 输入参数: ssh连接
# **************************************************************************** # 
def execute(ssh):
    flag = True
    errMsg = ""
    lang = py_java_env.get("lang")
    
    #不存在东芝SAS硬盘检查不通过。 modified 20150119 begin
    #检查系统版本号与热补丁版本号是否满足要求
    iRet = getCurSystemVersion(ssh)
    sysSpcVersion = iRet[0]
    cliRet = iRet[1]
    if not sysSpcVersion:
        flag = False
        if lang == "zh":
            errMsg = u"\n获取系统版本号失败。"
        else:
            errMsg = "\nFailed to obtain the device version."
        return (flag, cliRet, errMsg)
    
    #阵列软件版本与热补丁版本对应关系
    verDict = {'V100R002C00SPC015' : 'V100R002C00SPHf02',
               'V100R002C01SPC015' : 'V100R002C01SPHf02',
               'V100R002C00SPCg00' : 'V100R002C00SPHg02',
               'V100R002C01SPCg00' : 'V100R002C01SPHg02',
               'V100R002C00SPCh00' : 'V100R002C00SPHh01',
               'V100R002C01SPCh00' : 'V100R002C01SPHh01',
               'V100R005C00SPC700' : 'V100R005C00SPH701',
               'V100R005C01SPC700' : 'V100R005C01SPH701'
            }
    
    #系统版本不在需要检查的版本范围内，巡检通过
    if sysSpcVersion not in verDict:
        return (flag, cliRet, errMsg)

    #获取当前热补丁版本号
    isQrySucc, curHotPatchVer, cliRet1, errMsg = getCurHotPatchVersion(ssh, lang)
    cliRet += "\n" + cliRet1
    if not isQrySucc:
        return (False, cliRet, errMsg)
   
    #安装有效热补丁，巡检通过
    if curHotPatchVer >= verDict[sysSpcVersion]:
        return (flag, cliRet, errMsg)
    
    #检查系统是否存在东芝SAS硬盘
    diskPhysicInfo = ssh.execCmd("showdisk -physic")
    cliRet += '\n' + diskPhysicInfo
    lineList = diskPhysicInfo.splitlines()
    if len(lineList) < 7:
        #判断cli信息是否有效
        if not common.checkCliInfoValid(diskPhysicInfo, True):
            flag = False
            if lang == "zh":
                errMsg = u"\n硬盘信息无效。"
            else:
                errMsg = "\nInvalid disk physic information."       
        return (flag, cliRet, errMsg)

    toshibaSasList = []
    toshibaNoSasList = []
    noToshibaSasList = []
    noToshibaNoSasList = []
    
    diskList = parseHorizontal(diskPhysicInfo).getResult()
    for diskInfo in diskList:
        diskID = diskInfo.get('Disk Location')
        diskVendor = isDiskVendorToshiba(diskInfo.get('Vendor'))
        diskType = isDiskTypeSas(diskInfo.get('Type'))
        
        #是东芝SAS硬盘,则巡检不通过
        if diskVendor == IS_TOSHIBA_DISK and diskType == IS_SAS_DISK:
            toshibaSasList.append(diskID)
        
        #是东芝盘，但不确定是否为SAS硬盘
        elif diskVendor == IS_TOSHIBA_DISK and diskType == INFO_INVALID:
            toshibaNoSasList.append(diskID)
            
        #是SAS盘，但不确定是否为东芝盘
        elif diskVendor == INFO_INVALID and diskType == IS_SAS_DISK:
            noToshibaSasList.append(diskID)
            
        #既不确定是否为东芝盘，也不确定是否为SAS盘
        elif diskVendor == INFO_INVALID and diskType == INFO_INVALID:
            noToshibaNoSasList.append(diskID)
        
        #不是东芝SAS硬盘
        else:
            continue
        
    PY_LOGGER.info("[ToshiBaDisk]toshibaSasList is:" + unicode(toshibaSasList))
    PY_LOGGER.info("[ToshiBaDisk]toshibaNoSasList is:" + unicode(toshibaNoSasList))
    PY_LOGGER.info("[ToshiBaDisk]noToshibaSasList is:" + unicode(noToshibaSasList))
    PY_LOGGER.info("[ToshiBaDisk]noToshibaNoSasList is:" + unicode(noToshibaNoSasList))
    
    errMsg = _getErrMsg(toshibaSasList, TOSHIBA_SAS_DISK, lang, verDict[sysSpcVersion]) 
    errMsg += _getErrMsg(toshibaNoSasList, TOSHIBA_NOSAS_DISK, lang) 
    errMsg += _getErrMsg(noToshibaSasList, NOTOSHIBA_SAS_DISK, lang) 
    errMsg += _getErrMsg(noToshibaNoSasList, NOTOSHIBA_NOSAS_DISK, lang) 
    
    flag = WARNING if (toshibaSasList or toshibaNoSasList or noToshibaSasList or noToshibaNoSasList) else True
    
    #不存在东芝SAS硬盘检查不通过。 modified 20150119 end
    return (flag, cliRet, errMsg)
