# -*- coding: UTF-8 -*-
from java.io import File
from java.lang import Exception
from cbb.frame.util.tar_util import decompress_tar_all_file
import os
import re
import string
import common
import shutil
import traceback
import time

#安全红线界限版本
R2C00_SECURE_VERSION = 'V100R002C00SPC010'
R2C01_SECURE_VERSION = 'V100R002C01SPC007'
logger = PY_LOGGER 
# **************************************************************************** #
# 函数名称: checkSecure
# 功能说明: 判断当前版本是否为安全红线
# 输入参数: sysSpcVersion
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def checkSecure(sysSpcVersion):

    isSecure = False

    #根据设备版本号是否安全红线设备
    if bool(re.search('V100R001', sysSpcVersion, re.IGNORECASE)):
        isSecure = False
    elif bool(re.search('V100R002C00', sysSpcVersion, re.IGNORECASE)):
        if sysSpcVersion < R2C00_SECURE_VERSION:
            isSecure = False
        else:
            isSecure = True
    elif bool(re.search('V100R002C01', sysSpcVersion, re.IGNORECASE)):
        if sysSpcVersion < R2C01_SECURE_VERSION:
            isSecure = False
        else:
            isSecure = True
    elif bool(re.search('V100R005', sysSpcVersion, re.IGNORECASE)):
        isSecure = True
    else:
        isSecure = False
    #返回判断结果
    return isSecure

# **************************************************************************** #
# 函数名称: switchValidVersion
# 功能说明: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
# 输入参数: version
# **************************************************************************** # 
def switchValidVersion(version):
    temp = ""
    
    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17
    else:
        endIndex = startIndex + 11
    
    #提取有效的版本号信息
    temp = version[startIndex: endIndex]
    
    return temp

# **************************************************************************** #
# 函数名称: getDeviceSpcVersion
# 功能说明: 获取热补丁的SPC版本号
# 输入参数: cli
# **************************************************************************** #
def getDeviceSpcVersion(cli): 

    ctrlSpc = cli.execCmd('showupgradepkginfo -t 1')
    rowList = string.split(ctrlSpc, '\n')
    count = len(rowList)
    devVersion = ''
    for i in range(count):
        if bool(re.search('Controller ID', rowList[i], re.IGNORECASE)):
            columnList = string.split(rowList[i + 2])
            #P12I-3845 特殊版本号优化处理  modified 20130806 Begin
            devVersion = switchValidVersion(columnList[1])
            break
            #P12I-3845 特殊版本号优化处理  modified 20130806 End

    return (devVersion, ctrlSpc)

# **************************************************************************** #
# 函数名称: getDeviceSNNumber
# 功能说明: 获取设备SN号
# 输入参数: cli
# **************************************************************************** #
def getDeviceSNNumber(cli): 

    deviceSnNum = ""
    
    ctrlSpc = cli.execCmd('showsys')
    lineList = ctrlSpc.splitlines()
    
    for line in lineList:
        if re.search("Device Serial Number", line, re.IGNORECASE):
            deviceSnNum = line.split("|")[-1].strip()
            break

    return deviceSnNum

# **************************************************************************** #
# 函数名称: decompressPKG
# 功能说明: 解压告警文件
# 输入参数: filePath，depressPath
# **************************************************************************** #
def decompressPKG(filePath, depressPath):
    decompress_tar_all_file(filePath, depressPath)

# **************************************************************************** #
# 函数名称: secureGetEventFile
# 功能说明: 安全红线后的版本：获取告警文件
# 输入参数: cli
# **************************************************************************** #
def secureGetEventFile(cli):
    
    flag = True
    errMsg = ""
    remoteFile = ""
    lang = py_java_env.get("lang")
    
    #通过Cli命令导出告警文件
    cliRet = cli.execCmd("exportsysevent -t alarm")
    #Cli回文异常，直接返回不通过
    if not bool(re.search("Path", cliRet, re.IGNORECASE)):
        flag = False
        if "zh" == lang:
            errMsg = u"\n通过CLI命令生成告警文件失败，请通过登录ISM查看是否存在未恢复告警。"
        else:
            errMsg = "\nRunning cli command to creat alarm file failed, please check the unrecovered alarm by ISM."
        return (flag, errMsg, remoteFile)
    
    lineList = cliRet.splitlines()    
    for line in lineList:
        if re.search("Path :", line, re.IGNORECASE):
            remoteFile = line.replace("Path :", "").replace(" ", "")
            break 
    
    return (flag, errMsg, remoteFile)

# **************************************************************************** #
# 函数名称: changeCliToMml
# 功能说明: Cli模式转换成Mml模式
# 输入参数: ssh
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def changeCliToMml(ssh):
    
    iRet = changeCliToDebug(ssh)
    if True == iRet:
        iRet1 = ssh.execCmd("mml")
        if bool (re.search("MML>", iRet1, re.IGNORECASE)):
            return True
    
    return False

# **************************************************************************** #
# 函数名称: changeCliToDebug
# 功能说明: Cli模式转换成Debug模式
# 输入参数: ssh
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def changeCliToDebug(ssh):

    password = unicode(py_java_env.get("devPwd").get("developer"))
    ssh.execCmd("developer")
    ssh.execCmdNoLog(password)
    retDeveloper = ssh.execCmdNoLog("debug")
    if - 1 != retDeveloper.find('Password:'):
        ssh.execCmdNoLog(password)
    return True

# **************************************************************************** #
# 函数名称: noSecureGetEventFile
# 功能说明: 安全红线前的版本：获取告警文件
# 输入参数: cli
# **************************************************************************** #
def noSecureGetEventFile(cli):
    
    flag = True
    errMsg = ""
    remoteFile = ""
    lang = py_java_env.get("lang")
    
    iRet = changeCliToMml(cli)
    if not iRet:
        flag = False
        #设备登陆时输入developer密码无效 modified 20131130 Begin
        if lang == "zh":
            errMsg = u"\n进入MML模式失败，未恢复告警检查失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入调试密码。\n（2）添加设备时输入的调试密码无效。"
        else:
            errMsg = "\nLog in to MML model failed, can not execute unrecovered alarms check.The reason of failure may be:\n" \
                    + "(1) Did not enter a debug password when adding the device.\n(2) The debug password entered is incorrect." 
        return (flag, errMsg, remoteFile)

    #通过mml命令生成告警文件
    mmlRet = cli.execCmd("alarm exportevent")
    
    #退回到Cli模式
    cli.execCmd("exit")
    cli.execCmd("exit")
    cli.execCmd("exit")

    lineList = mmlRet.splitlines()
    if not bool(re.search("path", mmlRet, re.IGNORECASE)):
        flag = False
        if lang == "zh":
            errMsg = u"\n当前版本不支持此操作，请通过登录ISM查看是否存在未恢复告警。"
        else:
            errMsg = "\nThe operation is not supported, please check the unrecovered alarm by ISM."
        return (flag, errMsg, remoteFile)
    
    #解析命令回文
    lineList = mmlRet.splitlines()
    for line in lineList:
        if re.search("path", line, re.IGNORECASE):
            remoteFile = line[line.index(":/") + 1 :-2]
            break

    return (flag, errMsg, remoteFile)


# **************************************************************************** #
# 函数名称: deleteRemoteFile
# 功能说明: 清理设备临时文件
# 输入参数: collectRemotePath阵列端文件路径
# **************************************************************************** #
def deleteRemoteFile(collectRemotePath):
    try:
        sftp = py_java_env.get("sftp")
        #使用sftp自带接口删除远端临时文件
        if not collectRemotePath:
            logger.error("[deleteRemoteFile] delete file is:" + unicode(collectRemotePath))
            return False
        else:
            sftp.deleteFile(collectRemotePath)
            return True
    except:
        logger.error("[deleteRemoteFile] except trace back:" + unicode(traceback.format_exc()))
        return False

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 脚本执行接口
# 输入参数: cli
# **************************************************************************** #
def execute(cli):

    flag = True
    lang = py_java_env.get("lang")
    errMsg = ""

    #P12V-2951 未恢复告警检查项优化 modified 20130807 Begin
    #工具传入的SFTP为空，检查失败
    cliRet = ""
    sftp = py_java_env.get("sftp")
    if sftp is None:
        if "zh" == lang:
            errMsg = u"\n工具传入SFTP为空，检查失败，请通过登录ISM查看是否存在未恢复告警。"
        else:
            errMsg = "\nThe sftp provided by tool is None, check failed, please check the unrecovered alarm by ISM."
        return (False, cliRet, errMsg)
    
    #特殊的系统版本，工具不支持检查
    iRet = getDeviceSpcVersion(cli)
    spcVersion = iRet[0]
    cliRet = iRet[1]
    
    #获取系统版本失败
    if spcVersion == "":
        flag = False
        if "zh" == lang:
            errMsg = u"\n获取系统版本信息失败。"
        else:
            errMsg = "\nGet system information failed."
        return (flag, cliRet, errMsg)
    
    #获取设备SN号，唯一标识设备
    deviceSnNum = getDeviceSNNumber(cli)
    
    #判断是否为安全红线后版本
    secureVerFlag = checkSecure(spcVersion)

    #获取告警文件
    if secureVerFlag:
        iRet = secureGetEventFile(cli)
    else:
        iRet = noSecureGetEventFile(cli)
    
    #判断有效性
    if not iRet[0]:
        return (False, cliRet, iRet[1])
    
    #获取远程文件路径
    remoteFile = iRet[2]
    
    #创建文件保存路径
    devNode = py_java_env.get("devInfo")
    saveDir = 'pytemp' + os.sep + devNode.getDeviceSerialNumber() + "_" + time.strftime('%Y%m%d%H%M%S', time.localtime(time.time())) + os.sep 
    if not os.path.exists(saveDir):
        os.makedirs(saveDir)
    
    localFileName = saveDir + 'event_export.tar'
    try:
        sftp.getFile(remoteFile, localFileName, None)
    except Exception:
        shutil.rmtree(saveDir, True)
        if "zh" == lang:
            errMsg += u"\n告警文件" + remoteFile + u"不存在。"
        else:
            errMsg += "\nAlarm file " + remoteFile + " does not exist."
        return (False, cliRet, errMsg)
    finally:
        deleteRemoteFile(remoteFile)
    #解压文件
    decompressPKG(localFileName, saveDir)
    alarmFile = open(saveDir + os.sep + 'local_alm_file.txt', "r")
    alarmIDs = []
    cliRet = "Raised  Local Time    Alarm ID    Alarm Level    Recovered Local Time    Description\n"
    for line in alarmFile:
        if line.find("Event list") >= 0:
            break
        
        list = line.split()
        if len(list) < 5:
            continue
        if list[4] != "None":
            continue
        if list[3] not in ("Major", "Critical", "Warning"):
            continue
        
        flag = False
        # 将Suggestion字段之前的字符串添加到原始信息cliRet中。
        index = line.find("Step 1")
        cliRet += line[0:index]
        cliRet += "\n"

        alarmIDs.append(list[2])
    alarmFile.close()
    
    if not flag:
        if "zh" == lang:
            errMsg += u"\n存在未恢复告警，告警ID列表：" + unicode(alarmIDs).replace("[", "").replace("]", "") + u"。"
        else:
            errMsg += "\nThere are unrecovered alarms: " + unicode(alarmIDs).replace("[", "").replace("]", "")\
                      + "."
    
    #删除临时文件
    shutil.rmtree(saveDir, True)
    
    return (flag, cliRet, errMsg)
