# -*- coding: UTF-8 -*-
import os
import time
import shutil
from common import getDeviceSNNumber
from common import secureGetEventFile
from common import noSecureGetEventFile
from common import deleteRemoteFile
from common import decompressPKG
from common import getCurSystemVersion
from common import getCurHotPatchVersion
from common import getDeviceType
from common import selectErrorMsg
from common import getVaultDiskLockSet
from comm.cTV1R1 import parseHorizontal
from comm.checkSecure import checkSecure

CHK_DEV_TYPE_LIST = ['S3900-M200', 'S3900-M300', 'S5900-M100', 'S5900-M200', 'S6900-M100']
SLOW_DISK_ALM_ID_LIST = ['0xe0209000a', '0xe02090011']

java_env = py_java_env
logger = PY_LOGGER

# **************************************************************************** #
# 函数名称: execute
# 功能说明: 保险箱慢盘检查
# 输入参数: ssh连接
# **************************************************************************** # 
def execute(cli):
    lang = java_env.get("lang")

    isQryOk, deviceType, cliRet, errMsg = getDeviceType(cli, lang)
    if not isQryOk:
        return (False, cliRet, errMsg)
    
    if deviceType not in CHK_DEV_TYPE_LIST:
        return (True, cliRet, '')
        
    isQrySucc, sysSpcVersion, cliSysVerRet, errMsg = getCurSystemVersion(cli, lang)
    cliRet += "\n" + cliSysVerRet
    if not isQrySucc:
        return (False, cliRet, errMsg)
    
    if sysSpcVersion > 'V100R005C02SPC300':
        return (True, cliRet, errMsg)
    elif sysSpcVersion == 'V100R005C02SPC300':
        isQrySucc, curHotPatchVer, cliRet1, errMsg = getCurHotPatchVersion(cli, lang)
        cliRet += "\n" + cliRet1
        if not isQrySucc:
            return (False, cliRet, errMsg)
        
        if 'V100R005C02' in curHotPatchVer and curHotPatchVer >= 'V100R005C02SPH301':
            return (True, cliRet, errMsg)
        else:
            logger.warn('System has not installed hot patch!')
            #check slow disk alarm
    else:
        pass
        #check slow disk alarm
    
    #check slow disk alarm
    logger.info('Check slow disk alarm now!')
    isSecure = checkSecure(sysSpcVersion)
    isQryAlmOk, existAlm, almRet, slowDskAlmErrMsg = checkSlowDiskAlarm(cli, isSecure)
    
    cliRet += ('\n' + almRet)
    errMsg += slowDskAlmErrMsg
    
    if not isQryAlmOk:
        logger.error('Query slow disk alarm failed:' + errMsg)
        return (False, cliRet, errMsg)
    
    logger.info('Slow disk alarm check finished, check whether RAID contain vault disk!')
    isQryOk, isRgContainVaultDisk, rgRet, rgMsg = checkRgContainVaultDisk(cli)
    cliRet += ('\n' + rgRet)
    errMsg += rgMsg
    
    if not isQryOk:
        return (False, cliRet, errMsg)

    if existAlm:
        logger.error('Slow disk alarm exists, check not pass!')
        return (False, cliRet, errMsg)
       
    if isRgContainVaultDisk:
        logger.warn('RAID group contains vault disk, check warning!')
        return ('WARNING', cliRet, errMsg)
    else:
        logger.info('RAID group contains no vault disk, check pass!')
        return (True, cliRet, errMsg)
            
# *************************************************************#
# 函数名称: checkSlowDiskAlarm
# 功能说明: 检查慢盘告警
# 返回  :  (isQryAlmOk, isSlowDiskAlmExist, getCliRet, errMsg)
# *************************************************************#
def checkSlowDiskAlarm(cli, isSecure):
    deviceSnNum = getDeviceSNNumber(cli)
    lang = java_env.get("lang")
    sftp = java_env.get("sftp")
    isQryAlmOk = False
    isSlowDiskAlmExist = False
    
    #获取告警文件
    if isSecure:
        isGetSuccc, getCliRet, errMsg, remoteFile = secureGetEventFile(lang, cli)
    else:
        isGetSuccc, getCliRet, errMsg, remoteFile = noSecureGetEventFile(java_env, cli)
    
    #判断有效性
    if not isGetSuccc:
        logger.error('Get event file failed!')
        isQryAlmOk = False
        isSlowDiskAlmExist = False
        return (isQryAlmOk, isSlowDiskAlmExist, getCliRet, errMsg)
    
    logger.info('Get event file successfully!')
    
    #创建文件保存路径
    devNode = py_java_env.get("devInfo")
    saveDir = 'pytemp' + os.sep + devNode.getDeviceSerialNumber() + "_" + time.strftime('%Y%m%d%H%M%S', time.localtime(time.time())) + os.sep 
    if not os.path.exists(saveDir):
        os.makedirs(saveDir)
    
    localFileName = saveDir + 'event_export.tar'
    try:
        sftp.getFile(remoteFile, localFileName, None)
    except:
        logger.error('Get event file caught exception!')
        zhErrMsg = u"\n告警文件" + remoteFile + u"不存在。"
        enErrMsg = "\nAlarm file " + remoteFile + " does not exist."
        errMsg += selectErrorMsg(lang, zhErrMsg, enErrMsg)
        isQryAlmOk = False
        isSlowDiskAlmExist = False        
        return (isQryAlmOk, isSlowDiskAlmExist, getCliRet, errMsg)
    finally:
        deleteRemoteFile(java_env, remoteFile, logger)
        
    #解压文件
    decompressPKG(localFileName, saveDir)
    alarmFilePath = saveDir + os.sep + 'local_alm_file.txt'
    slowDskAlmLines = []
    almHeadLine = "Raised  Local Time    Alarm ID    Alarm Level    Recovered Local Time    Description\n"
    alarmFile = open(alarmFilePath, "r")
    for line in alarmFile:
        if "Event list" in line:
            logger.info('Search to Event area, stop searching!')
            break
        
        fields = line.split()
        if len(fields) < 5:
            continue
        if ((fields[2].strip() in  SLOW_DISK_ALM_ID_LIST)
            and fields[4] == 'None'):
            logger.info('Found unrecovered slow disk alarm.')
            slowDskAlmLines.append(line)
    alarmFile.close()

    #删除临时文件
    shutil.rmtree(saveDir, True)
    
    isQryAlmOk = True
            
    if not slowDskAlmLines:
        isSlowDiskAlmExist = False 
        logger.info('No unrecovered slow disk alarm exist!')
        return (isQryAlmOk, isSlowDiskAlmExist, getCliRet, errMsg)
    else:
        isSlowDiskAlmExist = True 
        logger.warn('Unrecovered slow disk alarm(s) exist!')
        slowDskAlmLines.insert(0, almHeadLine)
        
        zhErrMsg = u"\n系统存在未恢复的慢盘告警。"
        enErrMsg = "\nAlarms indicating slow disks exist in the system."
        errMsg += selectErrorMsg(lang, zhErrMsg, enErrMsg)
        
        getCliRet += ('\n' + ''.join(slowDskAlmLines)) 
                    
        return (isQryAlmOk, isSlowDiskAlmExist, getCliRet, errMsg)
    

# *************************************************************#
# 函数名称: checkRgContainVaultDisk
# 功能说明: 检查保险箱盘是否是RAID组成员盘。
# 返回  :  0：信息无效      1：是东芝硬盘         2：不是东芝硬盘
# *************************************************************#
def checkRgContainVaultDisk(ssh):
    lang = java_env.get("lang")
    rgCliRet = ssh.execCmd("showrg")
    lineList = rgCliRet.splitlines()
       
    if len(lineList) == 3 and 'success' in rgCliRet:
        logger.info('No RAID group exist!')
        return (True, False, rgCliRet, '')
            
    (isQryOk, vaultDiskLocSet, cliRet, errMsg) = getVaultDiskLockSet(ssh, logger, lang)
    if not isQryOk:
        logger.error('Query vault disk location list failed!')
        return (False, False, cliRet, errMsg)
    
    logger.info('Query vault disk location list successfully!')
    logger.info('Vault disk location list is:' + unicode(vaultDiskLocSet))
    
    try:
        rgInfoDicList = parseHorizontal(rgCliRet).getResult()
    except Exception, e:
        logger.error('Parse RAID CLI information exception:' + unicode(e))
        return (False, False, cliRet, errMsg)
    else:
        logger.info('Parse RAID CLI information successfully!')
    
    vaultDiskNumFoundInRg = 0    
    for rgInfoDict in rgInfoDicList:
        vaultMemDiskList = []
        rgId = rgInfoDict.get('Id')
        memList = rgInfoDict.get('Disk List').split(';')
        memList = [locStr.strip() for locStr in memList if locStr]
        memDiskLocList = []
        
        #将列表中的字符串形式(['1,2','3,4','0,1'])转换成元组形式([(1,2),(3,4),(0,1)])
        for locStr in memList:
            try:
                frameNo = int(locStr.split(',')[0].strip())
                slotNo = int(locStr.split(',')[1].strip())
                diskLoc = (frameNo, slotNo)
            except IndexError:
                logger.error('Parse RAID member disk location caught IndexError exception:' + unicode(locStr))
                continue
            except ValueError:
                logger.error('Parse RAID member disk location caught ValueError exception:' + unicode(locStr))
                continue
            else:
                memDiskLocList.append(diskLoc)
                
        #判断RAID组成员盘是否包含保险箱盘    
        for memDiskLoc in memDiskLocList:
            if memDiskLoc in vaultDiskLocSet:
                vaultMemDiskList.append(memDiskLoc)
                
        if vaultMemDiskList:
            vaultDiskNumFoundInRg += len(vaultMemDiskList)
            vaultMemDiskLocStrList = [unicode(loc) for loc in vaultMemDiskList]       
            logger.warn('Some RAID groups member disk contain vault disk!')
            zhErrMsg = u"\nRAID组（ID:" + unicode(rgId) + u')成员盘中存在保险箱盘：' + ';'.join(vaultMemDiskLocStrList)
            enErrMsg = "\nRAID group(ID:" + unicode(rgId) + ') contains member coffer disks:' + ';'.join(vaultMemDiskLocStrList)
            errMsg += selectErrorMsg(lang, zhErrMsg, enErrMsg) 
        else:
            logger.info('No RAID or all RAIDs member disk do do not contain vault disk!')
        
        if  vaultDiskNumFoundInRg >= 4:
            break
              
    if errMsg:
        return (True, True, rgCliRet, errMsg)
    else:
        return (True, False, rgCliRet, '')
    
