# -*- coding: UTF-8 -*-
import re

#枚举类型
START_INDEX = 0
END_INDEX = 1

#常用匹配规则定义

#连续=分隔符
patternEqu = re.compile("^\s*=+\s*$")

#连续-分隔符
patternSub = re.compile("^\s*-+\s*$")

#连续*分隔符
patternStar = re.compile("^\s*\*+\s*$")

#空格分隔符
patternBlank = re.compile("^\s*$")

#回显结束标志
patternEnd = re.compile("^.*>.*$")

#行中不包含字母和数字
patternSymbol = re.compile("^\W*$")

# *****************************************************************************************#
# 函数名称: _isSkipLine(line)
# 功能说明: 符号行需要跳过（符号行即此行不包含任何字母或数字的情况）
# 输入参数: line
# 输出参数: True or False
# *****************************************************************************************#
def _isSkipLine(line):
    return not re.search("[A-Za-z0-9]", line) or "The system is in single controller status" in line

# *****************************************************************************************#
# 函数名称: _isEndLine(line)
# 功能说明: 是否为结束行
# 输入参数: line
# 输出参数: True or False
# *****************************************************************************************#
def _isEndLine(line):
    if re.match("^.*>\s*$", line):
        return True
    
    if re.match("^.*~\s*#\s*$", line):
        return True
    
    if re.match("^.*spu ui>\s*$", line):
        return True

class cHorizontal:
    """
    # *****************************************************************************************#
    # 类名称: cHorizontal（通用类）
    # 功能说明: 对于CLI回显，一行为关键字，下面所有行为其对应的value值格式，格式化为一个关键字对应
    #           value值的list(dict)结果，value值的行数即为list的长度
    # 输入参数: cliValRetList：横向结构对应的val值区域信息
    #           fstKeyEndIndex：第一个key值对应的结束位置（用于判断value值是否存在换行的情况）
    #           keyIndexDict：key值对应value的起始，结束位置
    #           newLineSign：对于value值存在多行的情况，默认为直接连接
    # 传入格式说明：cliValRetList行为key值对应的value行，handle函数对外提供
    #                 (0, 0)  Normal           FC             Double                             37  
    #                 (0, 1)  Normal           FC             Double                             38  
    #                 (0, 2)  Normal           FC             Double                             41  
    #                 (0, 3)  Normal           FC             Double                             36  
    # 输出参数: list(dict)
    # *****************************************************************************************# 
    """
    def __init__(self, cliValRetList, fstKeyEndIndex, keyIndexDict, newLineSign=""):
        self.cliValRetList = cliValRetList
        self.fstKeyEndIndex = fstKeyEndIndex
        self.keyIndexDict = keyIndexDict
        self.newLineSign = newLineSign
    
    #入参检查
    def __checkParams(self):
        if not self.cliValRetList:
            return False
        if -1 == self.fstKeyEndIndex:
            return False
        if int != type(self.fstKeyEndIndex):
            return False
        if dict != type(self.keyIndexDict):
            return False
        return True
    
    #通用处理函数，格式化为一个关键字对应value值的list(dict)结果，value值的行数即为list的长度
    def handle(self, isAlign=False):
        """
        返回CLI回显解析结构
        """
        result = []
        oneLineDict = {}
        
        if not self.__checkParams():
            raise ParseException
        
        for valInfoLine in self.cliValRetList:
            if _isSkipLine(valInfoLine):
                continue
            
            if patternEnd.match(valInfoLine):
                break
            
            #用第一个key值下面对面是否有val值来判断此行是否是接在上行上
            if valInfoLine[:self.fstKeyEndIndex].split():
                #keyIndexDict传入字典，单个元素key值对应value的起始位置
                oneLineDict = {}
                for key in self.keyIndexDict:
                    #为了兼容不同版本关键字的修复，中间可以传入不存在的关键字，对应返回None
                    if [-1, -1] != self.keyIndexDict[key]: 
                        oneLineDict[key] = self.__getValue(valInfoLine, self.keyIndexDict[key], isAlign)
                    else:
                        oneLineDict[key] = None
                result.append(oneLineDict)
            #第一列ID为空，需要把相应位置的值增加到上一次的key值对应的value中
            elif oneLineDict:
                #keyIndexDict传入字典，单个元素key值对应value的起始位置
                result.remove(oneLineDict)
                for key in self.keyIndexDict.keys():
                    #为了兼容不同版本关键字的修复，中间可以传入不存在的关键字，对应返回None
                    if [-1, -1] != self.keyIndexDict[key]: 
                        oneLineDict[key] += self.newLineSign + self.__getValue(valInfoLine, self.keyIndexDict[key], True)
                    else:
                        oneLineDict[key] = None
                result.append(oneLineDict)
        return result
 
    #有些产品value可能没有对齐于key，需要补充起始位置之外value是否还有相连的字母
    def __getValue(self, valInfoLine, keyIndexDict, isAlign=False):
        
        #Value行的值已经严格对齐了,不需要调整
        if isAlign:
            return valInfoLine[keyIndexDict[START_INDEX]:keyIndexDict[END_INDEX]].rstrip()
            
        valuePosInfo = [-1, -1]
        
        valuePosInfo[START_INDEX] = valInfoLine[0:keyIndexDict[START_INDEX] + 1].rfind(" ")
        endIndex = valInfoLine[keyIndexDict[END_INDEX] - 1:].find(" ")
        if -1 == endIndex:
            valuePosInfo[END_INDEX] = len(valInfoLine)
        else:
            valuePosInfo[END_INDEX] = keyIndexDict[END_INDEX] - 1 + endIndex

        retVal = valInfoLine[valuePosInfo[START_INDEX]:valuePosInfo[END_INDEX]].strip()
        if '  ' in retVal:
            retVal = retVal.split('  ')[0]
            
        return retVal 
        
class cVertical():
    """
    # *****************************************************************************************#
    # 函数名称: cVertical（通用类）
    # 功能说明: 快速查找键值对，对于CLI回显，左侧为key值，右测为value值的情况
    # 输入参数: cliRet：CLI回显
    #           defaultSep：默认分隔符为"|"
    # 传入格式说明：cliRet为多段下述结构，handle函数对外提供
    #        ======================================================
    #                          System Information
    #        *-----------------------------------------------------*
    #         System Name           |  
    #         Device Model          |  OceanStor S5600
    #         Current System Mode   |  Normal Dual-Controller Mode
    #         Mirror Path           |  Normal
    #         Location              |  
    #         System Time           |  2014-05-04 07:36:11
    #         Device Serial Number  |  210231G407Z09C000013
    #         Product Version       |  V100R002C01
    #         iSCSI Device Name     |  --
    #        ======================================================
    # 输出参数:list(dict)
    # 约束条件:key和value值一定为defaultSep分开
    # *****************************************************************************************#
   """  
    __keyOriginOrder = []    #原始key值顺序列表
    __keyOrder = []    #key值顺序列表
    def __init__(self, cliRet, defaultSep="|", newLineSign=""):
        self.cliRet = cliRet
        self.defaultSep = defaultSep
        self.newLineSign = newLineSign
    
    #入参检查  
    def __checkParams(self):
        if not self.cliRet:
            return False
        self.cliRet = self.cliRet.encode('utf8')
        if  str != type(self.defaultSep):
            return False
        #分隔符不能只为数字和空格
        if  re.compile("^[\w\s]*$").match(self.defaultSep):
            return False
        if str != type(self.newLineSign):
            return False
        return True
        
    def handle(self):
        """
        返回CLI回显解析结构
        """
        cliValRetList = self.cliRet.splitlines()
        result = []
        secInfoDict = {}
        
        key = ''
        value = ""
        lastKey = ''
        
        if not self.__checkParams():
            raise ParseException("Base class param error(cliRet=" + unicode(self.cliRet) + \
                            ",defaultSep=" + unicode(self.defaultSep) + \
                            ",newLineSign=" + unicode(self.newLineSign) + ")")
        
        for lineInfo in cliValRetList:
            #需要跳过的行
            if _isSkipLine(lineInfo):
                continue
            
            if patternEnd.match(lineInfo):
                continue 
            
            index = lineInfo.find(self.defaultSep) 
            if index > 0:
                key = lineInfo[0:index].strip().title()
                
                if key not in self.__keyOrder:
                    self.__keyOrder.append(key)
                    self.__keyOriginOrder.append(lineInfo[0:index].strip())
                    
                value = lineInfo[index + 1:].strip()
                
                if key in secInfoDict:   #表示为下一组数据
                    result.append(secInfoDict)
                    secInfoDict = {}     #将存放临时数据的字典清空
                    
                if key:     #不换行
                    lastKey = key
                    secInfoDict[key] = value
                else:   #换行
                    secInfoDict[lastKey] += self.newLineSign + value
       
        #如果只有一组数据将数据追加到列表,如果有多组数据，将最后一组数据追加到字典
        if secInfoDict:
            result.append(secInfoDict)
        
        return result
        
    def getKeyOrder(self):
        """
        返回key值在CLI回显中的顺序
        """
        return self.__keyOrder
    
    def getOriginKeyOrder(self):
        """
        返回key值在CLI回显中的顺序
        """
        return self.__keyOriginOrder
    
class cHorizontalWithSep:
    """
    # *****************************************************************************************#
    # 类名称: cHorizontalWithSep（通用类）
    # 功能说明: 对于CLI回显，key值行和value行都是由固定的defaultSep分隔开,并且第一位和最后一位
    #           都是defaultSep的情况
    # 输入参数: cliRet：defaultSep分隔的CLI回显格式
    #           defaultSep：分隔符
    #           areaIndex：处理CLI回显的区域,一定要含有key值行
    # 传入格式说明：cliRet为defaultSep分隔的CLI回显格式，handle函数对外提供
    #|Port|LossOfSignal|Bad RX Char|Loss of Sync|Link Fail|RxEOFa|Dis Frame|Bad CRC|Proto error|
    #|  30|           0|          0|           0|       10|     0|        0|      0|         0|
    #|  31|           0|          0|           0|        0|     0|        0|      0|         0|
    #|  20|           0|          0|           0|        0|     0|        0|      0|         0|
    #|  21|           0|          0|           0|        0|     0|        0|      0|         0|
    # 输出参数: list(dict)
    # *****************************************************************************************# 
    """
    __keyLineInfo = "" #key值所在行的信息
    __keyOrder = []    #key值顺序列表
    __originKeyOrder = [] #原始key值顺序列表
    __parseResult = [] #解析接口
    def __init__(self, cliRet, defaultSep="|", areaIndex=[None, None]):
        cliRet = cliRet.encode('utf8')
        if not self.__checkParams(cliRet, defaultSep, areaIndex):
            raise ParseException("Input param error(defaultSep=" + unicode(defaultSep) + \
                        ",areaIndex=" + unicode(areaIndex) + ")")
          
        cliRet = cliRet.encode('utf8') 
        self.cliValRetList = self.__getCliRetDoArea(cliRet, areaIndex)
        self.defaultSep = defaultSep
        self.__handle()

    #入参检查  
    def __checkParams(self, cliRet, defaultSep, areaIndex):
        if not cliRet:
            return False
        if str != type(defaultSep):
            return False
        if list != type(areaIndex) and 2 == len(areaIndex):
            return False
        return True  
 
    #获取需要处理区域的CLI回显，请传入一个表头+N个value行的结构     
    def __getCliRetDoArea(self, cliRet, areaIndex=[None, None]):
        
        cliValRetList = cliRet.splitlines()
        if [None, None] == areaIndex:
            return cliValRetList
        elif areaIndex[0] > 0 and None == areaIndex[1]:
            return  cliValRetList[areaIndex[0]:len(cliValRetList)] 
        elif [None, None] != areaIndex:
            if 2 != len(areaIndex) or areaIndex[0] >= areaIndex[1]:
                return None
            return  cliValRetList[areaIndex[0]:areaIndex[1]]
        else:
            return cliValRetList
    
    def __getKeyInfo(self):
        keyWordPos = -1
        
        self.__keyLineInfo = ""
        
        for i in xrange(len(self.cliValRetList)):
            lineInfo = self.cliValRetList[i].strip()
            if len(lineInfo) < 2:
                continue
            if self.defaultSep == lineInfo[0] and self.defaultSep == lineInfo[-1]:
                keyWordPos = i
                self.__keyLineInfo = lineInfo
                break
        
        tmpKeySplit = self.__keyLineInfo.split(self.defaultSep)
        for key in tmpKeySplit:
            if key:
                self.__originKeyOrder.append(key.strip())
                self.__keyOrder.append(key.strip().title())
        
        return keyWordPos
    
    def __getValList(self, lineInfo):
        valList = []
        
        tmpValSplit = lineInfo.split(self.defaultSep)
        
        for value in tmpValSplit:
            if value:
                valList.append(value.strip())
        return valList
                   
    def __handle(self):
        keyWordPos = self.__getKeyInfo()
        if -1 == keyWordPos:
            return self.__parseResult
        
        for i in xrange(keyWordPos + 1, len(self.cliValRetList)):
            valueLineInfo = self.cliValRetList[i].strip()
            if len(valueLineInfo) < 2:
                continue
            
            if self.defaultSep == valueLineInfo[0] and self.defaultSep == valueLineInfo[-1]: 
                oneLineValList = self.__getValList(valueLineInfo)
                if len(self.__keyOrder) == len(oneLineValList):
                    self.__parseResult.append(dict(zip(self.__keyOrder, oneLineValList)))
        return 
    
    def getOriginKeyOrder(self):
        """
        返回原始key值在CLI回显中的顺序
        """
        return self.__originKeyOrder
    
    def getKeyOrder(self):
        """
        返回key值在CLI回显中的顺序
        """
        return self.__keyOrder
    
    def getResult(self):
        """
        获取CLI回显解析结果
        """
        return self.__parseResult
    
     
    def _getKeyLine(self):
        """
        测试使用：返回key值所在行的信息
        """
        return self.__keyLineInfo    

class ParseException(Exception):
    pass 
