# -*- coding: UTF-8 -*-

from com.huawei.ism.tool.obase.exception import ToolException

import re
from cbb.frame.util.tar_util import decompress_tar_all_file
import traceback
import hardware
from comm.cTV1R1 import parseHorizontal
import time
import resource
import os
import datetime

#日志下载枚举值
#日志下载受限版本
LIMMIT_VERSION = 0
#日志下载不受限版本
NOT_LIMMIT_VERSION = 1
#查询异常
ERROR_LIMMIT_VERSION = 2

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"
LOCAL_CONTROL_1= "Local    control  :1"
PRIVATE_LUN_RET = "Error: The object does not exist"
normalInfo = "command operates successfully|license|not exist|not support"
INDEPENDENT_ENGINE_TYPE_SET = ('S5600T', 'S5800T', 'S6800T',
                               'S5900-M100', 'S5900-M200',
                               'S6900-M100')
VAULT_DISK_CAPACITY_GAP_MIN = 23
VAULT_DISK_CAPACITY_GAP_MAX = 24

# **************************************************************************** #
# 函数名称: checkCliInfoValid
# 功能说明: 查询当前获取的Cli信息是否有效
# 输入参数: cliInfo：命令回文信息（String）
#                   necessaryData：系统中必须包含的配置为True（如控制器、框等）（boolean）
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def getCurDeviceInfo(py_java_env):
    """
    @summary: query the device info from java env context
    """
    dev = py_java_env.get("devInfo")
    
    return dev
    
def getPatchWarningDevs(py_java_env):
    '''
    @summary:  query the PatchWarningDev  from java env context
    @param py_java_env: context
    @return: patchWarningDevs:Map object in java
    '''
    return py_java_env.get("patchWarningDevs")
    
def checkCliInfoValid(cliInfo, necessaryData=False):
    
    #异常模式
    if re.search("-bash", cliInfo, re.IGNORECASE):
        return False
    
    #命令回文大于5行，说明必然有数据，检查通过
    lineList = cliInfo.splitlines()
    if len(lineList) >= 5:
        return True

    #系统中必须包含的配置（如控制器、框等）
    if necessaryData:
        return False

    #非必需配置项，按照如下标准检查
    flag = False
    if re.search(normalInfo, cliInfo, re.IGNORECASE):
        flag = True
    else:
        flag = False
        
    return flag

def getDeviceType(cli, lang):
    cliRet = cli.execCmd('showsys')
    lineList = cliRet.splitlines()
    deviceType = ''
    errMsg = ''
    isQryOk = True
    
    for line in lineList:
        if re.search("Device Type", line, re.IGNORECASE):
            deviceType = line.split()[-1].strip()
            break
    
    if not deviceType:
        zhErrMsg = u"\n查询设备型号失败。"
        enErrMsg = "\nFailed to query device type."
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        isQryOk = False        

    return (isQryOk, deviceType, cliRet, errMsg)

def isIndepentEngine(devType):
    return devType in INDEPENDENT_ENGINE_TYPE_SET

def getDeviceSNNumber(cli): 

    deviceSnNum = ""
    
    ctrlSpc = cli.execCmd('showsys')
    lineList = ctrlSpc.splitlines()
    
    for line in lineList:
        if re.search("Device Serial Number", line, re.IGNORECASE):
            deviceSnNum = line.split("|")[-1].strip()
            break

    return deviceSnNum

# **************************************************************************** #
# 函数名称: switchValidVersion
# 功能说明: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
# 输入参数: version
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def switchValidVersion(version):
    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  #格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  #格式：V100R005C02
    
    #提取有效的版本号信息
    temp = version[startIndex: endIndex]
    
    return temp

# *************************************************************#
# 函数名称: getCurSystemVersion
# 功能说明: 获取当前系统SPC版本号
# 其 他   :  无
# *************************************************************#
def getCurSystemVersion(ssh, lang):
    
    #查询系统版本号
    sysSpcVersion = ""
    errMsg = ""
    
    cliRet = ssh.execCmd("showupgradepkginfo -t 1")
    lines = cliRet.splitlines()
    # 命令执行结果小于7行，说明执行该命令后并没有输出有效信息
    if len(lines) < 7:
        if "zh" == lang:
            errMsg = u"\n查询系统版本信息失败。"
        else:
            errMsg = "\nFailed to query system version information."
        return (False, sysSpcVersion, cliRet, errMsg)
    # 从第7行开始，取出有效信息
    field = lines[6].split()
    sysSpcVersion = switchValidVersion(field[1])
    return (True, sysSpcVersion, cliRet, errMsg)

def switchHotPatchVersion(version):
    '''
    @summary: 将热补丁版本号转化成有效的版本号（V100R002C00SPC011SPHb04转变成V100R002C00SPHb04）
    @param version: the hot patch version of device
    @return: version
    '''
    startIndex = version.index("V100")
    #提取有效的版本号信息
    version = version[startIndex:]

    #老版本热补丁版本不规范：V100R002C00SPC011SPHb04
    if "SPC" in version:
        startIndex = version.index("SPC")
        endIndex = startIndex + 6
        delStr = version[startIndex: endIndex]
        version = version.replace(delStr, "")
    return version

# *************************************************************#
# 函数名称: getCurHotPatchVersion
# 功能说明: 获取当前系统热补丁版本号
# 其 他   :  无
# *************************************************************#
def getCurHotPatchVersion(ssh, lang):

    #查询当前设备已安装热补丁
    curHotPatchVer = ""
    errMsg = ""
    systemIfIsDoubleMode = True
    
    systemInfoStr = ssh.execCmd("showsys")
    
    #判断回文是正确
    if not re.search("System Name", systemInfoStr, re.IGNORECASE):
        if lang == "zh":
            errMsg = u"\n获取系统信息失败。"
        else:
            errMsg = "\nGetting system information failed."
        return (False, '', systemInfoStr, errMsg)
                
    #查询是否是单控型态
    if re.search("Single Controller", systemInfoStr, re.IGNORECASE):
        systemIfIsDoubleMode = False
    
    cliRet = ssh.execCmd("showupgradepkginfo -t 3")
    #不支持热补丁，检查通过
    if re.search("not exist.|Error : Invalid argument", cliRet, re.IGNORECASE):
        return (True, curHotPatchVer, cliRet, errMsg)

    #没有安装任何热补丁，返回空
    if re.search("command operates successfully", cliRet, re.IGNORECASE):
        return (True, curHotPatchVer, cliRet, errMsg)
    
    #查询所有热补丁版本号
    lineList = cliRet.splitlines()
    #信息少于7行，查询信息失败
    if len(lineList) < 7:
        if "zh" == lang:
            errMsg = u"\n获取热补丁信息失败。"
        else:
            errMsg = "\nFailed to query hot patch information."
        return (False, curHotPatchVer, cliRet, errMsg)
    
    #导入了热补丁包但是没有激活
    if not re.search("Running", cliRet, re.IGNORECASE):
        return (True, curHotPatchVer, cliRet, errMsg)
    
    hotPatchVerList = []
    hotPatchCtrl = []
    for line in lineList:
        if re.search("A|B", line) and re.search("Running", line, re.IGNORECASE):
            field = line.split()
            if len(field) >= 2:
                hotPatchCtrl.append(field[0])
                #兼容不规范的热补丁版本V100R002C00SPC011SPHb04
                curHotPatchVer = switchHotPatchVersion(field[1])
                hotPatchVerList.append(curHotPatchVer)
        else:
            continue

    #只有一端有补丁，或者两端控制器都有补丁但是版本号不相同，则报检查不通过
    ctrlHotpatchCounter = len(hotPatchVerList)
    if ctrlHotpatchCounter == 0 or ctrlHotpatchCounter > 2:
        if lang == "zh":
            errMsg = u"\n获取热补丁版本号失败。"
        else:
            errMsg = "\nGetting hot patch version failed."
        return (False, curHotPatchVer, cliRet, errMsg)  
    #针对双控在位，补丁不一致
    elif systemIfIsDoubleMode and 1 == ctrlHotpatchCounter:
        failHotPatchCtrl = ""
        if "A" == hotPatchCtrl[0]:
            failHotPatchCtrl = "B"
        else:
            failHotPatchCtrl = "A"
        #双控设备，只有一端加载热补丁
        if lang == "zh":
            errMsg = u"\n控制器%s的热补丁信息查询失败。" % failHotPatchCtrl
        else:
            errMsg = "\nFailed to query the hot patch of controller %s." % failHotPatchCtrl
        return (False, curHotPatchVer, cliRet, errMsg)
    elif ctrlHotpatchCounter == 2 and hotPatchVerList[0] != hotPatchVerList[1]:
        if lang == "zh":
            errMsg = u"\n两端控制器的热补丁版本号不相同。"
        else:
            errMsg = "\nThe hot patch versions of two controllers are not the same."
        return (False, curHotPatchVer, cliRet, errMsg) 

            
    return (True, curHotPatchVer, cliRet, errMsg)

# **************************************************************************** #
# 函数名称: secureGetEventFile
# 功能说明: 安全红线后的版本：获取告警文件
# 输入参数: cli
# **************************************************************************** #
def secureGetEventFile(lang, cli):
    
    flag = True
    errMsg = ""
    remoteFile = ""
    
    #通过Cli命令导出告警文件
    cliRet = cli.execCmd("exportsysevent -t alarm")
    #Cli回文异常，直接返回不通过
    if not bool(re.search("Path", cliRet, re.IGNORECASE)):
        flag = False
        if "zh" == lang:
            errMsg = u"\n通过CLI命令生成告警文件失败，请通过登录ISM查看是否存在未恢复告警。"
        else:
            errMsg = "\nRunning cli command to create alarm file failed, please check the unrecovered alarm by ISM."
        return (flag, cliRet, errMsg, remoteFile)
    
    lineList = cliRet.splitlines()    
    for line in lineList:
        if re.search("Path :", line, re.IGNORECASE):
            remoteFile = line.replace("Path :", "").replace(" ", "")
            break 
    
    return (flag, cliRet, errMsg, remoteFile)

# **************************************************************************** #
# 函数名称: noSecureGetEventFile
# 功能说明: 安全红线前的版本：获取告警文件
# 输入参数: cli
# **************************************************************************** #
def noSecureGetEventFile(java_env, cli):
    
    flag = True
    errMsg = ""
    remoteFile = ""
    lang = java_env.get('lang')
    iRet = changeCliToMml(java_env, cli)
    if not iRet:
        flag = False
        #设备登陆时输入developer密码无效 modified 20131130 Begin
        if lang == "zh":
            errMsg = u"\n进入MML模式失败，未恢复告警检查失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入调试密码。\n（2）添加设备时输入的调试密码无效。"
        else:
            errMsg = "\nLog in to MML model failed, can not execute unrecovered alarms check.The reason of failure may be:\n" \
                    + "(1) Did not enter a debug password when adding the device.\n(2) The debug password entered is incorrect." 
        return (flag, '', errMsg, remoteFile)

    #通过mml命令生成告警文件
    mmlRet = cli.execCmd("alarm exportevent")
    
    #退回到Cli模式
    cli.execCmd("exit")
    cli.execCmd("exit")
    cli.execCmd("exit")

    if not bool(re.search("path", mmlRet, re.IGNORECASE)):
        flag = False
        if lang == "zh":
            errMsg = u"\n当前版本不支持此操作，请通过登录ISM查看是否存在未恢复告警。"
        else:
            errMsg = "\nThe operation is not supported, please check the unrecovered alarm by ISM."
        return (flag, mmlRet, errMsg, remoteFile)
    
    #解析命令回文
    lineList = mmlRet.splitlines()
    for line in lineList:
        if re.search("path", line, re.IGNORECASE):
            remoteFile = line[line.index(":/") + 1 :-2]
            break

    return (flag, mmlRet, errMsg, remoteFile)

# **************************************************************************** #
# 函数名称: deleteRemoteFile
# 功能说明: 清理设备临时文件
# 输入参数: collectRemotePath阵列端文件路径
# **************************************************************************** #
def deleteRemoteFile(py_java_env, collectRemotePath, logger):
    try:
        sftp = py_java_env.get("sftp")
        #使用sftp自带接口删除远端临时文件
        if not collectRemotePath:
            logger.error("[deleteRemoteFile] delete file is:" + unicode(collectRemotePath))
            return False
        else:
            sftp.deleteFile(collectRemotePath)
            return True
    except:
        logger.error("[deleteRemoteFile] except trace back:" + unicode(traceback.format_exc()))
        return False

# **************************************************************************** #
# 函数名称: decompressPKG
# 功能说明: 解压告警文件
# 输入参数: filePath，depressPath
# **************************************************************************** #
def decompressPKG(filePath, depressPath):
    decompress_tar_all_file(filePath, depressPath)
        
# **************************************************************************** #
# 函数名称: changeCliToMml
# 功能说明: Cli模式转换成Mml模式
# 输入参数: ssh
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def changeCliToMml(java_env, ssh):
    
    iRet = changeCliToDebug(java_env, ssh)
    if True == iRet:
        iRet1 = ssh.execCmd("mml")
        if bool (re.search("MML>", iRet1, re.IGNORECASE)):
            return True
    
    return False

# **************************************************************************** #
# 函数名称: changeCliToDebug
# 功能说明: Cli模式转换成Debug模式
# 输入参数: ssh
# 输出参数: 无
# 返 回 值: True or False
# **************************************************************************** # 
def changeCliToDebug(java_env, ssh):

    password = unicode(java_env.get("devPwd").get("developer"))
    ssh.execCmd("developer")
    ssh.execCmdNoLog(password)
    retDeveloper = ssh.execCmdNoLog("debug")
    if - 1 != retDeveloper.find('Password:'):
        ssh.execCmdNoLog(password)
    return True

def selectErrorMsg(lang, zhErrMsg, enErrMsg):
    if lang == 'zh':
        return zhErrMsg
    else:
        return enErrMsg

def getParseExceptErrMsg(lang):
    """
    # *****************************************************************************************#
    # 函数名称: getParseExceptErrMsg(lang)
    # 功能说明: CLI命令解析异常时提示
    # 输入参数: lang
    # 返 回 值: errMsg
    # *****************************************************************************************#
    """
    if "zh" == lang:
        errMsg = u"\nCLI回显解析异常。"
    else:
        errMsg = u"\nCLI output resolving abnormality."
    return errMsg  

# *************************************************************#
# 函数名称: getVaultDiskLockSet
# 功能说明: 获取保险箱盘位置列表。
# 返回  :  (isQryOk, vaultDskLocSet, cliRet, errMsg)
# *************************************************************#
def getVaultDiskLockSet(cli, logger, lang='zh'):
    cli.execCmd('chgpagination off')
    dskPhyCliRet = cli.execCmd('showdisk -physic')
    dskLogicCliRet = cli.execCmd('showdisk -logic')
    
    errMsg = ''
    cliRet = dskPhyCliRet + '\n' + dskLogicCliRet
    vaultDskLocSet = set()
    
    if (not re.search('Disk Information', dskPhyCliRet, re.IGNORECASE)
     or not re.search('Disk Information', dskLogicCliRet, re.IGNORECASE)):
        zhErrMsg = u'\n查询硬盘信息失败。'
        enErrMsg = '\nQuery disk information failed.'
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        return (False, vaultDskLocSet, cliRet, errMsg)
    
    try:
        dskPhyInfoDictList = parseHorizontal(dskPhyCliRet).getResult()
        dskLogicInfoDictList = parseHorizontal(dskLogicCliRet).getResult()
        for dskPhyInfoDict in dskPhyInfoDictList:
            diskLocStr = dskPhyInfoDict.get('Disk Location').strip()
            (isParseOk, diskPhyLoc, errMsg) = convertStrLocToTupleLoc(diskLocStr, logger, lang)
            if not isParseOk:
                return (False, vaultDskLocSet, cliRet, errMsg)
            
            diskSlotNo = diskPhyLoc[1]
            if diskSlotNo not in [0,1,2,3]:#只有保险箱盘的槽位号才可能是0,1,2,3,4
                continue
            
            phyCapacityStr = dskPhyInfoDict.get('Raw Capacity(Gb)')
            phyCapacity = int(phyCapacityStr)
            
            for dskLogicInfoDict in dskLogicInfoDictList:
                diskLocStr = dskLogicInfoDict.get('Disk Location').strip()
                (isParseOk, diskLogicLoc, errMsg) = convertStrLocToTupleLoc(diskLocStr, logger, lang)
                if not isParseOk:
                    return (False, vaultDskLocSet, cliRet, errMsg)

                diskSlotNo = diskLogicLoc[1]
                if diskSlotNo not in [0,1,2,3]:#只有保险箱盘的槽位号才可能是0,1,2,3,4
                    continue
                                        
                if diskLogicLoc == diskPhyLoc:
                    diskLogicCapacityStr = dskLogicInfoDict.get('Usable Capacity(Gb)')
                    diskLogicCapacity = int(diskLogicCapacityStr)
            
                    capacityGap = phyCapacity - diskLogicCapacity
                    if capacityGap in [VAULT_DISK_CAPACITY_GAP_MIN, VAULT_DISK_CAPACITY_GAP_MAX]:
                        vaultDskLocSet.add(diskPhyLoc)
                        break
            if len(vaultDskLocSet) == 4:
                break
    except Exception, e:
        logger.error('Parse disk information exception:' + unicode(e))
        zhErrMsg = u'\n查询硬盘信息失败。'
        enErrMsg = '\nQuery disk information failed.'
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        return (False, vaultDskLocSet, cliRet, errMsg)       
    else:
        return (True, vaultDskLocSet, cliRet, errMsg)

# *************************************************************#
# 函数名称: convertStrLocToTupleLoc
# 功能说明: 将硬盘字符串形式的框槽号转化成元组形式:'(1,2)' ---->(1,2)
# 返回  :  (isParseOk, tupleLoc, errMsg)
# *************************************************************#
def convertStrLocToTupleLoc(diskLocStr, logger, lang):
    try:       
        frameNo = int(diskLocStr.split(',')[0].split('(')[1].strip())
        slotNo = int(diskLocStr.split(',')[1].split(')')[0].strip())
    except IndexError:
        logger.error('Convert string disk location to tuple location caught IndexError exception:' + unicode(diskLocStr))
        zhErrMsg = u'\n解析硬盘位置信息失败。'
        enErrMsg = '\nParse disk location failed.'
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        
        return (False, (), errMsg)
    except ValueError:
        logger.error('Convert string disk location to tuple location caught IndexError exception:' + unicode(diskLocStr))
        zhErrMsg = u'\n解析硬盘位置信息失败。'
        enErrMsg = '\nParse disk location failed.'
        errMsg = selectErrorMsg(lang, zhErrMsg, enErrMsg)
        logger.error('Convert string disk location to tuple location caught ValueError exception:' + unicode(diskLocStr))
        return (False, (), errMsg)
    else:
        return (True, (frameNo,slotNo), '')


def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    @注：该方法用于解析老产品（V100R005C00及之后版本）业务资源检查回显公共方法
    '''
    try:
        Dict = {}
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_line = re.compile("[a-zA-Z\s]+[:=][\s]*[\d]+") 
            findall_lineList = reg_line.findall(line)
            split_Mark_Colon =":"
            split_Mark_Equal ="="
            for item in findall_lineList:
                key = ""
                val = ""
                if split_Mark_Colon in item:
                    list = item.split(split_Mark_Colon)
                    key = list[0].strip()
                    val = list[1].strip()
                    Dict.setdefault(key,val)
                if split_Mark_Equal in item:
                    list = item.split(split_Mark_Equal)
                    key = list[0].strip()
                    val = list[1].strip()
                    Dict.setdefault(key,val)
        return Dict
    except:
        return {}

def getMsg(lang, msg, args="", resource=resource.ERROR_MESSAGE_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg

def execCmdLlist(cli,cmdlist):
    '''
    @summary: 获取CLI命令回显,不做判断只进行收集，单次执行
    @param cli: cli对象
    @param cmdlist: CLI命令列表
    @return: CLI回显
             True:获取成功
             False:获取失败返回空
    '''
    cliRet = ""
    cliRetAll = ""
    try:
        for cmd in cmdlist:
            cliRet = cli.execCmd(cmd)
            cliRetAll += cliRet
            
        return cliRetAll   
    except:
        return cliRetAll

def repeExecCmdLlist(cli,cmdlist,totaltime,interval, PY_JAVA_ENV, logger, isRefressProcess = False):
    '''
    @summary: 获取CLI命令回显,不做判断只进行收集，周期执行
    @param cli: cli对象
    @param cmdlist: CLI命令列表
    @param totaltime: 总时间
    @param interval: 间隔时间
    @return: CLI回显
             True:获取成功
             False:获取失败返回空
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    cliRetAll = ""
    
    try:
        #每10秒执行一次，持续观察1分钟，查询7次。
        retriedTimes = int(totaltime / interval)
        retriedTimes = retriedTimes + 1
        curentProcess = 0
        if isRefressProcess:
            curentProcess = 6
            singleStep = 90 / (len(cmdlist) * (totaltime / interval))
            nodestep = singleStep
        for i in range(0, retriedTimes):
            for cmd in cmdlist:                
                cliRet = cli.execCmd(cmd)
                cliRetAll+=cliRet
                if isRefressProcess:
                    if nodestep >1:
                        curentProcess += int(nodestep)
                        nodestep = singleStep
                        refreshProcess(PY_JAVA_ENV, curentProcess, logger)
                    else:
                        nodestep += singleStep    
            time.sleep(interval)
        return cliRetAll
    except:
        return cliRetAll
        
def getLunIdList(cliRet):
    '''
    @summary: 获取showlun的ID列表
    @param cliRet: cli命令回显
    @return: LUNID列表
    '''
    try:         
        list = cliRet.splitlines()
        lunIdList = []
        for line in list[6:]:
            lineList =line.split()
            if len(lineList) > 2:
                LUNID = str(lineList[0])
                lunIdList.append(LUNID)
                
        return lunIdList
    
    except:
        #处理异常反空
        return []
def checkLunCacheResourse(cli,lang,logger):
    '''
    @summary: LUN的Cache资源检查
    @param lunIdList: LUN的ID列表
    @return: 1、flag
             True：检查通过
             False：检查不通过
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    cliRetAll = ""
    #获取LUNID列表
    cliRet = cli.execCmd("showlun")
    cliRetAll += cliRet
    
    lunIdList = getLunIdList(cliRet)
    logger.info("lunIdList is" + str(lunIdList))
    if len(lunIdList) == 0:
        flag = True
        return flag, cliRet, errMsg 

    flagList = []
    #记录不通过的LUN，及其对应场景的次数,ID为key，[times,times,times,times]为值，times为对应各条件下不通过的次数
    notPassCountDict = {}
    
    #最多会循环三次执行
    for times in range(3):
        for ID in lunIdList:
            cmd = "showcacheluninfo -lun " + ID
            cliRet = cli.execCmd(cmd)
            cliRetAll += cliRet
            if not checkCliInfoValid(cliRet, False):
                flag = False
                errMsg += getMsg(lang, "invalid.cli.information")
                return (flag, cliRetAll, errMsg)
            
            #命令不存在时检查通过,ommand为Command兼容“C”的大小写情况
            if "not exist" in cliRet and "ommand" in cliRet:
                flag = True
                return (flag, cliRetAll, errMsg)
            
            #存在私有LUN情况时，跳过私有LUN继续检查剩余LUN情况
            if PRIVATE_LUN_RET in cliRet:
                continue
            
            Index = "Show Cache"
            try:
                ShowCacheIndex  = str(cliRet).rindex(Index)
            except:
                Index = "Show cache"
                ShowCacheIndex  = str(cliRet).rindex(Index)
            CliRet2Controller0 = ""    
            CliRet2Controller1 = ""    
            CliRet2Controller0 = cliRet[:(ShowCacheIndex)]
        
            #获取控制器1的回显并去掉结束符
            CliRet2Controller1 = cliRet[(ShowCacheIndex):].replace("developer: admin:/>","")
            CliRet2ControllerList =[] 
            logger.info("CliRet2Controller0 is " + str(CliRet2Controller0))
            logger.info("CliRet2Controller1 is " + str(CliRet2Controller1))
            CliRet2ControllerList.append(CliRet2Controller0)
            CliRet2ControllerList.append(CliRet2Controller1)
            
            #兼容单控情况，如果无法获取local control为1的信息则直接通过
            Local_Control_1 = ""
            for ret in CliRet2ControllerList:
                if LOCAL_CONTROL_1 in ret:
                    Local_Control_1 = ret
                    break   
            if Local_Control_1 == "":
                return (True, cliRetAll, errMsg)
    
            #获取回显字典
            cmdInfoDict = getHorizontalNostandardCliRet(Local_Control_1)
            
            #第一次检查时，需要添加该LUN的信息,并初始化所有条件计数为0
            if ID not in notPassCountDict:
                notPassCountDict[ID] = [0, 0, 0, 0]
            
            #阈值计算与判断方法入口   
            checkFlag, msg = calculateLunCacheResourse(cmdInfoDict, ID, notPassCountDict, logger, lang)
            errMsg += msg
            
            #去除notPassCountDict中不存在连续满足同一条件的lun；若times + 1在该字典中，则表明某个条件连续满足
            if (times + 1) not in notPassCountDict[ID]:
                notPassCountDict.pop(ID)
            
            #检查过程出错
            if not checkFlag:
                flag = False
        
        #下一轮循环，只需遍历存在问题的LUN        
        lunIdList = notPassCountDict.keys()
        lunIdList.sort()
        if len(lunIdList) == 0:
            break 
       
        #第三次检查完毕后，不用睡
        if times != 2:
            time.sleep(10)  
    
    #3次遍历完后，如果notPassCountDict还有元素，即lunIdList不为空，则存在问题
    if len(lunIdList) != 0:
        flag = False
        
    return (flag, cliRetAll, errMsg)

def checkCacheResourse(cli,lang,totaltime,interval,logger, py_java_env):
    '''
    @summary: Cache资源检查
    @param lunIdList: LUN的ID列表
    @return: 1、flag
             True：检查通过
             False：检查不通过
    '''
    
    flag = True
    cliRet = ""
    errMsg = ""
    cliRetAll = ""
    CliRet2Controller0 = ""    
    CliRet2Controller1 = ""    
    try:
        #获取A、B控高水位
        cliRet = cli.execCmd("showcache")
        cliRetAll += cliRet
        list = cliRet.splitlines()
        if not checkCliInfoValid(cliRet, False):
            flag = False
            errMsg += getMsg(lang, "Invalid.Cli.information")         
            return (flag, cliRetAll, errMsg)
        if len(list) < 5:
            flag = True
            return (flag, cliRetAll, errMsg)
        
        #回显判断
        Index = "Controller ID"
        ControllerIDIndex  = str(cliRet).rindex(Index)
        CliRet2Controller0 = cliRet[:(ControllerIDIndex)]
        #获取控制器1的回显并去掉结束符
        CliRet2Controller1 = cliRet[(ControllerIDIndex):].replace("developer: admin:/>","")
        DictA = getHorizontalStandardCliRet(CliRet2Controller0)                
        DictB = getHorizontalStandardCliRet(CliRet2Controller1)                
        High_Water_Level_A = float(DictA.get("High Water Level(%)"))*0.01
        High_Water_Level_B = float(DictB.get("High Water Level(%)"))*0.01
        
        #每10秒执行一次，持续观察1分钟，查询7次。
        retriedTimes = int(totaltime / interval)
        currentProcess = 6    
        singleStep = 85 / retriedTimes
        retriedTimes = retriedTimes + 1
        dictInfo = {"0":[0,0,0,0],"1":[0,0,0,0]}
        for i in range(0, retriedTimes):
            cliRet = cli.execCmd("showcachepartition -pr 0")
            cliRetAll += cliRet
            
            if not checkCliInfoValid(cliRet, False):
                flag = False
                errMsg += getMsg(lang, "Invalid.Cli.information")         
                return (flag, cliRetAll, errMsg)
            
            list = cliRet.splitlines()
            if len(list) < 5:
                flag = True
                return (flag, cliRetAll, errMsg)    
            #命令不存在时检查通过
            if "not exist" in cliRet:
                flag = True
                return (flag, cliRetAll, errMsg)
            #兼容大小写Cache
            Index = "Show Cache"
            try:
                ShowCacheIndex  = str(cliRet).rindex(Index)
            except:
                Index = "Show cache"
                ShowCacheIndex  = str(cliRet).rindex(Index)
              
            CliRet2Controller0 = cliRet[:(ShowCacheIndex)]
        
            #获取控制器1的回显并去掉结束符
            CliRet2Controller1 = cliRet[(ShowCacheIndex):].replace("developer: admin:/>","")
            logger.info("CliRet2Controller0 is " + str(CliRet2Controller0))
            logger.info("CliRet2Controller1 is " + str(CliRet2Controller1))
            
            #兼容单控情况，如果无法获取local control为1的信息则直接通过
            #获取回显字典
            CacheDictList = []
            CacheDict0 = getHorizontalNostandardCliRet(CliRet2Controller0)
            CacheDict1 = getHorizontalNostandardCliRet(CliRet2Controller1)
            CacheDictList.append(CacheDict0)
            CacheDictList.append(CacheDict1)
            #阈值计算与判断方法入口
            

            dictInfo = calculateCacheResourse(dictInfo, High_Water_Level_A,High_Water_Level_B,CacheDictList,logger,lang)
            
            time.sleep(interval)
            # 刷新进度条      
            currentProcess = refreshProcessByStep(currentProcess, singleStep, py_java_env, logger)
            
        for key in dictInfo:
            ag1 = dictInfo[key][0]
            ag2 = dictInfo[key][1]
            ag3 = dictInfo[key][2]
            ag4 = dictInfo[key][3]
            Controller_Id = key
            #控制器ID转换，将0,1转换为A,B
            if Controller_Id == "0":
                Controller_Id = "A"
            if Controller_Id == "1":
                Controller_Id = "B"
            
            #1 showcachepartition -pr 0回显中计算每个控制器写页面(Write alloced count/Write page quota -high water level)/1-high water level 每10秒执行一次，
            # 检查1分钟共7次，如果5次超过50%则巡检不通过；
            #兼容算法Write_alloced_count/float(Write_page_quota) - High_Water_Level)小于等于“0”的情况，将结果置为“0”
            if ag1 >= 5:
                flag = False
                errMsg += getMsg(lang, "control.cache.write.page",Controller_Id)
            
            #2 showcachepartition -pr 0回显中计算每个控制器刷盘并发请求数write disk req count/destage max req每10秒执行一次，
            # 检查1分钟共7次，如果5次超过90%则巡检不通过；
            if ag2 >= 5:
                flag = False
                errMsg += getMsg(lang, "control.cache.concurrent.disk.flushing.request",Controller_Id)
             
            #3 showcachepartition -pr 0回显中计算每个控制器刷盘并发页面数write disk page /destage max page每10秒执行一次，
            # 检查1分钟共7次，如果5次超过90%则巡检不通过；   
            if ag3 >= 5:
                flag = False
                errMsg += getMsg(lang, "control.cache.disk.flushing.page",Controller_Id)
            
            #4 showcachepartition -pr 0回显中计算每个控制器离散chunk数discrete dirty chunk /discrete dirty limit每10秒执行一次，
            # 检查1分钟共7次，如果5次超过90%则巡检不通过；
            if ag4 >= 5:
                flag = False
                errMsg += getMsg(lang, "control.cache.discrete.chunk",Controller_Id)

        logger.info("dictInfo is "  + str(dictInfo))
        return (flag, cliRetAll, errMsg)
    except:
        return (False, cliRetAll, errMsg) 
                    

def getHorizontalStandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    @注：该方法用于解析老产品（V100R005C00及之后版本）业务资源检查回显公共方法
    '''
    try:
        Dict = {}
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            split_Mark_Colon =":"
            split_Mark_Equal ="|"
            
            key = ""
            val = ""
            if split_Mark_Colon in line:
                list = line.split(split_Mark_Colon)
                key = list[0].strip()
                #对值进行有效性判断
                val = list[1].strip()
                Dict.setdefault(key,val)
            if split_Mark_Equal in line:
                list = line.split(split_Mark_Equal)
                key = list[0].strip()
                val = list[1].strip()
                Dict.setdefault(key,val)
        return Dict
    except:
        return {}

        
def calculateLunCacheResourse(cmdInfoDict, ID, notPassCountDict, logger, lang):
    '''
    @summary: LUN的Cache资源检查的阈值计算方法
    @param cliRet: 回显字典
    @param lang: 语言对象
    @return: 
             True：检查通过
             False:检查不通过
    '''
    errMsg = ""
    flag = True
    try:
        Dirty_count = float(cmdInfoDict.get("Dirty count"))
        Lun_high_level = float(cmdInfoDict.get("Lun high level"))
        Lun_quota = float(cmdInfoDict.get("Lun quota"))
        Send_Raid_Cur_Num = float(cmdInfoDict.get("Send Raid Cur Num"))
        Send_Raid_Max_Num = float(cmdInfoDict.get("Send Raid Max Num"))
        Destage_page_num = float(cmdInfoDict.get("Destage page num"))
        Max_destage_pages = float(cmdInfoDict.get("Max destage pages"))
        Discrete_chunk = float(cmdInfoDict.get("Discrete    chunk"))
        Discrete_quota = float(cmdInfoDict.get("Discrete quota"))

        #兼容算法Dirty_count - Lun_high_level为小于等于“0”的情况，将结果置为“0”
        divisor = Dirty_count - Lun_high_level
        if divisor <= float(0):
            divisor = float(0)
             
        #1 showcacheluninfo -lun lunid回显中计算local control为1控制器的单个LUN的水位，如果(dirty count - lun high level)/(lun quota - lun high level)超过50%则巡检不通过；
        if divisor/float(Lun_quota - Lun_high_level)> float(0.5):
            notPassCountDict[ID][0] += 1
            if notPassCountDict[ID][0] == 3:
                errMsg += getMsg(lang, "lun.cache.watermark", ID)

        #2 showcacheluninfo -lun lunid回显中计算local control为1控制器的单个LUN的刷盘并发数，如果send raid cur num/send raid max num超过90%则巡检不通过；
        if float(Send_Raid_Cur_Num)/float(Send_Raid_Max_Num)> float(0.9):
            notPassCountDict[ID][1] += 1
            if notPassCountDict[ID][1] == 3:
                errMsg += getMsg(lang, "lun.cache.concurrent.disk.flushing.request", ID)

        #3 showcacheluninfo -lun lunid回显中计算local control为1控制器的单个LUN的刷盘页面数，如果destage page num /max destage pages超过90%则巡检不通过；
        if float(Destage_page_num)/float(Max_destage_pages)> float(0.9):
            notPassCountDict[ID][2] += 1
            if notPassCountDict[ID][2] == 3:
                errMsg += getMsg(lang, "lun.cache.concurrent.disk.flushing.page", ID)

        #4 showcacheluninfo -lun lunid回显中计算local control为1控制器的单个LUN的离散chunk数，如果discrete chunk/discrete quota超过90%则巡检不通过；
        if float(Discrete_chunk)/float(Discrete_quota)> float(0.9):
            notPassCountDict[ID][3] += 1
            if notPassCountDict[ID][3] == 3:
                errMsg += getMsg(lang, "lun.cache.discrete.chunk",ID)
        
        return True ,errMsg
       
    except:
        logger.error("calculate Lun Cache Resourse failed" )
        flag = False 
        return flag ,errMsg  
      
def calculateCacheResourse(dictInfo, High_Water_Level_A,High_Water_Level_B,CacheDictList,logger,lang):
    '''
    @summary: Cache资源检查的阈值计算方法
    @param cliRet: 回显字典
    @param lang: 语言对象
    @return: 
             True：检查通过
             False:检查不通过
    '''
    try:

        for dict in CacheDictList:
            Controller_Id = dict.get("Controller Id")
            if float(dict.get("Controller Id")) == 0:
                High_Water_Level = High_Water_Level_A
            if float(dict.get("Controller Id")) == 1:
                High_Water_Level = High_Water_Level_B
            Write_alloced_count = float(dict.get("Write  alloced count"))    
            Write_page_quota = float(dict.get("Write    page   quota"))    
            Write_disk_req_count = float(dict.get("Write disk req  count"))    
            Destage_max_req = float(dict.get("Destage    max    req"))    
            Write_disk_page = float(dict.get("Write    disk   page"))    
            Destage_max_page = float(dict.get("Destage   max   page"))    
            Discrete_dirty_chunk = float(dict.get("Discrete dirty chunk"))    
            Discrete_dirty_limit = float(dict.get("Discrete dirty limit"))    

            if Controller_Id in dictInfo:
                ag1 = dictInfo[Controller_Id][0]
                ag2 = dictInfo[Controller_Id][1]
                ag3 = dictInfo[Controller_Id][2]
                ag4 = dictInfo[Controller_Id][3]

            Cache_Value = Write_alloced_count/Write_page_quota - High_Water_Level
            if Cache_Value <= float(0):
                Cache_Value = float(0)
                
            if Cache_Value/(float(1)-High_Water_Level)> float(0.5):
                ag1 = ag1 + 1
            
            if float(Write_disk_req_count)/float(Destage_max_req)> float(0.9):
                ag2 = ag2 + 1
                
            if float(Write_disk_page)/float(Destage_max_page)> float(0.9):
                ag3 = ag3 + 1
                
            if float(Discrete_dirty_chunk)/float(Discrete_dirty_limit)> float(0.96):
                ag4 = ag4 + 1
                
            dictInfo[Controller_Id] = [ag1, ag2, ag3, ag4]
        return dictInfo
       
    except:
        logger.error("calculate Cache Resourse Failed")
        return {}    
        
        
# *************************************************************#
# 函数名称: getDiskNameInfo
# 功能说明: 将硬盘的位置与硬盘的盘符信息映射一一对应关系
# 返回  :  diskDict---硬盘位置与盘符的键值对{"(0,4)":"sda"}
# *************************************************************#
def getDiskNameInfo(cliRet):
    diskDict = {}
    diskName = ""
    diskLocation = ""
    cliRetLines = cliRet.splitlines()
    for line in cliRetLines:
        if re.search('userid',line,re.IGNORECASE):
            useridList = line.split()[1].split(",")[0].split(":")
            if len(useridList) >= 2:
                userid = useridList[1]
            slotidList = line.split()[1].split(",")[2].split(":")
            if len(slotidList) >= 2:
                slotid = slotidList[1][:-1]
            diskLocation = "(" + userid + "," + slotid + ")"
        if re.search('Name',line,re.IGNORECASE):
            for info in line.split(","):
                if re.search('Name',info,re.IGNORECASE):
                    diskNameList = info.split()
                    if len(diskNameList) >= 2:
                        diskName = info.split()[1]
        if diskLocation != "" and diskName != "":
            diskDict.setdefault(diskLocation, diskName)
            diskLocation = ""
            diskName = ""
    return diskDict

# *************************************************************#
# 函数名称: getUsageInfo
# 功能说明: 获取盘符的利用率值(-iostat -x 2 -N 31回显的解析)
# 返回  :  usageDict---硬盘位置与盘符的键值对{"sda":"0.51,0.52"}
# *************************************************************#
def getUsageInfoByName(cliRet, nameList):
    tmplist = cliRet.split("avg-cpu:")
    usageDict = {}
    for str in tmplist[2:]:
        lineList = str.splitlines()
        for line in lineList:
            if len(line.split()) != 0:
                name = line.split()[0]
                usage = line.split()[-1]
                if re.match("[0-9]+.[0-9]+",usage) == None:
                    usage = 0.0
                if name in nameList and usage != "":
                    if usageDict.has_key(name):
                        usages = usageDict.get(name).append(float(usage))
                        usageDict.setdefault(name,usages)
                        
                    else:
                        usageDict.setdefault(name,[float(usage)])
    return usageDict

def getDiskTypeInfo(cliRet):
    # *************************************************************#
    # 函数名称: getDiskTypeInfo
    # 功能说明: 获取硬盘位置与硬盘类型的字典类
    #  返回  :  typeDict---硬盘位置与硬盘类型键值对{"(0,4)":"SAS"}
    # *************************************************************#
    dicttmp = hardware.getHorizontalCliRet("DISK", cliRet)
    locatList = dicttmp.get("Disk Location")
    typeList = dicttmp.get("Type")
    typeDict = {}
    for location,type in zip(locatList,typeList):
        if location != "" and type in ["SATA","NL SAS","SAS","FC"]:
            typeDict.setdefault(location,type)
            location = ""
            type = ""

    return typeDict

# *************************************************************#
# 函数名称: isSystemNormal
# 功能说明: 检查系统当前是否双控正常
# 其 他   :  无
# *************************************************************#
def isSystemNormal(ssh, lang):
    
    IsDoubleMode = False
    
    systemInfoStr = ssh.execCmd("showsys")
    
    #查询是否是双控正常态
    if re.search("Double Controllers Normal", systemInfoStr, re.IGNORECASE):
        IsDoubleMode = True
    return IsDoubleMode,systemInfoStr

# *************************************************************#
# 函数名称: getHotpatchVersion
# 功能说明: 查询当前设备已安装热补丁
# 其 他   :  无
# *************************************************************#
def getHotpatchVersion(ssh, lang):
    
    #查询当前设备已安装热补丁
    cliRet = ""

    cliRet = ssh.execCmd("showupgradepkginfo -t 3")
    #不支持热补丁，检查通过
    if re.search("not exist.|Error : Invalid argument", cliRet, re.IGNORECASE):
        return (True, cliRet,'')

    #没有安装任何热补丁，返回空
    if re.search("command operates successfully", cliRet, re.IGNORECASE):
        return (True, cliRet,'')

    #导入了热补丁包但是没有激活
    if not re.search("Running", cliRet, re.IGNORECASE):
        return (True, cliRet, '')
    
    lineList = cliRet.splitlines()
    #信息少于7行，查询信息失败
    if len(lineList) < 7:
        return (False, cliRet,'')
    
    #正常情况下获取Running的热补丁版本
    for line in lineList:
        if re.search("A|B", line) and re.search("Running", line, re.IGNORECASE):
            field = line.split()
            if len(field) >= 2:
                curHotPatchVer = _switchHotPatchVersion(field[1])
                break
        else:
            continue 
    
    #检查结果返回
    return (True, cliRet,curHotPatchVer)

# **************************************************************************** #
# 函数名称: _switchHotPatchVersion
# 功能说明: 将热补丁版本号转化成有效的版本号（V100R002C00SPC011SPHb04转变成V100R002C00SPHb04）
# 输入参数: version
# 输出参数: 无
# 返 回 值: version
# **************************************************************************** # 
def _switchHotPatchVersion(version):
    
    startIndex = version.index("V100")
    #提取有效的版本号信息
    version = version[startIndex:]

    #老版本热补丁版本不规范：V100R002C00SPC011SPHb04
    if "SPC" in version:
        startIndex = version.index("SPC")
        endIndex = startIndex + 6
        delStr = version[startIndex: endIndex]
        version = version.replace(delStr, "")
    return version


def FcPortConvert(FcPortID):
    '''
    @summary: FC端口转换方法
    @param cliRet: 底层FC端口号，例：110100
    @return: 槽位号和端口号列表
    '''
    SoltPortIDList = []
    
    #倒数第三位为槽位号
    SoltID = FcPortID[-3]
    SoltPortIDList.append(SoltID)
    #倒数第一位为端口号
    PortID = FcPortID[-1]
    SoltPortIDList.append(PortID)
    
    return SoltPortIDList

def getPerfSwitchStatus(cli):
# **************************************************************************** #
# 函数名称: getPerfSwitchStatus
# 功能说明: 获取当前系统的PerfSwitchStatus
# 输入参数: cliRet, status
# 输出参数: 无
# 返 回 值: version
# **************************************************************************** # 
    status = ""
    cliRet = cli.execCmd("showperfswitch")
    if re.search("off", cliRet, re.IGNORECASE):
        status = "off"
    if re.search("on", cliRet, re.IGNORECASE):
        status = "on"
    return (cliRet,status)

def chgPerfSwitchStatus(cli,status):
# **************************************************************************** #
# 函数名称: getPerfSwitchStatus
# 功能说明: 获取当前系统的PerfSwitchStatus
# 输入参数: cliRet, status
# 输出参数: 无
# 返 回 值: version
# **************************************************************************** # 

    flag = False
    cliRet = cli.execCmd("chgperfswitch -sw " + status)
    if re.search("successfully", cliRet, re.IGNORECASE):
        flag = True
    return (flag,cliRet)
    
def refreshProcess(py_java_env, percentNumber, LOGGER):
    """
    @summary: 设置巡检当前进度
    """
    observer = py_java_env.get("progressObserver")
    try:
        if observer != None:
            observer.updateProgress(int(percentNumber))
    except:
        LOGGER.logInfo(str(traceback.format_exc()))   	

def getTimeDifferenceMinutes(oldTime, nowTime, logger):
    '''
    @summary: 获取时差，格式：（年 - 月 - 日 - 时 - 分），"20171223170556"
                            单位：秒
    '''
    timeDifference = ""
    try:
        oldTimeLen = len(oldTime)
        nowTimeLen = len(nowTime)
        if (oldTimeLen > 12) and (nowTimeLen > 12):
            if oldTime[:8] == nowTime[:8]:
                #去掉秒数 
                oldTime = (oldTime.strip())[:-2]
                nowTime = (nowTime.strip())[:-2]

                oldTimeNumber = len(oldTime)
                nowTimeNumber = len(nowTime)

                #如果是标准格式内，则 长度因为12
                if (oldTimeNumber == 12) and (nowTimeNumber == 12):
                    oldTimeYear = int(oldTime[:4])
                    oldTimeMonth = int(oldTime[4:6])
                    oldTimeDay = int(oldTime[6:8])
                    oldTimeHour = int(oldTime[8:10])
                    oldTimeMin = int(oldTime[10:12])

                    nowTimeYear = int(nowTime[:4])
                    nowTimeMonth = int(nowTime[4:6])
                    nowTimeDay = int(nowTime[6:8])
                    nowTimeHour = int(nowTime[8:10])
                    nowTimeMin = int(nowTime[10:12])

                    #计算出相差秒数
                    timeDifference = (datetime.datetime(nowTimeYear, nowTimeMonth, nowTimeDay, nowTimeHour, nowTimeMin)
                                      - datetime.datetime(oldTimeYear, oldTimeMonth, oldTimeDay, oldTimeHour, oldTimeMin)).seconds

        return timeDifference

    except:
        logger.error(str(traceback.format_exc()))
        return ""
    
def refreshProcessByStep(currentProcess, stepProcess, py_java_env, logger):
    """
    @summary: 用于循环中刷新进度
    """
    
    nodeProcess = stepProcess
    if nodeProcess > 1:
        currentProcess += int(nodeProcess)
        nodeProcess = stepProcess
        # 刷新进度条
        refreshProcess(py_java_env,currentProcess, logger)  
    else:
        nodeProcess += stepProcess 
    
    return currentProcess


def reconnection_cli(cli, logger):
    try:
        cli.execCmd("showsys")
        logger.info(
            "it is afresh to connect to device by ssh gallery.")
        return True
    except (ToolException, Exception) as e:
        logger.error(str(e))
        cli.reConnect()