# -*- coding: UTF-8 -*-

import re
import traceback


#硬件配置文件路径
HWConfigFile = r"cfg/hardwareConfig.xml"

#光模块：BOM编码和厂商对应关系
BOM_ModelMap = {
"34060365":("FTLF8524P3BNL-HW",
            "PLRXPLVESG4HWN",
            "LTD8542-BE+",
            "PLRXPLVESG462N"),
"3406G000":("AFBR-57D9AMZ",),
"34060521":("AFBR-57D7APZ",
            "AFBR-57D7APZ-HW1",
            "AFBR-57D9AMZ-HW1",
            "PLRXPLVCSH4HW",
            "FTLF8528P3BCV"),
"34060494":("TPP1XEDSRCZAB00",
            "LTF8502-BC+",
            "PLRXPLSCS43HW",
            "AFBR-703SDZ-HW1",
            "FTLX8571D3BCL-HW",
            "TRS2000EN-S003"),
"34060779":("FTLF8529P3BCV",),
"3406G001":("FCBG410QB1C15",),
"34060662-001":("JQPR04SRAB1HW",
                "LTA8513-PC+",
                "FTL410QE3C-HU",
                "TR-QQ85S-N00",),
"34060874":("1064155015",
            "7120-028-01")
}
# 硬件包含Key对应关系
HARDWAREKey_Map = {
"CTRL":("Controller ID",),
"EXP":("Controller ID",
       "Enclosure ID"),
"FAN":("Enclosure ID",
        "Fan ID"),
"BBU":("Controller ID",
        "BBU ID"),
"DISK":("Disk Location",
        "Status",
        "Type",
        "Vendor",
        "Model",
        "Serial Number",
        "FW Version",
        "Speed(RPM)",
        "Rate(Gbps)",
        "Raw Capacity(GB)",
        "BarCode"
        ),
"FIBRE":("Controller ID",
        "Enclosure ID",
        "Module Type",
        "Module ID ",
        "Port ID",
        "Vendor",
        "Model",
        "SN",
        "Status",
        "Data Rate(Gbps)"
        ),
        
"POWER":("Enclosure ID",
        "Power ID",
        "Status",
        "Vendor",
        "Type",
        "Date",
        "Model",
        "SN"
        ),
"ENCLOSURE":("Enclosure ID",
        "Type",
        "Status",
        "Temperature(Celsius)",
        "Physical Type",
        "Serial Number",
        "MAC Address"
        ),
        
}

CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"

def getItemFromELabelByEncId(EncId,hardwaretype,cliRet):
    cliRetLinesList = getVerticalCliRet(cliRet)
    for cliRetLines in cliRetLinesList:
        encId = cliRetLines.get(HARDWAREKey_Map.get(hardwaretype)[0],"")
        if EncId == encId:
            itemList = cliRetLines.get("Electronic Label_3", "").split("=")
            if len(itemList) >= 2:
                item = str(itemList[1])
                if item:
                    return item
                else:
                    continue
        else:
            continue
    return ""

def getNormalBomCodeByEncId(encloureID,hardwareType,cliRet):
    '''
    @summary: 对回显为水平对齐格式的信息解析，当对应的encloureID存在SN不为空时直接返回SN对应的BOM编码值
    @param cliRet: cli回显
    @param hardwareType：硬件类型（可取值：POWER）
    @return: BOM编码值
    '''

    hardwareDict = getHorizontalCliRet(hardwareType,cliRet)
    snList = hardwareDict.get("SN","")
    encIdList = hardwareDict.get("Enclosure ID","")
    for (encId,sn) in zip(encIdList,snList):
        if encId == encloureID and  _switchSN2BomCode(sn): 
            return _switchSN2BomCode(sn)
    return ""

def getNormalBomCode(hardwareType,cliRet):
    '''
    @summary: 对回显为水平对齐格式的信息解析，存在SN不为空时直接返回SN对应的BOM编码值
    @param cliRet: cli回显
	@param hardwareType：硬件类型（可取值：POWER）
    @return: BOM编码值
    '''

    hardwareDict = getHorizontalCliRet(hardwareType,cliRet)
    snList = hardwareDict.get("SN","")
    for sn in snList:
        bom =  _switchSN2BomCode(sn)
        if bom:
            return bom
        else:
            continue
    return ""

def getHorizontalCliRet(hardwareType,cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    keyList = HARDWAREKey_Map.get(hardwareType)
    valueList = []
    start_index = 0
    keystr = ""
    indexlist = []
    cliRetList = cliRet.encode("utf8").splitlines()
    for line in cliRetList: 
        index = line.find(keyList[0])
        if index >= 1:
            keystr = line
            indexlist.append(index)
            break
        start_index +=1
    for key in keyList[1:]:
        indexlist.append(keystr.find(key))
    endlist = []
    for i in range(len(indexlist)-1):
        tmplist = []
        endlist = []
        for line in cliRetList[(start_index + 2): -1]:
            list2 = line.split()
            if len(list2) < 3:
                continue
            tmplist.append(line[indexlist[i]:indexlist[i+1]].strip())
            endlist.append(line[indexlist[len(indexlist)-1]:].strip())
            
        valueList.append(tmplist)
    valueList.append(endlist)
    dicttmp = dict(zip(keyList,valueList))
    return dicttmp
           
def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    dictList = []
    try:
        cliRetList = cliRet.encode("utf8").splitlines()
        lineDict = {}
        findTimes = 1
        
        for line in cliRetList:
            if CLI_RET_END_FLAG in line:  
                break
            if re.search("-+", line):
                if findTimes != 1:
                    dictList.append(lineDict.copy())
                    lineDict.clear()
                    
                findTimes += 1
            
            fields = line.split("|")
            if len(fields) < 2:
                continue
           
            key = fields[0].strip().decode("utf8")
           
            value = "|".join(fields[1:len(fields)]).strip().decode("utf8")
            if key == "":
                key = "Electronic Label"
            
            if lineDict.has_key(key):
                key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
             
            lineDict.setdefault(key, value)
        if len(lineDict) > 0:
            dictList.append(lineDict.copy())
    except:
        return dictList
    return dictList

def getHardwareBomCode(idList,hardwareType,cliRet): 
    '''
    @summary: 获取指定ID的硬件的BOM编码值
    @param idList: 硬件ID列表值
    @param hardwaretype: 硬件类型（取值范围：EXP、FAN、CTRL、BBU、DISK、POWER、FIBRE、ENCLOSURE）
    @return: BOM编码值
    '''
    if hardwareType in ["EXP","FAN","CTRL","BBU"]:
        return getItemFromELabel(hardwareType,cliRet,idList)
    if hardwareType in ["DISK","POWER" ,"FIBRE", "ENCLOSURE"]:
        return getHorizontalHardwareBomCode(idList,hardwareType,cliRet) 
    return ""
def getHorizontalHardwareBomCode(idList,hardwareType,cliRet):
    '''
    @summary: 获取水平回显中指定ID硬件的BomCode相关的字段值（SN值/Model值）
    @param idList: 硬件ID列表值
    @param hardwaretype: 硬件类型
    @return: SN值/Model值
    '''
    hardwareDict = getHorizontalCliRet(hardwareType,cliRet)
    if hardwareType == "ENCLOSURE":
        IdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[0],"")
        snList =  hardwareDict.get("Serial Number","")
        for (id,sn) in zip(IdList,snList):
            if id == idList[0] and sn:
                return  _switchSN2BomCode(sn)
            else:
                continue
    elif hardwareType == "DISK":
        IdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[0],"")
        snList =  hardwareDict.get("BarCode","")
        for (id,sn) in zip(IdList,snList):
            if id == idList[0] and sn:
                return  _switchSN2BomCode(sn)
            else:
                continue
    elif hardwareType == "POWER":
        encloureIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[0],"")
        powerIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[1],"")
        snList =  hardwareDict.get("SN","")
        for i in range(0,len(encloureIdList)):
            encloureId = encloureIdList[i]
            powerId = powerIdList[i]
            sn = snList[i]
            if encloureId == idList[0] and powerId ==idList[1] and sn:
                return  _switchSN2BomCode(sn)
            else:
                continue
    elif hardwareType == "FIBRE":
        ctrlIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[0],"")
        encloureIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[1],"")
        moduleIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[3],"")
        portIdList =  hardwareDict.get(HARDWAREKey_Map.get(hardwareType)[4],"")
        modelList =  hardwareDict.get("Model","")
        for i in range(0,len(ctrlIdList)):
            ctrlId = ctrlIdList[i]
            encloureId = encloureIdList[i]
            moduleId = moduleIdList[i]
            portId = portIdList[i]
            model = modelList[i]
            if ctrlId == idList[0] and encloureId == idList[1] and moduleId  == idList[2] and portId == idList[3] and model:
                return  getFibreModuleBOMByModel(model)
            else:
                continue
    
    return "" 

def getItemFromELabel(hardwaretype,cliRet,idList):
    '''
    @summary: 获取指定ID硬件的Electronic Label字段中的Item值
    @param idList: 硬件ID列表值（当idList取值为"No"时，返回回显中存在不为空的Item的值）
    @param hardwaretype: 硬件类型
    @return: Item编码
    '''
    cliRetLinesList = getVerticalCliRet(cliRet)
    if idList == "No":
        for cliRetLines in cliRetLinesList:
            itemList = cliRetLines.get("Electronic Label_3", "").split("=")
            if len(itemList) >= 2:
                item = str(itemList[1])
                if len(item) != 0:
                    return item
                else:
                    continue
        return ""   
    elif len(idList) == 1:
        
        for cliRetLines in cliRetLinesList:
            id = cliRetLines.get(HARDWAREKey_Map.get(hardwaretype)[0],"")
            if id == idList[0]:
                itemList = cliRetLines.get("Electronic Label_3", "").split("=")
                if len(itemList) >= 2:
                    item = str(itemList[1])
                    if len(item) != 0:
                        return item
            else:
                continue
        return ""
                
    elif len(idList) == 2:
        for cliRetLines in cliRetLinesList:
            encId = cliRetLines.get(HARDWAREKey_Map.get(hardwaretype)[0],"")
            fruId = cliRetLines.get(HARDWAREKey_Map.get(hardwaretype)[1],"")
            if encId == idList[0] and  fruId == idList[1]:
                itemList = cliRetLines.get("Electronic Label_3", "").split("=")
                if len(itemList) >= 2:
                    item = str(itemList[1])
                    if len(item) != 0:
                        return item
            else:
                continue
        return ""
    else:
        return ""

def _switchSN2BomCode(SerialNumber):
    '''
    @summary: 从SN号中获取BOM编码
    @param SerialNumber: 硬件的SN号
    @return: BOM编码
    '''

    BomCode = ""
    #（1）以21开头的20位数字或字母的SN中，第3-10位代表这个物料对应物料种类的BOM编码。
    #以210305G06M10DC000253为例，这种控制器的BOM编码是0305G06M；
    if SerialNumber.startswith("21") and len(SerialNumber) == 20:
        BomCode = SerialNumber[2:10]
    #（2）以02或03开头的16位数字或字母，第1-6位前面加上03凑成8位数字或字母，代表BOM编码。
    #以02G258D0BB001344为例，它代表一个级联板唯一存在，BOM编码为0302G258；
    elif (SerialNumber.startswith("02") or SerialNumber.startswith("03"))\
        and len(SerialNumber) == 16:
        BomCode = "03" + SerialNumber[0:6]

    return BomCode

def getFibreModuleBOMByModel(Model):
    '''
    @summary: 通过光模块的BOM编码信息
    @param Model：光模块型号
    @return: BOM编码
    '''

    for bom,modelList in BOM_ModelMap.items():
        if Model in modelList:
            return bom
    return ""  