# -*- coding: UTF-8 -*-
import traceback
import re,time
from com.huawei.ism.tool.obase.entity import DevNode

#########################################
#      公共判断脚本
#########################################

PRODUCT_VERSION = "V200R001C00"

#非超级用户CLI命令开放的SPC版本号
CLI_SUPPORT_VERSION = "V200R001C00SPC900"

#非超级用户CLI命令没开放的SPC版本号
NOT_SUPPORT_VERSION = ["V200R001C00SPC100","V200R001C00SPC200","V200R001C00SPC300","V200R001C00SPC400",\
                       "V200R001C00SPC500","V200R001C00SPC600","V200R001C00SPC700","V200R001C00SPC800"]

###*****   巡检结果标志,不支持（命令因权限等原因不支持情况），通过，不通过   *****###
CHECK_NOSUPPORT = "NOSUPPORT"
CHECK_PASS = "TRUE"
CHECK_NOPASS = "FLASE"
CHECK_WARNING = "WARNING"

#返回此种结果，需要根据后续的逻辑判断来确实是否通过
CHECK_UNKOWN = "UNKOWN"

###*****   模式标志   *****###
SUPER_USER_FLAG = "admin:/>"
DEVELOPER_MODEL_FLAG = "developer:/>"

CLI_RET_END_FLAG = ":/>"

#重试次数，防止多次重试死循环
MAX_RETRYS = 5

#黑名单中字符串一定存在于前5行
KEY_WORD_VALID_LINES = 5

###*****   异常回显黑名单，前四行包含下面关键字，巡检不通过，此名单中返回CHECK_NOPASS   *****###
nopassDict = [
{"key_word":"Safe mode",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},
              
{"key_word":"minisystem",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"}, 
                            
{"key_word":"-bash:",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},

{"key_word":"Storage:",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},
           
{"key_word":"System is upgrading",
 "msg_zh":u"\n系统处于升级模式",
 "msg_en":"\nSystem is in upgrade Mode"},
           
{"key_word":"The system is powering off",
 "msg_zh":u"\n系统不正常，处于下电模式",
 "msg_en":"\nSystem is abnormal, powering off"},
           
{"key_word":"The user is offline",
 "msg_zh":u"\n系统不正常，用户离线",
 "msg_en":"\nThe user is offline"}, 
 
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},
             
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},
              
{"key_word":"Receive message failed",
 "msg_zh":u"\n接受信息失败",
 "msg_en":"\nReceive message failed"},
 
{"key_word":"Cmd execute except",
 "msg_zh":u"\n命令执行异常",
 "msg_en":"\nCmd execute except"},
             
{"key_word":"The system has an internal error",
 "msg_zh":u"\n设备内部错误",
 "msg_en":"\nThe system has an internal error"},
              
{"key_word":"The user account has been locked",
 "msg_zh":u"\n用户账号被锁定",
 "msg_en":"\nThe user account has been locked"},
    
{"key_word":"Unknown error",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"},
              
{"key_word":"error:",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"}                    
             ]

###*****   行黑名单，针对捕捉到某行回显信息，判断当前行是否显示异常，巡检不通过，此名单中返回CHECK_NOPASS   *****###
nopassLineDict = [
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},
           
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},
                 
{"key_word":"The system has an internal error",
 "msg_zh":u"\n设备内部错误",
 "msg_en":"\nThe system has an internal error"},
 
{"key_word":"Cmd execute except",
 "msg_zh":u"\n命令执行异常",
 "msg_en":"\nCmd execute except"},
                  
{"key_word":"Unknown error",
 "msg_zh":u"\nCLI命令执行错误",
 "msg_en":"\nExecute CLI command error"}
             ]

###*****   不支持名单，针对捕捉到某行回显信息，判断当前回显是否支持，此名单中返回CHECK_NOSUPPORT   *****###
#命令当前版本不支持时
nosupportDict = [
                               
{"key_word":"\^",
 "msg_zh":u"\n当前版本不支持此命令",
 "msg_en":"\nCommand not support at current version"} 
              ]

###*****   白名单可能存在的情况，含有以下标记表示直接巡检通过   *****###
#命令执行成功
CLI_EXEC_SUCCESS = "Command executed successfully"
#命令当前版本不支持，需要使用超时接口cliExecRetWithTimeout
NOT_SUPPORT_CMD_FLAG = "\^"
CLI_EXEC_NO_LICENSE = "Suggestion: Import a valid license file."
CLI_EXEC_LICENSE_VALID = "license file is unavailable."

###*****   白名单，前四行包含下面关键字，直接巡检通过，此名单中返回CHECK_PASS  *****###
#白名单，回显中捕捉到这列关键字时，巡检通过，巡检通过时无法添加详细信息，只需要列出关键字即可
passList = [CLI_EXEC_SUCCESS,NOT_SUPPORT_CMD_FLAG,CLI_EXEC_NO_LICENSE,CLI_EXEC_LICENSE_VALID]


# **************************************************************************** #
# 函数名称: isChinese
# 功能说明: 判断当前语言是否为中文
# 输入参数: lang
# 输出参数: True or False
# **************************************************************************** # 
def isChinese(lang):
    if "zh"==lang:
        return True
    return False

# **************************************************************************** #
# 函数名称: isExecSuccess
# 功能说明: 为了加速判定，根据回显结果判定命令是否执行成功（一般针对无记录的情况）
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False:
# **************************************************************************** # 
def isExecSuccess(cliRet):
    if re.search(CLI_EXEC_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserRet
# 功能说明: 根据CLI回显判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserRet(cliRet):
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if SUPER_USER_FLAG == lineList[-1].replace(" ",""):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserCli
# 功能说明: 根据传入的SSH连接执行CLI命令后判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserCli(cli):
    cliRet = cli.execCmdNoLog("show system general")
    return isSuperUserRet(cliRet)

# **************************************************************************** #
# 函数名称: isSuperUserJava
# 功能说明: 根据根据框架传入的参数判断是否为超级管理员用户
# 输入参数: cliRet,py_java_env
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserJava(py_java_env):
    userName =  py_java_env.get("devInfoMap").get("userName")
    if "admin" ==  userName:
        return True
    return False

# **************************************************************************** #
# 函数名称: isDeveloperModle
# 功能说明: 判断当前捕捉到的CLI回显是否在Developer模式下
# 输入参数: cliRet
# 输出参数: True:Developer模式，False:非Developer模式
# **************************************************************************** # 
def isDeveloperModle(cliRet):
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, productmodel)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        productmodel：产品型号
    '''
    product_model = ""
    cmd = "show system general"
    result, cliRet, errMsg = cliExecRetWithTimeout(cli, cmd, 5*60, lang)
    if result == CHECK_NOPASS:
        return (False, cliRet, errMsg)
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, cliRet, product_model)
        
    return (False, cliRet, "")


# **************************************************************************** #
# 函数名称: getCurrentVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC，为了加快获取速率，
#           一次获取成功后保存到框架提供的全局变量中
# 输入参数: cli, py_java_env
# 输出参数: currentVesion
# **************************************************************************** # 
def getCurrentVesion(cli, py_java_env):  
    currentVesion = ""
    currentVesion = py_java_env.get("currentVesion")
    
    #未在全局标记中获取到有效的版本号，则通过执行CLI命令查看
    if None == currentVesion or "" == currentVesion:        
        cliRet = cli.execCmdNoLog("show upgrade package")
        lineList = cliRet.splitlines()
        
        #直接从包含版本号的行开始
        for line in lineList[5:-1]:
            columnList = line.split()
            #找到关键字，直接返回当前版本号
            for column in columnList[2:-1]:
                if -1 != column.find(PRODUCT_VERSION):
                    currentVesion = column[0:17]
                    #在框架提供的全局变量里设置当前的版本号,精确到SPC
                    py_java_env.put("currentVesion", currentVesion)
                    return currentVesion           
    return currentVesion

def getProductVersion(cli):
    """
    @summary: 通过show system general获取产品版本（只到C版本）
    """
    productVersion = ""
    
    cliRet = cli.execCmd("show system general")
    if not cliRet:
        return (False, cliRet, '')
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Version":
            productVersion = fieldValue
            
        if len(productVersion) != 0 and productVersion != "--":
            return (True, cliRet, productVersion)
        
    return (False, cliRet, "")

def isControllerVersionSame(softwareVersionList): 
    '''
    @summary:校验控制器版本是否一致
    @param softwareVersionList: 控制器版本列表
    @return: 
        True:控制器版本一致
        False:控制器软件版本不一致
    '''  
    VersionSet = set(softwareVersion["Current Version"] for softwareVersion in softwareVersionList)
    if len(VersionSet) !=1:
        return False
    return True

def parse_upgradePackage(cli, lang):
    '''
    @summary: 执行show upgrade packge命令，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型,并返回CLI回显
    @param cli: cli对象
    @param lang: 语言lang
    '''    
    cmd = "show upgrade package"
    softwareVersionList = []
    hotPatchVersionList = []
    flag = True
    (result, cliRet, errMsg) = isCliExecRetInBlackList(cli, cmd, lang, True)
    if result:
        return ((False, cliRet, errMsg),softwareVersionList,hotPatchVersionList)
    
    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = getHorizontalCliRet(cliRet[(softwareVersionIndex):hotPatchVersionIndex])
    hotPatchVersionList = getHorizontalCliRet(cliRet[(hotPatchVersionIndex):])
    
    if len(softwareVersionList) == 0 or len(hotPatchVersionList) == 0:
        if "zh" == lang: 
            errMsg = u"\n无法获取升级包信息。"
        else:
            errMsg = "\nFailed to obtain the upgrade package."
        return ((False, cliRet, errMsg) , softwareVersionList , hotPatchVersionList)
        
    for softwareVersion in softwareVersionList:
        if softwareVersion.get("Current Version") == "--":   
            flag = False
            if "zh" == lang: 
                errMsg += u"\n控制器[%s]:无法获取控制器软件版本。" % softwareVersion.get("Name")
            else:
                errMsg += "\nController[%s]: Failed to obtain the controller software version." % softwareVersion.get("Name")
                
    if not isControllerVersionSame(softwareVersionList):
        if "zh" == lang: 
            errMsg += u"\n控制器软件版本不一致，请修复后再进行巡检检查。"
        else:
            errMsg += "\nController software versions are inconsistent, rectify the fault before performing the inspection."
        return ((False, cliRet, errMsg),softwareVersionList,hotPatchVersionList)
    
    return ((flag, cliRet, errMsg),softwareVersionList,hotPatchVersionList)


# **************************************************************************** #
# 函数名称: isSupportVesion
# 功能说明: 判断当前版本是否支持
# 输入参数: cli,py_java_env
# 输出参数: True or False
# **************************************************************************** # 
def isSupportVersion(cli, py_java_env):
    currentVesion =  getCurrentVesion(cli,py_java_env)
    #没有查询到版本号，最少还可以切入developer模式执行命令，不至于卡主，因此返回为False
    if ""==currentVesion:
        return False
    
    if currentVesion not in NOT_SUPPORT_VERSION: 
        return True
    return False

# **************************************************************************** #
# 函数名称: getUserNameCliRet
# 功能说明: 根据CLI回显获取用户名
# 输入参数: cli,py_java_env
# 输出参数: userName
# **************************************************************************** # 
def getUserNameCliRet(cliRet):
    userName = ""
    
    #输入CLI回显是否有效
    if None == cliRet or "" == cliRet: 
        return userName
    
    #执行命令获取当前登录的用户名      
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if lineList[-1].find(CLI_RET_END_FLAG):
        userName = lineList[-1].replace(" ","").replace(CLI_RET_END_FLAG,"")
    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取当前用户名
# 输入参数: cli,py_java_env
# 输出参数: userName
# **************************************************************************** # 
def getUserName(cli,py_java_env):
    userName = ""
    
    #从框架提供的接口获取
    userName = py_java_env.get("devInfoMap").get("userName")
    if None != userName and "" != userName: 
        #在框架提供的全局变量里设置当前的用户名
        py_java_env.put("userName", userName)
        return userName
    
    #从脚本保存好的全局变量中获取，防止框架提供的接口中无法获取的情况
    userName = py_java_env.get("userName")
    if None != userName and "" != userName: 
        return userName
    
    #执行命令获取当前登录的用户名      
    cliRet = cli.execCmdNoLog("show system general")
    
    userName = getUserNameCliRet(cliRet)
    py_java_env.put("userName", userName)
     
    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取登录用户名的权限级别
# 输入参数: cli,py_java_env
# 输出参数: super_admin,guest或admin
# **************************************************************************** # 
def getUserPrivilege(cli,py_java_env):
    
    userLevel = ""
    userName = getUserName(cli,py_java_env) 
    if None == userName or "" == userName: 
        return ""
    
    userLevel = py_java_env.get("userLevel")
    #未在全局标记中获取到当前用户登录的权限级别，则通过执行CLI命令查看
    if None == userLevel or "" == userLevel:
        cliRet = cli.execCmdNoLog("show user "+userName)
        lineList = cliRet.splitlines()
        
        #用户名一定包含在最后一行中
        for line in lineList[4:-1]:
            columnList = line.split()
            if len(columnList)>2:
                if userName == columnList[0]:
                    userLevel = columnList[1]
                    py_java_env.put("userLevel", userLevel)
                    return userLevel
    return userLevel

# **************************************************************************** #
# 函数名称: change2cli
# 功能说明: 任何模式尝试切换到CLI模式        
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def anyModel2Cli(cli, py_java_env):
    index =0 
    cliRet = cli.execCmdNoLog("show system general")
    
    userName = getUserName(cli,py_java_env)
    if ""== userName:
        userName = "admin"
    
    #已经在CLI模式下，无需退出
    if userName == getUserNameCliRet(cliRet):
        return
    
    #密码输入错误时及正常情况从developer模式下退出
    if -1 != cliRet.find("Password") or -1 != cliRet.find(DEVELOPER_MODEL_FLAG):
        cliRet = cli.execCmd("exit")
        while(-1 == cliRet.find(userName+CLI_RET_END_FLAG)):
            index+=1
            cliRet = cli.execCmd("exit")
            if -1!=cliRet.find("Are you sure to exit?(y/n):"):
                cliRet = cli.execCmd("n")
            if index>MAX_RETRYS:
                break
    return
            
# **************************************************************************** #
# 函数名称: change2Developer
# 功能说明: 切换到developer模式
# 输入参数: cli,py_java_env
# 输出参数: True:，False:
# **************************************************************************** # 
def change2Developer(cli, py_java_env):
    errMsg = ""
    flag = False
    lang = py_java_env.get("lang")
    if None == cli:    
        return (flag,"",errMsg)
    
    #guest用户无法切换到developer模式
    userLevel = getUserPrivilege(cli,py_java_env)
    if "guest" ==userLevel or ""==userLevel:
        if isChinese(lang):
            errMsg = u"\n系统没有运行在admin模式，无法进入调试模式"
        else:
            errMsg = "\nSystem is not in admin model, can't enter to debug model"
        return (flag,"",errMsg)

    checkRet = isCliExecRetInBlackList(cli,"developer",lang,True)
    cliRet = checkRet[1]
    errMsg = checkRet[2]
    if checkRet[0]:
        flag = False
        anyModel2Cli(cli, py_java_env)
        return (flag, cliRet, errMsg)
    
    password = str(py_java_env.get("devPwd").get("developer"))
    if(None == password or ""== password):
        anyModel2Cli(cli, py_java_env)
        if isChinese(lang):
            errMsg = u"\n获取developer密码为空，请确定是否已配置密码"
        else:
            errMsg = "\nGet the password of developer is null, make sure to configure it right"
        return (flag, cliRet, errMsg)
    
    checkRet = isCliExecRetInBlackList(cli,password,lang,False)
    cliRet = checkRet[1]
    errMsg = checkRet[2]
    if checkRet[0]:
        flag = False
        anyModel2Cli(cli, py_java_env)
        return (flag, cliRet, errMsg)
    
    #已经成功进入Developer模式
    if isDeveloperModle(cliRet):
        flag = True
    else:
        anyModel2Cli(cli, py_java_env)
        
        #再次判断CLI回显是否在黑名单中，由于进入失败，需要全部判断
        lineList = cliRet.splitlines()
        for line in lineList:
            checkRet = isLineInBlackDict(line, lang)
            if checkRet[0]:
                flag = False
                errMsg = checkRet[1]     
                return (flag, cliRet, errMsg)   
                
        if isChinese(lang):
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):
                errMsg = u"\ndeveloper密码配置错误"
            else:
                errMsg = u"\n系统不能进入developer模式"
        else:
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):     
                errMsg = "\nThe password of developer is wrong"
            else:
                errMsg = "\nIt failed to access the developer mode" 
    return (flag, cliRet, errMsg)


def changeCli2DiagnoseWithCliRet(cli, py_java_env):
    '''
            函数名称: changeCli2Diagnose
            功能说明: 从CLI模式进入diagnose模式（需要确定使用之前为CLI模式）
            输入参数: ssh
            输出参数: 无
            返 回 值: True or False
    '''
    flag, cliRet0, errMsg = change2Developer(cli, py_java_env)
    if not flag:
        return flag, cliRet0, errMsg
    
    cliRet1 = cli.execCmd("debug")
    if re.search('diagnose>', cliRet1, re.IGNORECASE):
        return True, cliRet0 + "\n" + cliRet1, errMsg
   
    #进入diagnose模式失败，重新回到CLI模式
    anyModel2Cli(cli, py_java_env)
    return False, cliRet0 + "\n" + cliRet1, errMsg


# **************************************************************************** #
# 函数名称: developer2Cli
# 功能说明: developer模式下退出到CLI模式
# 输入参数: cli
# 输出参数: 无
# **************************************************************************** # 
def developer2Cli(cli):
    cli.execCmd("exit")
   
# **************************************************************************** #
# 函数名称: isEndLine
# 功能说明: 判断此行是否是回显结束行
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isEndLine(cliLine):
    
    #空行，无效行 非结束标志
    if "" == cliLine or None == cliLine:
        return False
    
    if -1 != cliLine.find(CLI_RET_END_FLAG):
        return True     
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithStart
# 功能说明: 跳过以某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithStart(cliLine, startWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(startWordList):
        return False
    

    for startWord in startWordList:
        #是以XXX起头的行，需要跳过
        if line.startswith(startWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithContain
# 功能说明: 跳过包含某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithContain(cliLine, containWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(containWordList):
        return False
    

    for containWord in containWordList:
        #包含XXX关键字的行，需要跳过
        if -1 != line.find(containWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isLineInBlackDict
# 功能说明: 判断某行回显是否在黑名单中，在黑名单中表示CLI回显存在异常，巡检不通过，
# 不用逻辑判断，最终应返回CHECK_NOPASS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInBlackDict(cliLine, lang):
    
    errMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于黑名单
    if None == cliLine or "" == cliLine:
        return (False, errMsg)
    
    for dictItems in nopassDict:
        #黑名单中找到，表示回显无效，直接退出
        if re.search(dictItems.get("key_word"), cliLine, re.IGNORECASE):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

# **************************************************************************** #
# 函数名称: isLineInWhiteList
# 功能说明: 判断某行回显是否在白名单中，在白名单中表示CLI执行完成，巡检通过，
# 不用逻辑判断，最终应返回CHECK_PASS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInWhiteList(cliLine):
       
    keyWordMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于白名单
    if None == cliLine or "" == cliLine:
        return (False, keyWordMsg)

    for keyWords in passList:      
        #在白名单中找到关键字，表示命令执行成功，直接退出
        if re.search(keyWords, cliLine, re.IGNORECASE):
            keyWordMsg = keyWords
            return (True, keyWordMsg)
        
        #无license情况通过
        if re.search("license", cliLine, re.IGNORECASE):
            if -1 != cliLine.find("Error:") or -1 != cliLine.find("Suggestion:"):
                keyWordMsg = "license"
                return (True, keyWordMsg)
        
    return (False,keyWordMsg)

# **************************************************************************** #
# 函数名称: isNormalLine
# 功能说明: 判断某行回显是否正常
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isNormalLine(cliLine, lang):
    errMsg = ""
    
    #空行，无效行正常
    if None == cliLine or "" == cliLine:
        return (True, errMsg)
    
    #异常的关键字存在于前几行，防止全部判断时误判
    #某行是否因为超时或是其它原因，导致端口信息查询不到的情况，返回不正常行
    for dictItems in nopassLineDict:
        if re.search(dictItems.get("key_word"), cliLine, re.IGNORECASE):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (False, errMsg)
    return (True, errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInBlackList
# 功能说明: 传入待执行CLI命令，判断回显是否在黑名单中，带回结果供各个巡检项调用
#           1）黑名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回异常回显对应的中英文提示信息
# **************************************************************************** # 
def isCliExecRetInBlackList(cli, cmd, lang, isHasLog):
    
    errMsg = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (True, cliRet, errMsg)
    
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，则回显正常，不在黑名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, errMsg)
    
    #回显在白名单中直接返回，不带回key值
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            return (False, cliRet, errMsg) 
    
    #是否在黑名单中，若是获取具体错误原因
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (True, cliRet, errMsg)  
    
    #不在黑名单中 
    return (False,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInWriteList
# 功能说明: 传入待执行CLI命令，判断回显是否在白名单中，带回结果供各个巡检项调用
#           1）白名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回捕捉到的白名单关键字信息
# **************************************************************************** # 
def isCliExecRetInWriteList(cli, cmd, lang, isHasLog):
    
    writeKeyWords = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        return (False, cliRet, writeKeyWords)
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，不在白名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, writeKeyWords)
    
    for line in lineList[0:KEY_WORD_VALID_LINES]:
               
        #是否在白名单中
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            writeKeyWords = checkRet[1]
            return (True, cliRet, writeKeyWords)
    
    #不在白名单中
    return (False,cliRet,writeKeyWords)


# **************************************************************************** #
# 函数名称: cliRetPreCheck（供公共接口cliExecRetJudgeWithDoubleList及
#           cliExecRetWithTimeout调用）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cliRet（CLI回显内容），lang
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 
def cliRetPreCheck(cliRet, lang):
    errMsg = ""
    
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (CHECK_NOPASS, cliRet, errMsg)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，说明回显执行一定有结果，需要通过后续逻辑去判断
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (CHECK_UNKOWN, cliRet, errMsg)
    
    #在白名单中，巡检可以直接通过
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CHECK_PASS, cliRet,"")
        
    #在黑名单中，巡检直接不通过    
    for line in lineList[0:KEY_WORD_VALID_LINES]:   
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CHECK_NOPASS, cliRet, errMsg) 
        
    #其它情况，说明需要通过回显来判断此项巡检是否通过   
    return (CHECK_UNKOWN,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: cliExecRetJudgeWithDoubleList（新增接口）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 
def cliExecRetJudgeWithDoubleList(cli, cmd, lang, isHasLog):
    
    #CLI命令执行是否打印日志 
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    return cliRetPreCheck(cliRet, lang)


# **************************************************************************** #
# 函数名称: cliExecRetWithTimeout（公共调用函数）
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项，本函数带超时处理机制
#           1）黑名单回显，返回：CHECK_NOPASS，巡检不通过
#           2）白名单回显，返回：CHECK_PASS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CHECK_UNKOWN，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CHECK_PASS，CHECK_NOPASS or CHECK_UNKOWN
# **************************************************************************** # 

def cliExecRetWithTimeout(cli, cmd, timeOut, lang):
    
    #执行带超时设置的CLI命令
    cliRet = cli.execCmdWithTimout(cmd,timeOut)
    
    return cliRetPreCheck(cliRet, lang)


def changUnit2GB(strValue):
    '''
    @summary: 根据传入的值转换单位为GB
    @return: True/False, floatValue
    '''
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip())
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) / 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) / 1024 / 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip()) / 1024 / 1024 / 1024 
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)
    

def getLogicEngineOfDisk(diskId, productModel):
    """
    @summary: 获取硬盘所属的逻辑引擎
    """
    #硬盘id的第一个数字即为该硬盘所属的逻辑引擎
    firstDigit = re.search('([A-Z]+)(\d)', diskId).group(2)
    if firstDigit is None:
        return 0
    
    firstDigit = int(firstDigit)
    
    logicEngine = firstDigit
    
    #6U4C大卡环境逻辑i引擎需特殊处理
    bigCardModelList = ["6800 V3", "6900 V3", "18500 V3", "18800 V3"]
    if productModel in bigCardModelList:
        secondDigit = re.search('([A-Z]+)(\d)(\d|[A-F])', diskId).group(3)
        if secondDigit is None:
            return firstDigit
        
        if secondDigit <= "7":
            logicEngine = firstDigit * 2 + 0
        else:
            logicEngine = firstDigit * 2 + 1
    
    return logicEngine


def isBugProductVersion(currentVersion, bugProductVersionList):
    '''
    @summary: 判断产品版本是否是某个已知产品问题的产品版本
    @param: bugProductVersionList
    eg:("V300R003C00", "V300R003C00SPC300]", "V300R003C10.*"]/V300R003C00到V300R003C00SPC300（不包含V300R003C00，包含V300R003C00SPC300），V300R003C10所有版本
    '''
    for bugProductVersion in bugProductVersionList:
        if bugProductVersion.find(",") > -1:
            startVersion = bugProductVersion.split(",")[0][1:].strip()
            endVersion = bugProductVersion.split(",")[1][:-1].strip()
            isContainStartVersion = bugProductVersion.startswith("[")
            isContainEndVersion = bugProductVersion.endswith("]")
            
            if isContainStartVersion and isContainEndVersion:
                if currentVersion >= startVersion and currentVersion <= endVersion:
                    return True
            
            if isContainStartVersion and (not isContainEndVersion):
                if currentVersion >= startVersion and currentVersion < endVersion:
                    return True
            
            if (not isContainStartVersion) and isContainEndVersion:
                if currentVersion > startVersion and currentVersion <= endVersion:
                    return True
            
            if (not isContainStartVersion) and (not isContainEndVersion):
                if currentVersion > startVersion and currentVersion < endVersion:
                    return True
        else:
            if re.match(bugProductVersion, currentVersion):
                return True
    
    return False


def checkProductModelAndVersion(cli, modelAndVersionListDict, LANG):
    """
    @summary: 检查设备型号和版本是否为指定的型号和版本
    @param modelAndVersionListDict: 问题型号与对应的版本，型号与版本为正则表达式，若问题版本为某个区间，用数学表达式区间表示，
            eg:{"(?!2800 V3)" : ["V200R002C20.*", "[V200R002C30,V200R002C30SPC200]"}，表示问题型号为非2800 V3，
                                        问题版本为所有V200R002C20版本，V200R002C30到V200R002C30SPC200版本，包含问题版本为所有V200R002C20版本与V200R002C30SPC200。
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLID回显
        isPass：True/False，检查是否通过（设备型号与版本不是问题版本时检查通过）
        productModel：设备型号
        productVersion：设备版本
    """
    isSucc = True
    allCliRet = ""
    
    isPass = True
    productModel = ""
    productVersion = ""
    
    #获取产品型号
    flag, cliRet, productModel = getProductModel(cli, LANG)
    allCliRet += cliRet
    if flag != True:
        isSucc = False
        isPass = False
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    checkRet, versionDictList, hotPatchVersionList = parse_upgradePackage(cli, LANG)
    allCliRet += checkRet[1]
    if checkRet[0] != True:
        isSucc = False
        isPass = False
        errMsg = checkRet[2]
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    #产品版本（版本全称）
    productVersion = versionDictList[0].get("Current Version")
    
    for model in modelAndVersionListDict:
        if not re.match(model, productModel):
            #设备型号不是问题型号
            continue
        
        versionList = modelAndVersionListDict.get(model)
        
        if isBugProductVersion(productVersion, versionList):
            #问题型号与版本均匹配
            isSucc = True
            isPass = False
            return (isSucc, allCliRet, isPass, productModel, productVersion)
        else:
            return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    return (isSucc, allCliRet, isPass, productModel, productVersion)


def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []


def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        
        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())
    
    return dictList


def getJavaDev(ip, py_java_env):
    """
    @summary: 将连接设备的信息封装为java的DevNode对象
    """
    dev = py_java_env.get("devInfo")
    javaDev = DevNode()
    javaDev.setPort(dev.getPort())
    javaDev.setLoginUser(dev.getLoginUser())
    javaDev.setSshForwardList(dev.getSshForwardList())
    javaDev.setSocks5Proxy(dev.getSocks5Proxy())
    deviceSN = dev.getDeviceSerialNumber()
    #每个设备只有一个SN，而java DevNode初始化时需要为每个控制器传一个不同的SN，此SN只是用来作为框架识别不同的SSH连接，不比是真实的设备SN
    deviceSN = deviceSN + str(time.time())
    javaDev.setDeviceSerialNumber(deviceSN)
    javaDev.setIp(ip)
    return javaDev


def getCilConnectionByIp(ip, py_java_env, LOGGER):
    """
    @summary: 获取指定ip的控制器ssh连接
    """
    cliConnection = None
    try:
        javaDev = getJavaDev(ip, py_java_env)
        cliConnection = py_java_env.get("sshManager").getSshConnection(javaDev)
        return cliConnection
    except Exception, exception:
        LOGGER.logException(exception)
        if cliConnection is not None:
            py_java_env.get("sshManager").releaseConnection(cliConnection)
        return cliConnection


def getContrManagementIp(cli, lang):
    """
    @summary: 获取设备上每个控制器的管理IP
    """
    contrManagementIpList = []
    
    cmd = "show system management_ip"
    cliRet = cli.execCmd(cmd)
    if not cliRet:
        errInfo = {"zh":u"\n获取信息失败",
                   "en":"\nThere is an error in getting information",} 
        return (False, cliRet, errInfo.get(lang), contrManagementIpList)
    
    infoDictList = getVerticalCliRet(cliRet)
    for infoDict in infoDictList:
        IPv4 = infoDict.get("IPv4 Address")
        if IPv4 not in contrManagementIpList:
            contrManagementIpList.append(IPv4)
    
    return (True, cliRet, "", contrManagementIpList)


def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
    True: 具有执行cli命令的权限
    False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
    if "does not have the permission" in line:
        return False
    return True

def isNotSupport(cliRet):
    '''
    @summary: 判断回显结果里面是否包含 does not support the VM service
    @param cliRet: cli回显
    @return: 
        True: cli回显包含 does not support the VM service
        False: cli回显不包含does not support the VM service
    '''
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "does not support" in lowerLine:
            return True
    
    return False


def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    cliExecuteCmdSuccess = "Command executed successfully"
    if re.search(cliExecuteCmdSuccess, cliRet, re.IGNORECASE):
        return True
    return False
