# -*- coding: UTF-8 -*-
import re

CLI_RET_END_FLAG =  ":/>"

#光模块：BOM编码和厂商对应关系
BOM_ModelMap = {
"34060365":("FTLF8524P3BNL-HW",
            "PLRXPLVESG4HWN",
            "LTD8542-BE+",
            "PLRXPLVESG462N"),
"3406G000":("AFBR-57D9AMZ",),
"34060521":("AFBR-57D7APZ",
            "AFBR-57D7APZ-HW1",
            "AFBR-57D9AMZ-HW1",
            "PLRXPLVCSH4HW",
            "FTLF8528P3BCV"),
"34060494":("TPP1XEDSRCZAB00",
            "LTF8502-BC+",
            "PLRXPLSCS43HW",
            "AFBR-703SDZ-HW1",
            "FTLX8571D3BCL-HW",
            "TRS2000EN-S003"),
"34060779":("FTLF8529P3BCV",),
"3406G001":("FCBG410QB1C15",),
"34060662-001":("JQPR04SRAB1HW",
                "LTA8513-PC+",
                "FTL410QE3C-HU",
                "TR-QQ85S-N00",),
"34060874":("1064155015",
            "7120-028-01")
}

HARDWARE_CMD = {"BBU":"show bbu general bbu_id=",
                "FAN":"show fan fan_id=",
}

def getHorizontalHardwareBomCode(hardwareID,hardwareType,cliRet):
    '''
    @summary: 获取水平回显中指定ID硬件的BomCode相关的字段值（SN值/Model值）
    @param idList: 硬件ID列表值
    @param hardwaretype: 硬件类型(ENCLOSURE/DISK/FIBRE)
    @return: SN值/Model值
    '''
    
    hardwareDictList = getHorizontalCliRet(cliRet)
    if hardwareType == "ENCLOSURE":
        for hardwareDict in hardwareDictList:
            id =  hardwareDict.get("ID","")
            sn =  hardwareDict.get("SN","")
            if id == hardwareID and sn:
                return  _switchSN2BomCode(sn)
            else:
                continue
    elif hardwareType == "DISK": 
        for hardwareDict in hardwareDictList:
            id =  hardwareDict.get("ID","")
            sn =  hardwareDict.get("Bar Code","")
            if id == hardwareID and sn:
                return  _switchSN2BomCode(sn)
            else:
                continue
    elif hardwareType == "FIBRE":
        for hardwareDict in hardwareDictList:
            id =  hardwareDict.get("PortID","")
            model =  hardwareDict.get("Model","")
            if id == hardwareID and model:
                return getFibreModuleBOMByModel(model)
            else:
                continue
    return ""

def getItemFromELabel(cliRet):
    '''
    @summary: 获取控制器信息中不为空的Item值，如果所有的Item都为空则返回空(所有控制器的item值一样)
    @param cliRet: 硬件的信息回文
    @return: Item：硬件Item值
    '''
    item = ""
    if "Item=" not in cliRet:
        return item
    
    for line in cliRet.splitlines():
        if "Item=" in line:
            field = line.split("=")
            if len(field) == 2 and field[1].strip():
               return field[1].strip()

    return item

def getStandardBomCode(idList,hardwareType,ssh):
    '''
    @summary: 从idList硬件中的找到不为空Item值
    @param idList: 硬件的idList
    @param hardwareType: 硬件的SN号
    @param ssh: 命令下发模式
    @return: BOM编码
    '''
    
    itemList = getHardwareItemList(idList,hardwareType,ssh)
    for item in itemList:
        if item:
            return item
        else:
            continue
    return ""

def getFANBomCodeByEncId(enclosureID,idList,itemList):
    '''
    @summary: 返回idList硬件中 与同框的Item值
    @param idList: 硬件的idList
    @param itemList: 硬件的 itemList(此值必需通过getHardwareItemList获取其中idList需统一)
    @return: Item
    '''    

    for (id,item) in zip(idList,itemList):
        encId = id.split(".")[0]
        if encId == enclosureID and item:
            return item
        else:
            continue
    return ""

def getPowerBomCodeByEncId(enclosureID,cliRet):
    '''
    @summary: 返回POWER硬件中 同enclosureID的存在不为空的BOM编码值
    @param enclosureID: 硬件的enclosureID
    @param cliRet: 硬件的 回显
    @return: BOM编码值
    '''  
    hardwareDictList = getHorizontalCliRet(cliRet)
    for hardwareDict in hardwareDictList: 
        id =  hardwareDict.get("ID","")
        sn =  hardwareDict.get("Serial Number","")
        encId = id.split(".")[0]
        if encId == enclosureID and _switchSN2BomCode(sn):
            return  _switchSN2BomCode(sn)
        else:
            continue
    return ""
    
def getHardwareItemList(idList,hardwareType,ssh):
    '''
    @summary: 返回idList硬件中的Item列表(主要针对BBU，FAN)
    @param idList: 硬件的idList
    @param hardwareType: 硬件的SN号
    @param ssh: 命令下发模式
    @return: Item列表
    '''
    
    itemList = []
    
    for id in idList:
        checkCmd = HARDWARE_CMD.get(hardwareType)
        checkCmd += id
        cliRet = ssh.execCmd(checkCmd)
        item = getItemFromELabel(cliRet)
        itemList.append(item)
    return itemList
       
def _switchSN2BomCode(SerialNumber):
    '''
    @summary: 从SN号中获取BOM编码
    @param SerialNumber: 硬件的SN号
    @return: BOM编码
    '''

    BomCode = ""
    #（1）以21开头的20位数字或字母的SN中，第3-10位代表这个物料对应物料种类的BOM编码。
    #以210305G06M10DC000253为例，这种控制器的BOM编码是0305G06M；
    if SerialNumber.startswith("21") and len(SerialNumber) == 20:
        BomCode = SerialNumber[2:10]
    #（2）以02或03开头的16位数字或字母，第1-6位前面加上03凑成8位数字或字母，代表BOM编码。
    #以02G258D0BB001344为例，它代表一个级联板唯一存在，BOM编码为0302G258；
    elif (SerialNumber.startswith("02") or SerialNumber.startswith("03"))\
        and len(SerialNumber) == 16:
        BomCode = "03" + SerialNumber[0:6]

    return BomCode

def getFibreModuleBOMByModel(Model):
    '''
    @summary: 通过光模块的BOM编码信息
    @param Model：光模块型号
    @return: BOM编码
    '''

    for bom,modelList in BOM_ModelMap.items():
        if Model in modelList:
            return bom
    return ""      

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key)
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip())
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
