#-*- coding:utf-8 -*-
import util
import cliCmdManager
from constant import CLI_CMD
from com.huawei.ism.tool.obase.entity import DevNode
import re

#当前系统模式
from constant import SSH_CLI_MODEL_TYPE
NOCHECK = 'NOCHECK'
CLI_RET_END_FLAG = ":/>"

def getProductModel(cli, logger):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, productmodel)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        productmodel：产品型号
    '''
    product_model = ""
    
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, CLI_CMD.SHOW_SYSTEM_GENERAL, logger)
    if not cmdExeSucc:
        return (False, cliRet, '')
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, cliRet, product_model)
        
    return (False, cliRet, "")
    
def getProductFullVersion(cli, logger):
    """
    Function name      : getUserLevel
    Function describe  : get user level
    Input              : (cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, productVersion)
    """    
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, CLI_CMD.SHOW_UPGRADE_PACKAGE, logger)
    if not cmdExeSucc:
        return (False, cliRet, '')
    
    endIndex = cliRet.find('HotPatch Version')
    if endIndex == -1:
        return (False, cliRet, '')
    
    try:
        dictList = util.formatList(cliRet[:endIndex])
        fullVersion = dictList[0]['Current Version']
        if fullVersion.endswith('T'):
            fullVersion = fullVersion[:-1]
    except:
        return (False, cliRet, '')
    else:
        return (True, cliRet, fullVersion)

def parseHotPatchVersion(cliRet, logger):
    """
    Function name      : getHotPatchVersion
    Function describe  : get hot patch version.
    Input              : (cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, hotPatchVer)
    """    
    
    endIndex = cliRet.find('HotPatch Version')
    if endIndex == -1:
        return (False, cliRet, '')
    
    hotPatchStartIndex = endIndex + len('HotPatch Version') +1
    
    try:
        dictList = util.formatList(cliRet[hotPatchStartIndex:])
        hotPatchVer = dictList[0]['Current Version']
        if hotPatchVer == '--':
            logger.info('Hot patch has not been installed!')
    except:
        logger.error('Parse hot patch version exception.')
        return (False, '')
    else:
        return (True, hotPatchVer)

def checkThinLUNExist(cli, logger):
    """
    Function name      : checkThinLUNExist
    Function describe  : check whether thin LUN exists.
    Input              : (cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, isThinLUNExist)
    """    
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, CLI_CMD.QUERY_THIN_LUN, logger)
    if not cmdExeSucc:
        return False, cliRet, False
    
    if len(cliRet.splitlines()) > 4:
        return True, cliRet, True
    else:
        return True, cliRet, False
        
def getUserLevel(userName, cli, logger):
    """
    Function name      : getUserLevel
    Function describe  : get user level
    Input              : (userName, cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, userLevel)
    """    
    qryUserLvlCmd = CLI_CMD.SHOW_USER + userName
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, qryUserLvlCmd, logger)
    if not cmdExeSucc:
        return (False, cliRet, '')
    
    try:
        dictList = util.formatList(cliRet)
        usrLvl = dictList[0]['Level']
    except:
        return (False, cliRet, '')
    else:
        return (True, cliRet, usrLvl)
    
def getLunIDList(cli, logger):
    """
    Function name      : getLunIDList
    Function describe  : get LUN id list.
    Input              : (cli, logger)
    Return             : (cmdExecuteSuccess, cliRet, lunIdList)
    """    
        
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, CLI_CMD.SHOW_LUN_GENERAL, logger)
    if not cmdExeSucc:
        return (False, cliRet, [])
   
    lunIdList = [] 
    try:
        lunInfoDictList = util.formatList(cliRet)
        for lunInfoDict in lunInfoDictList:
            lunIdList.append(lunInfoDict['ID'])
    except:
        return (False, cliRet, lunIdList)
    else:
        return (True, cliRet, lunIdList)

def copyDevNode(devNode):
    '''
            函数名称: copyDevNode
            功能说明: 复制DevNode对象的部分参数，创建新的DevNode对象，除IP外请不要修改其它成员变量，否则可能影响原DevNode对象的值
            输入参数: devNode
            输出参数: 无
            返 回 值: devNode    
    '''
    devTmp = DevNode()
    devTmp.setSocks5Proxy(devNode.getSocks5Proxy());
    devTmp.setIp(devNode.getIp())   
    devTmp.setPort(devNode.getPort())
    devTmp.setTlvPort(devNode.getTlvPort())
    devTmp.setLoginUser(devNode.getLoginUser())
    devTmp.setSshForwardList(devNode.getSshForwardList())
    devTmp.setDeveloperPwd(devNode.getDeveloperPwd())
    return devTmp

def getCtrlCmdMode(cli,logger):
    # **************************************************************************** #
    # 函数名称: getCtrlCmdMode
    # 功能说明: 获取当前SSH所在命令模式（通用）
    # 输入参数: ssh, logger
    # 输出参数: 无
    # 返 回 值: 
    # **************************************************************************** #   
 
    cliRet = cli.execCmd("show system general")
    if not cliRet:
        return SSH_CLI_MODEL_TYPE.SSH_NOTREADY_MODEL
    
    #判断是否在Cli模式下
    if re.search('Product Version', cliRet, re.IGNORECASE):
        if re.search('developer>', cliRet, re.IGNORECASE):
            return SSH_CLI_MODEL_TYPE.SSH_DEVELOPER_MODEL
        else:
            return SSH_CLI_MODEL_TYPE.SSH_CLI_MODEL
    elif re.search('minisystem>', cliRet, re.IGNORECASE):
        logger.info("Get system state in minisystem mode by CLI!")
        return SSH_CLI_MODEL_TYPE.SSH_MINISYSTEM_MODEL
    elif re.search('diagnose',cliRet,re.IGNORECASE):
        logger.info("Get system state in diagnose mode by CLI!")
        return SSH_CLI_MODEL_TYPE.SSH_DIAGNOSE_MODEL
    elif re.search('failed to be started|safe mode', cliRet, re.IGNORECASE):
        logger.error("Get system state in safe mode by CLI!")
        return SSH_CLI_MODEL_TYPE.SSH_SAFE_MODEL
    else:
        return SSH_CLI_MODEL_TYPE.SSH_UNKNOWN_MODEL


def getDiskSectorSize(cli, logger):
    '''
    @summary: 执行show disk general |filterColumn include columnList=ID,Sector\sSize命令，获取扇区大小
    '''
    
    sectorSizeDictList = []
    errMsgDict = {"zh":"", "en":""}
    
    cmd = "show disk general |filterColumn include columnList=ID,Sector\sSize"
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, cmd, logger)

    #兼容产品命令Bug
    if not hasCliExecPrivilege(cliRet):
        cmd = "show disk general |filterColumn include colunmList=ID,Sector\sSize"
        cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, cmd, logger)
  
    if not cmdExeSucc: 
        errMsgDict["zh"] +=  u"\n获取硬盘扇区大小信息失败。"
        errMsgDict["en"] += "\nFailed to obtain information about the disk sector size."
        return (False, cliRet, [], errMsgDict)
        
    sectorSizeDictList = getHorizontalCliRet(cliRet)
    return (True, cliRet, sectorSizeDictList, errMsgDict)

def is4KDiskWithSectorSize(diskSectorSizeDictList):
    '''
    @summary: 根据扇区大小判断是否为4K盘
    '''
    is4KDiskList = []
    errMsgDict = {"zh":"", "en":""}
    
    for diskSectorSizeDict in diskSectorSizeDictList:
        diskId = diskSectorSizeDict.get("ID")
        sectorSizeStr = diskSectorSizeDict.get("Sector Size")
        
        #4196B=4.0625KB，显示为4.062KB，需要处理
        if sectorSizeStr == "4.062KB":
            sectorSizeStr = "4.0625KB"
        result, sectorSizeNum = changUnit2B(sectorSizeStr)
        if not result:
            errMsgDict["zh"] +=  u"\n转换单位失败。"
            errMsgDict["en"] += "\nFailed to convert units."
            return (True, [], errMsgDict)
        
        if sectorSizeNum == 4096.0 or sectorSizeNum == 4160.0:
            is4KDiskList.append(diskId)
        
    if is4KDiskList:
        errMsgDict["zh"] +=  u"\n设备中存在扇区大小为4096B或4160B的硬盘[ID: %s]，当前系统版本不支持该类型的硬盘。" % ', '.join(is4KDiskList)
        errMsgDict["en"] += "\nDisks [ID: %s] whose sector size is 4096 bytes or 4160 bytes exist in the device. The system does not support such disks." % ', '.join(is4KDiskList)
        return (True, is4KDiskList, errMsgDict)
    
    return(False, [], errMsgDict)

def changUnit2B(strValue):
    '''
    @summary: 根据传入的值转换单位为B
    @return: True/False, sectorSizeNum
    '''
    floatValue = 0.0
    
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024 * 1024 * 1024 * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip()) * 1024 * 1024 * 1024
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) * 1024 * 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) * 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip())
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)
    

def changUnit2GB(strValue):
    '''
    @summary: 根据传入的值转换单位为GB
    @return: True/False, floatValue
    '''
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip())
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) / 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) / 1024 / 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip()) / 1024 / 1024 / 1024 
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)


def get4kDiskInfo(cli, logger):
    '''
    @summary: 判断是否存在故障硬盘
    @param cli: cli对象
    @param lang: 语言lang
    '''

    cliRet = ""
    errMsgDict = {"zh":"", "en":""}

    #判断硬盘是否存在于1TB的SAS盘
    diskSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=SAS " \
                "logicOp=and column=Capacity predict=greater_than value=1.000TB " \

    #判断硬盘是否存在大于5TB的NearLine SAS盘
    diskNearLineSASCmd = "show disk general |filterRow " \
                "column=Type predict=equal_to value=NearLine\sSAS " \
                "logicOp=and column=Capacity predict=greater_than value=5.000TB " \

    all4kDiskList = []
    isCmdExeSucc = True
    for diskCmd in ( diskSASCmd, diskNearLineSASCmd):
        isCmdExeSucc, diskCliRet, fkDiskList = exeQry4kDiskCmd(diskCmd, cli, logger)
        #命令执行失败或存在故障盘或存在4K
        cliRet += diskCliRet
        all4kDiskList += fkDiskList
        if not isCmdExeSucc:
            errMsgDict["zh"] = u"\n执行命令失败。"
            errMsgDict["en"] = "\nexecute command failed."
            return (False, cliRet, errMsgDict, [])

    #存在4K盘返回Flase
    if all4kDiskList:
        errMsgDict["zh"] = u"\n设备中存在大于1TB的SAS盘或大于5TB的NearLine SAS盘[ID: %s],当前系统版本不支持该类型的硬盘。" % ', '.join(all4kDiskList)
        errMsgDict["en"] = "\nSAS disks larger than 1 TB or NL-SAS disks larger than 5 TB [ID: %s] exist in the device. The current version does not support such disks." % ', '.join(all4kDiskList)
    
    return (True, cliRet, errMsgDict, all4kDiskList)

def exeQry4kDiskCmd(cmd, cli, logger):

    cliRet = ""
    errMsg = ""
    cmdExeSucc, cliRet = cliCmdManager.execCmdWithTimout(cli, cmd, logger)
   
    if not cmdExeSucc:
        return (False, cliRet, [])

    #判断命令回显是否为Command executed successfully
    isCmdExeSucc = queryResultWithNoRecord(cliRet)
    if isCmdExeSucc:
        return (True, cliRet, [])

    #按逐行字典的方式获取水平表格形式的cli回显集合
    diskInfoDictList = getHorizontalCliRet(cliRet)

    fkDiskList = []
    for diskInfoDict in diskInfoDictList:
        diskId = diskInfoDict.get('ID')
        #把大于1T的SAS盘或大于5T的NearLine SAS盘添加到bugDiskList
        fkDiskList.append(diskId)
        
    return (True, cliRet, fkDiskList)


def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    cliExecuteCmdSuccess = "Command executed successfully"
    if re.search(cliExecuteCmdSuccess, cliRet, re.IGNORECASE):
        return True
    return False

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "does not have the permission" in line:
            return False
    return True


def getLogicEngineOfDisk(diskId, productModel):
    """
    @summary: 获取硬盘所属的逻辑引擎
    """
    #硬盘id的第一个数字即为该硬盘所属的逻辑引擎
    firstDigit = re.search('([A-Z]+)(\d)', diskId).group(2)
    if firstDigit is None:
        return 0
    
    firstDigit = int(firstDigit)
    
    logicEngine = firstDigit
    
    #6U4C大卡环境逻辑i引擎需特殊处理
    bigCardModelList = ["6800 V3", "6900 V3", "18500 V3", "18800 V3"]
    if productModel in bigCardModelList:
        secondDigit = re.search('([A-Z]+)(\d)(\d|[A-F])', diskId).group(3)
        if secondDigit is None:
            return firstDigit
        
        if secondDigit <= "7":
            logicEngine = firstDigit * 2 + 0
        else:
            logicEngine = firstDigit * 2 + 1
    
    return logicEngine


def isBugProductVersion(currentVersion, bugProductVersionList):
    '''
    @summary: 判断产品版本是否是某个已知产品问题的产品版本
    @param: bugProductVersionList
    eg:("V300R003C00", "V300R003C00SPC300]", "V300R003C10.*"]/V300R003C00到V300R003C00SPC300（不包含V300R003C00，包含V300R003C00SPC300），V300R003C10所有版本
    '''
    for bugProductVersion in bugProductVersionList:
        if bugProductVersion.find(",") > -1:
            startVersion = bugProductVersion.split(",")[0][1:].strip()
            endVersion = bugProductVersion.split(",")[1][:-1].strip()
            isContainStartVersion = bugProductVersion.startswith("[")
            isContainEndVersion = bugProductVersion.endswith("]")
            
            if isContainStartVersion and isContainEndVersion:
                if currentVersion >= startVersion and currentVersion <= endVersion:
                    return True
            
            if isContainStartVersion and (not isContainEndVersion):
                if currentVersion >= startVersion and currentVersion < endVersion:
                    return True
            
            if (not isContainStartVersion) and isContainEndVersion:
                if currentVersion > startVersion and currentVersion <= endVersion:
                    return True
            
            if (not isContainStartVersion) and (not isContainEndVersion):
                if currentVersion > startVersion and currentVersion < endVersion:
                    return True
        else:
            if re.match(bugProductVersion, currentVersion):
                return True
    
    return False


def checkProductModelAndVersion(cli, modelAndVersionListDict, logger):
    """
    @summary: 检查设备型号和版本是否为指定的型号和版本
    @param modelAndVersionListDict: 问题型号与对应的版本，型号与版本为正则表达式，若问题版本为某个区间，用数学表达式区间表示，
            eg:{"(?!2800 V3)" : ["V200R002C20.*", "[V200R002C30,V200R002C30SPC200]"}，表示问题型号为非2800 V3，
                                        问题版本为所有V200R002C20版本，V200R002C30到V200R002C30SPC200版本，包含问题版本为所有V200R002C20版本与V200R002C30SPC200。
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLID回显
        isPass：True/False，检查是否通过（设备型号与版本不是问题版本时检查通过）
        productModel：设备型号
        productVersion：设备版本
    """
    isSucc = True
    allCliRet = ""
    
    isPass = True
    productModel = ""
    productVersion = ""
    
    #获取产品型号
    flag, cliRet, productModel = getProductModel(cli, logger)
    allCliRet += cliRet
    if flag != True:
        isSucc = False
        isPass = False
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    #获取产品软件版本
    flag, cliRet, productVersion = getProductFullVersion(cli, logger)
    allCliRet += cliRet
    if flag != True:
        isSucc = False
        isPass = False
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    
    for model in modelAndVersionListDict:
        if not re.match(model, productModel):
            #设备型号不是问题型号
            continue
        
        versionList = modelAndVersionListDict.get(model)
        
        if isBugProductVersion(productVersion, versionList):
            #问题型号与版本均匹配
            isSucc = True
            isPass = False
            return (isSucc, allCliRet, isPass, productModel, productVersion)
        else:
            return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    return (isSucc, allCliRet, isPass, productModel, productVersion)

