# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import math
import time

import common
import cliUtil
import dpa_common
from java.lang import Exception as JException
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
NO_FILE_FLAG = "no such file or directory"


def execute(dpa_rest):
    check_item = CheckItem(dpa_rest, LANG, LOGGER)
    return check_item.execute()


class CheckItem:
    def __init__(self, dpa_rest, lang, logger):
        self.dpa_rest = dpa_rest
        self.lang = lang
        self.logger = logger
        self.err_msg = str()
        self.cli_ret = list()
        self.directory_size_list = list()
        self.conf_ini_list = list()
        self.opt_avail_size_list = list()
        self.upgrade_dir_node_list = list()

    @staticmethod
    def need_check_dev():
        cur_device_ip = PY_JAVA_ENV.get("devInfo").getIp()
        return [dev_node
                for dev_node in PY_JAVA_ENV.get('selectDevs')
                if dev_node.getIp() in list(PY_JAVA_ENV.get('dpaRelationshipHost').get(cur_device_ip))]

    def execute(self):
        need_check_dev_list = CheckItem.need_check_dev()
        for dev_node in need_check_dev_list:
            connection = None
            try:
                dev_node = EntityUtils.cloneDevNode(dev_node)
                device_sn = dev_node.getIp() + str(time.time())
                dev_node.setDeviceSerialNumber(device_sn)
                connection = PY_JAVA_ENV.get("sshManager").getSshConnection(dev_node)
                self.get_directory_size(connection, dev_node)
                if self.err_msg:
                    return False, "\n".join(self.cli_ret), self.err_msg
                self.get_conf_file_info(connection, dev_node)
                self.get_opt_size(connection, dev_node)
                if self.err_msg:
                    return False, "\n".join(self.cli_ret), self.err_msg
            except JException as exp:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", dev_node.getIp())
                self.logger.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
            finally:
                PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
                self.logger.logInfo("SSH {} connection is release".format(dev_node.getDeviceSerialNumber()))
        self.check_data()
        if self.err_msg:
            return False, "\n".join(self.cli_ret), self.err_msg
        return True, "\n".join(self.cli_ret), ""

    def check_data(self):
        upload_size = 0.0
        backup_size = 0.0
        if not self.conf_ini_list:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.conf.ini.not.exist")
            return
        elif len(self.conf_ini_list) > 1:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.conf.ini.abnormal",
                                                         ",".join(self.upgrade_dir_node_list))
            return
        for cli_tem_var in list(self.conf_ini_list)[0]:
            if "upload_limit" in cli_tem_var:
                upload_size = float(cli_tem_var.split("=")[1])
            if "backup_limit" in cli_tem_var:
                backup_size = float(cli_tem_var.split("=")[1])
        if backup_size < upload_size * 2:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.back.less.upload.abnormal", self.upgrade_dir_node_list[0])
        total_size = upload_size + backup_size
        for size, ip in self.opt_avail_size_list:
            if size / 1024 <= total_size:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.avail.less.conf.ini.abnormal", (ip, str(size / 1024)))
        for size, ip in self.directory_size_list:
            if (math.ceil(size / 1024) + 1) * 1024 > upload_size:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.disk.usage.abnormal", (ip, str(size)))

    def get_opt_size(self, connection, dev_node):
        cmd = "df /opt"
        opt_size_info_ret = connection.execCmd(cmd)
        self.cli_ret.append(opt_size_info_ret)
        if NO_FILE_FLAG not in opt_size_info_ret.lower():
            opt_avail_size = float(opt_size_info_ret.splitlines()[2].split()[3])
            self.opt_avail_size_list.append([opt_avail_size, dev_node.getIp()])
        else:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.get.directory.size.abnormal", dev_node.getIp())

    def get_conf_file_info(self, connection, dev_node):
        cmd = "cat /opt/upgrade/CDMServerUpdate/etc/conf.ini"
        conf_ini_ret = connection.execCmd(cmd)
        self.cli_ret.append(conf_ini_ret)
        if NO_FILE_FLAG not in conf_ini_ret.lower():
            self.upgrade_dir_node_list.append(dev_node.getIp())
            self.conf_ini_list.append(conf_ini_ret.splitlines()[1:-1])

    def get_directory_size(self, connection, dev_node):
        cmd = "du -sm /opt/CDMServer"
        cdm_server_size_ret = connection.execCmd(cmd)
        self.cli_ret.append(cdm_server_size_ret)
        if len(cdm_server_size_ret.encode("utf8").splitlines()) == 3 and \
                NO_FILE_FLAG not in cdm_server_size_ret.lower():
            directory_size = float(cdm_server_size_ret.encode("utf8").splitlines()[1].split()[0])
            self.directory_size_list.append([directory_size, dev_node.getIp()])
        else:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.get.cdmserver.directory.size.abnormal", dev_node.getIp())
