#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
"""
检查扩容硬盘是否超过单引擎硬盘最大规格
"""
from collections import namedtuple

import common
import expandconfig
from cbb.common.conf.max_disk_num_per_engine import \
    NO_LIMIT_MAX_DISK_NUM_PER_ENGINE
from cbb.common.conf.max_disk_num_per_engine import get_max_disk_num_per_engine
from cbb.common.query.hardware.disk import \
    get_existing_member_disk_num_per_engine
from cbb.common.query.hardware.disk import query_member_disk_id_list
from cbb.common.query.hardware.engine import get_all_engines
from cbb.frame.cli.cliUtil import RESULT_NOCHECK
from cbb.frame.cli.cliUtil import RESULT_NOSUPPORT
from cbb.frame.util.common import fakeProgress
from cbb.frame.util.common import wrapAllExceptionLogged

global py_java_env
global PY_LOGGER

lang = common.getLang(py_java_env)
logger = common.getLogger(PY_LOGGER, __file__)


@wrapAllExceptionLogged(logger=PY_LOGGER)
@fakeProgress(py_java_env, totalSeconds=30, logger=logger, interval=1)
def execute(cli):
    """检查扩容硬盘是否超过单引擎硬盘最大规格

    :param cli:
    :return:
    """
    all_cli_ret_list = ['']
    try:
        # 获取设备型号
        dev_info = common.getCurDeviceInfo(py_java_env)
        product_model = str(dev_info.getDeviceType())
        product_version = str(dev_info.getProductVersion())
        internal_pdt_model = product_model
        if 'Dorado' in product_model:
            exp_info = py_java_env.get("expInfo")
            internal_pdt_model = exp_info.getInternalModel()
            # For not support internal model version.
            if not internal_pdt_model:
                internal_pdt_model = '_'.join(product_model.split())

            # Dorado5000 NVMe V3R1C21 版本查询到的内部型号
            # 和SAS型号都是Dorado5000_V3
            if 'Dorado5000' in product_model and exp_info.isNVMe():
                internal_pdt_model = 'Dorado5000_V3_NVMe'

        logger.logInfo("Internal product model:{inter_model}".format(
            inter_model=internal_pdt_model))

        max_disk_cnt_per_engine = get_max_disk_num_per_engine(
            internal_pdt_model,
            product_version)
        logger.logInfo("Max member disk cnt per engine:{max_spec}".format(
            max_spec=max_disk_cnt_per_engine))
        if NO_LIMIT_MAX_DISK_NUM_PER_ENGINE == max_disk_cnt_per_engine:
            return RESULT_NOSUPPORT, '', ''

        according_logic_engine = product_model in ['Dorado18000 V3',
                                                   '6800 V3', '6800F V3',
                                                   '6900 V3']

        exp_config_obj = expandconfig.ExpandConfig(py_java_env,
                                                   according_logic_engine,
                                                   logger=logger)
        exp_disk_info_dict = exp_config_obj.getExpDiskInfo()
        logger.logInfo("The expDiskInfoDict is %s" % exp_disk_info_dict)

        for disk_domain_id in exp_disk_info_dict:
            exp_disk_info_obj_list = exp_disk_info_dict.get(disk_domain_id, [])
            for exp_disk_info_obj in exp_disk_info_obj_list:
                logger.logInfo(
                    "The exp_disk_info_obj is %s" % exp_disk_info_obj.__dict__)

        all_new_dd_total_disk_num = \
            exp_config_obj.get_total_disk_num_of_all_new_dd()
        logger.logInfo("Total disk number of all newly created DD is %s"
                       % all_new_dd_total_disk_num)

        all_exp_disk_num_per_engine = \
            exp_config_obj.get_expansion_disk_num_per_engine()
        logger.logInfo("Expansion disk number per engine: %s"
                       % all_exp_disk_num_per_engine)

        query_disk_result = query_member_disk_id_list(cli, lang, logger)
        if query_disk_result.cmd_ret_flag is not True:
            return (query_disk_result.cmd_ret_flag,
                    query_disk_result.cmd_raw_output,
                    query_disk_result.err_msg)

        all_cli_ret_list.append(query_disk_result.cmd_raw_output)
        existing_member_disk_list = query_disk_result.cmd_handled_output
        existing_member_disk_cnt_per_engine = \
            get_existing_member_disk_num_per_engine(
                existing_member_disk_list,
                according_logic_engine)

        qry_succ, qry_cli_ret_list, qry_err_msg, engines = \
            get_all_engines(cli, lang, according_logic_engine)
        all_cli_ret_list.extend(qry_cli_ret_list)
        if not qry_succ:
            return RESULT_NOCHECK, '\n'.join(all_cli_ret_list), qry_err_msg

        logger.logInfo("All (logical)engines: %s" % engines)

        EngineExpDetail = namedtuple('EngineExpDetail',
                                     field_names=['engine_name',
                                                  'current_member_disk_cnt',
                                                  'support_add_disk_cnt_max',
                                                  'plan_exp_disk_cnt'])
        over_spec_l = []
        all_check_egines = list(set(engines +
                                    all_exp_disk_num_per_engine.keys()))
        all_check_egines.sort()
        logger.logInfo("all_check_egines %s" % all_check_egines)
        for engine in all_check_egines:
            existing_member_disk_cnt = \
                existing_member_disk_cnt_per_engine.get(engine, 0)
            exp_member_disk_cnt = all_exp_disk_num_per_engine.get(engine, 0)
            if (existing_member_disk_cnt + exp_member_disk_cnt +
                    all_new_dd_total_disk_num) > max_disk_cnt_per_engine:
                support_add_disk_cnt_max = (max_disk_cnt_per_engine -
                                            existing_member_disk_cnt)
                over_spec_l.append(
                    EngineExpDetail(engine,
                                    existing_member_disk_cnt,
                                    support_add_disk_cnt_max,
                                    exp_member_disk_cnt
                                    ))

        err_msg_list = handle_err_msg(all_new_dd_total_disk_num,
                                      max_disk_cnt_per_engine, over_spec_l)
        if err_msg_list:
            return False, '\n'.join(all_cli_ret_list), '\n'.join(
                err_msg_list)

        return True, '\n'.join(all_cli_ret_list), ''
    except Exception as exception:
        logger.logException(exception)
        return (RESULT_NOCHECK, '\n'.join(all_cli_ret_list),
                common.getMsg(lang, "query.result.abnormal"))


def handle_err_msg(all_new_dd_total_disk_num, max_disk_cnt_per_engine,
                   over_spec_l):
    """封装错误消息。

    :param all_new_dd_total_disk_num: 新建硬盘域成员盘数量之和
    :param max_disk_cnt_per_engine: 每个引擎最大成员盘数量
    :param over_spec_l: 超过最大规格的记录列表
    :return:
    """
    err_msg_list = []
    for over_spec in over_spec_l:
        if over_spec.plan_exp_disk_cnt > over_spec.support_add_disk_cnt_max:
            err_msg_list.append(
                common.getMsg(lang,
                              'single.engine.exp.member.disk.cnt.over.spec',
                              args=(over_spec.engine_name,
                                    max_disk_cnt_per_engine,
                                    over_spec.current_member_disk_cnt,
                                    over_spec.plan_exp_disk_cnt)))
        else:
            err_msg_list.append(
                common.getMsg(lang, 'single.engine.exp.and.new.dd.'
                                    'member.disk.cnt.may.over.spec',
                              args=(over_spec.engine_name,
                                    max_disk_cnt_per_engine,
                                    over_spec.current_member_disk_cnt,
                                    over_spec.plan_exp_disk_cnt,
                                    all_new_dd_total_disk_num
                                    )))
    return err_msg_list
